"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutomationStepSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cancellation_exception_1 = require("../domain/cancellation-exception");
const data_type_1 = require("../domain/data-type");
const non_retriable_exception_1 = require("../domain/non-retriable-exception");
const response_code_1 = require("../domain/response-code");
const approve_hook_1 = require("../interface/approve-hook");
const aws_invoker_1 = require("../interface/aws-invoker");
const observer_1 = require("../interface/observer");
const pause_hook_1 = require("../interface/pause-hook");
const api_run_command_hook_1 = require("../interface/run-command-hook/api-run-command-hook");
const sleep_hook_1 = require("../interface/sleep-hook");
const webhook_1 = require("../interface/webhook");
const approve_simulation_1 = require("./automation/approve-simulation");
const assert_aws_resource_simulation_1 = require("./automation/assert-aws-resource-simulation");
const aws_api_simulation_1 = require("./automation/aws-api-simulation");
const branch_simulation_1 = require("./automation/branch-simulation");
const change_instance_state_simulation_1 = require("./automation/change-instance-state-simulation");
const copy_image_simulation_1 = require("./automation/copy-image-simulation");
const create_image_simulation_1 = require("./automation/create-image-simulation");
const create_stack_simulation_1 = require("./automation/create-stack-simulation");
const create_tags_simulation_1 = require("./automation/create-tags-simulation");
const delete_image_simulation_1 = require("./automation/delete-image-simulation");
const delete_stack_simulation_1 = require("./automation/delete-stack-simulation");
const execute_script_simulation_1 = require("./automation/execute-script-simulation");
const execute_state_machine_simulation_1 = require("./automation/execute-state-machine-simulation");
const invoke_lambda_function_simulation_1 = require("./automation/invoke-lambda-function-simulation");
const invoke_webhook_simulation_1 = require("./automation/invoke-webhook-simulation");
const pause_simulation_1 = require("./automation/pause-simulation");
const run_command_simulation_1 = require("./automation/run-command-simulation");
const run_instance_simulation_1 = require("./automation/run-instance-simulation");
const sleep_simulation_1 = require("./automation/sleep-simulation");
const wait_for_resource_simulation_1 = require("./automation/wait-for-resource-simulation");
// eslint-disable-next-line
const jp = require('jsonpath');
class AutomationStepSimulation {
    constructor(step, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k;
        const awsInvoker = (_b = props.awsInvoker) !== null && _b !== void 0 ? _b : new aws_invoker_1.ReflectiveAwsInvoker();
        const sleepHook = (_c = props.sleepHook) !== null && _c !== void 0 ? _c : new sleep_hook_1.SleepImpl();
        this.props = {
            sleepHook,
            awsInvoker,
            pauseHook: (_d = props.pauseHook) !== null && _d !== void 0 ? _d : new pause_hook_1.PauseImpl(),
            inputObserver: (_e = props.inputObserver) !== null && _e !== void 0 ? _e : new observer_1.NoopObserver(),
            outputObserver: (_f = props.outputObserver) !== null && _f !== void 0 ? _f : new observer_1.NoopObserver(),
            approveHook: (_g = props.approveHook) !== null && _g !== void 0 ? _g : new approve_hook_1.ApproveImpl(),
            parameterResolver: (_h = props.parameterResolver) !== null && _h !== void 0 ? _h : { resolve: (x) => x },
            webhook: (_j = props.webhook) !== null && _j !== void 0 ? _j : new webhook_1.WebhookImpl(),
            runCommandHook: (_k = props.runCommandHook) !== null && _k !== void 0 ? _k : new api_run_command_hook_1.ApiRunCommandHook(awsInvoker, sleepHook),
        };
        this.step = step;
    }
    /**
     * Invokes the current step on the input and will return a SimulationResult.
     * @param inputs must contain all of the inputs declared by the current step.
     * @returns a SimulationResult with the step outputs in the case of success or stacktrace in the case of failure.
     */
    invoke(inputs) {
        console.log('Executing Step: ' + this.step.name);
        return this.invokeWithFallback(inputs);
    }
    /**
     * If fallback/retries are specified for this step, the retry or skip logic is handled in this function.
     */
    invokeWithFallback(allInputs) {
        try {
            const filteredInputs = this.filterInputs(allInputs);
            this.step.inputObserver.accept(filteredInputs);
            const response = this.executeWithRetries(filteredInputs);
            this.step.outputObserver.accept(response);
            Object.assign(allInputs, response);
            const nextStep = this.mySimulation().nextStep(allInputs);
            if (nextStep && !this.step.isEnd) {
                const nextStepRes = new AutomationStepSimulation(nextStep, this.props).invoke(allInputs);
                if (nextStepRes.responseCode != response_code_1.ResponseCode.SUCCESS) {
                    return {
                        responseCode: nextStepRes.responseCode,
                        stackTrace: nextStepRes.stackTrace,
                        executedSteps: this.prependSelf(nextStepRes.executedSteps),
                    };
                }
                else {
                    return {
                        responseCode: response_code_1.ResponseCode.SUCCESS,
                        outputs: { ...response, ...nextStepRes.outputs },
                        executedSteps: this.prependSelf(nextStepRes.executedSteps),
                    };
                }
            }
            else {
                return {
                    responseCode: response_code_1.ResponseCode.SUCCESS,
                    outputs: response,
                    executedSteps: [this.step.name],
                };
            }
            // JSII does not differentiate between exceptions. For the API we therefore return a ResponseCode
        }
        catch (error) {
            if (error instanceof non_retriable_exception_1.NonRetriableException) {
                throw error;
            }
            else if (error instanceof cancellation_exception_1.CancellationException) {
                if (this.step.onCancel) {
                    console.log('Step failed: ' + this.step.name + '. Executing onCancel step ' + this.step.onCancel.name);
                    return new AutomationStepSimulation(this.step.onCancel, this.props).invoke(allInputs);
                }
                else {
                    return {
                        responseCode: response_code_1.ResponseCode.CANCELED,
                        stackTrace: error.message + '\n' + error.stack,
                        executedSteps: [this.step.name],
                    };
                }
            }
            else {
                if (this.step.onFailure) {
                    console.log('Step failed: ' + this.step.name + '. Executing onFailure step ' + this.step.onFailure.name);
                    return new AutomationStepSimulation(this.step.onFailure, this.props).invoke(allInputs);
                }
                else {
                    return {
                        responseCode: response_code_1.ResponseCode.FAILED,
                        outputs: [],
                        stackTrace: error.message + '\n' + error.stack,
                        executedSteps: [this.step.name],
                    };
                }
            }
        }
    }
    /**
     * This function receives all of the available inputs and returns the subset of inputs that were requested.
     * If any input requested in the current step is not available in the received inputs, this will throw.
     * @param inputs all available inputs for the currently executing step
     * @returns the subset of inputs which are requested by the current step
     */
    filterInputs(inputs) {
        const foundAll = this.step.listInputs().every(required => Object.keys(inputs).includes(required));
        if (!foundAll) {
            throw new non_retriable_exception_1.NonRetriableException(`Not all inputs required were provided. Required: ${this.step.listInputs()}. Provided: ${JSON.stringify(inputs)}`);
        }
        const filteredEntries = Object.entries(inputs).filter(entry => {
            const [key] = entry;
            return key.startsWith('global:') || key.startsWith('automation:') || this.step.listInputs().includes(key);
        });
        return Object.fromEntries(filteredEntries);
    }
    /**
     * Adds this step name to the list of executed steps.
     * Since the steps are invoked as a chain, the current step is prepended as invocation stack is popped.
     */
    prependSelf(subsequentSteps) {
        return [this.step.name, ...subsequentSteps];
    }
    executeWithRetries(inputs) {
        let exception;
        for (let i = 0; i < this.step.maxAttempts; i++) {
            try {
                console.log('Invoking step ' + this.step.name + ': attempt ' + i);
                return this.tryExecute(inputs);
            }
            catch (error) {
                if (error instanceof non_retriable_exception_1.NonRetriableException) {
                    throw error;
                }
                else {
                    exception = error;
                }
            }
        }
        console.log('Exception occurred in this step: ' + this.step.name);
        throw exception;
    }
    tryExecute(inputs) {
        const start = Date.now();
        console.log(`Execute step ${this.step.name} start: ${start}`);
        const response = this.mySimulation().executeStep(inputs);
        console.log(`Execute step ${this.step.name} end: ${Date.now()}`);
        const relevantResponse = this.getSelectedResponse(response);
        this.checkExecutionTime(start);
        return relevantResponse;
    }
    mySimulation() {
        switch (this.step.action) {
            case 'aws:approve':
                return new approve_simulation_1.ApproveSimulation(this.step, this.props);
            case 'aws:assertAwsResourceProperty':
                return new assert_aws_resource_simulation_1.AssertAwsResourceSimulation(this.step, this.props);
            case 'aws:executeAwsApi':
                return new aws_api_simulation_1.AwsApiSimulation(this.step, this.props);
            case 'aws:branch':
                return new branch_simulation_1.BranchSimulation(this.step);
            case 'aws:changeInstanceState':
                return new change_instance_state_simulation_1.ChangeInstanceStateSimulation(this.step, this.props);
            case 'aws:copyImage':
                return new copy_image_simulation_1.CopyImageSimulation(this.step, this.props);
            case 'aws:createImage':
                return new create_image_simulation_1.CreateImageSimulation(this.step, this.props);
            case 'aws:createStack':
                return new create_stack_simulation_1.CreateStackSimulation(this.step, this.props);
            case 'aws:createTags':
                return new create_tags_simulation_1.CreateTagsSimulation(this.step, this.props);
            case 'aws:deleteImage':
                return new delete_image_simulation_1.DeleteImageSimulation(this.step, this.props);
            case 'aws:deleteStack':
                return new delete_stack_simulation_1.DeleteStackSimulation(this.step, this.props);
            case 'aws:executeScript':
                return new execute_script_simulation_1.ExecuteScriptSimulation(this.step);
            case 'aws:executeStateMachine':
                return new execute_state_machine_simulation_1.ExecuteStateMachineSimulation(this.step, this.props);
            case 'aws:invokeLambdaFunction':
                return new invoke_lambda_function_simulation_1.InvokeLambdaFunctionSimulation(this.step, this.props);
            case 'aws:invokeWebhook':
                return new invoke_webhook_simulation_1.InvokeWebhookSimulation(this.step, this.props);
            case 'aws:pause':
                return new pause_simulation_1.PauseSimulation(this.step, this.props);
            case 'aws:runCommand':
                return new run_command_simulation_1.RunCommandSimulation(this.step, this.props);
            case 'aws:runInstances':
                return new run_instance_simulation_1.RunInstanceSimulation(this.step, this.props);
            case 'aws:sleep':
                return new sleep_simulation_1.SleepSimulation(this.step, this.props);
            case 'aws:waitForAwsResourceProperty':
                return new wait_for_resource_simulation_1.WaitForResourceSimulation(this.step, this.props);
            default:
                throw new Error('No simulation available for action: ' + this.step.name);
        }
    }
    /**
     * A timeout may be set as a property of a step.
     * This function checks the timeout to ensure that the execution time for the step was not exceeded.
     * The execution will not be killed when timeout is reached.
     * Rather an error will be thrown after the fact if time is exceeded.
     */
    checkExecutionTime(start) {
        const executionTime = Date.now() - start;
        if (executionTime > this.step.timeoutSeconds * 1000) {
            throw new Error('Execution time exceeded timeout: timeout set to ' +
                this.step.timeoutSeconds + ' seconds but took ' + executionTime + ' ms');
        }
    }
    /**
     * Finds each declared step output using the json selector specified.
     * @returns the mapping of output name to the value selected from the step execution response.
     */
    getSelectedResponse(response) {
        const relevantResponse = {};
        this.step.listOutputs().forEach(declaredOutput => {
            if (response == undefined) {
                throw new Error(`Output ${declaredOutput.name} specified selector ${declaredOutput.selector} but response was undefined for step ${this.step.name}`);
            }
            // I cannot explain the hack below. But it needs to be reformed into an object.
            const hackedResponse = JSON.parse(JSON.stringify(response));
            const selectedResponse = jp.query(hackedResponse, declaredOutput.selector)[0];
            if (selectedResponse === undefined) {
                throw new Error(`Output ${declaredOutput.name} specified selector ${declaredOutput.selector} but not found in response ${JSON.stringify(response)}`);
            }
            relevantResponse[this.step.name + '.' + declaredOutput.name] = selectedResponse;
            if (!new data_type_1.DataType(declaredOutput.outputType).validateType(selectedResponse)) {
                throw new Error(`DataType of output ${declaredOutput.name}:${data_type_1.DataTypeEnum[declaredOutput.outputType]} does not match response type ${JSON.stringify(selectedResponse)}`);
            }
        });
        return relevantResponse;
    }
}
exports.AutomationStepSimulation = AutomationStepSimulation;
_a = JSII_RTTI_SYMBOL_1;
AutomationStepSimulation[_a] = { fqn: "cdk-ssm-documents.AutomationStepSimulation", version: "0.0.12" };
//# sourceMappingURL=data:application/json;base64,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