"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StringStep = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iso8601_duration_1 = require("iso8601-duration");
const choice_1 = require("../../domain/choice");
const data_type_1 = require("../../domain/data-type");
const operation_1 = require("../../domain/operation");
const string_variable_1 = require("../../interface/variables/string-variable");
const assert_aws_resource_step_1 = require("../../parent-steps/automation/assert-aws-resource-step");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const branch_step_1 = require("../../parent-steps/automation/branch-step");
const execute_script_step_1 = require("../../parent-steps/automation/execute-script-step");
const pause_step_1 = require("../../parent-steps/automation/pause-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const wait_for_resource_step_1 = require("../../parent-steps/automation/wait-for-resource-step");
const composite_step_1 = require("./composite-step");
// eslint-disable-next-line
const yaml = require('js-yaml');
/**
 * StringStep allows for including a step from an existing automation document verbatim.
 * This is useful in that it allows developers to integrate with existing document steps.
 * This step can be used just as you would use any other Step including simulation and deployment.
 */
class StringStep extends composite_step_1.CompositeAutomationStep {
    constructor(scope, id, props) {
        super(scope, id);
        const sharedProps = {};
        sharedProps.name = props.name;
        sharedProps.action = props.action;
        if (props.maxAttempts) {
            sharedProps.maxAttempts = props.maxAttempts;
        }
        if (props.timeoutSeconds) {
            sharedProps.timeoutSeconds = props.timeoutSeconds;
        }
        if (props.isEnd != undefined) {
            sharedProps.isEnd = props.isEnd;
        }
        if (props.isCritical != undefined) {
            sharedProps.isCritical = props.isCritical;
        }
        if (props.outputs != undefined) {
            sharedProps.outputs = this.toOutputs(props.outputs);
        }
        const { Service, Api, PropertySelector, DesiredValues, ...restParams } = props.inputs;
        switch (props.action) {
            case 'aws:executeAwsApi':
                this.automationStep = new aws_api_step_1.AwsApiStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    outputs: sharedProps.outputs,
                    ...sharedProps,
                });
                break;
            case 'aws:waitForAwsResourceProperty':
                this.automationStep = new wait_for_resource_step_1.WaitForResourceStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    selector: PropertySelector,
                    desiredValues: DesiredValues,
                    ...sharedProps,
                });
                break;
            case 'aws:assertAwsResourceProperty':
                this.automationStep = new assert_aws_resource_step_1.AssertAwsResourceStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    selector: PropertySelector,
                    desiredValues: DesiredValues,
                    ...sharedProps,
                });
                break;
            case 'aws:pause':
                this.automationStep = new pause_step_1.PauseStep(this, props.name, {
                    ...sharedProps,
                });
                break;
            case 'aws:sleep':
                this.automationStep = new sleep_step_1.SleepStep(this, props.name, {
                    sleepSeconds: iso8601_duration_1.toSeconds(iso8601_duration_1.parse(restParams.Duration)),
                    ...sharedProps,
                });
                break;
            case 'aws:executeScript':
                this.automationStep = new execute_script_step_1.ExecuteScriptStep(this, props.name, {
                    language: execute_script_step_1.ExecuteScriptStep.getLanguage(restParams.Runtime),
                    inputs: Object.keys(restParams.InputPayload),
                    inlineCode: restParams.Script,
                    handlerName: restParams.Handler,
                    ...sharedProps,
                });
                break;
            case 'aws:branch':
                this.automationStep = new branch_step_1.BranchStep(this, props.name, {
                    choices: this.toChoices(restParams.Choices),
                    defaultStepName: restParams.Default,
                    ...sharedProps,
                });
                break;
            default:
                throw new Error('Action not implemented as StringStep ' + props.action);
        }
    }
    /**
       * Builds a step object from a yaml declaration.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromYaml(scope, inputYaml) {
        return StringStep.fromObject(scope, yaml.load(inputYaml));
    }
    /**
       * Builds a step object from a json declaration.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromJson(scope, json) {
        return StringStep.fromObject(scope, JSON.parse(json));
    }
    /**
       * Builds a step object from an object.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromObject(scope, props) {
        return new StringStep(scope, props.name, props);
    }
    addToDocument(doc) {
        doc.addStep(this.automationStep);
    }
    toOutputs(declaredOutputs) {
        return declaredOutputs.map(declaredOutput => this.toOutput(declaredOutput));
    }
    toOutput(declaredOutput) {
        return {
            name: declaredOutput.Name,
            selector: declaredOutput.Selector,
            outputType: data_type_1.DataType.fromDataType(declaredOutput.Type).dataTypeEnum,
        };
    }
    toChoices(declaredChoices) {
        return declaredChoices.map(declaredChoice => this.toChoice(declaredChoice));
    }
    toChoice(declaredChoice) {
        const operationEntry = Object.entries(declaredChoice)
            .filter(entry => operation_1.OperationEvaluator.STRING_TO_OPERATION[entry[0]] != undefined)[0];
        return new choice_1.Choice({
            jumpToStepName: declaredChoice.NextStep,
            variable: this.toVariable(declaredChoice.Variable),
            operation: operation_1.OperationEvaluator.fromOperationName(operationEntry[0]),
            constant: operationEntry[1],
        });
    }
    toVariable(variable) {
        if (variable.includes('{{')) {
            return new string_variable_1.StringVariable(variable.replace('{{', '').replace('}}', '').trim());
        }
        return new string_variable_1.HardCodedString(variable.trim());
    }
}
exports.StringStep = StringStep;
_a = JSII_RTTI_SYMBOL_1;
StringStep[_a] = { fqn: "cdk-ssm-documents.StringStep", version: "0.0.12" };
//# sourceMappingURL=data:application/json;base64,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