"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiRunCommandHook = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const data_type_1 = require("../../domain/data-type");
const response_code_1 = require("../../domain/response-code");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const automation_step_simulation_1 = require("../../simulation/automation-step-simulation");
const string_list_variable_1 = require("../variables/string-list-variable");
const string_map_variable_1 = require("../variables/string-map-variable");
const string_variable_1 = require("../variables/string-variable");
/**
 * RunCommand implementation using AWS API
 */
class ApiRunCommandHook {
    constructor(awsInvoker, sleepHook) {
        this.awsInvoker = awsInvoker;
        this.sleepHook = sleepHook;
        this.props = { awsInvoker, sleepHook };
    }
    execute(props) {
        if (string_list_variable_1.isStringList(props.targets)) {
            if (props.targets.length > ApiRunCommandHook.MaximumMaxResultsValue) {
                throw new Error(`Only up to ${ApiRunCommandHook.MaximumMaxResultsValue} instance IDs can be specified.`);
            }
            console.log('RunCommand: Validating the instances are healthy');
            this.validateInstanceIdsAreHealthy(props.targets);
        }
        console.log(`RunCommand: Executing the document ${props.documentName}`);
        const executionId = this.runCommand(props);
        console.log(`RunCommand: Waiting for ${executionId} to complete`);
        const result = this.waitForCommand(executionId);
        console.log(`RunCommand: ${executionId} finished`);
        return result;
    }
    waitForCommand(executionId) {
        let status = this.getCommandStatus(executionId);
        while (status === null || ApiRunCommandHook.RunningStatuses.includes(status)) {
            console.log(`Command ${executionId} is not complete. Status: ${status}.`);
            new automation_step_simulation_1.AutomationStepSimulation(new sleep_step_1.SleepStep(new aws_cdk_lib_1.Stack(), 'sleep', {
                sleepSeconds: 2,
            }), this.props).invoke({});
            status = this.getCommandStatus(executionId);
        }
        if (status !== 'Success') {
            throw new Error(`${executionId} ended in status ${status}`);
        }
        const pluginStatus = this.getSinglePluginStatus(executionId);
        return {
            commandId: executionId,
            status: status,
            responseCode: pluginStatus === null || pluginStatus === void 0 ? void 0 : pluginStatus.ResponseCode,
            output: pluginStatus === null || pluginStatus === void 0 ? void 0 : pluginStatus.Output,
        };
    }
    getCommandStatus(executionId) {
        var _b;
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'listCommands', {
            service: 'SSM',
            pascalCaseApi: 'ListCommands',
            apiParams: {
                CommandId: executionId,
            },
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.MAP_LIST,
                    name: 'commands',
                    selector: '$.Commands',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Failed to get command ${executionId} status: ${result.stackTrace}`);
        }
        const commands = (_b = result.outputs) === null || _b === void 0 ? void 0 : _b['listCommands.commands'];
        if (commands.length === 0) {
            return null;
        }
        return commands[0].Status;
    }
    /**
       * We only return status code and output if there is exactly one invocation and one plug-in
       */
    getSinglePluginStatus(executionId) {
        var _b;
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'listCommandInvocations', {
            service: 'SSM',
            pascalCaseApi: 'ListCommandInvocations',
            apiParams: {
                CommandId: executionId,
                Details: true,
            },
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING_MAP,
                    name: 'result',
                    selector: '$',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Failed to get invocations for ${executionId}: ${result.stackTrace}`);
        }
        const response = (_b = result.outputs) === null || _b === void 0 ? void 0 : _b['listCommandInvocations.result'];
        if (!response.CommandInvocations || response.CommandInvocations.length !== 1) {
            return null;
        }
        const plugins = response.CommandInvocations[0].CommandPlugins;
        if (!plugins || plugins.length !== 1) {
            return null;
        }
        return plugins[0];
    }
    runCommand(props) {
        var _b;
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'sendCommand', {
            service: 'SSM',
            pascalCaseApi: 'SendCommand',
            apiParams: this.getSendCommandProps(props),
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING,
                    name: 'commandId',
                    selector: '$.Command.CommandId',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Failed to start command: ${result.stackTrace}`);
        }
        return (_b = result.outputs) === null || _b === void 0 ? void 0 : _b['sendCommand.commandId'];
    }
    getSendCommandProps(props) {
        const apiParamMap = {
            DocumentName: props.documentName,
            MaxConcurrency: props.maxConcurrency,
            MaxErrors: props.maxErrors,
            DocumentHash: props.documentHash,
            DocumentHashType: props.documentHashType,
            NotificationConfig: props.notificationConfig,
            TimeoutSeconds: props.timeoutSeconds,
            Comment: props.comment,
            OutputS3BucketName: props.outputS3BucketName,
            OutputS3KeyPrefix: props.outputS3KeyPrefix,
            ServiceRoleArn: props.serviceRoleArn,
            CloudWatchOutputConfig: props.cloudWatchOutputConfig,
        };
        const apiParams = {};
        for (const key of Object.keys(apiParamMap)) {
            const value = apiParamMap[key];
            if (value === undefined) {
                continue;
            }
            apiParams[key] = value;
        }
        if (string_list_variable_1.isStringList(props.targets)) {
            apiParams.InstanceIds = props.targets;
        }
        else {
            apiParams.Targets = props.targets;
        }
        if (props.parameters) {
            apiParams.Parameters = this.adaptRunCommandParameters(props.parameters);
        }
        return apiParams;
    }
    adaptRunCommandParameters(parameters) {
        const result = {};
        for (const key of Object.keys(parameters)) {
            const value = parameters[key];
            if (string_variable_1.isString(value)) {
                result[key] = [value];
                continue;
            }
            if (Array.isArray(value)) {
                if (!string_list_variable_1.isStringList(value)) {
                    throw new Error(`Parameter ${key} is a list but not a StringList. Only StringList is supported.`);
                }
                result[key] = value;
                continue;
            }
            if (string_map_variable_1.isStringMap(value)) {
                result[key] = [JSON.stringify(value)];
                continue;
            }
            throw new Error(`Parameter ${key} must be a String, StringList, or StringMap.`);
        }
        return result;
    }
    validateInstanceIdsAreHealthy(instanceIds) {
        const pingStatus = this.getPingStatusOfInstances(instanceIds);
        if (pingStatus.length !== instanceIds.length || !pingStatus.every(x => x === ApiRunCommandHook.SsmPingOnlineStatusValue)) {
            throw new Error('Not every instance has a healthy SSM agent');
        }
    }
    getPingStatusOfInstances(instanceIds) {
        var _b, _c;
        const apiParams = {
            MaxResults: ApiRunCommandHook.MaximumMaxResultsValue,
            InstanceInformationFilterList: [{
                    key: ApiRunCommandHook.SsmInstanceIdsFilterKey,
                    valueSet: instanceIds,
                }],
        };
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'describeInstanceInfo', {
            service: 'SSM',
            pascalCaseApi: 'DescribeInstanceInformation',
            apiParams: apiParams,
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.MAP_LIST,
                    name: 'instanceInformation',
                    selector: '$.InstanceInformationList',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Failed to load instance information: ${result.stackTrace}`);
        }
        return (_c = (_b = result.outputs) === null || _b === void 0 ? void 0 : _b['describeInstanceInfo.instanceInformation']) === null || _c === void 0 ? void 0 : _c.map(x => x.PingStatus);
    }
}
exports.ApiRunCommandHook = ApiRunCommandHook;
_a = JSII_RTTI_SYMBOL_1;
ApiRunCommandHook[_a] = { fqn: "cdk-ssm-documents.ApiRunCommandHook", version: "0.0.12" };
ApiRunCommandHook.MaximumMaxResultsValue = 50; // Only up to 50 instance IDs are allowed for aws:runCommand
ApiRunCommandHook.SsmInstanceIdsFilterKey = 'InstanceIds';
ApiRunCommandHook.SsmPingOnlineStatusValue = 'Online';
ApiRunCommandHook.RunningStatuses = ['Pending', 'InProgress', 'Cancelling'];
//# sourceMappingURL=data:application/json;base64,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