"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SsmDocument = exports.DocumentFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const constructs_1 = require("constructs");
const data_type_1 = require("../domain/data-type");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const yaml = require('js-yaml');
var DocumentFormat;
(function (DocumentFormat) {
    DocumentFormat[DocumentFormat["JSON"] = 0] = "JSON";
    DocumentFormat[DocumentFormat["YAML"] = 1] = "YAML";
})(DocumentFormat = exports.DocumentFormat || (exports.DocumentFormat = {}));
class SsmDocument extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id);
        this.documentName = (_b = props.documentName) !== null && _b !== void 0 ? _b : id;
        this.description = (_c = props.description) !== null && _c !== void 0 ? _c : this.documentName;
        this.header = props.header;
        this.assumeRole = props.assumeRole;
        this.docOutputs = (_d = props.docOutputs) !== null && _d !== void 0 ? _d : [];
        this.docInputs = (_e = props.docInputs) !== null && _e !== void 0 ? _e : [];
        if (this.assumeRole && !this.docInputs.map(i => i.name).includes(this.assumeRole)) {
            throw new Error('Assume rold specified but not provided in inputs: ' + this.assumeRole);
        }
        this.props = props;
        const isYaml = this.props.documentFormat == DocumentFormat.YAML;
        new aws_ssm_1.CfnDocument(this, this.node.id + 'CfnDoc', {
            ...this.props,
            ...{
                content: aws_cdk_lib_1.Lazy.any({
                    produce: () => {
                        const doc = this.buildSsmDocument();
                        return JSON.parse(JSON.stringify(doc));
                    },
                }),
                documentFormat: isYaml ? 'YAML' : 'JSON',
                documentType: this.documentType(),
                tags: [{ key: 'CdkGenerated', value: 'true' }, ...(this.props.tags || [])],
            },
        });
    }
    /**
     * The SSM Document inputs optionally specify a number of parameters including allowedValues, minItems etc.
     * This function builds an object containing the relevant (declared) input parameters.
     * The return object will be used to build the yaml/json representation of the document.
     */
    formatInputs() {
        const ssmInputs = {};
        this.docInputs.forEach(inp => {
            const nested = {
                type: new data_type_1.DataType(inp.inputType).toSsmString(),
            };
            if (inp.description != undefined) {
                nested.description = inp.description;
            }
            if (inp.defaultValue != undefined) {
                nested.default = inp.defaultValue;
            }
            if (inp.allowedValues != undefined) {
                nested.allowedValues = inp.allowedValues;
            }
            if (inp.allowedPattern != undefined) {
                nested.allowedPattern = inp.allowedPattern;
            }
            if (inp.minItems != undefined) {
                nested.minItems = inp.minItems;
            }
            if (inp.maxItems != undefined) {
                nested.maxItems = inp.maxItems;
            }
            ssmInputs[inp.name] = nested;
        });
        return ssmInputs;
    }
    /**
       * Synthesize before calling this function!
       * You can use this to Synthesize: cdk.SynthUtils.synthesize(stack);
       *
       * Converts the objects define in the SSM Document (including all of the steps) to an SSM document string.
       * The format is dependency on the documentFormat property provided to the class.
       * The yaml can be used as is and will behave (or at least should behave) as was simulated in the runSimulation().
       * @returns a string representation of this document as an SSM formatted yaml/json.
       */
    print() {
        const isYaml = this.props.documentFormat == DocumentFormat.YAML;
        if (isYaml) {
            // Prepend the document with the header if defined
            return this.headerWithComments() + yaml.dump(this.buildSsmDocument());
        }
        else {
            // There is no way to provide the header comment in a json document
            return JSON.stringify(this.buildSsmDocument());
        }
    }
    /**
       * Prepend header line with # if it does not start with #.
       * If no header is provided, this method will return an emptry string.
       * Headers are only available in yaml format.
       */
    headerWithComments() {
        if (this.header == undefined) {
            return '';
        }
        return this.header.split(os_1.EOL).map(line => line.trim().startsWith('#') ? line : '# ' + line).join(os_1.EOL) +
            // "---" demarks the end of the header and the start of the document.
            '\n---\n';
    }
}
exports.SsmDocument = SsmDocument;
_a = JSII_RTTI_SYMBOL_1;
SsmDocument[_a] = { fqn: "cdk-ssm-documents.SsmDocument", version: "0.0.12" };
//# sourceMappingURL=data:application/json;base64,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