from __future__ import annotations

from .deps import CONDA
from dataclasses import dataclass
from pathlib import Path
from typing import List, Optional


@dataclass
class CondaEnvSpec:
    name: str
    is_prefix: bool = False

    @classmethod
    async def current(cls) -> CondaEnvSpec:
        info = await conda_info()
        if "active_prefix" not in info:
            raise RuntimeError(
                "No active conda environment, and none specified in --prefix or --name"
            )

        if info["active_prefix_name"] == info["active_prefix"]:
            return CondaEnvSpec(info["active_prefix"], is_prefix=True)
        else:
            return CondaEnvSpec(info["active_prefix_name"], is_prefix=False)

    def __call__(self) -> List[str]:
        return ["--prefix" if self.is_prefix else "--name", self.name]

    async def get_python(self) -> Path:
        return (await self.get_prefix()) / "bin" / "python"

    async def get_prefix(self) -> Path:
        if self.is_prefix:
            return Path(self.name)
        else:
            return Path((await conda_env_export(self))["prefix"])


async def conda_env_export(env_spec: CondaEnvSpec):
    return await CONDA.json("env", "export", env_spec(), "--no-builds", "--json")


async def conda_info():
    return await CONDA.json("info", "--json")
