"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongControlPlaneEKS = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class KongControlPlaneEKS extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const rds_secrets = 'rds-secrets';
        const rds_secrets_provider_class = 'rds-secrets-provider-class';
        const kong_namespace = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongNamespace', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: props.namespace },
                },
            ],
        });
        const kong_cp_service_account = new aws_cdk_lib_1.aws_eks.ServiceAccount(this, 'KongCpSA', {
            cluster: props.cluster,
            name: 'kong-cp-service-account',
            namespace: props.namespace,
        });
        kong_cp_service_account.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
            ],
        }));
        kong_cp_service_account.node.addDependency(kong_namespace);
        const kong_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongCert', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: props.cacertname,
                        namespace: props.namespace,
                    },
                    spec: {
                        commonName: props.hostedZoneName,
                        dnsNames: props.dnsNames,
                        duration: '2160h0m0s',
                        issuerRef: {
                            group: 'awspca.cert-manager.io',
                            kind: 'AWSPCAClusterIssuer',
                            name: props.clusterIssuerName,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${props.clusterIssuerName}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        kong_certificate.node.addDependency(kong_namespace);
        // const secretNameParts = Fn.split('-', props.rds.secret?.secretName!);
        // const secretNameWithoutSuffix = Fn.join('-', [Fn.select(0, secretNameParts), Fn.select(1, secretNameParts)]);
        const secrets_crd_mount = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'SecretsMount', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: rds_secrets_provider_class,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        parameters: {
                            objects: `- objectName: "${(_a = props.rds.secret) === null || _a === void 0 ? void 0 : _a.secretFullArn}"\n  jmesPath:\n    - path: username\n      objectAlias: dbusername\n    - path: password\n      objectAlias: dbpassword\n    - path: dbname\n      objectAlias: dbname\n    - path: host\n      objectAlias: dbhost\n`,
                        },
                        secretObjects: [
                            {
                                secretName: rds_secrets,
                                type: 'Opaque',
                                data: [
                                    {
                                        objectName: 'dbusername',
                                        key: 'dbusername',
                                    },
                                    {
                                        objectName: 'dbpassword',
                                        key: 'dbpassword',
                                    },
                                    {
                                        objectName: 'dbname',
                                        key: 'dbname',
                                    },
                                    {
                                        objectName: 'dbhost',
                                        key: 'dbhost',
                                    },
                                ],
                            },
                        ],
                    },
                },
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: props.license_secret_name,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        secretObjects: [
                            {
                                secretName: props.license_secret_name,
                                type: 'opaque',
                                data: [
                                    {
                                        objectName: props.license_secret_name,
                                        key: 'license',
                                    },
                                ],
                            },
                        ],
                        parameters: {
                            objects: `- objectName: "${props.license_secret_name}"\n  objectType: "secretsmanager"\n`,
                        },
                    },
                },
            ],
        });
        secrets_crd_mount.node.addDependency(kong_namespace);
        const deploy_kong_cp_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'KongCpHelm', {
            ...props.HelmOptions,
            cluster: props.cluster,
            repository: 'https://charts.konghq.com',
            chart: 'kong',
            release: 'kong',
            namespace: props.namespace,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                // ingressController: {
                //   enabled: true,
                //   installCRDs: false,
                //   image: {
                //     repository: 'kong/kubernetes-ingress-controller',
                //     tag: '1.3.1-alpine',
                //   },
                // },
                // image: {
                //   repository: 'kong/kong-gateway',
                //   tag: '2.5.0.0-alpine',
                // },
                env: {
                    database: 'postgres',
                    role: 'control_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: '/etc/secrets/kong-cp-cluster-issuer-secret/tls.crt',
                    cluster_cert_key: '/etc/secrets/kong-cp-cluster-issuer-secret/tls.key',
                    cluster_ca_cert: '/etc/secrets/kong-cp-cluster-issuer-secret/ca.crt',
                    pg_user: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbusername',
                            },
                        },
                    },
                    pg_password: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbpassword',
                            },
                        },
                    },
                    pg_database: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbname',
                            },
                        },
                    },
                    pg_host: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbhost',
                            },
                        },
                    },
                },
                cluster: {
                    enabled: 'true',
                    type: 'LoadBalancer',
                    tls: {
                        enabled: 'true',
                        servicePort: 8005,
                        containerPort: 8005,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': 'cluster.kong-cp.internal',
                    },
                },
                clustertelemetry: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: 'true',
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': 'telemetry.kong-cp.internal',
                    },
                },
                proxy: {
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=proxy',
                        'external-dns.alpha.kubernetes.io/hostname': 'proxy.kong-cp.internal',
                    },
                },
                admin: {
                    enabled: true,
                    type: 'LoadBalancer',
                    labels: {
                        'enable-metrics': true,
                    },
                    http: {
                        enabled: true,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=admin',
                        'external-dns.alpha.kubernetes.io/hostname': 'admin.kong-cp.internal',
                    },
                },
                deployment: {
                    serviceAccount: {
                        create: false,
                        name: kong_cp_service_account.serviceAccountName,
                    },
                    userDefinedVolumes: [
                        {
                            name: props.license_secret_name,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: props.license_secret_name,
                                },
                            },
                        },
                        {
                            name: rds_secrets,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: rds_secrets_provider_class,
                                },
                            },
                        },
                    ],
                    userDefinedVolumeMounts: [
                        {
                            name: props.license_secret_name,
                            mountPath: '/mnt/license_secrets',
                            readOnly: true,
                        },
                        {
                            name: rds_secrets,
                            mountPath: '/mnt/rds_secrets',
                            readOnly: true,
                        },
                    ],
                },
                enterprise: {
                    enabled: true,
                    license_secret: props.license_secret_name,
                },
                serviceMonitor: {
                    enabled: true,
                },
                manager: {
                    enabled: true,
                    type: 'LoadBalancer',
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': 'manager.kong-cp.internal',
                    },
                },
                postgresql: {
                    enabled: false,
                },
                secretVolumes: [
                    'kong-cp-cluster-issuer-secret',
                ],
            },
        });
        deploy_kong_cp_helm.node.addDependency(secrets_crd_mount, kong_namespace, kong_cp_service_account);
    }
}
exports.KongControlPlaneEKS = KongControlPlaneEKS;
//# sourceMappingURL=data:application/json;base64,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