"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongAdminService = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_route53_1 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const aws_servicediscovery_1 = require("aws-cdk-lib/aws-servicediscovery");
class KongAdminService extends aws_ecs_1.FargateService {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        super(scope, id, props.kongServiceProps);
        this.zone = props.route53_zone;
        // Initialize the port variables or user defined input
        this.adminPort = props.kongFeaturesProps.adminProps.httpPort || 8001;
        this.adminHttpsPort = props.kongFeaturesProps.adminProps.httpsPort || 8444;
        this.managerPort = props.kongFeaturesProps.kongManagerProps.httpPort || 8002;
        this.managerHttpsPort = props.kongFeaturesProps.kongManagerProps.httpsPort || 8445;
        this.telemetryPort = props.kongFeaturesProps.clusterTelemetryProps.port || 8006;
        this.clusterPort = props.kongFeaturesProps.clusterProps.port || 8005;
        this.devPortalPort = props.kongFeaturesProps.devPortalProps.port || 8003;
        // Initialize the endpoints
        this.controlPlaneDnsEp = this.cluster.clusterName;
        this.adminEndpoint = 'admin.' + props.route53_zone.zoneName;
        this.managerEndpoint = 'manager.' + props.route53_zone.zoneName;
        this.devPortalEndpoint = 'devportal.' + props.route53_zone.zoneName;
        this.clusterEndpoint = 'cluster.' + props.route53_zone.zoneName;
        this.telemetryEndpoint = 'telemetry.' + props.route53_zone.zoneName;
        const cloudmap = this.enableCloudMap({
            name: 'kongControlPlaneCloudMapService',
            cloudMapNamespace: new aws_servicediscovery_1.PrivateDnsNamespace(this, 'KongCPNamespace', {
                name: this.controlPlaneDnsEp,
                vpc: props.kongServiceProps.cluster.vpc,
            }),
            dnsRecordType: aws_servicediscovery_1.DnsRecordType.A,
        });
        //Use Cloudmap to expose the https service instead of adding a NLB and map it to a DNS record.
        if (props.kongFeaturesProps.clusterProps.enabled) {
            new aws_route53_1.CnameRecord(this, 'kongAdminCNameRecord', {
                domainName: ((_a = this.cloudMapService) === null || _a === void 0 ? void 0 : _a.serviceName) + '.' + this.controlPlaneDnsEp || '',
                zone: props.route53_zone,
                recordName: this.clusterEndpoint,
            }).node.addDependency(cloudmap);
            this.connections.allowFrom(aws_ec2_1.Peer.ipv4(this.cluster.vpc.vpcCidrBlock), aws_ec2_1.Port.tcp(this.clusterPort));
            new aws_cdk_lib_1.CfnOutput(this, 'KongClusterEndpoint', {
                value: this.clusterEndpoint + ':' + this.clusterPort,
                exportName: 'KongClusterEndpoint',
            });
        }
        if (props.kongFeaturesProps.clusterTelemetryProps.enabled) {
            new aws_route53_1.CnameRecord(this, 'kongTelemetryCNameRecord', {
                domainName: ((_b = this.cloudMapService) === null || _b === void 0 ? void 0 : _b.serviceName) + '.' + this.controlPlaneDnsEp || '',
                zone: props.route53_zone,
                recordName: this.telemetryEndpoint,
            }).node.addDependency(cloudmap);
            this.connections.allowFrom(aws_ec2_1.Peer.ipv4(this.cluster.vpc.vpcCidrBlock), aws_ec2_1.Port.tcp(this.telemetryPort));
            new aws_cdk_lib_1.CfnOutput(this, 'KongTelemetryEndpoint', {
                value: this.telemetryEndpoint + ':' + this.telemetryPort,
                exportName: 'KongTelemetryEndpoint',
            });
        }
        this.registerKongFeatureTargets(443, this.adminHttpsPort, props.adminNlb, 'KongAdminHttps', (_d = (_c = props.kongServiceProps.taskDefinition.defaultContainer) === null || _c === void 0 ? void 0 : _c.containerName) !== null && _d !== void 0 ? _d : '', true, this.adminEndpoint);
        if (props.kongFeaturesProps.adminProps.enableHttp) {
            this.registerKongFeatureTargets(80, this.adminPort, props.adminNlb, 'KongAdmin', (_f = (_e = props.kongServiceProps.taskDefinition.defaultContainer) === null || _e === void 0 ? void 0 : _e.containerName) !== null && _f !== void 0 ? _f : '', false);
        }
        if (props.kongFeaturesProps.kongManagerProps.enabled && props.managerNlb) {
            this.registerKongFeatureTargets(443, this.managerHttpsPort, props.managerNlb, 'KongManager', (_h = (_g = props.kongServiceProps.taskDefinition.defaultContainer) === null || _g === void 0 ? void 0 : _g.containerName) !== null && _h !== void 0 ? _h : '', true, this.managerEndpoint);
            if (props.kongFeaturesProps.kongManagerProps.enableHttp) {
                this.registerKongFeatureTargets(80, this.managerPort, props.managerNlb, 'KongManagerHttps', (_k = (_j = props.kongServiceProps.taskDefinition.defaultContainer) === null || _j === void 0 ? void 0 : _j.containerName) !== null && _k !== void 0 ? _k : '', false);
            }
        }
        ;
        if (props.kongFeaturesProps.devPortalProps.enabled && props.devPortalNlb) {
            this.registerKongFeatureTargets(80, this.devPortalPort, props.devPortalNlb, 'KongDevPortal', (_m = (_l = props.kongServiceProps.taskDefinition.defaultContainer) === null || _l === void 0 ? void 0 : _l.containerName) !== null && _m !== void 0 ? _m : '', true, this.devPortalEndpoint);
        }
    }
    registerKongFeatureTargets(listenerPort, containerPort, lb, id, containerName, addARecord, recordName) {
        const listener = new aws_elasticloadbalancingv2_1.NetworkListener(this, 'KongCp' + id + 'Listener', {
            port: listenerPort,
            protocol: aws_elasticloadbalancingv2_1.Protocol.TCP,
            loadBalancer: lb,
        });
        if (addARecord) {
            new aws_route53_1.ARecord(this, 'KongCp' + id + 'ARecord', {
                zone: this.zone,
                recordName: recordName,
                target: aws_route53_1.RecordTarget.fromAlias(new targets.LoadBalancerTarget(lb)),
            });
        }
        this.registerLoadBalancerTargets({
            containerName: containerName,
            containerPort: containerPort,
            newTargetGroupId: id,
            listener: aws_ecs_1.ListenerConfig.networkListener(listener, {
                deregistrationDelay: aws_cdk_lib_1.Duration.seconds(60),
                port: listenerPort,
                healthCheck: {
                    port: containerPort.toString(),
                },
                preserveClientIp: false,
                proxyProtocolV2: false,
                protocol: aws_elasticloadbalancingv2_1.Protocol.TCP,
            }),
        });
        this.connections.allowFrom(aws_ec2_1.Peer.ipv4(this.cluster.vpc.vpcCidrBlock), aws_ec2_1.Port.tcp(containerPort));
        if (recordName) {
            new aws_cdk_lib_1.CfnOutput(this, 'KongCp' + id + 'Endpoint', {
                value: recordName,
                exportName: id,
            });
        }
    }
}
exports.KongAdminService = KongAdminService;
//# sourceMappingURL=data:application/json;base64,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