"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const cloud9_generated_1 = require("../lib/cloud9.generated");
/**
 * A Cloud9 Environment with Amazon EC2
 * @resource AWS::Cloud9::EnvironmentEC2
 */
class Ec2Environment extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.vpc = props.vpc;
        if (!props.subnetSelection && this.vpc.publicSubnets.length === 0) {
            throw new Error('no subnetSelection specified and no public subnet found in the vpc, please specify subnetSelection');
        }
        const vpcSubnets = (_a = props.subnetSelection) !== null && _a !== void 0 ? _a : { subnetType: ec2.SubnetType.PUBLIC };
        const c9env = new cloud9_generated_1.CfnEnvironmentEC2(this, 'Resource', {
            name: props.ec2EnvironmentName,
            description: props.description,
            instanceType: (_c = (_b = props.instanceType) === null || _b === void 0 ? void 0 : _b.toString()) !== null && _c !== void 0 ? _c : ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.MICRO).toString(),
            subnetId: this.vpc.selectSubnets(vpcSubnets).subnetIds[0],
            repositories: props.clonedRepositories ? props.clonedRepositories.map(r => ({
                repositoryUrl: r.repositoryUrl,
                pathComponent: r.pathComponent,
            })) : undefined,
        });
        this.environmentId = c9env.ref;
        this.ec2EnvironmentArn = c9env.getAtt('Arn').toString();
        this.ec2EnvironmentName = c9env.getAtt('Name').toString();
        this.ideUrl = `https://${this.stack.region}.console.aws.amazon.com/cloud9/ide/${this.environmentId}`;
    }
    /**
     * import from EnvironmentEc2Name
     */
    static fromEc2EnvironmentName(scope, id, ec2EnvironmentName) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.ec2EnvironmentName = ec2EnvironmentName;
                this.ec2EnvironmentArn = cdk.Stack.of(this).formatArn({
                    service: 'cloud9',
                    resource: 'environment',
                    resourceName: this.ec2EnvironmentName,
                });
            }
        }
        return new Import(scope, id);
    }
}
exports.Ec2Environment = Ec2Environment;
/**
 * The class for different repository providers
 */
class CloneRepository {
    constructor(repositoryUrl, pathComponent) {
        this.repositoryUrl = repositoryUrl;
        this.pathComponent = pathComponent;
    }
    /**
     * import repository to cloud9 environment from AWS CodeCommit
     *
     * @param repository the codecommit repository to clone from
     * @param path  the target path in cloud9 environment
     */
    static fromCodeCommit(repository, path) {
        return {
            repositoryUrl: repository.repositoryCloneUrlHttp,
            pathComponent: path,
        };
    }
}
exports.CloneRepository = CloneRepository;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZW52aXJvbm1lbnQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJlbnZpcm9ubWVudC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUNBLHdDQUF3QztBQUN4QyxxQ0FBcUM7QUFDckMsOERBQTREO0FBc0U1RDs7O0dBR0c7QUFDSCxNQUFhLGNBQWUsU0FBUSxHQUFHLENBQUMsUUFBUTtJQTZDOUMsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUEwQjs7UUFDdEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsR0FBRyxHQUFHLEtBQUssQ0FBQyxHQUFHLENBQUM7UUFDckIsSUFBSSxDQUFDLEtBQUssQ0FBQyxlQUFlLElBQUksSUFBSSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNqRSxNQUFNLElBQUksS0FBSyxDQUFDLG9HQUFvRyxDQUFDLENBQUM7U0FDdkg7UUFFRCxNQUFNLFVBQVUsU0FBRyxLQUFLLENBQUMsZUFBZSxtQ0FBSSxFQUFFLFVBQVUsRUFBRSxHQUFHLENBQUMsVUFBVSxDQUFDLE1BQU0sRUFBRSxDQUFDO1FBQ2xGLE1BQU0sS0FBSyxHQUFHLElBQUksb0NBQWlCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNwRCxJQUFJLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtZQUM5QixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsWUFBWSxjQUFFLEtBQUssQ0FBQyxZQUFZLDBDQUFFLFFBQVEscUNBQU0sR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxVQUFVLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLEVBQUU7WUFDcEksUUFBUSxFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7WUFDekQsWUFBWSxFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGtCQUFrQixDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUM7Z0JBQzFFLGFBQWEsRUFBRSxDQUFDLENBQUMsYUFBYTtnQkFDOUIsYUFBYSxFQUFFLENBQUMsQ0FBQyxhQUFhO2FBQy9CLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1NBQ2hCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQztRQUMvQixJQUFJLENBQUMsaUJBQWlCLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUN4RCxJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUMxRCxJQUFJLENBQUMsTUFBTSxHQUFHLFdBQVcsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLHNDQUFzQyxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUM7SUFDdkcsQ0FBQztJQW5FRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxzQkFBc0IsQ0FBQyxLQUFvQixFQUFFLEVBQVUsRUFBRSxrQkFBMEI7UUFDL0YsTUFBTSxNQUFPLFNBQVEsR0FBRyxDQUFDLFFBQVE7WUFBakM7O2dCQUNTLHVCQUFrQixHQUFHLGtCQUFrQixDQUFDO2dCQUN4QyxzQkFBaUIsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7b0JBQ3RELE9BQU8sRUFBRSxRQUFRO29CQUNqQixRQUFRLEVBQUUsYUFBYTtvQkFDdkIsWUFBWSxFQUFFLElBQUksQ0FBQyxrQkFBa0I7aUJBQ3RDLENBQUMsQ0FBQztZQUNMLENBQUM7U0FBQTtRQUNELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQy9CLENBQUM7Q0F1REY7QUFyRUQsd0NBcUVDO0FBRUQ7O0dBRUc7QUFDSCxNQUFhLGVBQWU7SUFjMUIsWUFBb0MsYUFBcUIsRUFBa0IsYUFBcUI7UUFBNUQsa0JBQWEsR0FBYixhQUFhLENBQVE7UUFBa0Isa0JBQWEsR0FBYixhQUFhLENBQVE7SUFBRyxDQUFDO0lBYnBHOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLGNBQWMsQ0FBQyxVQUFrQyxFQUFFLElBQVk7UUFDM0UsT0FBTztZQUNMLGFBQWEsRUFBRSxVQUFVLENBQUMsc0JBQXNCO1lBQ2hELGFBQWEsRUFBRSxJQUFJO1NBQ3BCLENBQUM7SUFDSixDQUFDO0NBR0Y7QUFmRCwwQ0FlQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNvZGVjb21taXQgZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVjb21taXQnO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ2ZuRW52aXJvbm1lbnRFQzIgfSBmcm9tICcuLi9saWIvY2xvdWQ5LmdlbmVyYXRlZCc7XG5cbi8qKlxuICogQSBDbG91ZDkgRW52aXJvbm1lbnRcbiAqXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUVjMkVudmlyb25tZW50IGV4dGVuZHMgY2RrLklSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgRW52aXJvbm1lbnRFYzJcbiAgICpcbiAgICogQGF0dHJpYnV0ZSBlbnZpcm9ubWVudEVjMk5hbWVcbiAgICovXG4gIHJlYWRvbmx5IGVjMkVudmlyb25tZW50TmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgYXJuIG9mIHRoZSBFbnZpcm9ubWVudEVjMlxuICAgKlxuICAgKiBAYXR0cmlidXRlIGVudmlyb25tZW50RTJBcm5cbiAgICovXG4gIHJlYWRvbmx5IGVjMkVudmlyb25tZW50QXJuOiBzdHJpbmc7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgRWMyRW52aXJvbm1lbnRcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFYzJFbnZpcm9ubWVudFByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSB0eXBlIG9mIGluc3RhbmNlIHRvIGNvbm5lY3QgdG8gdGhlIGVudmlyb25tZW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHQyLm1pY3JvXG4gICAqL1xuICByZWFkb25seSBpbnN0YW5jZVR5cGU/OiBlYzIuSW5zdGFuY2VUeXBlO1xuXG4gIC8qKlxuICAgKiBUaGUgc3VibmV0U2VsZWN0aW9uIG9mIHRoZSBWUEMgdGhhdCBBV1MgQ2xvdWQ5IHdpbGwgdXNlIHRvIGNvbW11bmljYXRlIHdpdGhcbiAgICogdGhlIEFtYXpvbiBFQzIgaW5zdGFuY2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gYWxsIHB1YmxpYyBzdWJuZXRzIG9mIHRoZSBWUEMgYXJlIHNlbGVjdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgc3VibmV0U2VsZWN0aW9uPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcblxuICAvKipcbiAgICogVGhlIFZQQyB0aGF0IEFXUyBDbG91ZDkgd2lsbCB1c2UgdG8gY29tbXVuaWNhdGUgd2l0aCB0aGUgQW1hem9uIEVsYXN0aWMgQ29tcHV0ZSBDbG91ZCAoQW1hem9uIEVDMikgaW5zdGFuY2UuXG4gICAqXG4gICAqL1xuICByZWFkb25seSB2cGM6IGVjMi5JVnBjO1xuXG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoZSBlbnZpcm9ubWVudFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGF1dG9tYXRpY2FsbHkgZ2VuZXJhdGVkIG5hbWVcbiAgICovXG4gIHJlYWRvbmx5IGVjMkVudmlyb25tZW50TmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogRGVzY3JpcHRpb24gb2YgdGhlIGVudmlyb25tZW50XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gZGVzY3JpcHRpb25cbiAgICovXG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgQVdTIENvZGVDb21taXQgcmVwb3NpdG9yeSB0byBiZSBjbG9uZWRcbiAgICpcbiAgICogQGRlZmF1bHQgLSBkbyBub3QgY2xvbmUgYW55IHJlcG9zaXRvcnlcbiAgICovXG4gIC8vIHJlYWRvbmx5IGNsb25lZFJlcG9zaXRvcmllcz86IENsb3VkOVJlcG9zaXRvcnlbXTtcbiAgcmVhZG9ubHkgY2xvbmVkUmVwb3NpdG9yaWVzPzogQ2xvbmVSZXBvc2l0b3J5W107XG59XG5cbi8qKlxuICogQSBDbG91ZDkgRW52aXJvbm1lbnQgd2l0aCBBbWF6b24gRUMyXG4gKiBAcmVzb3VyY2UgQVdTOjpDbG91ZDk6OkVudmlyb25tZW50RUMyXG4gKi9cbmV4cG9ydCBjbGFzcyBFYzJFbnZpcm9ubWVudCBleHRlbmRzIGNkay5SZXNvdXJjZSBpbXBsZW1lbnRzIElFYzJFbnZpcm9ubWVudCB7XG4gIC8qKlxuICAgKiBpbXBvcnQgZnJvbSBFbnZpcm9ubWVudEVjMk5hbWVcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUVjMkVudmlyb25tZW50TmFtZShzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgZWMyRW52aXJvbm1lbnROYW1lOiBzdHJpbmcpOiBJRWMyRW52aXJvbm1lbnQge1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIGNkay5SZXNvdXJjZSB7XG4gICAgICBwdWJsaWMgZWMyRW52aXJvbm1lbnROYW1lID0gZWMyRW52aXJvbm1lbnROYW1lO1xuICAgICAgcHVibGljIGVjMkVudmlyb25tZW50QXJuID0gY2RrLlN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICAgIHNlcnZpY2U6ICdjbG91ZDknLFxuICAgICAgICByZXNvdXJjZTogJ2Vudmlyb25tZW50JyxcbiAgICAgICAgcmVzb3VyY2VOYW1lOiB0aGlzLmVjMkVudmlyb25tZW50TmFtZSxcbiAgICAgIH0pO1xuICAgIH1cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBlbnZpcm9ubWVudCBuYW1lIG9mIHRoaXMgQ2xvdWQ5IGVudmlyb25tZW50XG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBlYzJFbnZpcm9ubWVudE5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGVudmlyb25tZW50IEFSTiBvZiB0aGlzIENsb3VkOSBlbnZpcm9ubWVudFxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZWMyRW52aXJvbm1lbnRBcm46IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGVudmlyb25tZW50IElEIG9mIHRoaXMgQ2xvdWQ5IGVudmlyb25tZW50XG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZW52aXJvbm1lbnRJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgY29tcGxldGUgSURFIFVSTCBvZiB0aGlzIENsb3VkOSBlbnZpcm9ubWVudFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGlkZVVybDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBWUEMgSURcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB2cGM6IGVjMi5JVnBjO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRWMyRW52aXJvbm1lbnRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLnZwYyA9IHByb3BzLnZwYztcbiAgICBpZiAoIXByb3BzLnN1Ym5ldFNlbGVjdGlvbiAmJiB0aGlzLnZwYy5wdWJsaWNTdWJuZXRzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdubyBzdWJuZXRTZWxlY3Rpb24gc3BlY2lmaWVkIGFuZCBubyBwdWJsaWMgc3VibmV0IGZvdW5kIGluIHRoZSB2cGMsIHBsZWFzZSBzcGVjaWZ5IHN1Ym5ldFNlbGVjdGlvbicpO1xuICAgIH1cblxuICAgIGNvbnN0IHZwY1N1Ym5ldHMgPSBwcm9wcy5zdWJuZXRTZWxlY3Rpb24gPz8geyBzdWJuZXRUeXBlOiBlYzIuU3VibmV0VHlwZS5QVUJMSUMgfTtcbiAgICBjb25zdCBjOWVudiA9IG5ldyBDZm5FbnZpcm9ubWVudEVDMih0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBuYW1lOiBwcm9wcy5lYzJFbnZpcm9ubWVudE5hbWUsXG4gICAgICBkZXNjcmlwdGlvbjogcHJvcHMuZGVzY3JpcHRpb24sXG4gICAgICBpbnN0YW5jZVR5cGU6IHByb3BzLmluc3RhbmNlVHlwZT8udG9TdHJpbmcoKSA/PyBlYzIuSW5zdGFuY2VUeXBlLm9mKGVjMi5JbnN0YW5jZUNsYXNzLkJVUlNUQUJMRTIsIGVjMi5JbnN0YW5jZVNpemUuTUlDUk8pLnRvU3RyaW5nKCksXG4gICAgICBzdWJuZXRJZDogdGhpcy52cGMuc2VsZWN0U3VibmV0cyh2cGNTdWJuZXRzKS5zdWJuZXRJZHNbMF0sXG4gICAgICByZXBvc2l0b3JpZXM6IHByb3BzLmNsb25lZFJlcG9zaXRvcmllcyA/IHByb3BzLmNsb25lZFJlcG9zaXRvcmllcy5tYXAociA9PiAoe1xuICAgICAgICByZXBvc2l0b3J5VXJsOiByLnJlcG9zaXRvcnlVcmwsXG4gICAgICAgIHBhdGhDb21wb25lbnQ6IHIucGF0aENvbXBvbmVudCxcbiAgICAgIH0pKSA6IHVuZGVmaW5lZCxcbiAgICB9KTtcbiAgICB0aGlzLmVudmlyb25tZW50SWQgPSBjOWVudi5yZWY7XG4gICAgdGhpcy5lYzJFbnZpcm9ubWVudEFybiA9IGM5ZW52LmdldEF0dCgnQXJuJykudG9TdHJpbmcoKTtcbiAgICB0aGlzLmVjMkVudmlyb25tZW50TmFtZSA9IGM5ZW52LmdldEF0dCgnTmFtZScpLnRvU3RyaW5nKCk7XG4gICAgdGhpcy5pZGVVcmwgPSBgaHR0cHM6Ly8ke3RoaXMuc3RhY2sucmVnaW9ufS5jb25zb2xlLmF3cy5hbWF6b24uY29tL2Nsb3VkOS9pZGUvJHt0aGlzLmVudmlyb25tZW50SWR9YDtcbiAgfVxufVxuXG4vKipcbiAqIFRoZSBjbGFzcyBmb3IgZGlmZmVyZW50IHJlcG9zaXRvcnkgcHJvdmlkZXJzXG4gKi9cbmV4cG9ydCBjbGFzcyBDbG9uZVJlcG9zaXRvcnkge1xuICAvKipcbiAgICogaW1wb3J0IHJlcG9zaXRvcnkgdG8gY2xvdWQ5IGVudmlyb25tZW50IGZyb20gQVdTIENvZGVDb21taXRcbiAgICpcbiAgICogQHBhcmFtIHJlcG9zaXRvcnkgdGhlIGNvZGVjb21taXQgcmVwb3NpdG9yeSB0byBjbG9uZSBmcm9tXG4gICAqIEBwYXJhbSBwYXRoICB0aGUgdGFyZ2V0IHBhdGggaW4gY2xvdWQ5IGVudmlyb25tZW50XG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21Db2RlQ29tbWl0KHJlcG9zaXRvcnk6IGNvZGVjb21taXQuSVJlcG9zaXRvcnksIHBhdGg6IHN0cmluZyk6IENsb25lUmVwb3NpdG9yeSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHJlcG9zaXRvcnlVcmw6IHJlcG9zaXRvcnkucmVwb3NpdG9yeUNsb25lVXJsSHR0cCxcbiAgICAgIHBhdGhDb21wb25lbnQ6IHBhdGgsXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IHJlcG9zaXRvcnlVcmw6IHN0cmluZywgcHVibGljIHJlYWRvbmx5IHBhdGhDb21wb25lbnQ6IHN0cmluZykge31cbn1cbiJdfQ==