"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const dynamodb_generated_1 = require("./dynamodb.generated");
const replica_provider_1 = require("./replica-provider");
const scalable_table_attribute_1 = require("./scalable-table-attribute");
const HASH_KEY_TYPE = 'HASH';
const RANGE_KEY_TYPE = 'RANGE';
// https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
const MAX_LOCAL_SECONDARY_INDEX_COUNT = 5;
const READ_DATA_ACTIONS = [
    'dynamodb:BatchGetItem',
    'dynamodb:GetRecords',
    'dynamodb:GetShardIterator',
    'dynamodb:Query',
    'dynamodb:GetItem',
    'dynamodb:Scan',
];
const READ_STREAM_DATA_ACTIONS = [
    'dynamodb:DescribeStream',
    'dynamodb:GetRecords',
    'dynamodb:GetShardIterator',
];
const WRITE_DATA_ACTIONS = [
    'dynamodb:BatchWriteItem',
    'dynamodb:PutItem',
    'dynamodb:UpdateItem',
    'dynamodb:DeleteItem',
];
class TableBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.regionalArns = new Array();
    }
    /**
     * Adds an IAM policy statement associated with this table to an IAM
     * principal's policy.
     * @param grantee The principal (no-op if undefined)
     * @param actions The set of actions to allow (i.e. "dynamodb:PutItem", "dynamodb:GetItem", ...)
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [
                this.tableArn,
                core_1.Lazy.stringValue({ produce: () => this.hasIndex ? `${this.tableArn}/index/*` : core_1.Aws.NO_VALUE }),
                ...this.regionalArns,
                ...this.regionalArns.map(arn => core_1.Lazy.stringValue({
                    produce: () => this.hasIndex ? `${arn}/index/*` : core_1.Aws.NO_VALUE,
                })),
            ],
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement associated with this table's stream to an
     * IAM principal's policy.
     * @param grantee The principal (no-op if undefined)
     * @param actions The set of actions to allow (i.e. "dynamodb:DescribeStream", "dynamodb:GetRecords", ...)
     */
    grantStream(grantee, ...actions) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.tableStreamArn],
            scope: this,
        });
    }
    /**
     * Permits an IAM principal all data read operations from this table:
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan.
     * @param grantee The principal to grant access to
     */
    grantReadData(grantee) {
        return this.grant(grantee, ...READ_DATA_ACTIONS);
    }
    /**
     * Permits an IAM Principal to list streams attached to current dynamodb table.
     *
     * @param grantee The principal (no-op if undefined)
     */
    grantTableListStreams(grantee) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: [
                core_1.Lazy.stringValue({ produce: () => `${this.tableArn}/stream/*` }),
                ...this.regionalArns.map(arn => core_1.Lazy.stringValue({ produce: () => `${arn}/stream/*` })),
            ],
        });
    }
    /**
     * Permits an IAM principal all stream data read operations for this
     * table's stream:
     * DescribeStream, GetRecords, GetShardIterator, ListStreams.
     * @param grantee The principal to grant access to
     */
    grantStreamRead(grantee) {
        this.grantTableListStreams(grantee);
        return this.grantStream(grantee, ...READ_STREAM_DATA_ACTIONS);
    }
    /**
     * Permits an IAM principal all data write operations to this table:
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem.
     * @param grantee The principal to grant access to
     */
    grantWriteData(grantee) {
        return this.grant(grantee, ...WRITE_DATA_ACTIONS);
    }
    /**
     * Permits an IAM principal to all data read/write operations to this table.
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan,
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem
     * @param grantee The principal to grant access to
     */
    grantReadWriteData(grantee) {
        return this.grant(grantee, ...READ_DATA_ACTIONS, ...WRITE_DATA_ACTIONS);
    }
    /**
     * Permits all DynamoDB operations ("dynamodb:*") to an IAM principal.
     * @param grantee The principal to grant access to
     */
    grantFullAccess(grantee) {
        return this.grant(grantee, 'dynamodb:*');
    }
    /**
     * Return the given named metric for this Table
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/DynamoDB',
            metricName,
            dimensions: {
                TableName: this.tableName,
            },
            ...props,
        });
    }
    /**
     * Metric for the consumed read capacity units this table
     *
     * @default sum over a minute
     */
    metricConsumedReadCapacityUnits(props) {
        return this.metric('ConsumedReadCapacityUnits', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the consumed write capacity units this table
     *
     * @default sum over a minute
     */
    metricConsumedWriteCapacityUnits(props) {
        return this.metric('ConsumedWriteCapacityUnits', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the system errors this table
     *
     * @default sum over a minute
     */
    metricSystemErrors(props) {
        return this.metric('SystemErrors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the user errors this table
     *
     * @default sum over a minute
     */
    metricUserErrors(props) {
        return this.metric('UserErrors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the conditional check failed requests this table
     *
     * @default sum over a minute
     */
    metricConditionalCheckFailedRequests(props) {
        return this.metric('ConditionalCheckFailedRequests', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the successful request latency this table
     *
     * @default avg over a minute
     */
    metricSuccessfulRequestLatency(props) {
        return this.metric('SuccessfulRequestLatency', { statistic: 'avg', ...props });
    }
}
/**
 * Provides a DynamoDB table.
 */
class Table extends TableBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.tableName,
        });
        this.keySchema = new Array();
        this.attributeDefinitions = new Array();
        this.globalSecondaryIndexes = new Array();
        this.localSecondaryIndexes = new Array();
        this.secondaryIndexNames = new Set();
        this.nonKeyAttributes = new Set();
        this.tableScaling = {};
        this.indexScaling = new Map();
        this.billingMode = props.billingMode || BillingMode.PROVISIONED;
        this.validateProvisioning(props);
        let streamSpecification;
        if (props.replicationRegions) {
            if (props.stream && props.stream !== StreamViewType.NEW_AND_OLD_IMAGES) {
                throw new Error('`stream` must be set to `NEW_AND_OLD_IMAGES` when specifying `replicationRegions`');
            }
            streamSpecification = { streamViewType: StreamViewType.NEW_AND_OLD_IMAGES };
            if (props.billingMode && props.billingMode !== BillingMode.PAY_PER_REQUEST) {
                throw new Error('The `PAY_PER_REQUEST` billing mode must be used when specifying `replicationRegions`');
            }
            this.billingMode = BillingMode.PAY_PER_REQUEST;
        }
        else if (props.stream) {
            streamSpecification = { streamViewType: props.stream };
        }
        this.table = new dynamodb_generated_1.CfnTable(this, 'Resource', {
            tableName: this.physicalName,
            keySchema: this.keySchema,
            attributeDefinitions: this.attributeDefinitions,
            globalSecondaryIndexes: core_1.Lazy.anyValue({ produce: () => this.globalSecondaryIndexes }, { omitEmptyArray: true }),
            localSecondaryIndexes: core_1.Lazy.anyValue({ produce: () => this.localSecondaryIndexes }, { omitEmptyArray: true }),
            pointInTimeRecoverySpecification: props.pointInTimeRecovery ? { pointInTimeRecoveryEnabled: props.pointInTimeRecovery } : undefined,
            billingMode: this.billingMode === BillingMode.PAY_PER_REQUEST ? this.billingMode : undefined,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5,
            },
            sseSpecification: props.serverSideEncryption ? { sseEnabled: props.serverSideEncryption } : undefined,
            streamSpecification,
            timeToLiveSpecification: props.timeToLiveAttribute ? { attributeName: props.timeToLiveAttribute, enabled: true } : undefined,
        });
        this.table.applyRemovalPolicy(props.removalPolicy);
        if (props.tableName) {
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.tableName);
        }
        this.tableArn = this.getResourceArnAttribute(this.table.attrArn, {
            service: 'dynamodb',
            resource: 'table',
            resourceName: this.physicalName,
        });
        this.tableName = this.getResourceNameAttribute(this.table.ref);
        this.tableStreamArn = streamSpecification ? this.table.attrStreamArn : undefined;
        this.scalingRole = this.makeScalingRole();
        this.addKey(props.partitionKey, HASH_KEY_TYPE);
        this.tablePartitionKey = props.partitionKey;
        if (props.sortKey) {
            this.addKey(props.sortKey, RANGE_KEY_TYPE);
            this.tableSortKey = props.sortKey;
        }
        if (props.replicationRegions) {
            this.createReplicaTables(props.replicationRegions);
        }
    }
    /**
     * Permits an IAM Principal to list all DynamoDB Streams.
     * @deprecated Use {@link #grantTableListStreams} for more granular permission
     * @param grantee The principal (no-op if undefined)
     */
    static grantListStreams(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: ['*'],
        });
    }
    /**
     * Creates a Table construct that represents an external table via table name.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableName The table's name.
     */
    static fromTableName(scope, id, tableName) {
        return Table.fromTableAttributes(scope, id, { tableName });
    }
    /**
     * Creates a Table construct that represents an external table via table arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableArn The table's ARN.
     */
    static fromTableArn(scope, id, tableArn) {
        return Table.fromTableAttributes(scope, id, { tableArn });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `TableAttributes` object.
     */
    static fromTableAttributes(scope, id, attrs) {
        class Import extends TableBase {
            constructor(_tableArn, tableName, tableStreamArn) {
                super(scope, id);
                this.tableArn = _tableArn;
                this.tableName = tableName;
                this.tableStreamArn = tableStreamArn;
            }
            get hasIndex() {
                return false;
            }
        }
        let name;
        let arn;
        const stack = core_1.Stack.of(scope);
        if (!attrs.tableName) {
            if (!attrs.tableArn) {
                throw new Error('One of tableName or tableArn is required!');
            }
            arn = attrs.tableArn;
            const maybeTableName = stack.parseArn(attrs.tableArn).resourceName;
            if (!maybeTableName) {
                throw new Error('ARN for DynamoDB table must be in the form: ...');
            }
            name = maybeTableName;
        }
        else {
            if (attrs.tableArn) {
                throw new Error('Only one of tableArn or tableName can be provided');
            }
            name = attrs.tableName;
            arn = stack.formatArn({
                service: 'dynamodb',
                resource: 'table',
                resourceName: attrs.tableName,
            });
        }
        return new Import(arn, name, attrs.tableStreamArn);
    }
    /**
     * Add a global secondary index of table.
     *
     * @param props the property of global secondary index
     */
    addGlobalSecondaryIndex(props) {
        this.validateProvisioning(props);
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const gsiKeySchema = this.buildIndexKeySchema(props.partitionKey, props.sortKey);
        const gsiProjection = this.buildIndexProjection(props);
        this.secondaryIndexNames.add(props.indexName);
        this.globalSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: gsiKeySchema,
            projection: gsiProjection,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5,
            },
        });
        this.indexScaling.set(props.indexName, {});
    }
    /**
     * Add a local secondary index of table.
     *
     * @param props the property of local secondary index
     */
    addLocalSecondaryIndex(props) {
        // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
        if (this.localSecondaryIndexes.length >= MAX_LOCAL_SECONDARY_INDEX_COUNT) {
            throw new RangeError(`a maximum number of local secondary index per table is ${MAX_LOCAL_SECONDARY_INDEX_COUNT}`);
        }
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const lsiKeySchema = this.buildIndexKeySchema(this.tablePartitionKey, props.sortKey);
        const lsiProjection = this.buildIndexProjection(props);
        this.secondaryIndexNames.add(props.indexName);
        this.localSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: lsiKeySchema,
            projection: lsiProjection,
        });
    }
    /**
     * Enable read capacity scaling for this table
     *
     * @returns An object to configure additional AutoScaling settings
     */
    autoScaleReadCapacity(props) {
        if (this.tableScaling.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        return this.tableScaling.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'ReadScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:ReadCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable write capacity scaling for this table
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleWriteCapacity(props) {
        if (this.tableScaling.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        return this.tableScaling.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'WriteScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:WriteCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable read capacity scaling for the given GSI
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleGlobalSecondaryIndexReadCapacity(indexName, props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this index');
        }
        return attributePair.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}ReadScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:ReadCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable write capacity scaling for the given GSI
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleGlobalSecondaryIndexWriteCapacity(indexName, props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this index');
        }
        return attributePair.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}WriteScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:WriteCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Validate the table construct.
     *
     * @returns an array of validation error message
     */
    validate() {
        const errors = new Array();
        if (!this.tablePartitionKey) {
            errors.push('a partition key must be specified');
        }
        if (this.localSecondaryIndexes.length > 0 && !this.tableSortKey) {
            errors.push('a sort key of the table must be specified to add local secondary indexes');
        }
        return errors;
    }
    /**
     * Validate read and write capacity are not specified for on-demand tables (billing mode PAY_PER_REQUEST).
     *
     * @param props read and write capacity properties
     */
    validateProvisioning(props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            if (props.readCapacity !== undefined || props.writeCapacity !== undefined) {
                throw new Error('you cannot provision read and write capacity for a table with PAY_PER_REQUEST billing mode');
            }
        }
    }
    /**
     * Validate index name to check if a duplicate name already exists.
     *
     * @param indexName a name of global or local secondary index
     */
    validateIndexName(indexName) {
        if (this.secondaryIndexNames.has(indexName)) {
            // a duplicate index name causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`a duplicate index name, ${indexName}, is not allowed`);
        }
        this.secondaryIndexNames.add(indexName);
    }
    /**
     * Validate non-key attributes by checking limits within secondary index, which may vary in future.
     *
     * @param nonKeyAttributes a list of non-key attribute names
     */
    validateNonKeyAttributes(nonKeyAttributes) {
        if (this.nonKeyAttributes.size + nonKeyAttributes.length > 20) {
            // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
            throw new RangeError('a maximum number of nonKeyAttributes across all of secondary indexes is 20');
        }
        // store all non-key attributes
        nonKeyAttributes.forEach(att => this.nonKeyAttributes.add(att));
    }
    buildIndexKeySchema(partitionKey, sortKey) {
        this.registerAttribute(partitionKey);
        const indexKeySchema = [
            { attributeName: partitionKey.name, keyType: HASH_KEY_TYPE },
        ];
        if (sortKey) {
            this.registerAttribute(sortKey);
            indexKeySchema.push({ attributeName: sortKey.name, keyType: RANGE_KEY_TYPE });
        }
        return indexKeySchema;
    }
    buildIndexProjection(props) {
        if (props.projectionType === ProjectionType.INCLUDE && !props.nonKeyAttributes) {
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-dynamodb-projectionobject.html
            throw new Error(`non-key attributes should be specified when using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.projectionType !== ProjectionType.INCLUDE && props.nonKeyAttributes) {
            // this combination causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`non-key attributes should not be specified when not using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.nonKeyAttributes) {
            this.validateNonKeyAttributes(props.nonKeyAttributes);
        }
        return {
            projectionType: props.projectionType ? props.projectionType : ProjectionType.ALL,
            nonKeyAttributes: props.nonKeyAttributes ? props.nonKeyAttributes : undefined,
        };
    }
    findKey(keyType) {
        return this.keySchema.find(prop => prop.keyType === keyType);
    }
    addKey(attribute, keyType) {
        const existingProp = this.findKey(keyType);
        if (existingProp) {
            throw new Error(`Unable to set ${attribute.name} as a ${keyType} key, because ${existingProp.attributeName} is a ${keyType} key`);
        }
        this.registerAttribute(attribute);
        this.keySchema.push({
            attributeName: attribute.name,
            keyType,
        });
        return this;
    }
    /**
     * Register the key attribute of table or secondary index to assemble attribute definitions of TableResourceProps.
     *
     * @param attribute the key attribute of table or secondary index
     */
    registerAttribute(attribute) {
        const { name, type } = attribute;
        const existingDef = this.attributeDefinitions.find(def => def.attributeName === name);
        if (existingDef && existingDef.attributeType !== type) {
            throw new Error(`Unable to specify ${name} as ${type} because it was already defined as ${existingDef.attributeType}`);
        }
        if (!existingDef) {
            this.attributeDefinitions.push({
                attributeName: name,
                attributeType: type,
            });
        }
    }
    /**
     * Return the role that will be used for AutoScaling
     */
    makeScalingRole() {
        // Use a Service Linked Role.
        // https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
        return iam.Role.fromRoleArn(this, 'ScalingRole', core_1.Stack.of(this).formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/dynamodb.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_DynamoDBTable',
        }));
    }
    /**
     * Creates replica tables
     *
     * @param regions regions where to create tables
     */
    createReplicaTables(regions) {
        const stack = core_1.Stack.of(this);
        if (!core_1.Token.isUnresolved(stack.region) && regions.includes(stack.region)) {
            throw new Error('`replicationRegions` cannot include the region where this stack is deployed.');
        }
        const provider = replica_provider_1.ReplicaProvider.getOrCreate(this);
        // Documentation at https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2gt_IAM.html
        // is currently incorrect. AWS Support recommends `dynamodb:*` in both source and destination regions
        // Permissions in the source region
        this.grant(provider.onEventHandler, 'dynamodb:*');
        this.grant(provider.isCompleteHandler, 'dynamodb:DescribeTable');
        let previousRegion;
        for (const region of new Set(regions)) { // Remove duplicates
            // Use multiple custom resources because multiple create/delete
            // updates cannot be combined in a single API call.
            const currentRegion = new core_1.CustomResource(this, `Replica${region}`, {
                serviceToken: provider.provider.serviceToken,
                resourceType: 'Custom::DynamoDBReplica',
                properties: {
                    TableName: this.tableName,
                    Region: region,
                },
            });
            // Deploy time check to prevent from creating a replica in the region
            // where this stack is deployed. Only needed for environment agnostic
            // stacks.
            if (core_1.Token.isUnresolved(stack.region)) {
                const createReplica = new core_1.CfnCondition(this, `StackRegionNotEquals${region}`, {
                    expression: core_1.Fn.conditionNot(core_1.Fn.conditionEquals(region, core_1.Aws.REGION)),
                });
                const cfnCustomResource = currentRegion.node.defaultChild;
                cfnCustomResource.cfnOptions.condition = createReplica;
            }
            // Save regional arns for grantXxx() methods
            this.regionalArns.push(stack.formatArn({
                region,
                service: 'dynamodb',
                resource: 'table',
                resourceName: this.tableName,
            }));
            // We need to create/delete regions sequentially because we cannot
            // have multiple table updates at the same time. The `isCompleteHandler`
            // of the provider waits until the replica is in an ACTIVE state.
            if (previousRegion) {
                currentRegion.node.addDependency(previousRegion);
            }
            previousRegion = currentRegion;
        }
        // Permissions in the destination regions (outside of the loop to
        // minimize statements in the policy)
        provider.onEventHandler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['dynamodb:*'],
            resources: this.regionalArns,
        }));
    }
    /**
     * Whether this table has indexes
     */
    get hasIndex() {
        return this.globalSecondaryIndexes.length + this.localSecondaryIndexes.length > 0;
    }
}
exports.Table = Table;
/**
 * Data types for attributes within a table
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes
 */
var AttributeType;
(function (AttributeType) {
    /** Up to 400KiB of binary data (which must be encoded as base64 before sending to DynamoDB) */
    AttributeType["BINARY"] = "B";
    /** Numeric values made of up to 38 digits (positive, negative or zero) */
    AttributeType["NUMBER"] = "N";
    /** Up to 400KiB of UTF-8 encoded text */
    AttributeType["STRING"] = "S";
})(AttributeType = exports.AttributeType || (exports.AttributeType = {}));
/**
 * DyanmoDB's Read/Write capacity modes.
 */
var BillingMode;
(function (BillingMode) {
    /**
     * Pay only for what you use. You don't configure Read/Write capacity units.
     */
    BillingMode["PAY_PER_REQUEST"] = "PAY_PER_REQUEST";
    /**
     * Explicitly specified Read/Write capacity units.
     */
    BillingMode["PROVISIONED"] = "PROVISIONED";
})(BillingMode = exports.BillingMode || (exports.BillingMode = {}));
/**
 * The set of attributes that are projected into the index
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Projection.html
 */
var ProjectionType;
(function (ProjectionType) {
    /** Only the index and primary keys are projected into the index. */
    ProjectionType["KEYS_ONLY"] = "KEYS_ONLY";
    /** Only the specified table attributes are projected into the index. The list of projected attributes is in `nonKeyAttributes`. */
    ProjectionType["INCLUDE"] = "INCLUDE";
    /** All of the table attributes are projected into the index. */
    ProjectionType["ALL"] = "ALL";
})(ProjectionType = exports.ProjectionType || (exports.ProjectionType = {}));
/**
 * When an item in the table is modified, StreamViewType determines what information
 * is written to the stream for this table.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_StreamSpecification.html
 */
var StreamViewType;
(function (StreamViewType) {
    /** The entire item, as it appears after it was modified, is written to the stream. */
    StreamViewType["NEW_IMAGE"] = "NEW_IMAGE";
    /** The entire item, as it appeared before it was modified, is written to the stream. */
    StreamViewType["OLD_IMAGE"] = "OLD_IMAGE";
    /** Both the new and the old item images of the item are written to the stream. */
    StreamViewType["NEW_AND_OLD_IMAGES"] = "NEW_AND_OLD_IMAGES";
    /** Only the key attributes of the modified item are written to the stream. */
    StreamViewType["KEYS_ONLY"] = "KEYS_ONLY";
})(StreamViewType = exports.StreamViewType || (exports.StreamViewType = {}));
//# sourceMappingURL=data:application/json;base64,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