"""
Visualization functions for cohort analysis using Plotly.

This module provides core visualization functions to help analyze and 
interpret cohort data generated by the generate_cohort_data function.
"""

from typing import Optional, Dict, Any, Literal, Union
import pandas as pd
import plotly.express as px
import plotly.graph_objects as go
from plotly.subplots import make_subplots
import numpy as np
from plotly.colors import sequential, diverging, cyclical, qualitative


def plot_cohort_heatmap(
    cohort_data: pd.DataFrame,
    title: Optional[str] = None,
    color_scale: str = "Blues",
    show_values: bool = True,
    value_format: str = ".0f",
    width: int = 800,
    height: int = 600,
    show_colorscale: bool = True,
    reverse_y_axis: bool = True
) -> go.Figure:
    """
    Create a heatmap visualization of cohort data.
    
    This function generates an interactive heatmap using Plotly to visualize cohort analysis 
    results. The heatmap displays cohorts on the y-axis and analysis periods on the x-axis, 
    with color intensity representing the metric values.

    :param cohort_data: Cohort data in pivot format (cohorts as rows, periods as columns)
    :type cohort_data: pd.DataFrame
    :param title: Title for the plot. If None, a default title will be generated, defaults to None
    :type title: Optional[str], optional
    :param color_scale: Color scale for the heatmap. Common options: "Blues", "Viridis", "RdYlBu", "Plasma", defaults to "Blues"
    :type color_scale: str, optional
    :param show_values: Whether to show values on the heatmap cells, defaults to True
    :type show_values: bool, optional
    :param value_format: Format string for displaying values (e.g., ".0f" for integers, ".1f" for 1 decimal), defaults to ".0f"
    :type value_format: str, optional
    :param width: Width of the plot in pixels, defaults to 800
    :type width: int, optional
    :param height: Height of the plot in pixels, defaults to 600
    :type height: int, optional
    :param show_colorscale: Whether to show the color scale bar on the right side of the plot, defaults to True
    :type show_colorscale: bool, optional
    :param reverse_y_axis: Whether to reverse the y-axis order (newer cohorts at top), defaults to True
    :type reverse_y_axis: bool, optional
    :raises ValueError: If input DataFrame is empty or invalid color scale is provided
    :return: Plotly figure object for the heatmap
    :rtype: go.Figure

    Examples::

        >>> cohort_data = generate_cohort_data(df, date_column='date', user_column='user')
        >>> fig = plot_cohort_heatmap(cohort_data, title="User Retention Cohorts")
        >>> fig.show()
    """
    # Validate input
    if cohort_data.empty:
        raise ValueError("Input DataFrame is empty")
    
    # Validate color scale using Plotly's built-in color scales

    # Gather all available color scale names from plotly
    valid_color_scales = (
        list(sequential.__dict__.keys())
        + list(diverging.__dict__.keys())
        + list(cyclical.__dict__.keys())
        + list(qualitative.__dict__.keys())
    )
    # Filter out private and non-list attributes
    valid_color_scales = [
        cs for cs in valid_color_scales
        if not cs.startswith("_") and isinstance(getattr(sequential, cs, None) or getattr(diverging, cs, None) or getattr(cyclical, cs, None) or getattr(qualitative, cs, None), list)
    ]
    # Also allow reversed versions
    valid_color_scales_with_reverse = valid_color_scales + [cs + '_r' for cs in valid_color_scales]
    
    if color_scale.lower() not in [cs.lower() for cs in valid_color_scales_with_reverse]:
        raise ValueError(f"Invalid color scale '{color_scale}'. Must be one of the valid Plotly color scales.")
    
    if title is None:
        title = "Cohort Analysis Heatmap"
    
    # Convert index to string for better display
    cohort_data_display = cohort_data.copy()
    cohort_data_display.index = cohort_data_display.index.astype(str)
    
    # Create heatmap using go.Heatmap for better control
    fig = go.Figure(data=go.Heatmap(
        z=cohort_data_display.values,
        x=[f"Period {i}" for i in cohort_data_display.columns],
        y=cohort_data_display.index,
        colorscale=color_scale,
        showscale=show_colorscale,
        hoverongaps=False,
        text=cohort_data_display.values if show_values else None,
        texttemplate=f"%{{text:{value_format}}}" if show_values else None,
        textfont={"size": 12, "color": "white"}
    ))
    
    # Update layout
    fig.update_layout(
        title=title,
        xaxis_title="Analysis Period",
        yaxis_title="Cohort (Acquisition Period)",
        font=dict(size=12),
        width=width,
        height=height,
        yaxis=dict(autorange='reversed' if reverse_y_axis else True)
    )
    
    return fig


def plot_retention_curves(
    retention_data: pd.DataFrame,
    title: Optional[str] = None,
    show_legend: bool = True,
    width: int = 900,
    height: int = 600,
    max_cohorts: Optional[int] = 10
) -> go.Figure:
    """
    Create line plots showing retention curves for different cohorts.
    
    This function generates an interactive line chart showing how retention rates change 
    over time for different cohorts. Each line represents a cohort, and the x-axis 
    represents analysis periods while the y-axis shows retention rates.

    :param retention_data: Retention rate data in pivot format (should be retention percentages)
    :type retention_data: pd.DataFrame
    :param title: Title for the plot, defaults to None
    :type title: Optional[str], optional
    :param show_legend: Whether to show the legend, defaults to True
    :type show_legend: bool, optional
    :param width: Width of the plot in pixels, defaults to 900
    :type width: int, optional
    :param height: Height of the plot in pixels, defaults to 600
    :type height: int, optional
    :param max_cohorts: Maximum number of cohorts to display (None for all), defaults to 10
    :type max_cohorts: Optional[int], optional
    :return: Plotly figure object for the retention curves
    :rtype: go.Figure

    Examples::

        >>> retention_data = generate_cohort_data(df, date_column='date', user_column='user', 
        ...                                       calculate_retention_rate=True)
        >>> fig = plot_retention_curves(retention_data)
        >>> fig.show()
    """
    if title is None:
        title = "Cohort Retention Curves"
    
    # Limit number of cohorts if specified
    data_to_plot = retention_data.copy()
    if max_cohorts is not None and len(data_to_plot) > max_cohorts:
        data_to_plot = data_to_plot.head(max_cohorts)
    
    fig = go.Figure()
    
    # Add a line for each cohort
    for cohort in data_to_plot.index:
        cohort_str = str(cohort)
        y_values = data_to_plot.loc[cohort].values
        x_values = list(range(len(y_values)))
        
        fig.add_trace(go.Scatter(
            x=x_values,
            y=y_values,
            mode='lines+markers',
            name=f'Cohort {cohort_str}',
            line=dict(width=2),
            marker=dict(size=6)
        ))
    
    # Update layout
    fig.update_layout(
        title=title,
        xaxis_title="Period",
        yaxis_title="Retention Rate (%)",
        showlegend=show_legend,
        width=width,
        height=height,
        hovermode='x unified'
    )
    
    # Add grid
    fig.update_xaxes(showgrid=True, gridwidth=1, gridcolor='lightgray')
    fig.update_yaxes(showgrid=True, gridwidth=1, gridcolor='lightgray')
    
    return fig