"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualGateway = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const gateway_route_1 = require("./gateway-route");
const mesh_1 = require("./mesh");
const utils_1 = require("./private/utils");
const virtual_gateway_listener_1 = require("./virtual-gateway-listener");
class VirtualGatewayBase extends cdk.Resource {
    /**
     * Utility method to add a new GatewayRoute to the VirtualGateway
     */
    addGatewayRoute(id, props) {
        return new gateway_route_1.GatewayRoute(this, id, {
            ...props,
            virtualGateway: this,
        });
    }
    grantStreamAggregatedResources(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['appmesh:StreamAggregatedResources'],
            resourceArns: [this.virtualGatewayArn],
        });
    }
}
/**
 * VirtualGateway represents a newly defined App Mesh Virtual Gateway
 *
 * A virtual gateway allows resources that are outside of your mesh to communicate to resources that
 * are inside of your mesh.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html
 */
class VirtualGateway extends VirtualGatewayBase {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.virtualGatewayName || cdk.Lazy.string({ produce: () => cdk.Names.uniqueId(this) }),
        });
        this.listeners = new Array();
        jsiiDeprecationWarnings._aws_cdk_aws_appmesh_VirtualGatewayProps(props);
        this.mesh = props.mesh;
        if (!props.listeners) {
            // Use listener default of http listener port 8080 if no listener is defined
            this.listeners.push(virtual_gateway_listener_1.VirtualGatewayListener.http().bind(this));
        }
        else {
            props.listeners.forEach(listener => this.listeners.push(listener.bind(this)));
        }
        const accessLogging = (_b = props.accessLog) === null || _b === void 0 ? void 0 : _b.bind(this);
        const node = new appmesh_generated_1.CfnVirtualGateway(this, 'Resource', {
            virtualGatewayName: this.physicalName,
            meshName: this.mesh.meshName,
            meshOwner: utils_1.renderMeshOwner(this.env.account, this.mesh.env.account),
            spec: {
                listeners: this.listeners.map(listener => listener.listener),
                backendDefaults: props.backendDefaults !== undefined
                    ? {
                        clientPolicy: {
                            tls: utils_1.renderTlsClientPolicy(this, (_c = props.backendDefaults) === null || _c === void 0 ? void 0 : _c.tlsClientPolicy),
                        },
                    }
                    : undefined,
                logging: accessLogging !== undefined ? {
                    accessLog: accessLogging.virtualGatewayAccessLog,
                } : undefined,
            },
        });
        this.virtualGatewayName = this.getResourceNameAttribute(node.attrVirtualGatewayName);
        this.virtualGatewayArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualGateway`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualGateway given an ARN
     */
    static fromVirtualGatewayArn(scope, id, virtualGatewayArn) {
        return new class extends VirtualGatewayBase {
            constructor() {
                super(...arguments);
                this.parsedArn = cdk.Fn.split('/', cdk.Stack.of(scope).splitArn(virtualGatewayArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName);
                this.mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', cdk.Fn.select(0, this.parsedArn));
                this.virtualGatewayArn = virtualGatewayArn;
                this.virtualGatewayName = cdk.Fn.select(2, this.parsedArn);
            }
        }(scope, id);
    }
    /**
     * Import an existing VirtualGateway given its attributes
     */
    static fromVirtualGatewayAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_appmesh_VirtualGatewayAttributes(attrs);
        return new class extends VirtualGatewayBase {
            constructor() {
                super(...arguments);
                this.mesh = attrs.mesh;
                this.virtualGatewayName = attrs.virtualGatewayName;
                this.virtualGatewayArn = cdk.Stack.of(this).formatArn({
                    service: 'appmesh',
                    resource: `mesh/${attrs.mesh.meshName}/virtualGateway`,
                    resourceName: this.virtualGatewayName,
                });
            }
        }(scope, id);
    }
}
exports.VirtualGateway = VirtualGateway;
_a = JSII_RTTI_SYMBOL_1;
VirtualGateway[_a] = { fqn: "@aws-cdk/aws-appmesh.VirtualGateway", version: "1.147.0" };
//# sourceMappingURL=data:application/json;base64,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