"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RouteSpec = exports.GrpcRetryEvent = exports.TcpRetryEvent = exports.HttpRetryEvent = exports.HttpRouteProtocol = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const http_route_path_match_1 = require("./http-route-path-match");
const utils_1 = require("./private/utils");
const shared_interfaces_1 = require("./shared-interfaces");
/**
 * Supported :scheme options for HTTP2
 */
var HttpRouteProtocol;
(function (HttpRouteProtocol) {
    /**
     * Match HTTP requests
     */
    HttpRouteProtocol["HTTP"] = "http";
    /**
     * Match HTTPS requests
     */
    HttpRouteProtocol["HTTPS"] = "https";
})(HttpRouteProtocol = exports.HttpRouteProtocol || (exports.HttpRouteProtocol = {}));
/**
 * HTTP events on which to retry.
 */
var HttpRetryEvent;
(function (HttpRetryEvent) {
    /**
     * HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511
     */
    HttpRetryEvent["SERVER_ERROR"] = "server-error";
    /**
     * HTTP status codes 502, 503, and 504
     */
    HttpRetryEvent["GATEWAY_ERROR"] = "gateway-error";
    /**
     * HTTP status code 409
     */
    HttpRetryEvent["CLIENT_ERROR"] = "client-error";
    /**
     * Retry on refused stream
     */
    HttpRetryEvent["STREAM_ERROR"] = "stream-error";
})(HttpRetryEvent = exports.HttpRetryEvent || (exports.HttpRetryEvent = {}));
/**
 * TCP events on which you may retry
 */
var TcpRetryEvent;
(function (TcpRetryEvent) {
    /**
     * A connection error
     */
    TcpRetryEvent["CONNECTION_ERROR"] = "connection-error";
})(TcpRetryEvent = exports.TcpRetryEvent || (exports.TcpRetryEvent = {}));
/**
 * gRPC events
 */
var GrpcRetryEvent;
(function (GrpcRetryEvent) {
    /**
     * Request was cancelled
     *
     * @see https://grpc.github.io/grpc/core/md_doc_statuscodes.html
     */
    GrpcRetryEvent["CANCELLED"] = "cancelled";
    /**
     * The deadline was exceeded
     *
     * @see https://grpc.github.io/grpc/core/md_doc_statuscodes.html
     */
    GrpcRetryEvent["DEADLINE_EXCEEDED"] = "deadline-exceeded";
    /**
     * Internal error
     *
     * @see https://grpc.github.io/grpc/core/md_doc_statuscodes.html
     */
    GrpcRetryEvent["INTERNAL_ERROR"] = "internal";
    /**
     * A resource was exhausted
     *
     * @see https://grpc.github.io/grpc/core/md_doc_statuscodes.html
     */
    GrpcRetryEvent["RESOURCE_EXHAUSTED"] = "resource-exhausted";
    /**
     * The service is unavailable
     *
     * @see https://grpc.github.io/grpc/core/md_doc_statuscodes.html
     */
    GrpcRetryEvent["UNAVAILABLE"] = "unavailable";
})(GrpcRetryEvent = exports.GrpcRetryEvent || (exports.GrpcRetryEvent = {}));
/**
 * Used to generate specs with different protocols for a RouteSpec
 */
class RouteSpec {
    /**
     * Creates an HTTP Based RouteSpec
     */
    static http(options) {
        jsiiDeprecationWarnings._aws_cdk_aws_appmesh_HttpRouteSpecOptions(options);
        return new HttpRouteSpec(options, shared_interfaces_1.Protocol.HTTP);
    }
    /**
     * Creates an HTTP2 Based RouteSpec
     *
     */
    static http2(options) {
        jsiiDeprecationWarnings._aws_cdk_aws_appmesh_HttpRouteSpecOptions(options);
        return new HttpRouteSpec(options, shared_interfaces_1.Protocol.HTTP2);
    }
    /**
     * Creates a TCP Based RouteSpec
     */
    static tcp(options) {
        jsiiDeprecationWarnings._aws_cdk_aws_appmesh_TcpRouteSpecOptions(options);
        return new TcpRouteSpec(options);
    }
    /**
     * Creates a GRPC Based RouteSpec
     */
    static grpc(options) {
        jsiiDeprecationWarnings._aws_cdk_aws_appmesh_GrpcRouteSpecOptions(options);
        return new GrpcRouteSpec(options);
    }
}
exports.RouteSpec = RouteSpec;
_a = JSII_RTTI_SYMBOL_1;
RouteSpec[_a] = { fqn: "@aws-cdk/aws-appmesh.RouteSpec", version: "1.147.0" };
class HttpRouteSpec extends RouteSpec {
    constructor(props, protocol) {
        var _b, _c;
        super();
        this.protocol = protocol;
        this.match = props.match;
        this.weightedTargets = props.weightedTargets;
        this.timeout = props.timeout;
        this.priority = props.priority;
        if (props.retryPolicy) {
            const httpRetryEvents = (_b = props.retryPolicy.httpRetryEvents) !== null && _b !== void 0 ? _b : [];
            const tcpRetryEvents = (_c = props.retryPolicy.tcpRetryEvents) !== null && _c !== void 0 ? _c : [];
            if (httpRetryEvents.length + tcpRetryEvents.length === 0) {
                throw new Error('You must specify one value for at least one of `httpRetryEvents` or `tcpRetryEvents`');
            }
            this.retryPolicy = {
                ...props.retryPolicy,
                httpRetryEvents: httpRetryEvents.length > 0 ? httpRetryEvents : undefined,
                tcpRetryEvents: tcpRetryEvents.length > 0 ? tcpRetryEvents : undefined,
            };
        }
    }
    bind(scope) {
        var _b, _c, _d, _e, _f, _g;
        const pathMatchConfig = ((_c = (_b = this.match) === null || _b === void 0 ? void 0 : _b.path) !== null && _c !== void 0 ? _c : http_route_path_match_1.HttpRoutePathMatch.startsWith('/')).bind(scope);
        // Set prefix path match to '/' if none of path matches are defined.
        const headers = (_d = this.match) === null || _d === void 0 ? void 0 : _d.headers;
        const queryParameters = (_e = this.match) === null || _e === void 0 ? void 0 : _e.queryParameters;
        utils_1.validateHttpMatchArrayLength(headers, queryParameters);
        const httpConfig = {
            action: {
                weightedTargets: renderWeightedTargets(this.weightedTargets),
            },
            match: {
                prefix: pathMatchConfig.prefixPathMatch,
                path: pathMatchConfig.wholePathMatch,
                headers: headers === null || headers === void 0 ? void 0 : headers.map(header => header.bind(scope).headerMatch),
                method: (_f = this.match) === null || _f === void 0 ? void 0 : _f.method,
                scheme: (_g = this.match) === null || _g === void 0 ? void 0 : _g.protocol,
                queryParameters: queryParameters === null || queryParameters === void 0 ? void 0 : queryParameters.map(queryParameter => queryParameter.bind(scope).queryParameterMatch),
            },
            timeout: renderTimeout(this.timeout),
            retryPolicy: this.retryPolicy ? renderHttpRetryPolicy(this.retryPolicy) : undefined,
        };
        return {
            priority: this.priority,
            httpRouteSpec: this.protocol === shared_interfaces_1.Protocol.HTTP ? httpConfig : undefined,
            http2RouteSpec: this.protocol === shared_interfaces_1.Protocol.HTTP2 ? httpConfig : undefined,
        };
    }
}
class TcpRouteSpec extends RouteSpec {
    constructor(props) {
        super();
        this.weightedTargets = props.weightedTargets;
        this.timeout = props.timeout;
        this.priority = props.priority;
    }
    bind(_scope) {
        return {
            priority: this.priority,
            tcpRouteSpec: {
                action: {
                    weightedTargets: renderWeightedTargets(this.weightedTargets),
                },
                timeout: renderTimeout(this.timeout),
            },
        };
    }
}
class GrpcRouteSpec extends RouteSpec {
    constructor(props) {
        var _b, _c, _d;
        super();
        this.weightedTargets = props.weightedTargets;
        this.match = props.match;
        this.timeout = props.timeout;
        this.priority = props.priority;
        if (props.retryPolicy) {
            const grpcRetryEvents = (_b = props.retryPolicy.grpcRetryEvents) !== null && _b !== void 0 ? _b : [];
            const httpRetryEvents = (_c = props.retryPolicy.httpRetryEvents) !== null && _c !== void 0 ? _c : [];
            const tcpRetryEvents = (_d = props.retryPolicy.tcpRetryEvents) !== null && _d !== void 0 ? _d : [];
            if (grpcRetryEvents.length + httpRetryEvents.length + tcpRetryEvents.length === 0) {
                throw new Error('You must specify one value for at least one of `grpcRetryEvents`, `httpRetryEvents` or `tcpRetryEvents`');
            }
            this.retryPolicy = {
                ...props.retryPolicy,
                grpcRetryEvents: grpcRetryEvents.length > 0 ? grpcRetryEvents : undefined,
                httpRetryEvents: httpRetryEvents.length > 0 ? httpRetryEvents : undefined,
                tcpRetryEvents: tcpRetryEvents.length > 0 ? tcpRetryEvents : undefined,
            };
        }
    }
    bind(scope) {
        const serviceName = this.match.serviceName;
        const methodName = this.match.methodName;
        const metadata = this.match.metadata;
        utils_1.validateGrpcRouteMatch(this.match);
        utils_1.validateGrpcMatchArrayLength(metadata);
        if (methodName && !serviceName) {
            throw new Error('If you specify a method name, you must also specify a service name');
        }
        return {
            priority: this.priority,
            grpcRouteSpec: {
                action: {
                    weightedTargets: renderWeightedTargets(this.weightedTargets),
                },
                match: {
                    serviceName: serviceName,
                    methodName: methodName,
                    metadata: metadata === null || metadata === void 0 ? void 0 : metadata.map(singleMetadata => singleMetadata.bind(scope).headerMatch),
                },
                timeout: renderTimeout(this.timeout),
                retryPolicy: this.retryPolicy ? renderGrpcRetryPolicy(this.retryPolicy) : undefined,
            },
        };
    }
}
/**
 * Utility method to add weighted route targets to an existing route
 */
function renderWeightedTargets(weightedTargets) {
    const renderedTargets = [];
    for (const t of weightedTargets) {
        renderedTargets.push({
            virtualNode: t.virtualNode.virtualNodeName,
            weight: t.weight || 1,
        });
    }
    return renderedTargets;
}
/**
 * Utility method to construct a route timeout object
 */
function renderTimeout(timeout) {
    return timeout
        ? {
            idle: (timeout === null || timeout === void 0 ? void 0 : timeout.idle) !== undefined
                ? {
                    unit: 'ms',
                    value: timeout === null || timeout === void 0 ? void 0 : timeout.idle.toMilliseconds(),
                }
                : undefined,
            perRequest: (timeout === null || timeout === void 0 ? void 0 : timeout.perRequest) !== undefined
                ? {
                    unit: 'ms',
                    value: timeout === null || timeout === void 0 ? void 0 : timeout.perRequest.toMilliseconds(),
                }
                : undefined,
        }
        : undefined;
}
function renderHttpRetryPolicy(retryPolicy) {
    return {
        maxRetries: retryPolicy.retryAttempts,
        perRetryTimeout: {
            unit: 'ms',
            value: retryPolicy.retryTimeout.toMilliseconds(),
        },
        httpRetryEvents: retryPolicy.httpRetryEvents,
        tcpRetryEvents: retryPolicy.tcpRetryEvents,
    };
}
function renderGrpcRetryPolicy(retryPolicy) {
    return {
        ...renderHttpRetryPolicy(retryPolicy),
        grpcRetryEvents: retryPolicy.grpcRetryEvents,
    };
}
//# sourceMappingURL=data:application/json;base64,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