"""Tests for the git diff scanner module."""

import os
import subprocess
import sys
from pathlib import Path
from unittest.mock import MagicMock, Mock, patch

import pytest

# Add the src directory to the path to import modules
sys.path.insert(0, os.path.join(os.path.dirname(__file__), "..", "src"))

from adversary_mcp_server.diff_scanner import (
    DiffChunk,
    GitDiffError,
    GitDiffParser,
    GitDiffScanner,
)
from adversary_mcp_server.scan_engine import ScanEngine
from adversary_mcp_server.threat_engine import Language, Severity


class TestDiffChunk:
    """Test cases for DiffChunk class."""

    def test_init(self):
        """Test DiffChunk initialization."""
        chunk = DiffChunk("test.py", 10, 5, 15, 8)
        assert chunk.file_path == "test.py"
        assert chunk.old_start == 10
        assert chunk.old_count == 5
        assert chunk.new_start == 15
        assert chunk.new_count == 8
        assert chunk.added_lines == []
        assert chunk.removed_lines == []
        assert chunk.context_lines == []

    def test_add_line_added(self):
        """Test adding added lines."""
        chunk = DiffChunk("test.py", 10, 5, 15, 8)
        chunk.add_line("+", 16, 'print("new code")')

        assert len(chunk.added_lines) == 1
        assert chunk.added_lines[0] == (16, 'print("new code")')

    def test_add_line_removed(self):
        """Test adding removed lines."""
        chunk = DiffChunk("test.py", 10, 5, 15, 8)
        chunk.add_line("-", 11, 'print("old code")')

        assert len(chunk.removed_lines) == 1
        assert chunk.removed_lines[0] == (11, 'print("old code")')

    def test_add_line_context(self):
        """Test adding context lines."""
        chunk = DiffChunk("test.py", 10, 5, 15, 8)
        chunk.add_line(" ", 12, 'print("context")')

        assert len(chunk.context_lines) == 1
        assert chunk.context_lines[0] == (12, 'print("context")')

    def test_get_changed_code(self):
        """Test getting changed code."""
        chunk = DiffChunk("test.py", 10, 5, 15, 8)
        chunk.add_line(" ", 12, "def function():")
        chunk.add_line("+", 16, '    print("new code")')
        chunk.add_line("-", 11, '    print("old code")')

        changed_code = chunk.get_changed_code()
        assert "def function():" in changed_code
        assert 'print("new code")' in changed_code
        assert 'print("old code")' not in changed_code

    def test_get_added_lines_only(self):
        """Test getting only added lines."""
        chunk = DiffChunk("test.py", 10, 5, 15, 8)
        chunk.add_line(" ", 12, "def function():")
        chunk.add_line("+", 16, '    print("new code")')
        chunk.add_line("+", 17, "    return True")

        added_code = chunk.get_added_lines_only()
        assert 'print("new code")' in added_code
        assert "return True" in added_code
        assert "def function():" not in added_code


class TestGitDiffParser:
    """Test cases for GitDiffParser class."""

    def test_init(self):
        """Test GitDiffParser initialization."""
        parser = GitDiffParser()
        assert parser.diff_header_pattern is not None
        assert parser.chunk_header_pattern is not None
        assert parser.file_header_pattern is not None

    def test_parse_diff_simple(self):
        """Test parsing a simple diff."""
        diff_output = """diff --git a/test.py b/test.py
index 1234567..abcdefg 100644
--- a/test.py
+++ b/test.py
@@ -1,3 +1,4 @@
 def hello():
+    print("new line")
     print("world")
-    return None
+    return True
"""
        parser = GitDiffParser()
        chunks = parser.parse_diff(diff_output)

        assert "test.py" in chunks
        assert len(chunks["test.py"]) == 1

        chunk = chunks["test.py"][0]
        assert chunk.old_start == 1
        assert chunk.new_start == 1
        assert len(chunk.added_lines) == 2
        assert len(chunk.removed_lines) == 1
        assert len(chunk.context_lines) == 2

    def test_parse_diff_multiple_files(self):
        """Test parsing diff with multiple files."""
        diff_output = """diff --git a/file1.py b/file1.py
index 1234567..abcdefg 100644
--- a/file1.py
+++ b/file1.py
@@ -1,2 +1,3 @@
 line1
+new line
 line2
diff --git a/file2.js b/file2.js
index 7890123..defghij 100644
--- a/file2.js
+++ b/file2.js
@@ -1,2 +1,2 @@
-console.log("old");
+console.log("new");
 console.log("same");
"""
        parser = GitDiffParser()
        chunks = parser.parse_diff(diff_output)

        assert "file1.py" in chunks
        assert "file2.js" in chunks
        assert len(chunks["file1.py"]) == 1
        assert len(chunks["file2.js"]) == 1

    def test_parse_diff_empty(self):
        """Test parsing empty diff."""
        parser = GitDiffParser()
        chunks = parser.parse_diff("")

        assert chunks == {}

    def test_parse_diff_no_changes(self):
        """Test parsing diff with no actual changes."""
        diff_output = """diff --git a/test.py b/test.py
index 1234567..abcdefg 100644
--- a/test.py
+++ b/test.py
"""
        parser = GitDiffParser()
        chunks = parser.parse_diff(diff_output)

        assert "test.py" in chunks
        assert len(chunks["test.py"]) == 0


class TestGitDiffScanner:
    """Test cases for GitDiffScanner class."""

    def test_init(self):
        """Test GitDiffScanner initialization."""
        scanner = GitDiffScanner()
        assert scanner.scan_engine is not None
        assert scanner.working_dir == Path.cwd()
        assert scanner.parser is not None

    def test_init_with_params(self):
        """Test GitDiffScanner initialization with parameters."""
        mock_scan_engine = Mock(spec=ScanEngine)
        working_dir = Path("/tmp")

        scanner = GitDiffScanner(scan_engine=mock_scan_engine, working_dir=working_dir)
        assert scanner.scan_engine == mock_scan_engine
        assert scanner.working_dir == working_dir

    def test_detect_language_from_path(self):
        """Test language detection from file paths."""
        scanner = GitDiffScanner()

        assert scanner._detect_language_from_path("test.py") == Language.PYTHON
        assert scanner._detect_language_from_path("test.js") == Language.JAVASCRIPT
        assert scanner._detect_language_from_path("test.jsx") == Language.JAVASCRIPT
        assert scanner._detect_language_from_path("test.ts") == Language.TYPESCRIPT
        assert scanner._detect_language_from_path("test.tsx") == Language.TYPESCRIPT
        assert scanner._detect_language_from_path("test.txt") is None
        assert scanner._detect_language_from_path("README.md") is None

    @patch("subprocess.run")
    def test_run_git_command_success(self, mock_run):
        """Test successful git command execution."""
        mock_run.return_value = Mock(stdout="success output", stderr="")

        scanner = GitDiffScanner()
        result = scanner._run_git_command(["status"])

        assert result == "success output"
        mock_run.assert_called_once_with(
            ["git", "status"],
            cwd=scanner.working_dir,
            capture_output=True,
            text=True,
            check=True,
        )

    @patch("subprocess.run")
    def test_run_git_command_failure(self, mock_run):
        """Test failed git command execution."""
        mock_run.side_effect = subprocess.CalledProcessError(
            1, "git", stderr="error message"
        )

        scanner = GitDiffScanner()
        with pytest.raises(GitDiffError, match="Git command failed: error message"):
            scanner._run_git_command(["status"])

    @patch("subprocess.run")
    def test_run_git_command_not_found(self, mock_run):
        """Test git command not found."""
        mock_run.side_effect = FileNotFoundError()

        scanner = GitDiffScanner()
        with pytest.raises(GitDiffError, match="Git command not found"):
            scanner._run_git_command(["status"])

    @patch("adversary_mcp_server.diff_scanner.GitDiffScanner._run_git_command")
    def test_validate_branches_success(self, mock_run_git):
        """Test successful branch validation."""
        mock_run_git.return_value = "commit_hash"

        scanner = GitDiffScanner()
        scanner._validate_branches("feature", "main")

        assert mock_run_git.call_count == 2
        mock_run_git.assert_any_call(
            ["rev-parse", "--verify", "feature^{commit}"], None
        )
        mock_run_git.assert_any_call(["rev-parse", "--verify", "main^{commit}"], None)

    @patch("adversary_mcp_server.diff_scanner.GitDiffScanner._run_git_command")
    def test_validate_branches_failure(self, mock_run_git):
        """Test branch validation failure."""
        mock_run_git.side_effect = GitDiffError("Branch not found")

        scanner = GitDiffScanner()
        with pytest.raises(GitDiffError, match="Branch validation failed"):
            scanner._validate_branches("feature", "main")

    @patch("adversary_mcp_server.diff_scanner.GitDiffScanner._run_git_command")
    def test_get_diff_changes_success(self, mock_run_git):
        """Test successful diff changes retrieval."""
        diff_output = """diff --git a/test.py b/test.py
index 1234567..abcdefg 100644
--- a/test.py
+++ b/test.py
@@ -1,2 +1,3 @@
 def hello():
+    print("new line")
     print("world")
"""
        mock_run_git.side_effect = ["commit_hash", "commit_hash", diff_output]

        scanner = GitDiffScanner()
        changes = scanner.get_diff_changes("feature", "main")

        assert "test.py" in changes
        assert len(changes["test.py"]) == 1
        chunk = changes["test.py"][0]
        assert len(chunk.added_lines) == 1
        assert chunk.added_lines[0][1] == '    print("new line")'

    @patch("adversary_mcp_server.diff_scanner.GitDiffScanner._run_git_command")
    def test_get_diff_changes_no_diff(self, mock_run_git):
        """Test diff changes with no differences."""
        mock_run_git.side_effect = ["commit_hash", "commit_hash", ""]

        scanner = GitDiffScanner()
        changes = scanner.get_diff_changes("feature", "main")

        assert changes == {}

    @patch("adversary_mcp_server.diff_scanner.GitDiffScanner.get_diff_changes")
    def test_scan_diff_no_changes(self, mock_get_diff):
        """Test scanning diff with no changes."""
        mock_get_diff.return_value = {}

        scanner = GitDiffScanner()
        results = scanner.scan_diff("feature", "main")

        assert results == {}

    @patch("adversary_mcp_server.diff_scanner.GitDiffScanner.get_diff_changes")
    def test_scan_diff_with_unsupported_files(self, mock_get_diff):
        """Test scanning diff with unsupported file types."""
        mock_chunk = Mock()
        mock_chunk.get_added_lines_only.return_value = "some content"
        mock_chunk.added_lines = [(1, "some content")]
        mock_get_diff.return_value = {
            "README.md": [mock_chunk],
            "config.yml": [mock_chunk],
        }

        scanner = GitDiffScanner()
        results = scanner.scan_diff("feature", "main")

        assert results == {}

    @patch("adversary_mcp_server.diff_scanner.GitDiffScanner.get_diff_changes")
    def test_scan_diff_with_supported_files(self, mock_get_diff):
        """Test scanning diff with supported file types."""
        mock_chunk = Mock()
        mock_chunk.get_added_lines_only.return_value = (
            "print('hello')\neval(user_input)"
        )
        mock_chunk.added_lines = [(1, "print('hello')"), (2, "eval(user_input)")]

        mock_get_diff.return_value = {"test.py": [mock_chunk]}

        # Mock the scan engine
        mock_scan_engine = Mock()
        mock_scan_result = Mock()
        mock_scan_result.all_threats = []
        mock_scan_engine.scan_code.return_value = mock_scan_result

        scanner = GitDiffScanner(scan_engine=mock_scan_engine)
        results = scanner.scan_diff("feature", "main")

        assert "test.py" in results
        assert len(results["test.py"]) == 1
        mock_scan_engine.scan_code.assert_called_once()

    @patch("adversary_mcp_server.diff_scanner.GitDiffScanner.get_diff_changes")
    def test_get_diff_summary_success(self, mock_get_diff):
        """Test getting diff summary successfully."""
        mock_chunk1 = Mock()
        mock_chunk1.added_lines = [(1, "line1"), (2, "line2")]
        mock_chunk1.removed_lines = [(1, "old_line")]

        mock_chunk2 = Mock()
        mock_chunk2.added_lines = [(1, "new_line")]
        mock_chunk2.removed_lines = []

        mock_chunk3 = Mock()
        mock_chunk3.added_lines = [(1, "readme_line")]
        mock_chunk3.removed_lines = []

        mock_get_diff.return_value = {
            "test.py": [mock_chunk1],
            "script.js": [mock_chunk2],
            "README.md": [mock_chunk3],  # Unsupported
        }

        scanner = GitDiffScanner()
        summary = scanner.get_diff_summary("feature", "main")

        assert summary["source_branch"] == "feature"
        assert summary["target_branch"] == "main"
        assert summary["total_files_changed"] == 3
        assert summary["supported_files"] == 2
        assert summary["total_chunks"] == 3
        assert summary["lines_added"] == 4  # 2 + 1 + 1 = 4
        assert summary["lines_removed"] == 1
        assert "test.py" in summary["scannable_files"]
        assert "script.js" in summary["scannable_files"]
        assert "README.md" not in summary["scannable_files"]

    @patch("adversary_mcp_server.diff_scanner.GitDiffScanner.get_diff_changes")
    def test_get_diff_summary_git_error(self, mock_get_diff):
        """Test getting diff summary with git error."""
        mock_get_diff.side_effect = GitDiffError("Git error")

        scanner = GitDiffScanner()
        summary = scanner.get_diff_summary("feature", "main")

        assert summary["source_branch"] == "feature"
        assert summary["target_branch"] == "main"
        assert "error" in summary
        assert summary["error"] == "Failed to get diff summary"


class TestGitDiffScannerIntegration:
    """Integration tests for GitDiffScanner."""

    def test_full_scan_workflow(self):
        """Test the full scanning workflow with mocked components."""
        # Mock git diff output
        diff_output = """diff --git a/test.py b/test.py
index 1234567..abcdefg 100644
--- a/test.py
+++ b/test.py
@@ -1,3 +1,4 @@
 def vulnerable_function():
+    eval(user_input)
     print("hello")
     return True
"""

        # Mock scan engine and results
        mock_scan_engine = Mock()
        mock_threat = Mock()
        mock_threat.line_number = 2
        mock_threat.rule_id = "eval_usage"
        mock_scan_result = Mock()
        mock_scan_result.all_threats = [mock_threat]
        mock_scan_engine.scan_code.return_value = mock_scan_result

        scanner = GitDiffScanner(scan_engine=mock_scan_engine)

        # Mock the git commands
        with patch.object(scanner, "_run_git_command") as mock_run_git:
            mock_run_git.side_effect = [
                "commit_hash",  # validate source branch
                "commit_hash",  # validate target branch
                diff_output,  # get diff
            ]

            results = scanner.scan_diff("feature", "main", use_llm=False)

            assert "test.py" in results
            assert len(results["test.py"]) == 1
            scan_result = results["test.py"][0]
            assert len(scan_result.all_threats) == 1

            # Verify the scan was called with the changed code
            mock_scan_engine.scan_code.assert_called_once()
            call_args = mock_scan_engine.scan_code.call_args
            assert "eval(user_input)" in call_args[1]["source_code"]
            assert call_args[1]["file_path"] == "test.py"
            assert call_args[1]["language"] == Language.PYTHON

    def test_scan_with_severity_filtering(self):
        """Test scanning with severity threshold filtering."""
        mock_scan_engine = Mock()
        mock_scan_result = Mock()
        mock_scan_result.all_threats = []
        mock_scan_engine.scan_code.return_value = mock_scan_result

        scanner = GitDiffScanner(scan_engine=mock_scan_engine)

        with patch.object(scanner, "get_diff_changes") as mock_get_diff:
            mock_chunk = Mock()
            mock_chunk.get_added_lines_only.return_value = "print('hello')"
            mock_chunk.added_lines = [(1, "print('hello')")]
            mock_get_diff.return_value = {"test.py": [mock_chunk]}

            results = scanner.scan_diff(
                "feature", "main", severity_threshold=Severity.HIGH
            )

            # Verify severity threshold was passed
            mock_scan_engine.scan_code.assert_called_once()
            call_args = mock_scan_engine.scan_code.call_args
            assert call_args[1]["severity_threshold"] == Severity.HIGH

    def test_scan_with_llm_enabled(self):
        """Test scanning with LLM analysis enabled."""
        mock_scan_engine = Mock()
        mock_scan_result = Mock()
        mock_scan_result.all_threats = []
        mock_scan_engine.scan_code.return_value = mock_scan_result

        scanner = GitDiffScanner(scan_engine=mock_scan_engine)

        with patch.object(scanner, "get_diff_changes") as mock_get_diff:
            mock_chunk = Mock()
            mock_chunk.get_added_lines_only.return_value = "print('hello')"
            mock_chunk.added_lines = [(1, "print('hello')")]
            mock_get_diff.return_value = {"test.py": [mock_chunk]}

            results = scanner.scan_diff("feature", "main", use_llm=True)

            # Verify use_llm was passed
            mock_scan_engine.scan_code.assert_called_once()
            call_args = mock_scan_engine.scan_code.call_args
            assert call_args[1]["use_llm"] is True


class TestGitDiffScannerEdgeCases:
    """Test edge cases and error conditions."""

    def test_scan_with_empty_chunks(self):
        """Test scanning with empty diff chunks."""
        scanner = GitDiffScanner()

        with patch.object(scanner, "get_diff_changes") as mock_get_diff:
            mock_chunk = Mock()
            mock_chunk.get_added_lines_only.return_value = ""  # Empty
            mock_chunk.added_lines = []  # Empty list
            mock_get_diff.return_value = {"test.py": [mock_chunk]}

            results = scanner.scan_diff("feature", "main")

            assert results == {}

    def test_scan_with_scan_engine_error(self):
        """Test scanning when scan engine throws error."""
        mock_scan_engine = Mock()
        mock_scan_engine.scan_code.side_effect = Exception("Scan failed")

        scanner = GitDiffScanner(scan_engine=mock_scan_engine)

        with patch.object(scanner, "get_diff_changes") as mock_get_diff:
            mock_chunk = Mock()
            mock_chunk.get_added_lines_only.return_value = "print('hello')"
            mock_chunk.added_lines = [(1, "print('hello')")]
            mock_get_diff.return_value = {"test.py": [mock_chunk]}

            # Should not raise, but should log error and continue
            results = scanner.scan_diff("feature", "main")

            # Should return empty results for the failed file
            assert results == {}

    def test_line_number_mapping(self):
        """Test that line numbers are correctly mapped."""
        mock_scan_engine = Mock()
        mock_threat = Mock()
        mock_threat.line_number = 2  # Line in combined code
        mock_scan_result = Mock()
        mock_scan_result.all_threats = [mock_threat]
        mock_scan_engine.scan_code.return_value = mock_scan_result

        scanner = GitDiffScanner(scan_engine=mock_scan_engine)

        with patch.object(scanner, "get_diff_changes") as mock_get_diff:
            mock_chunk = Mock()
            mock_chunk.get_added_lines_only.return_value = "line1\nline2\nline3"
            mock_chunk.added_lines = [
                (10, "line1"),
                (11, "line2"),
                (12, "line3"),
            ]  # Original line numbers
            mock_get_diff.return_value = {"test.py": [mock_chunk]}

            results = scanner.scan_diff("feature", "main")

            # Line number should be mapped correctly
            # Note: The exact mapping depends on the implementation
            # This test verifies the mapping logic is called
            assert "test.py" in results
            assert len(results["test.py"]) == 1
