# Copyright 2021 Okera Inc. All Rights Reserved.
#
# Collect of data gen utilities
#
from enum import Enum
import os
import pytest

from okera.tests import pycerebro_test_common as common

# List of datasets/test configurations that are standard. This can be easily
# loaded per test case.
#
# Test cases should be authorized like this:
#
# def test_case():
#   create_test_data.require(create_test_data.DataHarness.BIG_CATALOG)
#   # Use big catalog
#
# The testdata is loaded once on demand so that running a single perf test
# is still fast.
class DataHarness(Enum):
    LARGE_DB_WIDE_TABLES = 1

# Map of test harness enums to impls
TEST_DATA_HARNESS_FUNCTIONS = {}

def require(perf, conn, ds, db=None):
    if perf.skip_data_load():
        print("\nSkipping data load.")
        return db

    globals = {}
    key = (ds, db)
    # Use globals to store loaded test data across tests
    if hasattr(pytest, 'okera_globals'):
        globals = pytest.okera_globals
    else:
        pytest.okera_globals = globals

    if 'loaded_test_data' not in globals:
        globals['loaded_test_data'] = {}

    if key in globals['loaded_test_data']:
        print('Dataset "%s" already loaded.' % key)
        return globals['loaded_test_data'][key]

    if ds not in TEST_DATA_HARNESS_FUNCTIONS:
        raise ValueError('Unknown dataset: %s' % ds)

    print('Loading dataset: %s...' % ds)
    fn = TEST_DATA_HARNESS_FUNCTIONS[ds]
    db = fn(conn, db)
    globals['loaded_test_data'][key] = db
    print('...loaded dataset')
    return db

def create_large_db_wide_tables_attributes(conn, db):
    sql = '''
    CREATE EXTERNAL TABLE %s.%s (
      valid_event_for_session INT,
      segment_user_id STRING,
      parsed_ip_area_code INT,
      properties_end_user_email STRING,
      properties_tag_id BIGINT,
      properties_old_state STRING,
      properties_topics_in_all_others_section_count BIGINT,
      properties_sentiment_score BIGINT,
      properties_media_type STRING,
      parsed_ip_city STRING,
      properties_filter_name STRING,
      properties_team_member_name STRING,
      parsed_ip_country_name STRING,
      properties_install_order_volume BIGINT,
      event_created_at STRING,
      properties_card_header STRING,
      properties_is_invalid_pinterest_token BIGINT,
      properties_phase BIGINT,
      properties_search_medium STRING,
      properties_is_has_reviews_tab BOOLEAN,
      properties_error_text STRING,
      properties_is_has_star_rating BOOLEAN,
      properties_is3_stars_checked BOOLEAN,
      properties_old_plarform STRING,
      properties_form_field_lead_type_c STRING,
      properties_feature_update_source STRING,
      properties_updated_signature STRING,
      properties_updated_background_color STRING,
      properties_anonymized_email_count BIGINT,
      _metadata_unbundled ARRAY<STRING>,
      days_from_purchase_changed STRING,
      properties_topics_shown BIGINT,
      properties_url STRING,
      properties_post_published STRING,
      properties_form_field_lead_source STRING,
      properties_attributes_score BIGINT,
      properties_top_positive_sentences ARRAY<
          STRUCT<product_name:STRING,review_id:BIGINT,score:BIGINT,sentence:STRING>>,
      properties_end_user_id BIGINT,
      properties_year BIGINT,
      properties_moderated_text_is_product_album BOOLEAN,
      properties_plan_name STRING,
      properties_updated_header_text_color STRING,
      properties_updated_reviews_type STRING,
      properties_admin BOOLEAN,
      name STRING,
      properties_locked_menu_name STRING,
      properties_is_has_sent_mai BOOLEAN,
      properties_account_selected_points BIGINT,
      properties_updated_map_state BOOLEAN,
      properties_is_enabled_coupons BOOLEAN,
      properties_import_status STRING,
      properties_integration_type STRING,
      properties_end_user_name STRING,
      parsed_ip_postal_code STRING,
      properties_comment_body STRING,
      properties_picture_id STRING,
      properties_verified_file STRING,
      properties_store_domain STRING,
      channel STRING,
      event_sent_to_segment_at STRING,
      properties_media_source STRING,
      properties_is_first_invite BOOLEAN,
      properties_updated_header_customization_checkbox BOOLEAN,
      properties_is_invalid_facebook_token_date STRING,
      properties_link_count BIGINT,
      properties_reviewer_first_letter STRING,
      properties_days_invalid_facebook_token BIGINT,
      properties_is2_stars_checked BOOLEAN,
      properties_days_invalid_twitter_token BIGINT,
      properties_clicked_update_npsscore BIGINT,
      properties_neutral_opinions_count BIGINT,
      properties_yotpo_product_score_v2 BIGINT,
      context_campaign_cokuchen20silicon20form20tent STRING,
      properties_url_for_verified STRING,
      properties_chargify_revenue BIGINT,
      properties_interaction_id BIGINT,
      properties_is_email_template_subject_save_succeeded BOOLEAN,
      properties_sentiment_change STRING,
      properties_is_widget_font_updated BOOLEAN,
      properties_updated_subject STRING,
      properties_invited_count BIGINT,
      is_out_of_date_range INT,
      properties_updated_stars_color STRING,
      properties_role STRING,
      context_integration_name STRING,
      properties_is_first_album BOOLEAN,
      context_campaign_sq STRING,
      properties_product_id STRING,
      properties_added_to_albums STRING,
      properties_name_on_credit_card STRING,
      properties_reviewer_type STRING,
      properties_collection_name STRING,
      properties_media_id STRING,
      properties_breakdown_by STRING,
      properties_is_has_convert_to_site_review BOOLEAN,
      properties_search_title STRING,
      context_campaign_name STRING,
      properties_days_in_dunning BIGINT,
      properties_cta_title STRING,
      properties_org_name STRING,
      properties_sentiment_filter_selection BOOLEAN,
      properties_from_date STRING,
      properties_signup_utmterm STRING,
      properties_is_invalid_twitter_token_date STRING,
      properties__locked_feature_name STRING,
      properties_review_body STRING,
      properties_order_date STRING,
      properties__deleted_users ARRAY<STRING>,
      properties_form_field_mkto_person_notes STRING,
      context_campaign_content STRING,
      properties_thirty_day_order_volume BIGINT,
      properties_to_date STRING,
      parsed_user_agent_browser_major STRING,
      properties_post_author STRING,
      properties_cta_url STRING,
      properties_item_name STRING,
      properties_media_sub_source STRING,
      properties_is_reviews_widget_installed BOOLEAN,
      properties_next_charge_date STRING,
      properties_updated_ctabutton_text STRING,
      properties_is_invalid_order_id BOOLEAN,
      properties_filter_location STRING,
      properties_is_has_tpr BOOLEAN,
      properties_review_star_count BIGINT,
      properties_last_conversion_order_time STRING,
      event STRING,
      properties_tpr_upload_count BIGINT,
      properties_failed_reviews BIGINT,
      properties_form_fieldreferrer_token STRING,
      context_campaign_term STRING,
      properties_filter_type STRING,
      properties_tag_name STRING,
      parsed_ip_region_name STRING,
      properties_multiple_products_interval BIGINT,
      parsed_ip_region STRING,
      properties_opinion_sentiment STRING,
      properties_mandatory_count BIGINT,
      properties_number_orders STRING,
      properties_tag_location STRING,
      properties_source STRING,
      properties_item_id STRING,
      properties_customer_points BIGINT,
      properties_batch_action BOOLEAN,
      properties_platform_plan STRING,
      properties_is_enabled_promoted_products_email BOOLEAN,
      properties_email_id STRING,
      properties_search_text STRING,
      properties_is_enabled_map BOOLEAN,
      properties_uninstall_source STRING,
      properties_report_type STRING,
      properties_is_receive_review_notifications_star4 BOOLEAN,
      properties_reviews_type STRING,
      properties_campaign_id BIGINT,
      page_referrer STRING,
      properties_campaign_name STRING,
      properties_is1_star_checked BOOLEAN,
      properties_signup_utmsource STRING,
      properties_button_clicked BOOLEAN,
      properties_updated_send_after_amount BIGINT,
      properties_user_plan_name STRING,
      properties_is_has_sent_map BOOLEAN,
      properties_is_product_enablement BOOLEAN,
      properties_sorting_content STRING,
      properties_failure_reason STRING,
      context_page_title STRING,
      properties_search_type STRING,
      properties_is_enabled_custom_reviews BOOLEAN,
      properties_updated_body STRING,
      parsed_user_agent_browser_patch STRING,
      properties_is_receive_review_notifications_star5 BOOLEAN,
      properties_email_template_version_id STRING,
      properties_new_package_id BIGINT,
      parsed_user_agent_browser_minor STRING,
      properties_ui_version STRING,
      properties_star_rating_filter_selection STRING,
      properties_is_subject_updated BOOLEAN,
      properties_signup_utmcampaign STRING,
      properties_product_with_reviews_count BIGINT,
      properties_is4_stars_checked BOOLEAN,
      properties_is_email_templates_reset_succeeded BOOLEAN,
      properties_signup_utmcontent STRING,
      properties_failure_message STRING,
      properties_agent_id STRING,
      properties_updated_testimonial_link_url STRING,
      properties_tab_name STRING,
      parsed_ip_latitude FLOAT,
      properties_phone STRING,
      properties_total_search_generated_purchases_products ARRAY<
          STRUCT<name:STRING,total_search_generated_purchases:BIGINT>>,
      properties_search_keyword STRING,
      properties_search_phase STRING,
      properties_ad_id BIGINT,
      properties_ad_network STRING,
      properties_review_submission STRING,
      properties_is_body_updated BOOLEAN,
      properties_is_has_reviews_carousel BOOLEAN,
      properties_is_has_custom_reviews BOOLEAN,
      properties_product_name STRING,
      properties_platform STRING,
      properties_updated_text_size BIGINT,
      properties_maps BIGINT,
      properties_recurring_payment_interval BIGINT,
      properties_top_overall_conversion_of_searching_users ARRAY<
            STRUCT<name:STRING,overall_conversion_of_searching_users:DOUBLE>>,
      properties_hashtag_count BIGINT,
      properties_moderation_action STRING,
      properties_video_host STRING,
      properties_products_app_id BIGINT,
      properties_billing_provider STRING,
      original_timestamp STRING,
      properties_download_link STRING,
      properties_total_searches_in_domain_key_products ARRAY<
            STRUCT<name:STRING,total_searches_in_domain_key:BIGINT>>,
      properties_referrer STRING,
      properties_url_for_errors STRING,
      properties_cta_type STRING,
      properties_form_field_segment_id_c STRING,
      properties_moderation_source STRING,
      properties_is_invalid_twitter_token BIGINT,
      properties_date_range STRING,
      properties_syndicated_inherit BOOLEAN,
      properties_shopify_plan STRING,
      properties_review_product_name STRING,
      properties_updated_show_navigation_arrows_checkbox BOOLEAN,
      properties_is_has_social_push BOOLEAN,
      properties_cta_text STRING,
      properties_send_to_email STRING ATTRIBUTE misc.guid,
      properties_removed_from_albums STRING,
      properties_is_email_template_content_changed BOOLEAN,
      event_name STRING,
      properties_intercom_user_hash STRING,
      page_path STRING,
      properties_is_online BOOLEAN,
      properties_amount_of_words BIGINT,
      properties_order_amount STRING,
      properties_traits_email STRING ATTRIBUTE misc.guid,
      properties_conversion_type STRING,
      properties_status STRING,
      properties_is_receive_review_notifications BOOLEAN,
      page_title STRING,
      context_page_path STRING,
      properties_is_receive_system_notifications BOOLEAN,
      properties_error STRING,
      properties_closed_question_count BIGINT,
      properties_is_questions_and_answers_subscribed BOOLEAN,
      properties_is_free_text_profanity BOOLEAN,
      properties_section STRING,
      properties_signup_employee_count STRING,
      properties_products_shown BIGINT,
      properties_step STRING,
      properties_configuration_action STRING,
      properties_review_type STRING,
      properties_meeting_time BIGINT,
      parsed_ip_dma_code INT,
      properties_exception_name STRING,
      properties_sidebar_open BOOLEAN,
      properties_insights_api_end_point STRING,
      properties_package_categories STRING,
      properties_is_has_pla BOOLEAN,
      properties_last_name STRING,
      properties_state_name STRING,
      properties_number_reviews BIGINT,
      properties_product_star_rating DOUBLE,
      properties_form_field_email STRING,
      properties_from_product STRING,
      properties_post_words STRING,
      properties_is_email_template_subject_changed BOOLEAN,
      app_key STRING,
      properties_duration BIGINT,
      parsed_ip_timezone STRING,
      properties_developer_email STRING,
      properties_start_date STRING,
      properties_lpg_action_value STRING,
      properties_send_after BIGINT,
      properties_new_platform STRING,
      properties_signup_country STRING,
      properties_email_template_version_order BIGINT,
      properties_new_admin_email STRING,
      properties_agent_name STRING,
      properties_total_products_count BIGINT,
      properties_form_field_subscribed_to_blog_c STRING,
      properties_is_has_rich_snippets BOOLEAN,
      properties_sort_by STRING,
      properties_text STRING,
      properties_path STRING,
      properties_uploader_name STRING,
      properties_results_per_page BIGINT,
      properties_is_invalid_facebook_token BIGINT,
      properties_tag_domain_key STRING,
      properties_map_type STRING,
      properties_is_has_sent_tsr BOOLEAN,
      properties_revenue DOUBLE,
      properties_product STRING,
      properties_is_tab_position_updated BOOLEAN,
      context_ip STRING,
      properties_signup_utmmedium STRING,
      properties_topic_name STRING,
      properties_cta_location STRING,
      properties_form_fieldform_sfdccampaign STRING,
      properties_is_mapcustom_fields_enabled BOOLEAN,
      properties_conversation_link STRING,
      properties_is_invalid_pinterest_token_date STRING,
      properties_updated_primary_color STRING,
      properties_form_field_utm_content_c STRING,
      properties_is_explicit_review BOOLEAN,
      properties_old_package STRING,
      properties_external_order_id STRING,
      properties_facebook_spend STRING,
      properties_interaction_response ARRAY<STRUCT<fieldId:STRING,fieldRequired:BOOLEAN,fieldType:STRING,label:STRING,value:STRING>>,
      properties_is_has_new_main_widget_layout BOOLEAN,
      properties_is_sentiment_filter BOOLEAN,
      user_id STRING,
      parsed_user_agent_device_brand STRING,
      properties_invited_user_email STRING,
      properties_form_fieldsubscribedto_email_course STRING,
      properties_updated_show_total_count_checkbox BOOLEAN,
      properties_video_title STRING,
      properties_reviewer_name STRING,
      properties_app_id STRING,
      properties_updated_widget_font STRING,
      review_count BIGINT,
      properties_moderated_text_selected_albums ARRAY<STRING>,
      properties_old_package_id BIGINT,
      properties_email STRING ATTRIBUTE misc.guid,
      properties_birthday_selected_points BIGINT,
      properties_package STRING,
      properties_is_has_sent_mas BOOLEAN,
      properties_is_has_badge BOOLEAN,
      properties_search_location STRING,
      properties_email_campaign_id BIGINT,
      properties_is_has_tsr BOOLEAN,
      properties_charge_price STRING,
      properties_is_has_comments BOOLEAN,
      properties_team_member_id BIGINT,
      properties_is_newsletter_subscribed BOOLEAN,
      properties_is_include_product_photo BOOLEAN,
      properties_is_has_css_editor BOOLEAN,
      properties_updated_promoted_products_title_check_box BOOLEAN,
      properties_existing_baseline_version STRING,
      properties_profanity_filter_selection BOOLEAN,
      properties_is_auto_publish_enabled BOOLEAN,
      properties_new_state STRING,
      properties_album_id STRING,
      properties_message_id STRING,
      parsed_user_agent_device_model STRING,
      properties_gross_margin DOUBLE,
      properties_map_state BOOLEAN,
      properties_expiration_period_in_hours BIGINT,
      properties_is_enabled_promoted_products_widget BOOLEAN,
      properties_pct_reached DOUBLE,
      properties_is_has_dedicated_page BOOLEAN,
      properties_picture_url ARRAY<STRING>,
      properties_user_days_to_renewal BIGINT,
      properties_pixel_version BIGINT,
      properties_form_field_comments_c STRING,
      properties_visitors STRING,
      properties_is_downgrade_reason BOOLEAN,
      properties_filter_text STRING,
      properties_updated_review_submission STRING,
      properties_is_primary_color_updated BOOLEAN,
      properties_reviewer_email STRING ATTRIBUTE misc.guid,
      properties_meeting_duration BIGINT,
      properties_updated_testimonial_link_text STRING,
      properties_imported_reviews BIGINT,
      properties_form_fieldform_category STRING,
      parsed_user_agent_os_patch STRING,
      properties_is_using_packages_service BOOLEAN,
      _metadata_bundled ARRAY<STRING>,
      properties_title STRING,
      properties_social_publish_link STRING,
      properties_store_count BIGINT,
      properties_updated_post_title_text_length BIGINT,
      properties_top_negative_period ARRAY<STRUCT<product_name:STRING,review_id:BIGINT,score:BIGINT,sentence:STRING>>,
      message_id STRING,
      properties_updated_ctabutton_text_check_box BOOLEAN,
      properties_action STRING,
      properties_tsr_upload_count BIGINT,
      properties_total_errors BIGINT,
      properties_bottom_overall_conversion_of_searching_users ARRAY<STRUCT<name:STRING,overall_conversion_of_searching_users:BIGINT>>,
      properties_card_name STRING,
      properties_invited_user_id BIGINT,
      properties_form_field_domain_c STRING,
      properties_new_admin_name STRING,
      properties_album_name STRING,
      properties_category STRING,
      properties_email_template_key STRING,
      properties_tag_source STRING,
      properties_image_id STRING,
      properties_activation_url STRING,
      properties_send_to_email_address STRING,
      properties_comment_id BIGINT,
      properties_lpg_action_type STRING,
      properties_name STRING,
      properties_negative_topics ARRAY<STRUCT<mentions_count:BIGINT,rank:BIGINT,sentiment_score:BIGINT,topic:STRING>>,
      properties_image_ids ARRAY<STRUCT<external_id:STRING,source:STRING>>,
      properties_has_pos BOOLEAN,
      properties_moderated_text_removed_albums ARRAY<STRING>,
      properties_neutral_reviews_count BIGINT,
      properties_invited_user_name STRING,
      properties_facebook_ad_account_id STRING,
      properties_updated_autoplay_speed_checkbox BOOLEAN,
      properties_conversation_duration BIGINT,
      properties_is_public BOOLEAN,
      properties_message_body STRING,
      properties_questions_count BIGINT,
      properties_is_has_qn_a BOOLEAN,
      properties_page_number BIGINT,
      properties_has_referrer_first_name_variable BOOLEAN,
      properties_is_receive_review_notifications_star2 BOOLEAN,
      properties_is_review_notifications_subscribed BOOLEAN,
      properties_count_orders BIGINT,
      properties_cta_name STRING,
      properties_is_enabled_rich_snippets BOOLEAN,
      properties_is_email_rendering_succeeded BOOLEAN,
      properties_is5_stars_checked BOOLEAN,
      properties_rad_prefill_email STRING,
      properties_email_subject STRING,
      properties_hashtag STRING,
      properties_is_deleted_user BOOLEAN,
      properties_new_package STRING,
      properties_is_has_shoppable_instagram BOOLEAN,
      properties_is_has_promoted_products_widget BOOLEAN,
      properties__rrs_sent_in_billing_cycle BIGINT,
      properties_new_package_monthly_price BIGINT,
      properties_reminder_id BIGINT,
      properties_website_publish_link STRING,
      properties_agent_username STRING,
      properties_search_phrase STRING,
      properties_yotpo_product_score BIGINT,
      anonymous_id STRING,
      event_type STRING,
      properties_page_category STRING,
      properties_app_name STRING,
      properties_top_negative_sentences ARRAY<STRUCT<product_name:STRING,review_id:BIGINT,score:BIGINT,sentence:STRING>>,
      properties_default_image_used BOOLEAN,
      properties_domain STRING,
      properties_is_enabled_pla BOOLEAN,
      properties_result_type STRING,
      properties_signup_monthly_orders_count STRING,
      properties_total_count BIGINT,
      properties_override_date STRING,
      properties_thirty_day_revenue_cents BIGINT,
      properties_account_id BIGINT,
      properties_is_email_templates_reset_confirmed BOOLEAN,
      properties_is_cross_sell BOOLEAN,
      properties_is_customer_tab_clicked BOOLEAN,
      properties_search_term STRING,
      properties_card_id BIGINT,
      properties_is_multi_product BOOLEAN,
      parsed_user_agent_os_major STRING,
      properties_positive_topics ARRAY<STRUCT<mentions_count:BIGINT,rank:BIGINT,sentiment_score:BIGINT,topic:STRING>>,
      properties_topics_in_favorites_section_count BIGINT,
      properties_social_network STRING,
      properties_is_invalid_order_amount BOOLEAN,
      properties__rrmonthly_limit BIGINT,
      context_page_url STRING,
      properties_popup_name STRING,
      properties_traits_name STRING,
      properties_updated_autoplay_speed BIGINT,
      properties_favorite_action BOOLEAN,
      properties_widget_visible BOOLEAN,
      properties_end_date STRING,
      properties_form_field_utm_term_c STRING,
      properties_negative_opinions_count BIGINT,
      properties_is_all_reviews STRING,
      context_page_referrer STRING,
      properties_album_picture_count BIGINT,
      properties_form_fieldform_name STRING,
      context_campaign_cokuchcake20silicon20molden20silicon20form20tent STRING,
      properties_gallery_id STRING,
      properties_days_to_renewal BIGINT,
      properties_is_receive_product_didnt_arrive_notification BOOLEAN,
      properties_is_users_count_limit_reached BOOLEAN,
      properties_is_has_gsr BOOLEAN,
      properties_is_agency BOOLEAN,
      properties_original_source STRING,
      properties_post_comments STRING,
      properties_current_app STRING,
      properties_sentiment_value DOUBLE,
      properties_updated_link_to_testimonial_page_checkbox BOOLEAN,
      properties_package_id BIGINT,
      properties_org_key STRING,
      properties_friend_discount BIGINT,
      properties_is_receive_review_notifications_star3 BOOLEAN,
      properties_instance_id BIGINT,
      properties_open_text_question_count BIGINT,
      parsed_user_agent_os_family STRING,
      properties_group_id STRING,
      properties_traits_plan STRING,
      properties_pictures_count BIGINT,
      properties_publish_everywhere_link STRING,
      properties_is_referrals_selected BOOLEAN,
      properties__package_extensions STRING,
      properties_updated_number_of_reviews BIGINT,
      properties_subscription_state STRING,
      properties_website STRING,
      properties_form_fieldutm_source_c STRING,
      properties_dest_app STRING,
      properties_invalid_reason STRING,
      properties_multiple_products_review_request_logic STRING,
      properties_credit_card_type STRING,
      properties_number_of_reviews BIGINT,
      properties_country_name STRING,
      parsed_ip_longitude FLOAT,
      properties_creation_location STRING,
      properties_referral_code STRING,
      properties_emails_sent BIGINT,
      properties_insights_api_error STRING,
      properties_is_friend_tab_clicked BOOLEAN,
      properties_form_fieldhashtag STRING,
      properties_topics_count BIGINT,
      properties_org_admin_email STRING,
      properties_amount_of_products BIGINT,
      context_page_search STRING,
      properties_is_has_coupons BOOLEAN,
      properties_yotpo_ip STRING,
      properties_negative_reviews_count BIGINT,
      properties_errors_file STRING,
      properties_is_receive_receive_newsletters BOOLEAN,
      properties_marketo_form_id STRING,
      properties_is_system_notifications_subscribed BOOLEAN,
      properties_gallery_name STRING,
      properties_is_receive_review_notifications_star1 BOOLEAN,
      properties_chargify_balance BIGINT,
      properties_is_email_template_content_save_succeeded BOOLEAN,
      properties_industry_average STRING,
      properties_is_receive_sentiment_notification BOOLEAN,
      properties_meeting_time_zone STRING,
      properties_updated_automatic_frequency_drop_down STRING,
      properties_emails_attempted BIGINT,
      properties_orders STRING,
      context_campaign_medium STRING,
      properties_new_baseline_version STRING,
      properties_first_name STRING,
      properties_csv_path STRING,
      properties_review_moderation_link STRING,
      properties_error_message STRING,
      properties_is_product_tag_missing BOOLEAN,
      properties_is_signature_updated BOOLEAN,
      properties_layout STRING,
      properties_updated_background_color_checkbox BOOLEAN,
      parsed_user_agent_device_family STRING,
      received_at STRING,
      properties_view STRING,
      properties_feature STRING,
      page_search STRING,
      context_campaign_source STRING,
      properties_pending_rrs BIGINT,
      properties_charge_id BIGINT,
      properties_updated_star_rating_color_check_box BOOLEAN,
      properties_form_field_gclid_c STRING,
      properties_is_pull_past_orders_enabled BOOLEAN,
      properties_old_instance_version STRING,
      properties_is_stars_color_updated BOOLEAN,
      context_campaign_contepowerreviewsnt STRING,
      properties_is_checked BIGINT,
      properties_search_results_count BIGINT,
      properties_positive_opinions_count BIGINT,
      properties_updated_post_title_text STRING,
      properties_attributes_feedback STRING,
      properties_utm_campaign STRING,
      properties_link_expiration_period_in_hours BIGINT,
      properties_url_for_skipped STRING,
      properties_coupon_notification_type STRING,
      properties_updated_ctabutton_color STRING,
      properties_end_anonymous_user_id BIGINT,
      context_campaign_source_c STRING,
      properties_review_title STRING,
      properties_pays_via STRING,
      properties_google_spend STRING,
      properties_updated_star_rating_color STRING,
      properties_is_classic_editor BOOLEAN,
      context_campaign_campaign_c STRING,
      properties_updated_header_text STRING,
      properties_is_invited_user_store_restricted BOOLEAN,
      properties_sentences STRING,
      properties_is_has_sent_site_reminder BOOLEAN,
      properties_created_at STRING,
      properties_content_type_filter_selection STRING,
      properties_sub_status_filter_selection STRING,
      properties_min_star_rating BIGINT,
      properties_positive_reviews_count BIGINT,
      properties_watched_pct STRING,
      context_library_version STRING,
      properties_amount_of_products_in_file BIGINT,
      properties_updated_only_my_instagram_photos_check_box BOOLEAN,
      properties_blog_category STRING,
      properties_is_has_upload_button BOOLEAN,
      properties_multiple_products_max_emails_amount BIGINT,
      properties_review_product_url STRING,
      properties_reviewer_id BIGINT,
      properties_form_fieldutm_medium_c STRING,
      properties_avg_distinct_queries_in_domain_key_products ARRAY<STRUCT<avg_distinct_queries_in_domain_key:BIGINT,name:STRING>>,
      properties_is_spam_filter_checkbox_checked STRING,
      properties_opinions STRING,
      properties_order_currency STRING,
      properties_view_by STRING,
      properties_updated_alignment STRING,
      properties_review_product_id BIGINT,
      context_library_name STRING,
      properties_star_rating BIGINT,
      properties_marketo_form_loaded STRING,
      properties_review_source STRING,
      properties_old_pacakge_monthly_price BIGINT,
      properties_form_fieldsubscribedto_shopify_plus_email_course STRING,
      properties_to_product STRING,
      properties_updated_page_title_check_box BOOLEAN,
      context_integration_version STRING,
      properties_is_has_personal_info BOOLEAN,
      parsed_ip_country_code STRING,
      properties_is_enabled_gsr BOOLEAN,
      properties_new_instance_version STRING,
      properties_form_field_last_name STRING,
      properties_max_orders BIGINT,
      parsed_user_agent_os_minor STRING,
      properties_user_renewal_date STRING,
      properties_cta_is_open BOOLEAN,
      properties_subject STRING,
      properties_question_id STRING,
      properties_source_name STRING,
      properties_is_skip_email BOOLEAN,
      properties_error_name STRING,
      properties_form_fieldutm_campaign_c STRING,
      properties_company_name STRING,
      page_url STRING,
      properties_form_field_phone STRING,
      properties_is_has_sent_tpr BOOLEAN,
      properties_signature STRING,
      properties_month BIGINT,
      properties_summary STRING,
      properties_is_secondary_color_updated BOOLEAN,
      properties_is_synced_gallery_enabled BOOLEAN,
      properties_tag_type STRING,
      properties_orders_count BIGINT,
      properties_last_locked_feature STRING,
      context_campaign_medium_c STRING,
      properties_status_filter_selection STRING,
      properties_review_id BIGINT,
      properties_phase_name STRING,
      properties_merchant_id BIGINT,
      properties_conversation_id STRING,
      properties_search STRING,
      properties_is_receive_profanity_notification BOOLEAN,
      properties_new_admin_id BIGINT,
      properties_form_field_first_name STRING,
      properties_quick_filter_selection STRING,
      properties_auto_publish_enabled BOOLEAN,
      properties_is_enabled_qn_a BOOLEAN,
      properties_feed_type STRING,
      properties_updated_post_frequency_check_box STRING,
      properties_num_of_points BIGINT,
      properties_admin_name STRING,
      properties_send_after_amount BIGINT,
      properties_purchase_selected_points BIGINT,
      properties_minimum_opinions BIGINT,
      properties_app_developer_name STRING,
      properties_top_positive_period ARRAY<STRUCT<product_name:STRING,review_id:BIGINT,score:BIGINT,sentence:STRING>>,
      properties_days_invalid_pinterest_token BIGINT,
      parsed_ip_metro_code INT,
      parsed_user_agent_browser_family STRING,
      properties_is_has_promoted_products_email BOOLEAN,
      properties_products_count STRING,
      properties_order_id STRING,
      properties_updated_page_title_text STRING,
      properties_credit_card_last4_digits STRING,
      properties_updated_selecting_reviews STRING,
      properties_object_type STRING,
      properties_item_sentiment_score BIGINT,
      properties_moderated_text_allow_remove_from_shoppable_instagram BOOLEAN,
      context_user_agent STRING,
      properties_renewal_date STRING,
      properties_form_field_title STRING,
      session_id STRING
    )
    PARTITIONED BY (
      year INT,
      month INT,
      day INT
    )
    WITH SERDEPROPERTIES ('path'='s3://yotpo-data-lake/warehouse/segment/full/last', 'serialization.format'='1')
    STORED AS PARQUET
    LOCATION 's3a://yotpo-data-lake/warehouse/segment/full/last';
    '''
    if not db:
      db = 'perf_test_db_many_wide_tables'
    NUM_TABLES = 25
    conn.execute_ddl('DROP DATABASE IF EXISTS %s CASCADE' % db)
    conn.execute_ddl('CREATE DATABASE %s' % db)

    conn.create_attribute('misc', 'guid')
    conn.delete_attribute(db, 'email')
    conn.create_attribute(db, 'email')
    conn.delete_attribute(db, 'dummy_tag')
    conn.create_attribute(db, 'dummy_tag')

    for i in range(NUM_TABLES):
        tbl = 'tbl_' + str(i)
        conn.execute_ddl(sql % (db, tbl))
        conn.assign_attribute(db, 'email', db, tbl,
                              'properties_send_to_email_address')
        conn.assign_attribute(db, 'email', db, tbl,
                              'properties_traits_email')
        conn.assign_attribute(db, 'email', db, tbl,
                              'properties_email')
        conn.assign_attribute(db, 'email', db, tbl,
                              'properties_reviewer_email')
    return db

TEST_DATA_HARNESS_FUNCTIONS[DataHarness.LARGE_DB_WIDE_TABLES] =\
    create_large_db_wide_tables_attributes

def create_xilinx_tbl1(conn, db, tbl1, role, testuser):
      flash_view_wtd_path = os.path.join(
          os.path.dirname(os.path.realpath(__file__)),
          'assets', 'xilinx_flash_view_wtd.sql')

      ddls = [
          "DROP DATABASE IF EXISTS %s CASCADE" % (db),
          "CREATE DATABASE %s" % (db),
          common.read_sql_file(flash_view_wtd_path, db=db, tbl=tbl1),

          "create attribute if not exists flash.cogs_bklg_cra",
          "create attribute if not exists flash.cogs_bklg_si",
          "create attribute if not exists flash.cogs_bklg_st",
          "create attribute if not exists flash.cogs_hist_cra",
          "create attribute if not exists flash.cogs_hist_si",
          "create attribute if not exists flash.cogs_hist_st",
          "create attribute if not exists flash.dimension",
          "create attribute if not exists flash.measure",
          "create attribute if not exists flash.qty_bklg_cra",
          "create attribute if not exists flash.qty_bklg_si",
          "create attribute if not exists flash.qty_bklg_st",
          "create attribute if not exists flash.qty_hist_cra",
          "create attribute if not exists flash.qty_hist_si",
          "create attribute if not exists flash.qty_hist_st",
          "create attribute if not exists flash.resale_bklg_st",
          "create attribute if not exists flash.resale_hist_st",
          "create attribute if not exists flash.revenue_bklg_cra",
          "create attribute if not exists flash.revenue_bklg_si",
          "create attribute if not exists flash.revenue_bklg_st",
          "create attribute if not exists flash.revenue_hist_cra",
          "create attribute if not exists flash.revenue_hist_si",
          "create attribute if not exists flash.revenue_hist_st",
          "create attribute if not exists flash.revenue_cover_st",
          "create attribute if not exists flash.asp_hist_st",
          "create attribute if not exists flash.asp_hist_si",
          "create attribute if not exists flash.prod_margin_bklg_st",
          "create attribute if not exists pii.person",

          "DROP ROLE IF EXISTS %s" % (role),
          "CREATE ROLE %s WITH GROUPS %s" % (role, testuser),

          """GRANT SELECT ON DATABASE `%s`
          HAVING ATTRIBUTE IN (`flash`.`cogs_bklg_cra`, `flash`.`cogs_bklg_si`, `flash`.`cogs_bklg_st`, `flash`.`cogs_hist_cra`,  `flash`.`cogs_hist_si`, `flash`.`cogs_hist_st`, `flash`.`qty_bklg_cra`, `flash`.`qty_bklg_si`, `flash`.`qty_bklg_st`, `flash`.`qty_hist_cra`, `flash`.`qty_hist_si`, `flash`.`qty_hist_st`, `flash`.`revenue_bklg_cra`, `flash`.`revenue_bklg_si`, `flash`.`revenue_bklg_st`,`flash`.`resale_bklg_st`, `flash`.`revenue_hist_cra`, `flash`.`revenue_hist_si`, `flash`.`revenue_hist_st`, `flash`.`resale_hist_st`, `flash`.`prod_margin_bklg_st`, `flash`.`asp_hist_st`, `flash`.`asp_hist_si`, `flash`.`revenue_cover_st`)
          WHERE ((in_set('ALL', user_attribute('customer_tier')) OR in_set(customer_tier, user_attribute('customer_tier')))
          AND (in_set('ALL', user_attribute('external_markets')) OR in_set(external_markets, user_attribute('external_markets')))
          AND (in_set('ALL', user_attribute('external_markets_alias')) OR in_set(external_markets_alias, user_attribute('external_markets_alias')))
          AND (in_set('ALL', user_attribute('internal_reporting_markets')) OR in_set(internal_reporting_markets, user_attribute('internal_reporting_markets')))
          AND (in_set('ALL', user_attribute('primary_end_markets')) OR in_set(primary_end_markets, user_attribute('primary_end_markets')))
          -- AND (in_set('ALL', user_attribute('prim_sec_end_markets')) OR in_set(primary_secondary_combo, user_attribute('prim_sec_end_markets')))
          AND (in_set('ALL', user_attribute('sub_seg_level1')) OR in_set(sub_seg_level1, user_attribute('sub_seg_level1')))
          AND (in_set('ALL', user_attribute('channel_geography')) OR in_set(channel_geography, user_attribute('channel_geography')))
          AND (in_set('ALL', user_attribute('channel_hier3')) OR in_set(channel_hier3, user_attribute('channel_hier3')))
          AND (in_set('ALL', user_attribute('customer_xu')) OR in_set(customer_xu, user_attribute('customer_xu')))
          AND (in_set('ALL', user_attribute('rep_name')) OR in_set(rep_name, user_attribute('rep_name')))
          AND (in_set('ALL', user_attribute('territory')) OR in_set(territory, user_attribute('territory')))
          -- AND (in_set('ALL', user_attribute('sector')) OR in_set(sector, user_attribute('sector')))
          -- AND (in_set('ALL', user_attribute('area')) OR in_set(area, user_attribute('area')))
          -- AND (in_set('ALL', user_attribute('internal_product_line')) OR in_set(internal_product_line, user_attribute('internal_product_line')))
          )
          TO ROLE `%s`""" % (db, role),

          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE fiscal_quarter  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE fiscal_quarter_org  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE fiscal_qtr_std  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE year_quarter    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE year_quarter_week   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtr_alias   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE week_alias  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE year_alias  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE fiscal_year flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE channel_distribution_type   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE channel_direct_type flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE channel_hier3   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE channel_geography   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE channel_geo_hier2   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE channel_distributor_name    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE part    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE extnd_product_dppg  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE extnd_product_device    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE extnd_product_family    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE extnd_product_sub_group flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE extnd_product_sub_group_alias   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE extnd_product_group flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE extnd_product_group_alias   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE extended_product_header flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE geometry    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE internal_product_dppg   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE internal_product_group  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE internal_product_family flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE internal_product_line   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE internal_product_division   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE internal_product_roll_up    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE part_external   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE part_pkg_pin    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE part_grade  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE part_speed  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE part_logic_cells    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE part_node   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE primary_end_markets flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE primary_secondary_combo flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE secondary_end_markets   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE external_markets    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE external_markets_alias  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE ext_mkt_level1  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE primary_end_market_alias    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE internal_reporting_markets  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE int_mkt_level1  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE sub_seg_level1  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE sub_segments    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE territory   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE sector  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE area    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE region  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE rep_name    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE rep_alias   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE default_territory   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE default_sector  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE default_area    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE default_region  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE default_rep_name    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE default_rep_alias   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE customer_tier   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE customer_tier_detail    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE customer_xu flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE customer_gu flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE hub_party   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE ship_customer_tier  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE ship_customer_tier_detail   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE ship_customer_xu    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE ship_customer_gu    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE ship_hub_party  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE scenario    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE record_type flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE demand_creation_category    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE demand_creation_category_alias  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE demand_creation_type    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE business_type   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_actual_curr_qtr_backlog flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_actual_curr_qtr_backlog flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_actual_curr_qtr_backlog flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_actual_delinquency flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_allocated_curr_qtr_backlog  flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_allocated_curr_qtr_backlog  flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_allocated_curr_qtr_backlog  flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_allocated_delinquency  flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_1_curr_qtr_backlog  flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_2_curr_qtr_backlog  flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_3_curr_qtr_backlog  flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_csd    flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_curr_qtr_delinquency  flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_cfd    flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtr_plus2_bklg_rev  flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_actual_revenue  flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_actual_revenue  flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_actual_revenue  flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_revenue_adjustments flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_revenue_adjustments flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_revenue_adjustments flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_allocated_revenue   flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_allocated_revenue   flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_allocated_revenue   flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_1_revenue   flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_2_revenue   flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_3_revenue   flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtd_revenue flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_revenue flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_actual_curr_qtr_backlog_qty flash.measure flash.qty_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_allocated_curr_qtr_backlog_qty  flash.measure flash.qty_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_actual_curr_qtr_backlog_qty flash.measure flash.qty_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_allocated_curr_qtr_backlog_qty  flash.measure flash.qty_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_actual_curr_qtr_backlog_qty flash.measure flash.qty_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_allocated_curr_qtr_backlog_qty  flash.measure flash.qty_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_actual_delinquency_qty flash.measure flash.qty_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_allocated_delinquency_qty  flash.measure flash.qty_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_1_curr_qtr_backlog_qty  flash.measure flash.qty_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_2_curr_qtr_backlog_qty  flash.measure flash.qty_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_3_curr_qtr_backlog_qty  flash.measure flash.qty_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_csd_qty    flash.measure flash.qty_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_curr_qtr_delinquency_qty  flash.measure flash.qty_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_qty    flash.measure flash.qty_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_actual_curr_qtr_backlog_cogs    flash.cogs_bklg_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_allocated_curr_qtr_backlog_cogs flash.cogs_bklg_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_actual_curr_qtr_backlog_cogs    flash.cogs_bklg_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_allocated_curr_qtr_backlog_cogs flash.cogs_bklg_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_actual_curr_qtr_backlog_cogs    flash.cogs_bklg_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_allocated_curr_qtr_backlog_cogs flash.cogs_bklg_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_actual_delinquency_cogs    flash.cogs_bklg_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_allocated_delinquency_cogs flash.cogs_bklg_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_1_curr_qtr_backlog_cogs flash.cogs_bklg_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_2_curr_qtr_backlog_cogs flash.cogs_bklg_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_3_curr_qtr_backlog_cogs flash.cogs_bklg_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_csd_cogs   flash.cogs_bklg_st flash.measure pii.person" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_curr_qtr_delinquency_cogs flash.cogs_bklg_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_cogs   flash.cogs_bklg_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_actual_qty  flash.measure flash.qty_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_allocated_qty   flash.measure flash.qty_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_adjustments_qty flash.measure flash.qty_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_actual_qty  flash.measure flash.qty_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_allocated_qty   flash.measure flash.qty_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_adjustments_qty flash.measure flash.qty_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_actual_qty  flash.measure flash.qty_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_allocated_qty   flash.measure flash.qty_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_adjustments_qty flash.measure flash.qty_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_1_qty   flash.measure flash.qty_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_2_qty   flash.measure flash.qty_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_3_qty   flash.measure flash.qty_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtd_qty flash.measure flash.qty_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_qty flash.measure flash.qty_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_actual_cogs flash.cogs_hist_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_allocated_cogs  flash.cogs_hist_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_adjustments_cogs    flash.cogs_hist_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_actual_cogs flash.cogs_hist_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_allocated_cogs  flash.cogs_hist_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_adjustments_cogs    flash.cogs_hist_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_actual_cogs flash.cogs_hist_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_allocated_cogs  flash.cogs_hist_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_adjustments_cogs    flash.cogs_hist_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_1_cogs  flash.cogs_hist_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_2_cogs  flash.cogs_hist_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_3_cogs  flash.cogs_hist_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtd_cogs    flash.cogs_hist_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_cogs    flash.cogs_hist_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtr_plus2_bklg_qty  flash.measure flash.qty_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtr_plus2_bklg_cogs flash.cogs_bklg_st flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_actual_curr_qtr_backlog_cra flash.measure flash.revenue_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_actual_curr_qtr_backlog_cra flash.measure flash.revenue_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_actual_curr_qtr_backlog_cra flash.measure flash.revenue_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtr_plus2_bklg_rev_cra  flash.measure flash.revenue_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_allocated_curr_qtr_backlog_cra  flash.measure flash.revenue_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_allocated_curr_qtr_backlog_cra  flash.measure flash.revenue_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_allocated_curr_qtr_backlog_cra  flash.measure flash.revenue_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_1_curr_qtr_backlog_cra  flash.measure flash.revenue_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_2_curr_qtr_backlog_cra  flash.measure flash.revenue_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_3_curr_qtr_backlog_cra  flash.measure flash.revenue_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_csd_cra    flash.measure flash.revenue_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_cra    flash.measure flash.revenue_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_actual_revenue_cra  flash.measure flash.revenue_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_actual_revenue_cra  flash.measure flash.revenue_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_actual_revenue_cra  flash.measure flash.revenue_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_revenue_adjustments_cra flash.measure flash.revenue_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_revenue_adjustments_cra flash.measure flash.revenue_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_revenue_adjustments_cra flash.measure flash.revenue_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_1_revenue_cra   flash.measure flash.revenue_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_2_revenue_cra   flash.measure flash.revenue_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_3_revenue_cra   flash.measure flash.revenue_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtd_revenue_cra flash.measure flash.revenue_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_revenue_cra flash.measure flash.revenue_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_sell_in    flash.measure flash.revenue_bklg_si" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtd_revenue_sell_in flash.measure flash.revenue_hist_si" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_sell_in flash.measure flash.revenue_hist_si" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE actual_curr_qtr_backlog flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE allocated_curr_qtr_backlog  flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_actual_and_allocation  flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE adjustment  flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_actual_curr_qtr_backlog_qty_cra flash.measure flash.qty_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_actual_curr_qtr_backlog_qty_cra flash.measure flash.qty_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_actual_curr_qtr_backlog_qty_cra flash.measure flash.qty_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtr_plus2_bklg_qty_cra  flash.measure flash.qty_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_allocated_curr_qtr_backlog_qty_cra  flash.measure flash.qty_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_allocated_curr_qtr_backlog_qty_cra  flash.measure flash.qty_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_allocated_curr_qtr_backlog_qty_cra  flash.measure flash.qty_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_1_curr_qtr_backlog_qty_cra  flash.measure flash.qty_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_2_curr_qtr_backlog_qty_cra  flash.measure flash.qty_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_3_curr_qtr_backlog_qty_cra  flash.measure flash.qty_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_csd_qty_cra    flash.measure flash.qty_bklg_cra pii.person" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_qty_cra    flash.measure flash.qty_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_actual_qty_cra  flash.measure flash.qty_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_actual_qty_cra  flash.measure flash.qty_hist_cra pii.person" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_actual_qty_cra  flash.measure flash.qty_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_adjustments_qty_cra flash.measure flash.qty_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_adjustments_qty_cra flash.measure flash.qty_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_adjustments_qty_cra flash.measure flash.qty_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_1_qty_cra   flash.measure flash.qty_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_2_qty_cra   flash.measure flash.qty_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_3_qty_cra   flash.measure flash.qty_hist_cra pii.person" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtd_qty_cra flash.measure flash.qty_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_qty_cra flash.measure flash.qty_hist_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_qty_sell_in    flash.measure flash.qty_bklg_si" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtd_qty_sell_in flash.measure flash.qty_hist_si" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_qty_sell_in flash.measure flash.qty_hist_si" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_actual_curr_qtr_backlog_cogs_cra    flash.cogs_bklg_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_actual_curr_qtr_backlog_cogs_cra    flash.cogs_bklg_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_actual_curr_qtr_backlog_cogs_cra    flash.cogs_bklg_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtr_plus2_bklg_cogs_cra flash.cogs_bklg_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_allocated_curr_qtr_backlog_cogs_cra flash.cogs_bklg_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_allocated_curr_qtr_backlog_cogs_cra flash.cogs_bklg_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_allocated_curr_qtr_backlog_cogs_cra flash.cogs_bklg_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_1_curr_qtr_backlog_cogs_cra flash.cogs_bklg_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_2_curr_qtr_backlog_cogs_cra flash.cogs_bklg_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_3_curr_qtr_backlog_cogs_cra flash.cogs_bklg_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_csd_cogs_cra   flash.cogs_bklg_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_cogs_cra   flash.cogs_bklg_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_actual_cogs_cra flash.cogs_hist_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_actual_cogs_cra flash.cogs_hist_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_actual_cogs_cra flash.cogs_hist_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_adjustments_cogs_cra    flash.cogs_hist_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_adjustments_cogs_cra    flash.cogs_hist_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_adjustments_cogs_cra    flash.cogs_hist_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_1_cogs_cra  flash.cogs_hist_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_2_cogs_cra  flash.cogs_hist_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_3_cogs_cra  flash.cogs_hist_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtd_cogs_cra    flash.cogs_hist_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_cogs_cra    flash.cogs_hist_cra flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_cogs_sell_in   flash.cogs_bklg_si flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtd_cogs_sell_in    flash.cogs_hist_si flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_cogs_sell_in    flash.cogs_hist_si flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_actual_curr_qtr_backlog_resale  flash.measure flash.resale_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_actual_curr_qtr_backlog_resale  flash.measure flash.resale_bklg_st pii.person" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_actual_curr_qtr_backlog_resale  flash.measure flash.resale_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_actual_delinquency_resale  flash.measure flash.resale_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_allocated_curr_qtr_backlog_resale   flash.measure flash.resale_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_allocated_curr_qtr_backlog_resale   flash.measure flash.resale_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_allocated_curr_qtr_backlog_resale   flash.measure flash.resale_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_allocated_delinquency_resale   flash.measure flash.resale_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_1_curr_qtr_backlog_resale   flash.measure flash.resale_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_2_curr_qtr_backlog_resale   flash.measure flash.resale_bklg_st pii.person" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_3_curr_qtr_backlog_resale   flash.measure flash.resale_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_resale_csd flash.measure flash.resale_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_curr_qtr_delinquency_resale   flash.measure flash.resale_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE curr_qtr_backlog_resale flash.measure flash.resale_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtr_plus2_bklg_res  flash.measure" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_actual_resale   flash.measure flash.resale_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_actual_resale   flash.measure flash.resale_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_actual_resale   flash.measure flash.resale_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_resale_adjustments  flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_resale_adjustments  flash.measure flash.revenue_hist_st pii.person" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_resale_adjustments  flash.measure flash.revenue_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_1_allocated_resale    flash.measure flash.resale_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_2_allocated_resale    flash.measure flash.resale_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE month_3_allocated_resale    flash.measure flash.resale_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_1_resale    flash.measure flash.resale_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_2_resale    flash.measure flash.resale_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE total_month_3_resale    flash.measure flash.resale_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE qtd_resale  flash.measure flash.resale_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_resale  flash.measure flash.resale_hist_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_revenue_qtr_ago flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_revenue_cra_qtr_ago flash.measure flash.revenue_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_sell_in_qtr_ago flash.measure flash.revenue_bklg_si" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_revenue_qtr_ago_turns   flash.measure flash.revenue_bklg_st" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_revenue_cra_qtr_ago_turns   flash.measure flash.revenue_bklg_cra" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flash_total_sell_in_qtr_ago_turns   flash.measure flash.revenue_bklg_si" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE dm_time_key flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE dm_part_key flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE dm_geography_key    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE dm_channel_key  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE dm_cust_key_disti   flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE dm_cust_key_end flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE dm_city_key flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE dm_cust_key_end_xcm flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE dm_city_key_xcm flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE dm_sales_hierarchy_key  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE dm_mkt_seg_key  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE dm_default_sales_hierarchy_key  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE dm_cust_key_ship_xcm    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE dm_business_type_key    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE dm_demand_creation_key  flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE flag_14_week    flash.dimension" % (db, tbl1),
          "ALTER TABLE %s.%s ADD COLUMN ATTRIBUTE week_no flash.dimension" % (db, tbl1),
      ]

      for ddl in ddls:
          conn.execute_ddl(ddl)


def create_xilinx_tbl2(conn, db, tbl, role, testuser):
      flash_view_wtd_path = os.path.join(
          os.path.dirname(os.path.realpath(__file__)),
          'assets', 'xilinx_flash_view2.sql')

      ddls = [
            "DROP DATABASE IF EXISTS %s CASCADE" % (db),
            "CREATE DATABASE %s" % (db),
            common.read_sql_file(flash_view_wtd_path, db=db, tbl=tbl),

            "create attribute if not exists catalog.approved",
            "create attribute if not exists catalog.dimension",
            "create attribute if not exists catalog.measure",
            "create attribute if not exists fin.cogs_bklg_cra",
            "create attribute if not exists fin.cogs_bklg_si",
            "create attribute if not exists fin.cogs_bklg_st",
            "create attribute if not exists fin.cogs_cover_si",
            "create attribute if not exists fin.cogs_cover_st",
            "create attribute if not exists fin.cogs_hist_cra",
            "create attribute if not exists fin.cogs_hist_si",
            "create attribute if not exists fin.cogs_hist_st",
            "create attribute if not exists fin.prod_margin_hist_si",
            "create attribute if not exists fin.prod_margin_hist_st",
            "create attribute if not exists fin.qty_bklg_cra",
            "create attribute if not exists fin.qty_bklg_si",
            "create attribute if not exists fin.qty_bklg_st",
            "create attribute if not exists fin.qty_hist_cra",
            "create attribute if not exists fin.qty_hist_si",
            "create attribute if not exists fin.qty_hist_st",
            "create attribute if not exists fin.resale_bklg_st",
            "create attribute if not exists fin.resale_hist_st",
            "create attribute if not exists fin.revenue_bklg_cra",
            "create attribute if not exists fin.revenue_bklg_si",
            "create attribute if not exists fin.revenue_bklg_st",
            "create attribute if not exists fin.revenue_cover_cra",
            "create attribute if not exists fin.revenue_cover_si",
            "create attribute if not exists fin.revenue_cover_st",
            "create attribute if not exists fin.revenue_hist_cra",
            "create attribute if not exists fin.revenue_hist_si",
            "create attribute if not exists fin.revenue_hist_st",
            "create attribute if not exists flash.asp_hist_si",
            "create attribute if not exists flash.asp_hist_st",
            "create attribute if not exists flash.cogs_bklg_cra",
            "create attribute if not exists flash.cogs_bklg_si",
            "create attribute if not exists flash.cogs_bklg_st",
            "create attribute if not exists flash.cogs_hist_cra",
            "create attribute if not exists flash.cogs_hist_si",
            "create attribute if not exists flash.cogs_hist_st",
            "create attribute if not exists flash.prod_margin_bklg_st",
            "create attribute if not exists flash.qty_bklg_cra",
            "create attribute if not exists flash.qty_bklg_si",
            "create attribute if not exists flash.qty_bklg_st",
            "create attribute if not exists flash.qty_hist_cra",
            "create attribute if not exists flash.qty_hist_si",
            "create attribute if not exists flash.qty_hist_st",
            "create attribute if not exists flash.resale_bklg_st",
            "create attribute if not exists flash.resale_hist_st",
            "create attribute if not exists flash.revenue_bklg_cra",
            "create attribute if not exists flash.revenue_bklg_si",
            "create attribute if not exists flash.revenue_bklg_st",
            "create attribute if not exists flash.revenue_cover_st",
            "create attribute if not exists flash.revenue_hist_cra",
            "create attribute if not exists flash.revenue_hist_si",
            "create attribute if not exists flash.revenue_hist_st",
            "create attribute if not exists pii.domain_name",
            "create attribute if not exists pii.person",

            "alter table %s.%s add attribute catalog.approved pii.address" % (db, tbl),
            "alter table %s.%s add column attribute fiscal_quarter  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute year_quarter    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute today   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute flash_week  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute fiscal_qtr_std  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute qtr_alias   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute week_alias  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute year_alias  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute flash_week_std  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute fiscal_quarter_org  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute fiscal_year catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute week_no catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute flash_week_org  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute flash_key   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute scenario    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute channel_distribution_type   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute channel_direct_type catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute channel_hier3   catalog.dimension pii.person" % (db, tbl),
            "alter table %s.%s add column attribute channel_geography   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute channel_geo_hier2   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute channel_distributor_name    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute part    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute extnd_product_dppg  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute extnd_product_device    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute extnd_product_family    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute extnd_product_sub_group catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute extnd_product_sub_group_alias   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute extnd_product_group catalog.dimension pii.person" % (db, tbl),
            "alter table %s.%s add column attribute extnd_product_group_alias   catalog.dimension pii.person" % (db, tbl),
            "alter table %s.%s add column attribute extended_product_header catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute geometry    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute internal_product_dppg   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute internal_product_group  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute internal_product_family catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute internal_product_line   catalog.dimension pii.domain_name" % (db, tbl),
            "alter table %s.%s add column attribute internal_product_division   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute internal_product_roll_up    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute part_external   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute part_pkg_pin    catalog.dimension pii.person" % (db, tbl),
            "alter table %s.%s add column attribute part_grade  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute part_speed  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute part_logic_cells    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute part_node   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute primary_end_markets catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute primary_secondary_combo catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute secondary_end_markets   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute external_markets    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute external_markets_alias  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute ext_mkt_level1  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute primary_end_market_alias    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute internal_reporting_markets  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute int_mkt_level1  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute sub_segments    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute sub_seg_level1  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute territory   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute sector  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute area    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute region  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute rep_name    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute rep_alias   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute default_territory   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute default_sector  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute default_area    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute default_region  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute default_rep_name    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute default_rep_alias   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute customer_tier   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute customer_tier_detail    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute customer_xu catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute customer_gu catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute hub_party   catalog.dimension pii.domain_name" % (db, tbl),
            "alter table %s.%s add column attribute ship_customer_tier  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute ship_customer_tier_detail   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute ship_customer_xu    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute ship_customer_gu    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute ship_hub_party  catalog.dimension pii.domain_name" % (db, tbl),
            "alter table %s.%s add column attribute demand_creation_category    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute demand_creation_category_alias  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute demand_creation_type    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute business_type   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute dm_time_key catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute dm_part_key catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute dm_geography_key    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute dm_channel_key  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute dm_cust_key_disti   catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute dm_cust_key_end catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute dm_city_key catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute dm_cust_key_end_xcm catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute dm_city_key_xcm catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute dm_sales_hierarchy_key  catalog.dimension pii.person" % (db, tbl),
            "alter table %s.%s add column attribute dm_mkt_seg_key  catalog.dimension pii.person" % (db, tbl),
            "alter table %s.%s add column attribute dm_default_sales_hierarchy_key  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute dm_cust_key_ship_xcm    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute dm_business_type_key    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute dm_demand_creation_key  catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute flag_14_week    catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute record_type catalog.dimension" % (db, tbl),
            "alter table %s.%s add column attribute month_1_actual_curr_qtr_backlog catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_actual_curr_qtr_backlog catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_actual_curr_qtr_backlog catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_actual_delinquency catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_allocated_curr_qtr_backlog  catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_allocated_curr_qtr_backlog  catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_allocated_curr_qtr_backlog  catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_allocated_delinquency  catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_1_curr_qtr_backlog  catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_2_curr_qtr_backlog  catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_3_curr_qtr_backlog  catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_csd    catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute total_curr_qtr_delinquency  catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_cfd    catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute qtr_plus2_bklg_rev  catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_actual_revenue  catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_actual_revenue  catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_actual_revenue  catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_revenue_adjustments catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_revenue_adjustments catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_revenue_adjustments catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_allocated_revenue   catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_allocated_revenue   catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_allocated_revenue   catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_1_revenue   catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_2_revenue   catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_3_revenue   catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute qtd_revenue catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute flash_total_revenue catalog.measure fin.revenue_cover_st pii.person" % (db, tbl),
            "alter table %s.%s add column attribute month_1_actual_curr_qtr_backlog_qty catalog.measure fin.qty_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_allocated_curr_qtr_backlog_qty  catalog.measure fin.qty_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_actual_curr_qtr_backlog_qty catalog.measure fin.qty_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_allocated_curr_qtr_backlog_qty  catalog.measure fin.qty_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_actual_curr_qtr_backlog_qty catalog.measure fin.qty_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_allocated_curr_qtr_backlog_qty  catalog.measure fin.qty_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_actual_delinquency_qty catalog.measure fin.qty_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_allocated_delinquency_qty  catalog.measure fin.qty_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_1_curr_qtr_backlog_qty  catalog.measure fin.qty_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_2_curr_qtr_backlog_qty  catalog.measure fin.qty_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_3_curr_qtr_backlog_qty  catalog.measure fin.qty_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_csd_qty    catalog.measure fin.qty_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute total_curr_qtr_delinquency_qty  catalog.measure fin.qty_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_qty    catalog.measure fin.qty_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_actual_curr_qtr_backlog_cogs    catalog.measure fin.cogs_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_allocated_curr_qtr_backlog_cogs catalog.measure fin.cogs_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_actual_curr_qtr_backlog_cogs    catalog.measure fin.cogs_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_allocated_curr_qtr_backlog_cogs catalog.measure fin.cogs_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_actual_curr_qtr_backlog_cogs    catalog.measure fin.cogs_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_allocated_curr_qtr_backlog_cogs catalog.measure fin.cogs_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_actual_delinquency_cogs    catalog.measure fin.cogs_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_allocated_delinquency_cogs catalog.measure fin.cogs_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_1_curr_qtr_backlog_cogs catalog.measure fin.cogs_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_2_curr_qtr_backlog_cogs catalog.measure fin.cogs_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_3_curr_qtr_backlog_cogs catalog.measure fin.cogs_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_csd_cogs   catalog.measure fin.cogs_bklg_st pii.person" % (db, tbl),
            "alter table %s.%s add column attribute total_curr_qtr_delinquency_cogs catalog.measure fin.cogs_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_cogs   catalog.measure fin.cogs_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_actual_qty  catalog.measure fin.qty_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_allocated_qty   catalog.measure fin.qty_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_adjustments_qty catalog.measure fin.qty_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_actual_qty  catalog.measure fin.qty_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_allocated_qty   catalog.measure fin.qty_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_adjustments_qty catalog.measure fin.qty_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_actual_qty  catalog.measure fin.qty_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_allocated_qty   catalog.measure fin.qty_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_adjustments_qty catalog.measure fin.qty_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_1_qty   catalog.measure fin.qty_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_2_qty   catalog.measure fin.qty_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_3_qty   catalog.measure fin.qty_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute qtd_qty catalog.measure fin.qty_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute flash_total_qty catalog.measure fin.qty_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_actual_cogs catalog.measure fin.cogs_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_allocated_cogs  catalog.measure fin.cogs_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_adjustments_cogs    catalog.measure fin.cogs_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_actual_cogs catalog.measure fin.cogs_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_allocated_cogs  catalog.measure fin.cogs_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_adjustments_cogs    catalog.measure fin.cogs_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_actual_cogs catalog.measure fin.cogs_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_allocated_cogs  catalog.measure fin.cogs_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_adjustments_cogs    catalog.measure fin.cogs_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_1_cogs  catalog.measure fin.cogs_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_2_cogs  catalog.measure fin.cogs_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_3_cogs  catalog.measure fin.cogs_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute qtd_cogs    catalog.measure fin.cogs_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute flash_total_cogs    catalog.measure fin.cogs_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute qtr_plus2_bklg_qty  catalog.measure fin.qty_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute qtr_plus2_bklg_cogs catalog.measure fin.cogs_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_actual_curr_qtr_backlog_cra catalog.measure fin.revenue_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_2_actual_curr_qtr_backlog_cra catalog.measure fin.revenue_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_3_actual_curr_qtr_backlog_cra catalog.measure fin.revenue_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute qtr_plus2_bklg_rev_cra  catalog.measure fin.revenue_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_1_allocated_curr_qtr_backlog_cra  catalog.measure fin.revenue_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_2_allocated_curr_qtr_backlog_cra  catalog.measure fin.revenue_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_3_allocated_curr_qtr_backlog_cra  catalog.measure fin.revenue_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_1_curr_qtr_backlog_cra  catalog.measure fin.revenue_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_2_curr_qtr_backlog_cra  catalog.measure fin.revenue_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_3_curr_qtr_backlog_cra  catalog.measure fin.revenue_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_csd_cra    catalog.measure fin.revenue_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_cra    catalog.measure fin.revenue_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_1_actual_revenue_cra  catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_2_actual_revenue_cra  catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_3_actual_revenue_cra  catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_1_revenue_adjustments_cra catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_2_revenue_adjustments_cra catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_3_revenue_adjustments_cra catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_1_revenue_cra   catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_2_revenue_cra   catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_3_revenue_cra   catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute qtd_revenue_cra catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute flash_total_revenue_cra catalog.measure fin.revenue_cover_cra" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_sell_in    catalog.measure fin.revenue_bklg_si" % (db, tbl),
            "alter table %s.%s add column attribute qtd_revenue_sell_in catalog.measure fin.revenue_hist_si" % (db, tbl),
            "alter table %s.%s add column attribute flash_total_sell_in catalog.measure fin.revenue_hist_si" % (db, tbl),
            "alter table %s.%s add column attribute actual_curr_qtr_backlog catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute allocated_curr_qtr_backlog  catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_actual_and_allocation  catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute adjustment  catalog.measure fin.revenue_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_actual_curr_qtr_backlog_qty_cra catalog.measure fin.qty_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_2_actual_curr_qtr_backlog_qty_cra catalog.measure fin.qty_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_3_actual_curr_qtr_backlog_qty_cra catalog.measure fin.qty_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute qtr_plus2_bklg_qty_cra  catalog.measure fin.qty_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_1_allocated_curr_qtr_backlog_qty_cra  catalog.measure fin.qty_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_2_allocated_curr_qtr_backlog_qty_cra  catalog.measure fin.qty_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_3_allocated_curr_qtr_backlog_qty_cra  catalog.measure fin.qty_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_1_curr_qtr_backlog_qty_cra  catalog.measure fin.qty_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_2_curr_qtr_backlog_qty_cra  catalog.measure fin.qty_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_3_curr_qtr_backlog_qty_cra  catalog.measure fin.qty_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_csd_qty_cra    catalog.measure fin.qty_bklg_cra pii.person" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_qty_cra    catalog.measure fin.qty_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_1_actual_qty_cra  catalog.measure fin.qty_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_2_actual_qty_cra  catalog.measure fin.qty_hist_cra pii.person" % (db, tbl),
            "alter table %s.%s add column attribute month_3_actual_qty_cra  catalog.measure fin.qty_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_1_adjustments_qty_cra catalog.measure fin.qty_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_2_adjustments_qty_cra catalog.measure fin.qty_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_3_adjustments_qty_cra catalog.measure fin.qty_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_1_qty_cra   catalog.measure fin.qty_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_2_qty_cra   catalog.measure fin.qty_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_3_qty_cra   catalog.measure fin.qty_hist_cra pii.person" % (db, tbl),
            "alter table %s.%s add column attribute qtd_qty_cra catalog.measure fin.qty_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute flash_total_qty_cra catalog.measure fin.qty_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_qty_sell_in    catalog.measure fin.qty_bklg_si" % (db, tbl),
            "alter table %s.%s add column attribute qtd_qty_sell_in catalog.measure fin.qty_hist_si" % (db, tbl),
            "alter table %s.%s add column attribute flash_total_qty_sell_in catalog.measure fin.qty_hist_si" % (db, tbl),
            "alter table %s.%s add column attribute month_1_actual_curr_qtr_backlog_cogs_cra    catalog.measure fin.cogs_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_2_actual_curr_qtr_backlog_cogs_cra    catalog.measure fin.cogs_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_3_actual_curr_qtr_backlog_cogs_cra    catalog.measure fin.cogs_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute qtr_plus2_bklg_cogs_cra catalog.measure fin.cogs_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_1_allocated_curr_qtr_backlog_cogs_cra catalog.measure fin.cogs_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_2_allocated_curr_qtr_backlog_cogs_cra catalog.measure fin.cogs_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_3_allocated_curr_qtr_backlog_cogs_cra catalog.measure fin.cogs_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_1_curr_qtr_backlog_cogs_cra catalog.measure fin.cogs_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_2_curr_qtr_backlog_cogs_cra catalog.measure fin.cogs_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_3_curr_qtr_backlog_cogs_cra catalog.measure fin.cogs_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_csd_cogs_cra   catalog.measure fin.cogs_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_cogs_cra   catalog.measure fin.cogs_bklg_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_1_actual_cogs_cra catalog.measure fin.cogs_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_2_actual_cogs_cra catalog.measure fin.cogs_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_3_actual_cogs_cra catalog.measure fin.cogs_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_1_adjustments_cogs_cra    catalog.measure fin.cogs_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_2_adjustments_cogs_cra    catalog.measure fin.cogs_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute month_3_adjustments_cogs_cra    catalog.measure fin.cogs_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_1_cogs_cra  catalog.measure fin.cogs_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_2_cogs_cra  catalog.measure fin.cogs_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute total_month_3_cogs_cra  catalog.measure fin.cogs_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute qtd_cogs_cra    catalog.measure fin.cogs_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute flash_total_cogs_cra    catalog.measure fin.cogs_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_cogs_sell_in   catalog.measure fin.cogs_bklg_si" % (db, tbl),
            "alter table %s.%s add column attribute qtd_cogs_sell_in    catalog.measure fin.cogs_hist_si" % (db, tbl),
            "alter table %s.%s add column attribute flash_total_cogs_sell_in    catalog.measure fin.cogs_hist_si" % (db, tbl),
            "alter table %s.%s add column attribute month_1_actual_curr_qtr_backlog_resale  catalog.measure fin.resale_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_actual_curr_qtr_backlog_resale  catalog.measure fin.resale_bklg_st pii.person" % (db, tbl),
            "alter table %s.%s add column attribute month_3_actual_curr_qtr_backlog_resale  catalog.measure fin.resale_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_actual_delinquency_resale  catalog.measure fin.resale_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_allocated_curr_qtr_backlog_resale   catalog.measure fin.resale_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_allocated_curr_qtr_backlog_resale   catalog.measure fin.resale_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_allocated_curr_qtr_backlog_resale   catalog.measure fin.resale_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_allocated_delinquency_resale   catalog.measure fin.resale_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_1_curr_qtr_backlog_resale   catalog.measure fin.resale_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_2_curr_qtr_backlog_resale   catalog.measure fin.resale_bklg_st pii.person" % (db, tbl),
            "alter table %s.%s add column attribute total_month_3_curr_qtr_backlog_resale   catalog.measure fin.resale_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_resale_csd catalog.measure fin.resale_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute total_curr_qtr_delinquency_resale   catalog.measure fin.resale_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute curr_qtr_backlog_resale catalog.measure fin.resale_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute qtr_plus2_bklg_res  catalog.measure fin.resale_bklg_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_actual_resale   catalog.measure fin.resale_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_actual_resale   catalog.measure fin.resale_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_actual_resale   catalog.measure fin.resale_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_resale_adjustments  catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_resale_adjustments  catalog.measure fin.revenue_hist_st pii.person" % (db, tbl),
            "alter table %s.%s add column attribute month_3_resale_adjustments  catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_1_allocated_resale    catalog.measure fin.resale_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_2_allocated_resale    catalog.measure fin.resale_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute month_3_allocated_resale    catalog.measure fin.resale_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_1_resale    catalog.measure fin.resale_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_2_resale    catalog.measure fin.resale_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute total_month_3_resale    catalog.measure fin.resale_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute qtd_resale  catalog.measure fin.resale_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute flash_total_resale  catalog.measure fin.resale_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_day_turns    catalog.measure fin.revenue_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_day_turns_cra    catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute pr_day_turns_sell_in    catalog.measure fin.revenue_hist_si" % (db, tbl),
            "alter table %s.%s add column attribute wtd_turns   catalog.measure fin.revenue_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute wtd_turns_wk1   catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute wtd_turns_cra   catalog.measure fin.revenue_cover_cra" % (db, tbl),
            "alter table %s.%s add column attribute wtd_turns_cra_wk1   catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute wtd_turns_sell_in   catalog.measure fin.revenue_cover_si" % (db, tbl),
            "alter table %s.%s add column attribute wtd_turns_sell_in_wk1   catalog.measure fin.revenue_hist_si" % (db, tbl),
            "alter table %s.%s add column attribute avg_turns_wk    catalog.measure fin.revenue_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute avg_turns_wk_cra    catalog.measure fin.revenue_cover_cra" % (db, tbl),
            "alter table %s.%s add column attribute avg_turns_wk_sell_in    catalog.measure fin.revenue_cover_si" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr_ago  catalog.measure fin.revenue_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr_wk_ago   catalog.measure fin.revenue_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr_wk_end   catalog.measure fin.revenue_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr_wtd_turns    catalog.measure fin.revenue_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr_wtd_turns_13 catalog.measure fin.revenue_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr_wtd_turns_cra    catalog.measure fin.revenue_cover_cra" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr_wtd_turns_cra_13 catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr_wtd_turns_sell_in    catalog.measure fin.revenue_cover_si" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr_wtd_turns_sell_in_13 catalog.measure fin.revenue_hist_si" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr2_wtd_turns   pii.person" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr2_wtd_turns_cra   catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr2_wtd_turns_sell_in   pii.person" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr3_wtd_turns   pii.person" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr3_wtd_turns_cra   catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr3_wtd_turns_sell_in   pii.person" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr4_wtd_turns   pii.person" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr4_wtd_turns_cra   catalog.measure fin.revenue_hist_cra" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr4_wtd_turns_sell_in   catalog.measure fin.revenue_hist_si" % (db, tbl),
            "alter table %s.%s add column attribute pr_day_flash_total  catalog.measure fin.revenue_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_day_flash_total_cra  catalog.measure fin.revenue_cover_cra" % (db, tbl),
            "alter table %s.%s add column attribute pr_day_flash_total_sell_in  catalog.measure fin.revenue_cover_si" % (db, tbl),
            "alter table %s.%s add column attribute pr_day_flash_total_cogs catalog.measure fin.cogs_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_day_flash_total_cogs_sell_in catalog.measure fin.cogs_cover_si" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr_flash_total  catalog.measure fin.revenue_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr_flash_total_sell_in  catalog.measure fin.revenue_cover_si" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr_end_flash_total  catalog.measure fin.revenue_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr_end_flash_total_sell_in  catalog.measure fin.revenue_cover_si" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr_end_flash_total_cogs catalog.measure fin.cogs_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_qtr_end_flash_total_cogs_sell_in catalog.measure fin.cogs_cover_si" % (db, tbl),
            "alter table %s.%s add column attribute pr_wk_flash_total   catalog.measure fin.revenue_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_wkend_flash_total    catalog.measure fin.revenue_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_wk_flash_total_sell_in   catalog.measure fin.revenue_cover_si pii.person" % (db, tbl),
            "alter table %s.%s add column attribute pr_yr_flash_total   catalog.measure fin.revenue_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_yr_flash_total_cra   catalog.measure fin.revenue_cover_cra" % (db, tbl),
            "alter table %s.%s add column attribute pr_yr_flash_total_sell_in   catalog.measure fin.revenue_cover_si pii.person" % (db, tbl),
            "alter table %s.%s add column attribute product_margin  catalog.measure fin.prod_margin_hist_st" % (db, tbl),
            "alter table %s.%s add column attribute product_margin_sell_in  catalog.measure fin.prod_margin_hist_si" % (db, tbl),
            "alter table %s.%s add column attribute pr_4qtr_avg_wtd_turns   catalog.measure fin.revenue_cover_st" % (db, tbl),
            "alter table %s.%s add column attribute pr_4qtr_avg_wtd_turns_cra   catalog.measure fin.revenue_cover_cra" % (db, tbl),
            "alter table %s.%s add column attribute pr_4qtr_avg_wtd_turns_sell_in   catalog.measure fin.revenue_cover_si" % (db, tbl),

            "drop role if exists %s" % (role),
            "create role %s with groups %s" % (role, testuser),
            """GRANT SELECT ON DATABASE `%s`
               HAVING ATTRIBUTE IN (fin.revenue_cover_cra, `flash`.`cogs_bklg_cra`, `flash`.`cogs_bklg_si`, `flash`.`cogs_bklg_st`, `flash`.`cogs_hist_cra`,  `flash`.`cogs_hist_si`, `flash`.`cogs_hist_st`, `flash`.`qty_bklg_cra`, `flash`.`qty_bklg_si`, `flash`.`qty_bklg_st`, `flash`.`qty_hist_cra`, `flash`.`qty_hist_si`, `flash`.`qty_hist_st`, `flash`.`revenue_bklg_cra`, `flash`.`revenue_bklg_si`, `flash`.`revenue_bklg_st`,`flash`.`resale_bklg_st`, `flash`.`revenue_hist_cra`, `flash`.`revenue_hist_si`, `flash`.`revenue_hist_st`, `flash`.`resale_hist_st`, `flash`.`prod_margin_bklg_st`, `flash`.`asp_hist_st`, `flash`.`asp_hist_si`, `flash`.`revenue_cover_st`)
               WHERE ((in_set('ALL', user_attribute('customer_tier')) OR in_set(customer_tier, user_attribute('customer_tier')))
               AND (in_set('ALL', user_attribute('external_markets')) OR in_set(external_markets, user_attribute('external_markets')))
               AND (in_set('ALL', user_attribute('external_markets_alias')) OR in_set(external_markets_alias, user_attribute('external_markets_alias')))
               AND (in_set('ALL', user_attribute('internal_reporting_markets')) OR in_set(internal_reporting_markets, user_attribute('internal_reporting_markets')))
               AND (in_set('ALL', user_attribute('primary_end_markets')) OR in_set(primary_end_markets, user_attribute('primary_end_markets')))
               -- AND (in_set('ALL', user_attribute('prim_sec_end_markets')) OR in_set(primary_secondary_combo, user_attribute('prim_sec_end_markets')))
               AND (in_set('ALL', user_attribute('sub_seg_level1')) OR in_set(sub_seg_level1, user_attribute('sub_seg_level1')))
               AND (in_set('ALL', user_attribute('channel_geography')) OR in_set(channel_geography, user_attribute('channel_geography')))
               AND (in_set('ALL', user_attribute('channel_hier3')) OR in_set(channel_hier3, user_attribute('channel_hier3')))
               AND (in_set('ALL', user_attribute('customer_xu')) OR in_set(customer_xu, user_attribute('customer_xu')))
               AND (in_set('ALL', user_attribute('rep_name')) OR in_set(rep_name, user_attribute('rep_name')))
               AND (in_set('ALL', user_attribute('territory')) OR in_set(territory, user_attribute('territory')))
               -- AND (in_set('ALL', user_attribute('sector')) OR in_set(sector, user_attribute('sector')))
               -- AND (in_set('ALL', user_attribute('area')) OR in_set(area, user_attribute('area')))
               -- AND (in_set('ALL', user_attribute('internal_product_line')) OR in_set(internal_product_line, user_attribute('internal_product_line')))
               )
               TO ROLE `%s`""" % (db, role)
        ]

      for ddl in ddls:
          conn.execute_ddl(ddl)

def create_parquet_table_with_stats(conn, db, tbl):
    ddls = [
        "DROP DATABASE IF EXISTS %s CASCADE" % (db),
        "CREATE DATABASE %s" % (db),
        """
        CREATE EXTERNAL TABLE %s.%s (
                trip_id                 INT,
                vendor_id               STRING,
                pickup_datetime         TIMESTAMP,
                dropoff_datetime        TIMESTAMP,
                store_and_fwd_flag      STRING,
                rate_code_id            SMALLINT,
                pickup_longitude        DOUBLE,
                pickup_latitude         DOUBLE,
                dropoff_longitude       DOUBLE,
                dropoff_latitude        DOUBLE,
                passenger_count         SMALLINT,
                trip_distance           DOUBLE,
                fare_amount             DOUBLE,
                extra                   DOUBLE,
                mta_tax                 DOUBLE,
                tip_amount              DOUBLE,
                tolls_amount            DOUBLE,
                ehail_fee               DOUBLE,
                improvement_surcharge   DOUBLE,
                total_amount            DOUBLE,
                payment_type            STRING,
                trip_type               SMALLINT,
                pickup                  STRING,
                dropoff                 STRING,

                cab_type                STRING,

                pickup_nyct2010_gid     SMALLINT,
                pickup_ctlabel          STRING,
                pickup_borocode         SMALLINT,
                pickup_boroname         STRING,
                pickup_ct2010           STRING,
                pickup_boroct2010       STRING,
                pickup_cdeligibil       STRING,
                pickup_ntacode          STRING,
                pickup_ntaname          STRING,
                pickup_puma             STRING,

                dropoff_nyct2010_gid    SMALLINT,
                dropoff_ctlabel         STRING,
                dropoff_borocode        SMALLINT,
                dropoff_boroname        STRING,
                dropoff_ct2010          STRING,
                dropoff_boroct2010      STRING,
                dropoff_cdeligibil      STRING,
                dropoff_ntacode         STRING,
                dropoff_ntaname         STRING,
                dropoff_puma            STRING
            ) STORED AS parquet
            LOCATION 's3://cerebrodata-test/nytaxi-data/parquet/'
        """ % (db, tbl),
        ]
    for ddl in ddls:
        conn.execute_ddl(ddl)

def create_orc_table_with_stats(conn, db, tbl):
    ddls = [
        "DROP DATABASE IF EXISTS %s CASCADE" % (db),
        "CREATE DATABASE %s" % (db),
        """
        CREATE EXTERNAL TABLE %s.%s (
            trip_id                 INT,
            vendor_id               STRING,
            pickup_datetime         TIMESTAMP,
            dropoff_datetime        TIMESTAMP,
            store_and_fwd_flag      STRING,
            rate_code_id            SMALLINT,
            pickup_longitude        DOUBLE,
            pickup_latitude         DOUBLE,
            dropoff_longitude       DOUBLE,
            dropoff_latitude        DOUBLE,
            passenger_count         SMALLINT,
            trip_distance           DOUBLE,
            fare_amount             DOUBLE,
            extra                   DOUBLE,
            mta_tax                 DOUBLE,
            tip_amount              DOUBLE,
            tolls_amount            DOUBLE,
            ehail_fee               DOUBLE,
            improvement_surcharge   DOUBLE,
            total_amount            DOUBLE,
            payment_type            STRING,
            trip_type               SMALLINT,
            pickup                  STRING,
            dropoff                 STRING,

            cab_type                STRING,

            pickup_nyct2010_gid     SMALLINT,
            pickup_ctlabel          STRING,
            pickup_borocode         SMALLINT,
            pickup_boroname         STRING,
            pickup_ct2010           STRING,
            pickup_boroct2010       STRING,
            pickup_cdeligibil       STRING,
            pickup_ntacode          STRING,
            pickup_ntaname          STRING,
            pickup_puma             STRING,

            dropoff_nyct2010_gid    SMALLINT,
            dropoff_ctlabel         STRING,
            dropoff_borocode        SMALLINT,
            dropoff_boroname        STRING,
            dropoff_ct2010          STRING,
            dropoff_boroct2010      STRING,
            dropoff_cdeligibil      STRING,
            dropoff_ntacode         STRING,
            dropoff_ntaname         STRING,
            dropoff_puma            STRING
        ) STORED AS ORC
        LOCATION 's3a://cerebrodata-test/orc/nytaxi/orc_data/'
        """ % (db, tbl),
    ]
    for ddl in ddls:
        conn.execute_ddl(ddl)
