class Labels:  # pragma: no cover
    """Class for Labels used in CoreMS

    These labels are used to define:
    * types of columns in plaintext data inputs,
    * types of data/mass spectra
    * types of assignment for ions

    """

    mz = "m/z"
    abundance = "Peak Height"
    rp = "Resolving Power"
    s2n = "S/N"

    label = "label"
    bruker_profile = "Bruker_Profile"
    thermo_profile = "Thermo_Profile"
    simulated_profile = "Simulated Profile"
    booster_profile = "Booster Profile"
    bruker_frequency = "Bruker_Frequency"
    midas_frequency = "Midas_Frequency"
    thermo_centroid = "Thermo_Centroid"
    corems_centroid = "CoreMS_Centroid"
    gcms_centroid = "Thermo_Centroid"

    unassigned = "unassigned"

    radical_ion = "RADICAL"
    protonated_de_ion = "DE_OR_PROTONATED"
    protonated = "protonated"
    de_protonated = "de-protonated"
    adduct_ion = "ADDUCT"
    neutral = "neutral"
    ion_type = "IonType"

    ion_type_translate = {
        "protonated": "DE_OR_PROTONATED",
        "de-protonated": "DE_OR_PROTONATED",
        "radical": "RADICAL",
        "adduct": "ADDUCT",
        "ADDUCT": "ADDUCT",
    }


class Atoms:  # pragma: no cover
    """Class for Atoms in CoreMS

    This class includes key properties of atoms (and the electron) and isotopes, including their exact masses, relative abundances, and covalences.
    It also associates which isotopes are for the same element, and provides an ordering of elements.

    IUPAC definition of monoisotopic mass is based on the most abundant isotopes of each element present.
    Here, we will use atom symbols with isotope numbers for all isotopes excluding the most abundant one.
    This list has been corrected up to Iodine.

    References
    ----------

    1. NIST - Last Accessed 2019-06-12
    https://www.nist.gov/pml/atomic-weights-and-isotopic-compositions-relative-atomic-masses

    """

    electron_mass = 0.0005_485_799_090_65  # NIST value

    atomic_masses = {
        "H": 1.00782503223,
        "D": 2.01410177812,
        "T": 3.0160492779,  # consider removing
        "3He": 3.0160293201,
        "He": 4.00260325413,
        "6Li": 6.0151228874,
        "Li": 7.0160034366,
        "Be": 9.012183065,
        "10B": 10.01293695,
        "B": 11.00930536,
        "C": 12.0,
        "13C": 13.00335483507,
        "14C": 14.0032419884,
        "N": 14.00307400443,
        "15N": 15.00010889888,
        "O": 15.99491461957,
        "17O": 16.99913175650,
        "18O": 17.99915961286,
        "F": 18.99840316273,
        "Ne": 19.9924401762,
        "21Ne": 20.993846685,
        "22Ne": 21.991385114,
        "Na": 22.9897692820,
        "Mg": 23.985041697,
        "25Mg": 24.985836976,
        "26Mg": 25.982592968,
        "Al": 26.98153853,
        "Si": 27.97692653465,
        "29Si": 28.97649466490,
        "30Si": 29.973770136,
        "P": 30.97376199842,
        "S": 31.9720711744,
        "33S": 32.9714589098,
        "34S": 33.967867004,
        "36S": 35.96708071,
        "Cl": 34.968852682,
        "37Cl": 36.965902602,
        "36Ar": 35.967545105,
        "38Ar": 37.96273211,
        "Ar": 39.9623831237,
        "K": 38.9637064864,
        "40K": 39.963998166,
        "41K": 40.9618252579,
        "Ca": 39.962590863,
        "42Ca": 41.95861783,
        "43Ca": 42.95876644,
        "44Ca": 43.95548156,
        "46Ca": 45.9536890,
        "48Ca": 47.95252276,
        "Sc": 44.95590828,
        "46Ti": 45.95262772,
        "47Ti": 46.95175879,
        "Ti": 47.94794198,
        "49Ti": 48.94786568,
        "50Ti": 49.94478689,
        "50V": 49.94715601,
        "V": 50.94395704,
        "50Cr": 49.94604183,
        "Cr": 51.94050623,
        "53Cr": 52.94064815,
        "54Cr": 53.93887916,
        "Mn": 54.93804391,
        "54Fe": 53.93960899,
        "Fe": 55.93493633,
        "57Fe": 56.93539284,
        "58Fe": 57.93327443,
        "Co": 58.93319429,
        "Ni": 57.93534241,
        "60Ni": 59.93078588,
        "61Ni": 60.93105557,
        "62Ni": 61.92834537,
        "64Ni": 63.92796682,
        "Cu": 62.92959772,
        "65Cu": 64.92778970,
        "Zn": 63.92914201,
        "66Zn": 65.92603381,
        "67Zn": 66.92712775,
        "68Zn": 67.92484455,
        "70Zn": 69.9253192,
        "Ga": 68.9255735,
        "71Ga": 70.92470258,
        "70Ge": 69.92424875,
        "72Ge": 71.922075826,
        "73Ge": 72.923458956,
        "Ge": 73.921177761,
        "76Ge": 75.921402726,
        "As": 74.92159457,
        "74Se": 73.922475934,
        "76Se": 75.919213704,
        "77Se": 76.919914154,
        "78Se": 77.91730928,
        "Se": 79.9165218,
        "82Se": 81.9166995,
        "Br": 78.9183376,
        "81Br": 80.9162897,
        "78Kr": 77.92036494,
        "80Kr": 79.91637808,
        "82Kr": 81.91348273,
        "83Kr": 82.91412716,
        "Kr": 83.9114977282,
        "86Kr": 85.9106106269,
        "Rb": 84.9117897379,
        "87Rb": 86.9091805310,
        "84Sr": 83.9134191,
        "86Sr": 85.9092606,
        "87Sr": 86.9088775,
        "Sr": 87.9056125,
        "Y": 88.9058403,
        "Zr": 89.9046977,
        "91Zr": 90.9056396,
        "92Zr": 91.9050347,
        "94Zr": 93.9063108,
        "96Zr": 95.9082714,
        "Nb": 92.9063730,
        "92Mo": 91.90680796,
        "94Mo": 93.90508490,
        "95Mo": 94.90583877,
        "96Mo": 95.90467612,
        "97Mo": 96.90601812,
        "Mo": 97.90540482,
        "100Mo": 99.9074718,
        #'97Tc': 96.9063667,
        #'98Tc': 97.9072124,
        "Tc": 98.9062508,  # consider removing technetium as it is radioactive
        "96Ru": 95.90759025,
        "98Ru": 97.9052868,
        "99Ru": 98.9059341,
        "100Ru": 99.9042143,
        "101Ru": 100.9055769,
        "Ru": 101.9043441,
        "104Ru": 103.9054275,
        "Rh": 102.9054980,
        "102Pd": 101.9056022,
        "104Pd": 103.9040305,
        "105Pd": 104.9050796,
        "Pd": 105.9034804,
        "108Pd": 107.9038916,
        "110Pd": 109.9051722,
        "Ag": 106.9050916,
        "109Ag": 108.9047553,
        "106Cd": 105.9064599,
        "108Cd": 107.9041834,
        "110Cd": 109.90300661,
        "111Cd": 110.90418287,
        "Cd": 111.90276287,
        "113Cd": 112.90440813,
        "114Cd": 113.90336509,
        "116Cd": 115.90476315,
        "113In": 112.90406184,
        "In": 114.903878776,
        "112Sn": 111.90482387,
        "114Sn": 113.9027827,
        "115Sn": 114.903344699,
        "116Sn": 115.90174280,
        "117Sn": 116.90295398,
        "118Sn": 117.90160657,
        "119Sn": 118.90331117,
        "Sn": 119.90220163,
        "122Sn": 121.9034438,
        "124Sn": 123.9052766,
        "Sb": 120.9038120,
        "123Sb": 122.9042132,
        "120Te": 119.9040593,
        "122Te": 121.9030435,
        "123Te": 122.9042698,
        "124Te": 123.9028171,
        "125Te": 124.9044299,
        "126Te": 125.9033109,
        "128Te": 127.90446128,
        "Te": 129.906222748,
        "I": 126.9044719,
        "124Xe": 123.9058920,
        "126Xe": 125.9042983,
        "128Xe": 127.9035310,
        "129Xe": 128.9047808611,
        "130Xe": 129.903509349,
        "131Xe": 130.90508406,
        "Xe": 131.9041550856,
        "134Xe": 133.90539466,
        "136Xe": 135.907214484,
        "Cs": 132.9054519610,
        "130Ba": 129.9063207,
        "132Ba": 131.9050611,
        "134Ba": 133.90450818,
        "135Ba": 134.90568838,
        "136Ba": 135.90457573,
        "137Ba": 136.90582714,
        "Ba": 137.90524700,
        "138La": 137.9071149,
        "La": 138.9063563,
        "136Ce": 135.90712921,
        "138Ce": 137.905991,
        "Ce": 139.9054431,
        "142Ce": 141.9092504,
        "Pr": 140.9076576,
        "Nd": 141.9077290,
        "143Nd": 142.9098200,
        "144Nd": 143.9100930,
        "145Nd": 144.9125793,
        "146Nd": 145.9131226,
        "148Nd": 147.9168993,
        "150Nd": 149.9209022,
        "145Pm": 144.9127559,
        "147Pm": 146.9151450,
        "144Sm": 143.9120065,
        "147Sm": 146.9149044,
        "148Sm": 147.9148292,
        "149Sm": 148.9171921,
        "150Sm": 149.9172829,
        "Sm": 151.9197397,
        "154Sm": 153.9222169,
        "151Eu": 150.9198578,
        "Eu": 152.9212380,
        "152Gd": 151.9197995,
        "154Gd": 153.9208741,
        "155Gd": 154.9226305,
        "156Gd": 155.9221312,
        "157Gd": 156.9239686,
        "Gd": 157.9241123,
        "160Gd": 159.9270624,
        "Tb": 158.9253547,
        "156Dy": 155.9242847,
        "158Dy": 157.9244159,
        "160Dy": 159.9252046,
        "161Dy": 160.9269405,
        "162Dy": 161.9268056,
        "163Dy": 162.9287383,
        "Dy": 163.9291819,
        "Ho": 164.9303288,
        "162Er": 161.9287884,
        "164Er": 163.9292088,
        "Er": 165.9302995,
        "167Er": 166.9320546,
        "168Er": 167.9323767,
        "170Er": 169.9354702,
        "Tm": 168.9342179,
        "168Yb": 167.9338896,
        "170Yb": 169.9347664,
        "171Yb": 170.9363302,
        "172Yb": 171.9363859,
        "173Yb": 172.9382151,
        "Yb": 173.9388664,
        "176Yb": 175.9425764,
        "Lu": 174.9407752,
        "176Lu": 175.9426897,
        "174Hf": 173.9400461,
        "176Hf": 175.9414076,
        "177Hf": 176.9432277,
        "178Hf": 177.9437058,
        "179Hf": 178.9458232,
        "Hf": 179.9465570,
        "180Ta": 179.9474648,
        "Ta": 180.9479958,
        "180W": 179.9467108,
        "182W": 181.94820394,
        "183W": 182.95022275,
        "W": 183.95093092,
        "186W": 185.9543628,
        "185Re": 184.9529545,
        "Re": 186.9557501,
        "184Os": 183.9524885,
        "186Os": 185.9538350,
        "187Os": 186.9557474,
        "188Os": 187.9558352,
        "189Os": 188.9581442,
        "190Os": 189.9584437,
        "192Os": 191.9614770,
        "191Ir": 190.9605893,
        "Ir": 192.9629216,
        "190Pt": 189.9599297,
        "192Pt": 191.9610387,
        "194Pt": 193.9626809,
        "Pt": 194.9647917,
        "196Pt": 195.96495209,
        "198Pt": 197.9678949,
        "Au": 196.96656879,
        "196Hg": 195.9658326,
        "198Hg": 197.96676860,
        "199Hg": 198.96828064,
        "200Hg": 199.96832659,
        "201Hg": 200.97030284,
        "Hg": 201.97064340,
        "204Hg": 203.97349398,
        "203Tl": 202.9723446,
        "Tl": 204.9744278,
        "204Pb": 203.9730440,
        "206Pb": 205.9744657,
        "207Pb": 206.9758973,
        "Pb": 207.9766525,
        "Bi": 208.9803991,
        "209Po": 208.9824308,
        "210Po": 209.9828741,
        "210At": 209.9871479,
        "211At": 210.9874966,
        "211Rn": 210.9906011,
        "220Rn": 220.0113941,
        "222Rn": 222.0175782,
        "223Fr": 223.0197360,
        "223Ra": 223.0185023,
        "224Ra": 224.0202120,
        "226Ra": 226.0254103,
        "228Ra": 228.0310707,
        "227Ac": 227.0277523,
        "230Th": 230.0331341,
        "Th": 232.0380558,
        "Pa": 231.0358842,
        "233U": 233.0396355,
        "234U": 234.0409523,
        "235U": 235.0439301,
        "236U": 236.0455682,
        "U": 238.0507884,
        "236Np": 236.046570,
        "237Np": 237.0481736,
        "238Pu": 238.0495601,
        "239Pu": 239.0521636,
        "240Pu": 240.0538138,
        "241Pu": 241.0568517,
        "242Pu": 242.0587428,
        "244Pu": 244.0642053,
        "241Am": 241.0568293,
        "243Am": 243.0613813,
        "243Cm": 243.0613893,
        "244Cm": 244.0627528,
        "245Cm": 245.0654915,
        "246Cm": 246.0672238,
        "247Cm": 247.0703541,
        "248Cm": 248.0723499,
        "247Bk": 247.0703073,
        "249Bk": 249.0749877,
        "249Cf": 249.0748539,
        "250Cf": 250.0764062,
        "251Cf": 251.0795886,
        "252Cf": 252.0816272,
        "252Es": 252.082980,
        "257Fm": 257.0951061,
        "258Md": 258.0984315,
        "260Md": 260.10365,
        "259No": 259.10103,
        "262Lr": 262.10961,
        "267Rf": 267.12179,
        "268Db": 268.12567,
        "271Sg": 271.13393,
        "272Bh": 272.13826,
        "270Hs": 270.13429,
        "276Mt": 276.15159,
        "281Ds": 281.16451,
        "280Rg": 280.16514,
        "285Cn": 285.17712,
        "284Nh": 284.17873,
        "289Fl": 289.19042,
        "288Mc": 288.19274,
        "293Lv": 293.20449,
        "292Ts": 292.20746,
        "294Og": 294.21392,
    }

    # This list orders atoms, starting with the most abundant isotopes first, ordered as described.
    # Less abundant isotopes are ordered by mass at the end of the list.
    atoms_order = [
        "C",
        "H",
        "O",
        "N",
        "P",
        "S",  # CHONPS
        "F",
        "Cl",
        "Br",
        "I",
        "At",  # Halogens
        "Li",
        "Na",
        "K",
        "Rb",
        "Cs",
        "Fr",  # Alkali
        "He",
        "Ne",
        "Ar",
        "Kr",
        "Xe",
        "Rn",  # Noble gasses
        "Be",
        "B",  # Row 2
        "Mg",
        "Al",
        "Si",  # Row 3
        "Ca",
        "Sc",
        "Ti",
        "V",
        "Cr",
        "Mn",
        "Fe",
        "Co",
        "Ni",
        "Cu",
        "Zn",
        "Ga",
        "Ge",
        "As",
        "Se",  # Row 4
        "Sr",
        "Y",
        "Zr",
        "Nb",
        "Mo",
        "Tc",
        "Ru",
        "Rh",
        "Pd",
        "Ag",
        "Cd",
        "In",
        "Sn",
        "Sb",
        "Te",  # Row 5
        "Ba",
        "La",
        "Hf",
        "Ta",
        "W",
        "Re",
        "Os",
        "Ir",
        "Pt",
        "Au",
        "Hg",
        "Tl",
        "Pb",
        "Bi",
        "Po",  # Row 6
        "Ra",
        "Ac",
        "Rf",
        "Db",
        "Sg",
        "Bh",
        "Hs",
        "Mt",
        "Ds",
        "Rg",
        "Cn",
        "Nh",
        "Fl",
        "Mc",
        "Lv",
        "Ts",
        "Og",  # Row 7
        "Ce",
        "Pr",
        "Nd",
        "Pm",
        "Sm",
        "Eu",
        "Gd",
        "Tb",
        "Dy",
        "Ho",
        "Er",
        "Tm",
        "Yb",
        "Lu",  # Lanthanides
        "Th",
        "Pa",
        "U",
        "Np",
        "Pu",
        "Am",
        "Cm",
        "Bk",
        "Cf",
        "Es",
        "Fm",
        "Md",
        "No",
        "Lr",  # Actinides
        # Less abundant isotopes follow
        "D",
        "6Li",
        "10B",
        "13C",
        "15N",
        "17O",
        "18O",
        "22Ne",
        "25Mg",
        "26Mg",
        "29Si",
        "30Si",
        "33S",
        "34S",
        "36S",
        "37Cl",
        "40Ca",
        "41K",
        "44Ca",
        "46Ti",
        "47Ti",
        "49Ti",
        "50Cr",
        "50Ti",
        "50V",
        "53Cr",
        "54Cr",
        "54Fe",
        "57Fe",
        "58Fe",
        "60Ni",
        "61Ni",
        "62Ni",
        "65Cu",
        "66Zn",
        "67Zn",
        "68Zn",
        "70Ge",
        "71Ga",
        "72Ge",
        "73Ge",
        "76Ge",
        "76Se",
        "77Se",
        "78Se",
        "81Br",
        "80Kr",
        "82Kr",
        "82Se",
        "83Kr",
        "85Rb",
        "86Kr",
        "86Sr",
        "87Rb",
        "87Sr",
        "88Sr",
        "91Zr",
        "92Mo",
        "92Zr",
        "94Mo",
        "94Zr",
        "95Mo",
        "96Mo",
        "96Ru",
        "96Zr",
        "97Mo",
        "98Ru",
        "99Ru",
        "100Mo",
        "100Ru",
        "101Ru",
        "102Pd",
        "104Pd",
        "104Ru",
        "105Pd",
        "106Cd",
        "106Pd",
        "108Cd",
        "108Pd",
        "109Ag",
        "110Cd",
        "110Pd",
        "111Cd",
        "112Cd",
        "112Sn",
        "113Cd",
        "113In",
        "114Cd",
        "114Sn",
        "115In",
        "115Sn",
        "116Cd",
        "116Sn",
        "117Sn",
        "118Sn",
        "119Sn",
        "120Sn",
        "120Te",
        "121Sb",
        "122Sn",
        "122Te",
        "123Sb",
        "123Te",
        "124Sn",
        "124Te",
        "124Xe",
        "125Te",
        "126Te",
        "126Xe",
        "128Te",
        "128Xe",
        "129Xe",
        "130Ba",
        "130Te",
        "130Xe",
        "131Xe",
        "132Ba",
        "132Xe",
        "134Ba",
        "134Xe",
        "135Ba",
        "136Ba",
        "136Xe",
        "137Ba",
        "138Ba",
        "174Hf",
        "176Hf",
        "177Hf",
        "178Hf",
        "179Hf",
        "180Hf",
        "180W",
        "182W",
        "183W",
        "184Os",
        "184W",
        "185Re",
        "186Os",
        "186W",
        "187Os",
        "187Re",
        "188Os",
        "189Os",
        "190Os",
        "190Pt",
        "191Ir",
        "192Ir",
        "192Os",
        "192Pt",
        "194Pt",
        "195Pt",
        "196Hg",
        "196Pt",
        "198Hg",
        "198Pt",
        "199Hg",
        "200Hg",
        "201Hg",
        "202Hg",
        "203Tl",
        "204Hg",
        "204Pb",
        "205Tl",
        "206Pb",
        "207Pb",
        "208Pb",
    ]

    atoms_covalence = {
        "C": (4),
        "13C": (4),
        "N": (3),
        "O": (2),
        "S": (2),
        "H": (1),
        "F": (1, 0),
        "Cl": (1, 0),
        "Br": (1, 0),
        "I": (1, 0),
        "At": (1),
        "Li": (1, 0),
        "Na": (1, 0),
        "K": (1, 0),
        "Rb": (1),
        "Cs": (1),
        "Fr": (1),
        "B": (4, 3, 2, 1),
        "In": (3, 2, 1),
        "Al": (3, 1, 2),
        "P": (3, 5, 4, 2, 1),
        "Ga": (3, 1, 2),
        "Mg": (2, 1),
        "Be": (2, 1),
        "Ca": (2, 1),
        "Sr": (2, 1),
        "Ba": (2),
        "Ra": (2),
        "V": (5, 4, 3, 2, 1),
        "Fe": (3, 2, 4, 5, 6),
        "Si": (4, 3, 2),
        "Sc": (3, 2, 1),
        "Ti": (4, 3, 2, 1),
        "Cr": (1, 2, 3, 4, 5, 6),
        "Mn": (1, 2, 3, 4, 5, 6, 7),
        "Co": (1, 2, 3, 4, 5),
        "Ni": (1, 2, 3, 4),
        "Cu": (2, 1, 3, 4),
        "Zn": (2, 1),
        "Ge": (4, 3, 2, 1),
        "As": (5, 3, 2, 1),
        "Se": (6, 4, 2, 1),
        "Y": (3, 2, 1),
        "Zr": (4, 3, 2, 1),
        "Nb": (5, 4, 3, 2, 1),
        "Mo": (6, 5, 4, 3, 2, 1),
        "Tc": (7, 6, 5, 4, 3, 2, 1),
        "Ru": (8, 7, 6, 5, 4, 3, 2, 1),
        "Rh": (6, 5, 4, 3, 2, 1),
        "Pd": (4, 2, 1),
        "Ag": (0, 1, 2, 3, 4),
        "Cd": (2, 1),
        "Sn": (4, 2),
        "Sb": (5, 3),
        "Te": (6, 5, 4, 2),
        "La": (3, 2),
        "Hf": (4, 3, 2),
        "Ta": (5, 4, 3, 2),
        "W": (6, 5, 4, 3, 2, 1),
        "Re": (4, 7, 6, 5, 3, 2, 1),
        "Os": (4, 8, 7, 6, 5, 3, 2, 1),
        "Ir": (4, 8, 6, 5, 3, 2, 1),
        "Pt": (4, 6, 5, 3, 2, 1),
        "Au": (3, 5, 2, 1),
        "Hg": (1, 2, 4),
        "Tl": (3, 1),
        "Pb": (4, 2),
        "Bi": (3, 1, 5),
        "Po": (2, 4, 6),
        "Ac": (3, 2),
    }

    isotopic_abundance = {
        "H": 0.999885,
        "D": 0.000115,
        "T": 0,  # Consider removing.
        "3He": 0.00000134,
        "He": 0.99999866,
        "6Li": 0.0759,
        "Li": 0.9241,
        "Be": 1.0,
        "10B": 0.199,
        "B": 0.801,
        "C": 0.9893,
        "13C": 0.0107,
        "14C": 0,
        "N": 0.99636,
        "15N": 0.00364,
        "O": 0.99757,
        "17O": 0.00038,
        "18O": 0.00205,
        "F": 1.0,
        "Ne": 0.9048,
        "21Ne": 0.0027,
        "22Ne": 0.0925,
        "Na": 1.0,
        "Mg": 0.7899,
        "25Mg": 0.1000,
        "26Mg": 0.1101,
        "Al": 1.0,
        "Si": 0.92223,
        "29Si": 0.04685,
        "30Si": 0.03092,
        "P": 1.0,
        "S": 0.9499,
        "33S": 0.0075,
        "34S": 0.0425,
        "36S": 0.0001,
        "Cl": 0.7576,
        "37Cl": 0.2424,
        "36Ar": 0.003336,
        "38Ar": 0.000629,
        "Ar": 0.996035,
        "K": 0.932581,
        "40K": 0.000117,
        "41K": 0.067302,
        "Ca": 0.96941,
        "42Ca": 0.00647,
        "43Ca": 0.00135,
        "44Ca": 0.02086,
        "46Ca": 0.00004,
        "48Ca": 0.001872,
        "Sc": 1.0,
        "46Ti": 0.0825,
        "47Ti": 0.0744,
        "Ti": 0.7372,
        "49Ti": 0.0541,
        "50Ti": 0.0518,
        "50V": 0.00250,
        "V": 0.9975,
        "50Cr": 0.04345,
        "Cr": 0.83789,
        "53Cr": 0.09501,
        "54Cr": 0.02365,
        "Mn": 1.0,
        "54Fe": 0.05845,
        "Fe": 0.91754,
        "57Fe": 0.02119,
        "58Fe": 0.00282,
        "Co": 1.0,
        "Ni": 0.68077,
        "60Ni": 0.26223,
        "61Ni": 0.011399,
        "62Ni": 0.036346,
        "64Ni": 0.009255,
        "Cu": 0.6915,
        "65Cu": 0.3085,
        "Zn": 0.4917,
        "66Zn": 0.2773,
        "67Zn": 0.0404,
        "68Zn": 0.1845,
        "70Zn": 0.0061,
        "Ga": 0.60108,
        "71Ga": 0.39892,
        "70Ge": 0.2057,
        "72Ge": 0.2745,
        "73Ge": 0.0775,
        "Ge": 0.3650,
        "76Ge": 0.0773,
        "As": 1.0,
        "74Se": 0.0089,
        "76Se": 0.0937,
        "77Se": 0.0763,
        "78Se": 0.2377,
        "Se": 0.4961,
        "82Se": 0.0873,
        "Br": 0.5069,
        "81Br": 0.4931,
        "78Kr": 0.00355,
        "80Kr": 0.02286,
        "82Kr": 0.11593,
        "83Kr": 0.11500,
        "Kr": 0.56987,
        "86Kr": 0.17279,
        "Rb": 0.7217,
        "87Rb": 0.2783,
        "84Sr": 0.0056,
        "86Sr": 0.0986,
        "87Sr": 0.0700,
        "Sr": 0.8258,
        "Y": 1.0,
        "Zr": 0.5145,
        "91Zr": 0.1122,
        "92Zr": 0.1715,
        "94Zr": 0.1738,
        "96Zr": 0.0280,
        "Nb": 1.0,
        "92Mo": 0.1453,
        "94Mo": 0.0915,
        "95Mo": 0.1584,
        "96Mo": 0.1667,
        "97Mo": 0.0960,
        "Mo": 0.2439,
        "100Mo": 0.0982,
        "99Tc": 0,  # consider removing
        "96Ru": 0.0554,
        "98Ru": 0.0187,
        "99Ru": 0.1276,
        "100Ru": 0.1260,
        "101Ru": 0.1706,
        "Ru": 0.3155,
        "104Ru": 0.1862,
        "Rh": 1.0,
        "102Pd": 0.0102,
        "104Pd": 0.1114,
        "105Pd": 0.2233,
        "Pd": 0.2733,
        "108Pd": 0.2646,
        "110Pd": 0.1172,
        "Ag": 0.51839,
        "109Ag": 0.48161,
        "106Cd": 0.0125,
        "108Cd": 0.0089,
        "110Cd": 0.1249,
        "111Cd": 0.1280,
        "Cd": 0.2413,
        "113Cd": 0.1222,
        "114Cd": 0.2873,
        "116Cd": 0.0749,
        "113In": 0.0429,
        "In": 0.9571,
        "112Sn": 0.0097,
        "114Sn": 0.0066,
        "115Sn": 0.0034,
        "116Sn": 0.1454,
        "117Sn": 0.0768,
        "118Sn": 0.2422,
        "119Sn": 0.0859,
        "Sn": 0.3258,
        "122Sn": 0.0463,
        "124Sn": 0.0579,
        "Sb": 0.5721,
        "123Sb": 0.4279,
        "120Te": 0.0009,
        "122Te": 0.0255,
        "123Te": 0.0089,
        "124Te": 0.0474,
        "125Te": 0.0707,
        "126Te": 0.1884,
        "128Te": 0.3174,
        "Te": 0.3408,
        "I": 1.0,
        "124Xe": 0.000952,
        "126Xe": 0.000890,
        "128Xe": 0.019102,
        "129Xe": 0.264006,
        "130Xe": 0.040710,
        "131Xe": 0.212324,
        "Xe": 0.269086,
        "134Xe": 0.104357,
        "136Xe": 0.088573,
        "Cs": 1.0,
        "130Ba": 0.00106,
        "132Ba": 0.00101,
        "134Ba": 0.02417,
        "135Ba": 0.06592,
        "136Ba": 0.07854,
        "137Ba": 0.11232,
        "Ba": 0.71698,
        "138La": 0.0008881,
        "La": 0.9991119,
        "136Ce": 0.00185,
        "138Ce": 0.00251,
        "Ce": 0.88450,
        "142Ce": 0.11114,
        "Pr": 1.0,
        "Nd": 0.27152,
        "143Nd": 0.12174,
        "144Nd": 0.23798,
        "145Nd": 0.08293,
        "146Nd": 0.17189,
        "148Nd": 0.05756,
        "150Nd": 0.05638,
        "145Pm": 0,
        "147Pm": 0,
        "144Sm": 0.0307,
        "147Sm": 0.1499,
        "148Sm": 0.1124,
        "149Sm": 0.1382,
        "150Sm": 0.0738,
        "Sm": 0.2675,
        "154Sm": 0.2275,
        "151Eu": 0.4781,
        "Eu": 0.5219,
        "152Gd": 0.0020,
        "154Gd": 0.0218,
        "155Gd": 0.1480,
        "156Gd": 0.2047,
        "157Gd": 0.1565,
        "Gd": 0.2484,
        "160Gd": 0.2186,
        "Tb": 1.0,
        "156Dy": 0.00056,
        "158Dy": 0.00095,
        "160Dy": 0.02329,
        "161Dy": 0.18889,
        "162Dy": 0.25475,
        "163Dy": 0.24896,
        "Dy": 0.28260,
        "Ho": 1.0,
        "162Er": 0.00139,
        "164Er": 0.01601,
        "Er": 0.33503,
        "167Er": 0.22869,
        "168Er": 0.26978,
        "170Er": 0.14910,
        "Tm": 1.0,
        "168Yb": 0.00123,
        "170Yb": 0.02982,
        "171Yb": 0.1409,
        "172Yb": 0.2168,
        "173Yb": 0.16103,
        "Yb": 0.32026,
        "176Yb": 0.12996,
        "Lu": 0.97401,
        "176Lu": 0.02599,
        "174Hf": 0.0016,
        "176Hf": 0.0526,
        "177Hf": 0.1860,
        "178Hf": 0.2728,
        "179Hf": 0.1362,
        "Hf": 0.3508,
        "180Ta": 0.0001201,
        "Ta": 0.9998799,
        "180W": 0.0012,
        "182W": 0.2650,
        "183W": 0.1431,
        "W": 0.3064,
        "186W": 0.2843,
        "185Re": 0.3740,
        "Re": 0.6260,
        "184Os": 0.0002,
        "186Os": 0.0159,
        "187Os": 0.0196,
        "188Os": 0.1324,
        "189Os": 0.1615,
        "190Os": 0.2626,
        "Os": 0.4078,
        "191Ir": 0.373,
        "Ir": 0.627,
        "190Pt": 0.00012,
        "192Pt": 0.00782,
        "194Pt": 0.3286,
        "Pt": 0.3378,
        "196Pt": 0.2521,
        "198Pt": 0.07356,
        "Au": 1.0,
        "196Hg": 0.0015,
        "198Hg": 0.0997,
        "199Hg": 0.16872,
        "200Hg": 0.2310,
        "201Hg": 0.1318,
        "Hg": 0.2986,
        "204Hg": 0.0687,
        "203Tl": 0.2952,
        "Tl": 0.7048,
        "204Pb": 0.014,
        "206Pb": 0.241,
        "207Pb": 0.221,
        "Pb": 0.524,
        "Bi": 1.0,
        "209Po": 0,
        "210Po": 0,
        "210At": 0,
        "211At": 0,
        "211Rn": 0,
        "220Rn": 0,
        "222Rn": 0,
        "223Fr": 0,
        "223Ra": 0,
        "224Ra": 0,
        "226Ra": 0,
        "228Ra": 0,
        "227Ac": 0,
        "230Th": 0,
        "Th": 1.0,
        "Pa": 1.0,
        "233U": 0,
        "234U": 0.000054,
        "235U": 0.007204,
        "236U": 0,
        "U": 0.992742,
        "236Np": 0,
        "237Np": 0,
        "238Pu": 0,
        "239Pu": 0,
        "240Pu": 0,
        "241Pu": 0,
        "242Pu": 0,
        "244Pu": 0,
        "241Am": 0,
        "243Am": 0,
        "243Cm": 0,
        "244Cm": 0,
        "245Cm": 0,
        "246Cm": 0,
        "247Cm": 0,
        "248Cm": 0,
        "247Bk": 0,
        "249Bk": 0,
        "249Cf": 0,
        "250Cf": 0,
        "251Cf": 0,
        "252Cf": 0,
        "252Es": 0,
        "257Fm": 0,
        "258Md": 0,
        "260Md": 0,
        "259No": 0,
        "262Lr": 0,
        "267Rf": 0,
        "268Db": 0,
        "271Sg": 0,
        "272Bh": 0,
        "270Hs": 0,
        "276Mt": 0,
        "281Ds": 0,
        "280Rg": 0,
        "285Cn": 0,
        "284Nh": 0,
        "289Fl": 0,
        "288Mc": 0,
        "293Lv": 0,
        "292Ts": 0,
        "294Og": 0,
    }

    # Isotopes here is a dictionary of symbol, including full name,
    # and then the isotopes which arent the most abundant one, sorted by abundance.
    # None indicates no stable isotopes/naturally occuring ones.
    # This has been manually checked as far as Iodine only.
    isotopes = {
        "H": ["Hydrogen", ["D", "T"]],
        "He": ["Helium", ["3He"]],
        "Li": ["Lithium", ["6Li"]],
        "Be": ["Beryllium", [None]],
        "B": ["Boron", ["10B"]],
        "C": ["Carbon", ["13C"]],
        "N": ["Nitrogen", ["15N"]],
        "O": ["Oxygen", ["18O", "17O"]],
        "F": ["Fluorine", [None]],
        "Ne": ["Neon", ["22Ne", "21Ne"]],
        "Na": ["Sodium", [None]],
        "Mg": ["Magnesium", ["26Mg", "25Mg"]],
        "Al": ["Aluminum", [None]],
        "Si": ["Silicon", ["29Si", "30Si"]],
        "P": ["Phosphorus", [None]],
        "S": ["Sulfur", ["34S", "33S", "36S"]],
        "Cl": ["Chlorine", ["37Cl"]],
        "Ar": ["Argon", ["36Ar", "38Ar"]],
        "K": ["Potassium", ["41K", "40K"]],
        "Ca": ["Calcium", ["44Ca", "48Ca", "43Ca", "42Ca", "46Ca"]],
        "Sc": ["Scandium", [None]],
        "Ti": ["Titanium", ["46Ti", "47Ti", "49Ti", "50Ti"]],
        "V": ["Vanadium", ["50V"]],
        "Cr": ["Chromium", ["53Cr", "50Cr", "54Cr"]],
        "Mn": ["Manganese", [None]],
        "Fe": ["Iron", ["54Fe", "57Fe", "58Fe"]],
        "Co": ["Cobalt", [None]],
        "Ni": ["Nickel", ["60Ni", "62Ni", "61Ni", "64Ni"]],
        "Cu": ["Copper", ["65Cu"]],
        "Zn": ["Zinc", ["66Zn", "68Zn", "67Zn", "70Zn"]],
        "Ga": ["Gallium", ["71Ga"]],
        "Ge": ["Germanium", ["72Ge", "70Ge", "73Ge", "76Ge"]],
        "As": ["Arsenic", [None]],
        "Se": ["Selenium", ["78Se", "76Se", "82Se", "77Se", "74Se"]],
        "Br": ["Bromine", ["81Br"]],
        "Kr": ["Krypton", ["86Kr", "82Kr", "83Kr", "80Kr"]],
        "Rb": ["Rubidium", ["87Rb"]],
        "Sr": ["Strontium", ["86Sr", "87Sr", "84Sr"]],
        "Y": ["Yttrium", [None]],
        "Zr": ["Zirconium", ["94Zr", "92Zr", "91Zr", "96Zr"]],
        "Nb": ["Niobium", [None]],
        "Mo": ["Molybdenum", ["96Mo", "95Mo", "92Mo", "100Mo", "97Mo", "94Mo"]],
        "Tc": ["Technetium", [None]],  # consider removing
        "Ru": ["Ruthenium", ["104Ru", "101Ru", "99Ru", "100Ru", "96Ru", "98Ru"]],
        "Rh": ["Rhodium", [None]],
        "Pd": ["Palladium", ["108Pd", "105Pd", "110Pd", "104Pd", "102Pd"]],
        "Ag": ["Silver", ["109Ag"]],
        "Cd": [
            "Cadmium",
            ["114Cd", "111Cd", "110Cd", "113Cd", "116Cd", "106Cd", "108Cd"],
        ],
        "In": ["Indium", ["113In"]],
        "Sn": [
            "Tin",
            [
                "118Sn",
                "116Sn",
                "119Sn",
                "117Sn",
                "124Sn",
                "122Sn",
                "112Sn",
                "114Sn",
                "115Sn",
            ],
        ],
        "Sb": ["Antimony", ["123Sb"]],
        "Te": [
            "Tellurium",
            ["128Te", "126Te", "125Te", "124Te", "122Te", "123Te", "120Te"],
        ],
        "I": ["Iodine", [None]],
        "Xe": ["Xenon", ["129Xe", "131Xe", "134Xe", "136Xe", "130Xe", "128Xe"]],
        "Cs": ["Cesium", [None]],
        "Ba": ["Barium", ["137Ba", "136Ba", "135Ba", "134Ba"]],
        "La": ["Lanthanum", ["138La"]],
        "Hf": ["Hafnium", ["178Hf", "177Hf", "179Hf", "176Hf"]],
        "Ta": ["Tantalum", ["180Ta"]],
        "W": ["Tungsten", ["186W", "182W", "183W"]],
        "Re": ["Rhenium", ["185Re"]],
        "Os": ["Osmium", ["190Os", "189Os", "188Os", "187Os", "186Os"]],
        "Ir": ["Iridium", ["191Ir"]],
        "Pt": ["Platinum", ["194Pt", "196Pt", "198Pt", "192Pt"]],
        "Au": ["Gold", [None]],
        "Hg": ["Mercury", ["200Hg", "199Hg", "201Hg", "198Hg", "204Hg"]],
        "Tl": ["Thallium", ["203Tl"]],
        "Pb": ["Lead", ["206Pb", "207Pb", "204Pb"]],
        "Bi": ["Bismuth", [None]],
        "Po": ["Polonium", [None]],
        "At": ["Astatine", [None]],
        "Rn": ["Radon", [None]],
        "Fr": ["Francium", [None]],
        "Ra": ["Radium", [None]],
        "Ac": ["Actinium", [None]],
        "Rf": ["Rutherfordium", [None]],
        "Db": ["Dubnium", [None]],
        "Sg": ["Seaborgium", [None]],
        "Bh": ["Bohrium", [None]],
        "Hs": ["Hassium", [None]],
        "Mt": ["Meitnerium", [None]],
        "Ds": ["Darmstadtium", [None]],
        "Rg": ["Roentgenium", [None]],
        "Cn": ["Copernicium", [None]],
        "Nh": ["Nihonium", [None]],
        "Fl": ["Flerovium", [None]],
        "Mc": ["Moscovium", [None]],
        "Lv": ["Livermorium", [None]],
        "Ts": ["Tennessine", [None]],
        "Og": ["Oganesson", [None]],
        "Ce": ["Cerium", ["142Ce", "138Ce" "136Ce"]],
        "Pr": ["Praseodymium", [None]],
        "Nd": ["Neodymium", [None]],
        "Pm": ["Promethium", [None]],
        "Sm": ["Samarium", [None]],
        "Eu": ["Europium", [None]],
        "Gd": ["Gadolinium", [None]],
        "Tb": ["Terbium", [None]],
        "Dy": ["Dysprosium", [None]],
        "Ho": ["Holmium", [None]],
        "Er": ["Erbium", [None]],
        "Tm": ["Thulium", [None]],
        "Yb": ["Ytterbium", [None]],
        "Lu": ["Lutetium", ["176Lu"]],
        "Th": ["Thorium", [None]],
        "Pa": ["Protactinium", [None]],
        "U": ["Uranium", ["235U", "234U"]],
        "Np": ["Neptunium", [None]],
        "Pu": ["Plutonium", [None]],
        "Am": ["Americium", [None]],
        "Cm": ["Curium", [None]],
        "Bk": ["Berkelium", [None]],
        "Cf": ["Californium", [None]],
        "Es": ["Einsteinium", [None]],
        "Fm": ["Fermium", [None]],
        "Md": ["Mendelevium", [None]],
        "No": ["Nobelium", [None]],
        "Lr": ["Lawrencium", [None]],
    }
