import requests
from requests import Response
from tldextract import TLDExtract
from tldextract.tldextract import ExtractResult
from sourceRank.models.source_credibility_models import (SuffixRankDoc, OpenRankDoc,
                                                         CategoryAnalysisDoc, NewsAPISourceDoc,
                                                         WhoisAnalysisDoc, TwitterRankAnalysisDoc)
from sourceRank.helper.country_domain_list import top_level_country_domains
from sourceRank.helper.settings import logger
from sourceRank.helper.utils import compute_importance_from_exponential_distribution
from sourceRank.connectors.news_api_connector import NewsAPIConnector
from sourceRank.connectors.twitter_api_connector import TwitterConnector
from tweepy.models import User


class SourceCredibilityAnalysis(object):
    def __init__(self, open_rank_api_key: str, whois_api_key: str, news_api_key: str,
                 consumer_key: str, consumer_secret: str, access_token: str,
                 access_token_secret: str):
        self.open_rank_api_key: str = open_rank_api_key
        self.whois_api_key: str = whois_api_key
        self.news_api_key: str = news_api_key
        self.news_api_connector: NewsAPIConnector = NewsAPIConnector(
            api_key=self.news_api_key)
        self.twitter_connector: TwitterConnector = TwitterConnector(
            consumer_key=consumer_key,
            consumer_secret=consumer_secret,
            access_token=access_token,
            access_token_secret=access_token_secret)

    @staticmethod
    def parse_url(url: str) -> ExtractResult:
        tld_res: ExtractResult = ExtractResult(
            domain="", subdomain="", suffix="")
        try:
            tld_extract_obj: TLDExtract = TLDExtract(
                include_psl_private_domains=True)
            tld_res: ExtractResult = tld_extract_obj(
                url=url, include_psl_private_domains=True)

        except Exception as e:
            logger.error(e)
        return tld_res

    def get_open_page_rank(self, domain: str) -> Response:
        response: Response = Response()
        try:
            headers: dict = {"API-OPR": self.open_rank_api_key}
            url: str = "https://openpagerank.com/api/v1.0/getPageRank"
            params: dict = {"domains[]": domain}
            response: Response = requests.get(
                url=url, params=params,
                headers=headers)
        except Exception as e:
            logger.error(e)
        return response

    def get_whois_data_from_domain(self, domain: str) -> Response:
        response: Response = Response()
        try:
            url: str = "https://www.whoisxmlapi.com/whoisserver/WhoisService"
            params: dict = {"apiKey": self.whois_api_key,
                            "domainName": domain,
                            "outputFormat": "JSON"}
            response = requests.get(
                url=url, params=params)
        except Exception as e:
            logger.error(e)
        return response

    def get_whois_analysis(self, domain: str) -> WhoisAnalysisDoc:
        whois_analysis: WhoisAnalysisDoc = object.__new__(WhoisAnalysisDoc)
        try:
            response_whois: Response = self.get_whois_data_from_domain(
                domain=domain)

            # If the result is fine
            if response_whois.status_code == 200:
                whois_data: dict = response_whois.json().get("WhoisRecord", {})
                whois_analysis: WhoisAnalysisDoc = WhoisAnalysisDoc(
                    whois_data)

        except Exception as e:
            logger.error(e)
        return whois_analysis

    def get_open_rank_analysis(self, domain: str) -> OpenRankDoc:
        open_rank_analysis: OpenRankDoc = OpenRankDoc(
            domain=domain, page_rank_decimal=-1, rank=-1, last_updated="",
            analysed=False)
        try:
            response_open_rank: Response = self.get_open_page_rank(domain=domain)
            if response_open_rank.status_code == 200:
                response_open_rank_dict = response_open_rank.json()
                result_dict: dict = response_open_rank_dict.get("response")[0]

                open_rank_analysis: OpenRankDoc = OpenRankDoc(
                    domain= result_dict.get("domain", domain),
                    page_rank_decimal=result_dict.get(
                        "page_rank_decimal", -1),
                    rank=float(result_dict.get(
                        "rank", -1)),
                    last_updated=response_open_rank_dict.get(
                        "last_updated", ""),
                    analysed=True)

        except Exception as e:
            logger.error(e)
        return open_rank_analysis

    @staticmethod
    def get_suffix_mapping() -> dict:
        suffix_mapping: dict = {"edu": 5, "gov": 5, "org": 3,
                                "mil": 3, "net": 2, "com": 2}

        # Retrieve country domain
        country_domains: dict = dict(zip(
            top_level_country_domains.keys(),
            [4 for i in range(len(top_level_country_domains.keys()))]))

        # Update mapping with country domain
        suffix_mapping.update(country_domains)
        return suffix_mapping

    def get_suffix_analysis(self, suffix: str) -> SuffixRankDoc:
        suffix_analysis: SuffixRankDoc = object.__new__(SuffixRankDoc)
        try:
            suffix_mapping: dict = self.get_suffix_mapping()
            importance: float = suffix_mapping.get(suffix, 1)
            rank: float = round(compute_importance_from_exponential_distribution(
                x=importance), 1)
            suffix_analysis: SuffixRankDoc = SuffixRankDoc(
                suffix=suffix, importance=importance, rank=rank)
        except Exception as e:
            logger.error(e)
        return suffix_analysis

    @staticmethod
    def get_category_mapping() -> dict:
        category_mapping: dict = {'science': 5,
                                  'technology': 4,
                                  'entertainment': 3,
                                  'sports': 3,
                                  'health': 4,
                                  'general': 4,
                                  'business': 4}
        return category_mapping

    def get_category_analysis(self, url: str, category: str = None,
                              country_code: str = None, language: str = None) -> CategoryAnalysisDoc:
        category_analysis: CategoryAnalysisDoc = object.__new__(CategoryAnalysisDoc)
        try:
            candidate: NewsAPISourceDoc = self.news_api_connector.get_source_matching_from_url(
                candidate_source=url,
                category=category,
                country_code=country_code,
                language=language)

            category_mapping: dict = self.get_category_mapping()
            importance: float = category_mapping.get(candidate.category, 1)
            rank: float = round(compute_importance_from_exponential_distribution(
                x=importance), 1)

            # Create output object
            category_analysis: CategoryAnalysisDoc = CategoryAnalysisDoc(
                id=candidate.id,
                name=candidate.name,
                description=candidate.description,
                url=candidate.url,
                category=candidate.category,
                language=candidate.language,
                country_code=candidate.country,
                importance=importance,
                rank=rank)
        except Exception as e:
            logger.error(e)
        return category_analysis

    def get_twitter_rank(self, url: str) -> TwitterRankAnalysisDoc:
        twitter_rank_analysis: CategoryAnalysisDoc = object.__new__(CategoryAnalysisDoc)

        try:
            # 1. Start connection
            if self.twitter_connector.api is None:
                self.twitter_connector.set_up_twitter_api_connection()

            # 2. Extract Twitter account
            user_data: User = self.twitter_connector.extract_twitter_account_by_searching(
                api=self.twitter_connector.api, keyword=url)
            response_twitter_credibility: dict= self.twitter_connector.get_credibility_rank_from_user_account(
                user=user_data)

            # 3. Compute Twitter Rank
            twitter_rank_analysis: TwitterRankAnalysisDoc = TwitterRankAnalysisDoc(
                activity=response_twitter_credibility.get("activity"),
                popularity=response_twitter_credibility.get("popularity"),
                influence=response_twitter_credibility.get("influence"),
                rank=response_twitter_credibility.get("credibility"))
        except Exception as e:
            logger.error(e)
        return twitter_rank_analysis

