"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInsightsVersion = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const region_info_1 = require("@aws-cdk/region-info");
// This is the name of the mapping that will be added to the CloudFormation template, if a stack is region agnostic
const DEFAULT_MAPPING_PREFIX = 'LambdaInsightsVersions';
// To add new versions, update fact-tables.ts `CLOUDWATCH_LAMBDA_INSIGHTS_ARNS` and create a new `public static readonly VERSION_A_B_C_D`
/**
 * Version of CloudWatch Lambda Insights.
 *
 * @stability stable
 */
class LambdaInsightsVersion {
    constructor() {
        /**
         * The arn of the Lambda Insights extension.
         *
         * @stability stable
         */
        this.layerVersionArn = '';
    }
    /**
     * Use the insights extension associated with the provided ARN.
     *
     * Make sure the ARN is associated
     * with same region as your function
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-extension-versions.html
     * @stability stable
     */
    static fromInsightVersionArn(arn) {
        class InsightsArn extends LambdaInsightsVersion {
            constructor() {
                super(...arguments);
                this.layerVersionArn = arn;
            }
        }
        return new InsightsArn();
    }
    // Use the verison to build the object. Not meant to be called by the user -- user should use e.g. VERSION_1_0_54_0
    static fromInsightsVersion(insightsVersion) {
        // Check if insights version is valid. This should only happen if one of the public static readonly versions are set incorrectly
        const versionExists = region_info_1.RegionInfo.regions.some(regionInfo => regionInfo.cloudwatchLambdaInsightsArn(insightsVersion));
        if (!versionExists) {
            throw new Error(`Insights version ${insightsVersion} does not exist.`);
        }
        class InsightsVersion extends LambdaInsightsVersion {
            constructor() {
                super(...arguments);
                this.layerVersionArn = core_1.Lazy.uncachedString({
                    produce: (context) => getVersionArn(context, insightsVersion),
                });
            }
        }
        return new InsightsVersion();
    }
}
exports.LambdaInsightsVersion = LambdaInsightsVersion;
_a = JSII_RTTI_SYMBOL_1;
LambdaInsightsVersion[_a] = { fqn: "@aws-cdk/aws-lambda.LambdaInsightsVersion", version: "1.134.0" };
/**
 * Version 1.0.54.0.
 *
 * @stability stable
 */
LambdaInsightsVersion.VERSION_1_0_54_0 = LambdaInsightsVersion.fromInsightsVersion('1.0.54.0');
/**
 * Version 1.0.86.0.
 *
 * @stability stable
 */
LambdaInsightsVersion.VERSION_1_0_86_0 = LambdaInsightsVersion.fromInsightsVersion('1.0.86.0');
/**
 * Version 1.0.89.0.
 *
 * @stability stable
 */
LambdaInsightsVersion.VERSION_1_0_89_0 = LambdaInsightsVersion.fromInsightsVersion('1.0.89.0');
/**
 * Version 1.0.98.0.
 *
 * @stability stable
 */
LambdaInsightsVersion.VERSION_1_0_98_0 = LambdaInsightsVersion.fromInsightsVersion('1.0.98.0');
/**
 * Function to retrieve the correct Lambda Insights ARN from RegionInfo,
 * or create a mapping to look it up at stack deployment time.
 *
 * This function is run on CDK synthesis.
 */
function getVersionArn(context, insightsVersion) {
    const scopeStack = core_1.Stack.of(context.scope);
    const region = scopeStack.region;
    // Region is defined, look up the arn, or throw an error if the version isn't supported by a region
    if (region !== undefined && !core_1.Token.isUnresolved(region)) {
        const arn = region_info_1.RegionInfo.get(region).cloudwatchLambdaInsightsArn(insightsVersion);
        if (arn === undefined) {
            throw new Error(`Insights version ${insightsVersion} is not supported in region ${region}`);
        }
        return arn;
    }
    // Otherwise, need to add a mapping to be looked up at deployment time
    /**
     * See this for the context as to why the mappings are the way they are
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html
     *
     * Mappings have to have a structure like this, and no functions can be used inside them:
     * <Alphanumeric only>
     * - <Can be non-alphanumeric>
     * -- { <alphanumeric>: "value1"},
     * -- { <alphanumeric>: "value2"}
     *
     * So we cannot have an otherwise ideal mapping like this, because '1.0.98.0' is non-alphanumeric:
     * LambdaInsightsVersions
     * - us-east-1
     * -- {'1.0.98.0': 'arn1'},
     * -- {'1.0.89.0': 'arn2'}
     *
     * To get around this limitation, this is the mapping structure:
     * LambdaInsightsVersions10980 // for version 1.0.98.0
     * - us-east-1
     * -- {'arn': 'arn1'},
     * - us-east-2
     * -- {'arn': 'arn2'}
     * LambdaInsightsVersions10890 // a separate mapping version 1.0.89.0
     * - us-east-1
     * -- {'arn': 'arn3'},
     * - us-east-2
     * -- {'arn': 'arn4'}
     */
    const mapName = DEFAULT_MAPPING_PREFIX + insightsVersion.split('.').join('');
    const mapping = {};
    const region2arns = region_info_1.RegionInfo.regionMap(region_info_1.FactName.cloudwatchLambdaInsightsVersion(insightsVersion));
    for (const [reg, arn] of Object.entries(region2arns)) {
        mapping[reg] = { arn };
    }
    // Only create a given mapping once. If another version of insights is used elsewhere, that mapping will also exist
    if (!scopeStack.node.tryFindChild(mapName)) {
        new core_1.CfnMapping(scopeStack, mapName, { mapping });
    }
    // The ARN will be looked up at deployment time from the mapping we created
    return core_1.Fn.findInMap(mapName, core_1.Aws.REGION, 'arn');
}
//# sourceMappingURL=data:application/json;base64,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