"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QualifiedFunctionBase = exports.FunctionBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const event_invoke_config_1 = require("./event-invoke-config");
const event_source_mapping_1 = require("./event-source-mapping");
const lambda_generated_1 = require("./lambda.generated");
const util_1 = require("./util");
/**
 * @stability stable
 */
class FunctionBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Mapping of invocation principals to grants. Used to de-dupe `grantInvoke()` calls.
         * @internal
         */
        this._invocationGrants = {};
    }
    /**
     * Adds a permission to the Lambda resource policy.
     *
     * @param id The id for the permission construct.
     * @param permission The permission to grant to this Lambda function.
     * @see Permission for details.
     * @stability stable
     */
    addPermission(id, permission) {
        var _c, _d, _e, _f, _g;
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_Permission(permission);
        if (!this.canCreatePermissions) {
            // FIXME: @deprecated(v2) - throw an error if calling `addPermission` on a resource that doesn't support it.
            return;
        }
        const principal = this.parsePermissionPrincipal(permission.principal);
        const { sourceAccount, sourceArn } = (_c = this.parseConditions(permission.principal)) !== null && _c !== void 0 ? _c : {};
        const action = (_d = permission.action) !== null && _d !== void 0 ? _d : 'lambda:InvokeFunction';
        const scope = (_e = permission.scope) !== null && _e !== void 0 ? _e : this;
        new lambda_generated_1.CfnPermission(scope, id, {
            action,
            principal,
            functionName: this.functionArn,
            eventSourceToken: permission.eventSourceToken,
            sourceAccount: (_f = permission.sourceAccount) !== null && _f !== void 0 ? _f : sourceAccount,
            sourceArn: (_g = permission.sourceArn) !== null && _g !== void 0 ? _g : sourceArn,
        });
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     *
     * @stability stable
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Access the Connections object.
     *
     * Will fail if not a VPC-enabled Lambda Function
     *
     * @stability stable
     */
    get connections() {
        if (!this._connections) {
            // eslint-disable-next-line max-len
            throw new Error('Only VPC-associated Lambda Functions have security groups to manage. Supply the "vpc" parameter when creating the Lambda, or "securityGroupId" when importing it.');
        }
        return this._connections;
    }
    /**
     * The `$LATEST` version of this function.
     *
     * Note that this is reference to a non-specific AWS Lambda version, which
     * means the function this version refers to can return different results in
     * different invocations.
     *
     * To obtain a reference to an explicit version which references the current
     * function configuration, use `lambdaFunction.currentVersion` instead.
     *
     * @stability stable
     */
    get latestVersion() {
        if (!this._latestVersion) {
            this._latestVersion = new LatestVersion(this);
        }
        return this._latestVersion;
    }
    /**
     * Whether or not this Lambda function was bound to a VPC.
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     *
     * @stability stable
     */
    get isBoundToVpc() {
        return !!this._connections;
    }
    /**
     * Adds an event source that maps to this AWS Lambda function.
     *
     * @stability stable
     */
    addEventSourceMapping(id, options) {
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_EventSourceMappingOptions(options);
        return new event_source_mapping_1.EventSourceMapping(this, id, {
            target: this,
            ...options,
        });
    }
    /**
     * Grant the given identity permissions to invoke this Lambda.
     *
     * @stability stable
     */
    grantInvoke(grantee) {
        const identifier = `Invoke${grantee.grantPrincipal}`; // calls the .toString() of the principal
        // Memoize the result so subsequent grantInvoke() calls are idempotent
        let grant = this._invocationGrants[identifier];
        if (!grant) {
            grant = iam.Grant.addToPrincipalOrResource({
                grantee,
                actions: ['lambda:InvokeFunction'],
                resourceArns: [this.functionArn],
                // Fake resource-like object on which to call addToResourcePolicy(), which actually
                // calls addPermission()
                resource: {
                    addToResourcePolicy: (_statement) => {
                        // Couldn't add permissions to the principal, so add them locally.
                        this.addPermission(identifier, {
                            principal: grantee.grantPrincipal,
                            action: 'lambda:InvokeFunction',
                        });
                        const permissionNode = this._functionNode().tryFindChild(identifier);
                        if (!permissionNode) {
                            throw new Error('Cannot modify permission to lambda function. Function is either imported or $LATEST version. '
                                + 'If the function is imported from the same account use `fromFunctionAttributes()` API with the `sameEnvironment` flag.');
                        }
                        return { statementAdded: true, policyDependable: permissionNode };
                    },
                    node: this.node,
                    stack: this.stack,
                    env: this.env,
                },
            });
            this._invocationGrants[identifier] = grant;
        }
        return grant;
    }
    /**
     * Adds an event source to this function.
     *
     * Event sources are implemented in the @aws-cdk/aws-lambda-event-sources module.
     *
     * The following example adds an SQS Queue as an event source:
     * ```
     * import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
     * myFunction.addEventSource(new SqsEventSource(myQueue));
     * ```
     *
     * @stability stable
     */
    addEventSource(source) {
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_IEventSource(source);
        source.bind(this);
    }
    /**
     * Configures options for asynchronous invocation.
     *
     * @stability stable
     */
    configureAsyncInvoke(options) {
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_EventInvokeConfigOptions(options);
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this,
            ...options,
        });
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * For use internally for constructs, when the tree is set up in non-standard ways. Ex: SingletonFunction.
     * @internal
     */
    _functionNode() {
        return this.node;
    }
    /**
     * Given the function arn, check if the account id matches this account
     *
     * Function ARNs look like this:
     *
     *   arn:aws:lambda:region:account-id:function:function-name
     *
     * ..which means that in order to extract the `account-id` component from the ARN, we can
     * split the ARN using ":" and select the component in index 4.
     *
     * @returns true if account id of function matches the account specified on the stack, false otherwise.
     *
     * @internal
     */
    _isStackAccount() {
        if (core_1.Token.isUnresolved(this.stack.account) || core_1.Token.isUnresolved(this.functionArn)) {
            return false;
        }
        return this.stack.splitArn(this.functionArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).account === this.stack.account;
    }
    /**
     * Translate IPrincipal to something we can pass to AWS::Lambda::Permissions
     *
     * Do some nasty things because `Permission` supports a subset of what the
     * full IAM principal language supports, and we may not be able to parse strings
     * outright because they may be tokens.
     *
     * Try to recognize some specific Principal classes first, then try a generic
     * fallback.
     */
    parsePermissionPrincipal(principal) {
        // Try some specific common classes first.
        // use duck-typing, not instance of
        // @deprecated: after v2, we can change these to 'instanceof'
        if ('conditions' in principal) {
            // eslint-disable-next-line dot-notation
            principal = principal['principal'];
        }
        if ('accountId' in principal) {
            return principal.accountId;
        }
        if ('service' in principal) {
            return principal.service;
        }
        if ('arn' in principal) {
            return principal.arn;
        }
        // Try a best-effort approach to support simple principals that are not any of the predefined
        // classes, but are simple enough that they will fit into the Permission model. Main target
        // here: imported Roles, Users, Groups.
        //
        // The principal cannot have conditions and must have a single { AWS: [arn] } entry.
        const json = principal.policyFragment.principalJson;
        if (Object.keys(principal.policyFragment.conditions).length === 0 && json.AWS) {
            if (typeof json.AWS === 'string') {
                return json.AWS;
            }
            if (Array.isArray(json.AWS) && json.AWS.length === 1 && typeof json.AWS[0] === 'string') {
                return json.AWS[0];
            }
        }
        throw new Error(`Invalid principal type for Lambda permission statement: ${principal.constructor.name}. ` +
            'Supported: AccountPrincipal, ArnPrincipal, ServicePrincipal');
    }
    parseConditions(principal) {
        if (this.isPrincipalWithConditions(principal)) {
            const conditions = principal.policyFragment.conditions;
            const conditionPairs = util_1.flatMap(Object.entries(conditions), ([operator, conditionObjs]) => Object.keys(conditionObjs).map(key => { return { operator, key }; }));
            const supportedPrincipalConditions = [{ operator: 'ArnLike', key: 'aws:SourceArn' }, { operator: 'StringEquals', key: 'aws:SourceAccount' }];
            const unsupportedConditions = conditionPairs.filter((condition) => !supportedPrincipalConditions.some((supportedCondition) => supportedCondition.operator === condition.operator && supportedCondition.key === condition.key));
            if (unsupportedConditions.length == 0) {
                return {
                    sourceAccount: conditions.StringEquals['aws:SourceAccount'],
                    sourceArn: conditions.ArnLike['aws:SourceArn'],
                };
            }
            else {
                throw new Error(`PrincipalWithConditions had unsupported conditions for Lambda permission statement: ${JSON.stringify(unsupportedConditions)}. ` +
                    `Supported operator/condition pairs: ${JSON.stringify(supportedPrincipalConditions)}`);
            }
        }
        else {
            return null;
        }
    }
    isPrincipalWithConditions(principal) {
        return 'conditions' in principal;
    }
}
exports.FunctionBase = FunctionBase;
_a = JSII_RTTI_SYMBOL_1;
FunctionBase[_a] = { fqn: "@aws-cdk/aws-lambda.FunctionBase", version: "1.134.0" };
/**
 * @stability stable
 */
class QualifiedFunctionBase extends FunctionBase {
    constructor() {
        super(...arguments);
        /**
         * The construct node where permissions are attached.
         *
         * @stability stable
         */
        this.permissionsNode = this.node;
    }
    /**
     * The `$LATEST` version of this function.
     *
     * Note that this is reference to a non-specific AWS Lambda version, which
     * means the function this version refers to can return different results in
     * different invocations.
     *
     * To obtain a reference to an explicit version which references the current
     * function configuration, use `lambdaFunction.currentVersion` instead.
     *
     * @stability stable
     */
    get latestVersion() {
        return this.lambda.latestVersion;
    }
    /**
     * Configures options for asynchronous invocation.
     *
     * @stability stable
     */
    configureAsyncInvoke(options) {
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_EventInvokeConfigOptions(options);
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the qualified function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this.lambda,
            qualifier: this.qualifier,
            ...options,
        });
    }
}
exports.QualifiedFunctionBase = QualifiedFunctionBase;
_b = JSII_RTTI_SYMBOL_1;
QualifiedFunctionBase[_b] = { fqn: "@aws-cdk/aws-lambda.QualifiedFunctionBase", version: "1.134.0" };
/**
 * The $LATEST version of a function, useful when attempting to create aliases.
 */
class LatestVersion extends FunctionBase {
    constructor(lambda) {
        super(lambda, '$LATEST');
        this.version = '$LATEST';
        this.permissionsNode = this.node;
        this.canCreatePermissions = false;
        this.lambda = lambda;
    }
    get functionArn() {
        return `${this.lambda.functionArn}:${this.version}`;
    }
    get functionName() {
        return `${this.lambda.functionName}:${this.version}`;
    }
    get grantPrincipal() {
        return this.lambda.grantPrincipal;
    }
    get latestVersion() {
        return this;
    }
    get role() {
        return this.lambda.role;
    }
    addAlias(aliasName, options = {}) {
        return util_1.addAlias(this, this, aliasName, options);
    }
    get edgeArn() {
        throw new Error('$LATEST function version cannot be used for Lambda@Edge');
    }
}
//# sourceMappingURL=data:application/json;base64,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