# -*- coding: utf-8 -*-

"""
@date: 2022/4/3 下午1:40
@file: infer.py
@author: zj
@description: 
"""

import time

from typing import Tuple
from yacs.config import CfgNode

import torch
import torch.nn as nn
from torch.utils.data import DataLoader

from ..util.meter import AverageMeter
from ..util.prefetcher import data_prefetcher
from ..util.metric import accuracy
from ..util.distributed import reduce_tensor
from ..util.misc import to_python_float

from zcls2.util import logging

logger = logging.get_logger(__name__)


def validate(cfg: CfgNode, val_loader: DataLoader, model: nn.Module, criterion: nn.Module) -> Tuple[float, float]:
    batch_time = AverageMeter()
    losses = AverageMeter()
    top1 = AverageMeter()
    top5 = AverageMeter()

    # switch to evaluate mode
    model.eval()

    end = time.time()

    prefetcher = data_prefetcher(cfg, val_loader)
    input, target = prefetcher.next()
    i = 0
    while input is not None:
        i += 1

        # compute output
        with torch.no_grad():
            output = model(input)
            loss = criterion(output, target)

        # measure accuracy and record loss
        prec1, prec5 = accuracy(output.data, target, topk=(1, 5))

        if cfg.DISTRIBUTED:
            reduced_loss = reduce_tensor(cfg.NUM_GPUS, loss.data)
            prec1 = reduce_tensor(cfg.NUM_GPUS, prec1)
            prec5 = reduce_tensor(cfg.NUM_GPUS, prec5)
        else:
            reduced_loss = loss.data

        losses.update(to_python_float(reduced_loss), input.size(0))
        top1.update(to_python_float(prec1), input.size(0))
        top5.update(to_python_float(prec5), input.size(0))

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()

        # TODO:  Change timings to mirror train().
        if cfg.RANK_ID == 0 and i % cfg.PRINT_FREQ == 0:
            logger.info('Test: [{0}/{1}] '
                        'Time {batch_time.val:.3f} ({batch_time.avg:.3f}) '
                        'Speed {2:.3f} ({3:.3f}) '
                        'Loss {loss.val:.4f} ({loss.avg:.4f}) '
                        'Prec@1 {top1.val:.3f} ({top1.avg:.3f}) '
                        'Prec@5 {top5.val:.3f} ({top5.avg:.3f})'.format(
                i, len(val_loader),
                cfg.NUM_GPUS * cfg.DATALOADER.TRAIN_BATCH_SIZE / batch_time.val,
                cfg.NUM_GPUS * cfg.DATALOADER.TRAIN_BATCH_SIZE / batch_time.avg,
                batch_time=batch_time, loss=losses,
                top1=top1, top5=top5))

        input, target = prefetcher.next()

    logger.info(' * Prec@1 {top1.avg:.3f} Prec@5 {top5.avg:.3f}'
                .format(top1=top1, top5=top5))

    return top1.avg, top5.avg
