from typing import Optional, Union

from pydantic import AnyUrl, BaseModel, EmailStr, Extra, validator


class Contact(BaseModel):
    """Contact information for the exposed API."""

    name: Optional[str] = None
    """
    The identifying name of the contact person/organization.
    """

    url: Optional[AnyUrl] = None
    """
    The URL pointing to the contact information.
    MUST be in the form of a URL.
    """

    email: Optional[Union[EmailStr, str]] = None
    """
    The email address of the contact person/organization.
    MUST be in the form of an email address.
    """

    @validator("email", pre=True)
    def validate_email(  # pylint: disable=no-self-argument
        cls,
        v: Union[EmailStr, str],
    ) -> EmailStr:
        """Validate email address.

        Args:
            v: Holds the email string to be validated

        Raises:
            ValueError: Value is not a valid email address

        Returns:
            Validated email string.
        """
        return EmailStr(v) if isinstance(v, str) else v

    class Config:
        extra = Extra.ignore
        schema_extra = {
            "examples": [
                {"name": "API Support", "url": "http://www.example.com/support", "email": "support@example.com"}
            ]
        }
