"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Semaphore = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_dynamodb_1 = require("aws-cdk-lib/aws-dynamodb");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
/**
 * Generates a semaphore for a StepFunction job (or chained set of jobs) to limit parallelism across executions.
 */
class Semaphore extends aws_stepfunctions_1.StateMachineFragment {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The DynamoDB table used to store semaphores.
         */
        this.tableName = 'StateMachineSempahoreTable920751a65a584e8ab7583460f6db686a';
        const stackTracker = this.setUpMap();
        this.semaphoreTable = this.ensureTable(props);
        let lockInfo = stackTracker.get(props.lockName);
        if (lockInfo) {
            if (props.reuseLock) {
                if (lockInfo.limit != props.limit) {
                    throw new Error(`The reused \`lockName\` "${props.lockName}" was given a different \`limit\` than previously defined. Given: ${props.limit}, Previous: ${lockInfo.limit}.`);
                }
                else {
                    lockInfo = { limit: lockInfo.limit, timesUsed: lockInfo.timesUsed + 1 };
                    stackTracker.set(props.lockName, lockInfo);
                }
            }
            else {
                throw new Error(`The \`lockName\` "${props.lockName}" was reused without explicitly allowing reuse. Set \`reuseLock\` to \`true\` if you want to reuse the lock.`);
            }
        }
        else {
            lockInfo = { limit: props.limit, timesUsed: 1 };
            stackTracker.set(props.lockName, lockInfo);
        }
        const getLock = new aws_stepfunctions_1.Parallel(this, `Get ${props.lockName} Lock: ${lockInfo.timesUsed}`, { resultPath: aws_stepfunctions_1.JsonPath.DISCARD });
        const acquireLock = new aws_stepfunctions_tasks_1.DynamoUpdateItem(this, `Acquire ${props.lockName} Lock: ${lockInfo.timesUsed}`, {
            comment: props.comments ? `Acquire a lock using a conditional update to DynamoDB. This update will do two things:
          1) increment a counter for the number of held locks
          2) add an attribute to the DynamoDB Item with a unique key for this execution and with a value of the time when the lock was Acquired
          The Update includes a conditional expression that will fail under two circumstances:
          1) if the maximum number of locks have already been distributed
          2) if the current execution already owns a lock. The latter check is important to ensure the same execution doesn't increase the counter more than once
          If either of these conditions are not met, then the task will fail with a DynamoDB.ConditionalCheckFailedException error, retry a few times, then if it is still not successful \
          it will move off to another branch of the workflow. If this is the first time that a given lockname has been used, there will not be a row in DynamoDB \
          so the update will fail with DynamoDB.AmazonDynamoDBException. In that case, this state sends the workflow to state that will create that row to initialize.
          ` : undefined,
            table: this.semaphoreTable,
            key: { LockName: aws_stepfunctions_tasks_1.DynamoAttributeValue.fromString(props.lockName) },
            expressionAttributeNames: {
                '#currentlockcount': 'currentlockcount',
                '#lockownerid.$': '$$.Execution.Id',
            },
            expressionAttributeValues: {
                ':increase': aws_stepfunctions_tasks_1.DynamoAttributeValue.fromNumber(1),
                ':limit': aws_stepfunctions_tasks_1.DynamoAttributeValue.fromNumber(props.limit),
                ':lockacquiredtime': aws_stepfunctions_tasks_1.DynamoAttributeValue.fromString(aws_stepfunctions_1.JsonPath.stringAt('$$.State.EnteredTime')),
            },
            updateExpression: 'SET #currentlockcount = #currentlockcount + :increase, #lockownerid = :lockacquiredtime',
            conditionExpression: 'currentlockcount <> :limit and attribute_not_exists(#lockownerid)',
            returnValues: aws_stepfunctions_tasks_1.DynamoReturnValues.UPDATED_NEW,
            resultPath: '$.lockinfo.acquirelock',
        });
        const initializeLockItem = new aws_stepfunctions_tasks_1.DynamoPutItem(this, `Initialize ${props.lockName} Lock Item: ${lockInfo.timesUsed}`, {
            comment: props.comments ? `This state handles the case where an item hasn't been created for this lock yet. \
      In that case, it will insert an initial item that includes the lock name as the key and currentlockcount of 0. \ 
      The Put to DynamoDB includes a conditonal expression to fail if the an item with that key already exists, which avoids a race condition if multiple executions start at the same time. \ 
      There are other reasons that the previous state could fail and end up here, so this is safe in those cases too.` : undefined,
            table: this.semaphoreTable,
            item: {
                LockName: aws_stepfunctions_tasks_1.DynamoAttributeValue.fromString(props.lockName),
                currentlockcount: aws_stepfunctions_tasks_1.DynamoAttributeValue.fromNumber(0),
            },
            conditionExpression: 'LockName <> :lockname',
            expressionAttributeValues: {
                ':lockname': aws_stepfunctions_tasks_1.DynamoAttributeValue.fromString(props.lockName),
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        const getCurrentLockRecord = new aws_stepfunctions_tasks_1.DynamoGetItem(this, `Get Current ${props.lockName} Lock Record: ${lockInfo.timesUsed}`, {
            comment: props.comments ? 'This state is called when the execution is unable to acquire a lock because there limit has either been exceeded or because this execution already holds a lock. \
      In that case, this task loads info from DDB for the current lock item so that the right decision can be made in subsequent states.' : undefined,
            table: this.semaphoreTable,
            key: { LockName: aws_stepfunctions_tasks_1.DynamoAttributeValue.fromString(props.lockName) },
            expressionAttributeNames: { '#lockownerid.$': '$$.Execution.Id' },
            projectionExpression: [new aws_stepfunctions_tasks_1.DynamoProjectionExpression().withAttribute('#lockownerid')],
            resultSelector: {
                'Item.$': '$.Item',
                'ItemString.$': 'States.JsonToString($.Item)',
            },
            resultPath: '$.lockinfo.currentlockitem',
        });
        const checkIfLockAcquired = new aws_stepfunctions_1.Choice(this, `Check if ${props.lockName} Lock Already Acquired: ${lockInfo.timesUsed}`, {
            comment: props.comments ? `This state checks to see if the current execution already holds a lock. It can tell that by looking for Z, which will be indicative of the timestamp value. \ 
      That will only be there in the stringified version of the data returned from DDB if this execution holds a lock.` : undefined,
        });
        const continueBecauseLockWasAlreadyAcquired = new aws_stepfunctions_1.Pass(this, `Continue Because ${props.lockName} Lock Was Already Acquired: ${lockInfo.timesUsed}`, {
            comment: props.comments ? 'In this state, we have confimed that lock is already held, so we pass the original execution input into the the function that does the work.' : undefined,
        });
        const waitToGetLock = new aws_stepfunctions_1.Wait(this, `Wait to Get ${props.lockName} Lock: ${lockInfo.timesUsed}`, {
            comment: props.comments ? 'If the lock indeed not been succesfully Acquired, then wait for a bit before trying again.' : undefined,
            time: aws_stepfunctions_1.WaitTime.duration(aws_cdk_lib_1.Duration.seconds(3)),
        });
        acquireLock.addRetry({ errors: ['DynamoDB.AmazonDynamoDBException'], maxAttempts: 0 })
            .addRetry({ maxAttempts: 6, backoffRate: 2 })
            .addCatch(initializeLockItem, { errors: ['DynamoDB.AmazonDynamoDBException'], resultPath: '$.lockinfo.acquisitionerror' })
            .addCatch(getCurrentLockRecord, { errors: ['DynamoDB.ConditionalCheckFailedException'], resultPath: '$.lockinfo.acquisitionerror' });
        initializeLockItem.addCatch(acquireLock, { resultPath: aws_stepfunctions_1.JsonPath.DISCARD });
        getCurrentLockRecord.next(checkIfLockAcquired);
        checkIfLockAcquired.when(aws_stepfunctions_1.Condition.and(aws_stepfunctions_1.Condition.isPresent('$.lockinfo.currentlockitem.ItemString'), aws_stepfunctions_1.Condition.stringMatches('$.lockinfo.currentlockitem.ItemString', '*Z')), continueBecauseLockWasAlreadyAcquired);
        checkIfLockAcquired.otherwise(waitToGetLock);
        waitToGetLock.next(acquireLock);
        const releaseLock = new aws_stepfunctions_tasks_1.DynamoUpdateItem(this, `Release ${props.lockName} Lock: ${lockInfo.timesUsed}`, {
            table: this.semaphoreTable,
            key: { LockName: aws_stepfunctions_tasks_1.DynamoAttributeValue.fromString(props.lockName) },
            expressionAttributeNames: {
                '#currentlockcount': 'currentlockcount',
                '#lockownerid.$': '$$.Execution.Id',
            },
            expressionAttributeValues: {
                ':decrease': aws_stepfunctions_tasks_1.DynamoAttributeValue.fromNumber(1),
            },
            updateExpression: 'SET #currentlockcount = #currentlockcount - :decrease REMOVE #lockownerid',
            conditionExpression: 'attribute_exists(#lockownerid)',
            returnValues: aws_stepfunctions_tasks_1.DynamoReturnValues.UPDATED_NEW,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        releaseLock.addRetry({ errors: ['DynamoDB.ConditionalCheckFailedException'], maxAttempts: 0 })
            .addRetry({ maxAttempts: 5, backoffRate: 1.5 })
            .addCatch(props.nextState, { errors: ['DynamoDB.ConditionalCheckFailedException'], resultPath: '$.lockinfo.acquisitionerror' })
            .next(props.nextState);
        getLock.branch(acquireLock);
        getLock.endStates.forEach(j => j.next(props.job));
        props.job.next(releaseLock);
        this.startState = getLock;
        this.endStates = props.nextState.endStates;
    }
    setUpMap() {
        const stackId = aws_cdk_lib_1.Names.uniqueId(aws_cdk_lib_1.Stack.of(this));
        const existing = aws_cdk_lib_1.Stack.of(this).node.tryFindChild(this.tableName);
        if (existing) {
            return (Semaphore.semaphoreTracker.get(stackId));
        }
        else {
            const m = new Map();
            Semaphore.semaphoreTracker.set(stackId, m);
            return m;
        }
    }
    ensureTable(props) {
        const existing = aws_cdk_lib_1.Stack.of(this).node.tryFindChild(this.tableName);
        if (existing) {
            if (props.tableReadWriteCapacity) {
                throw new Error('`tableReadWriteCapacity` can only be specified on the first instance of the `Semaphore` construct in each stack.');
            }
            // Just assume this is true
            return existing;
        }
        else {
            return new aws_dynamodb_1.Table(aws_cdk_lib_1.Stack.of(this), this.tableName, {
                partitionKey: {
                    name: 'LockName',
                    type: aws_dynamodb_1.AttributeType.STRING,
                },
                readCapacity: props.tableReadWriteCapacity?.readCapacity,
                writeCapacity: props.tableReadWriteCapacity?.writeCapacity,
                billingMode: props.tableReadWriteCapacity ? aws_dynamodb_1.BillingMode.PROVISIONED : aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            });
        }
    }
}
exports.Semaphore = Semaphore;
_a = JSII_RTTI_SYMBOL_1;
Semaphore[_a] = { fqn: "@dontirun/state-machine-semaphore.Semaphore", version: "0.1.93" };
/**
 * The names and associated concurrency limits and number of uses of the sempahores.
 */
Semaphore.semaphoreTracker = new Map();
//# sourceMappingURL=data:application/json;base64,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