"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * VirtualService represents a service inside an AppMesh
 *
 * It routes traffic either to a Virtual Node or to a Virtual Router.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
 */
class VirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualServiceName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        if (props.virtualNode && props.virtualRouter) {
            throw new Error('Must provide only one of virtualNode or virtualRouter for the provider');
        }
        this.mesh = props.mesh;
        // Check which provider to use node or router (or neither)
        if (props.virtualRouter) {
            this.virtualServiceProvider = this.addVirtualRouter(props.virtualRouter.virtualRouterName);
        }
        if (props.virtualNode) {
            this.virtualServiceProvider = this.addVirtualNode(props.virtualNode.virtualNodeName);
        }
        const svc = new appmesh_generated_1.CfnVirtualService(this, 'Resource', {
            meshName: this.mesh.meshName,
            virtualServiceName: this.physicalName,
            spec: {
                provider: this.virtualServiceProvider,
            },
        });
        this.virtualServiceName = this.getResourceNameAttribute(svc.attrVirtualServiceName);
        this.virtualServiceArn = this.getResourceArnAttribute(svc.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualService`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualService given an ARN
     */
    static fromVirtualServiceArn(scope, id, virtualServiceArn) {
        return new ImportedVirtualService(scope, id, {
            virtualServiceArn,
        });
    }
    /**
     * Import an existing VirtualService given mesh and service names
     */
    static fromVirtualServiceName(scope, id, meshName, virtualServiceName) {
        return new ImportedVirtualService(scope, id, {
            meshName,
            virtualServiceName,
        });
    }
    addVirtualRouter(name) {
        return {
            virtualRouter: {
                virtualRouterName: name,
            },
        };
    }
    addVirtualNode(name) {
        return {
            virtualNode: {
                virtualNodeName: name,
            },
        };
    }
}
exports.VirtualService = VirtualService;
/**
 * Returns properties that allows a VirtualService to be imported
 */
class ImportedVirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualServiceArn) {
            this.virtualServiceArn = props.virtualServiceArn;
            this.virtualServiceName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualServiceArn).resourceName));
        }
        else if (props.virtualServiceName && props.meshName) {
            this.virtualServiceName = props.virtualServiceName;
            this.virtualServiceArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualService`,
                resourceName: this.virtualServiceName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidmlydHVhbC1zZXJ2aWNlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidmlydHVhbC1zZXJ2aWNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEscUNBQXFDO0FBRXJDLDJEQUF3RDtBQWdFeEQ7Ozs7OztHQU1HO0FBQ0gsTUFBYSxjQUFlLFNBQVEsR0FBRyxDQUFDLFFBQVE7SUFpQzlDLFlBQVksS0FBb0IsRUFBRSxFQUFVLEVBQUUsS0FBMEI7UUFDdEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUM7U0FDdEcsQ0FBQyxDQUFDO1FBRUgsSUFBSSxLQUFLLENBQUMsV0FBVyxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDNUMsTUFBTSxJQUFJLEtBQUssQ0FBQyx3RUFBd0UsQ0FBQyxDQUFDO1NBQzNGO1FBRUQsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDO1FBRXZCLDBEQUEwRDtRQUMxRCxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDdkIsSUFBSSxDQUFDLHNCQUFzQixHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLGlCQUFpQixDQUFDLENBQUM7U0FDNUY7UUFDRCxJQUFJLEtBQUssQ0FBQyxXQUFXLEVBQUU7WUFDckIsSUFBSSxDQUFDLHNCQUFzQixHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxlQUFlLENBQUMsQ0FBQztTQUN0RjtRQUVELE1BQU0sR0FBRyxHQUFHLElBQUkscUNBQWlCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNsRCxRQUFRLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRO1lBQzVCLGtCQUFrQixFQUFFLElBQUksQ0FBQyxZQUFZO1lBQ3JDLElBQUksRUFBRTtnQkFDSixRQUFRLEVBQUUsSUFBSSxDQUFDLHNCQUFzQjthQUN0QztTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDcEYsSUFBSSxDQUFDLGlCQUFpQixHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFO1lBQzdELE9BQU8sRUFBRSxTQUFTO1lBQ2xCLFFBQVEsRUFBRSxRQUFRLEtBQUssQ0FBQyxJQUFJLENBQUMsUUFBUSxpQkFBaUI7WUFDdEQsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO1NBQ2hDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFqRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMscUJBQXFCLENBQUMsS0FBb0IsRUFBRSxFQUFVLEVBQUUsaUJBQXlCO1FBQzdGLE9BQU8sSUFBSSxzQkFBc0IsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFHO1lBQzVDLGlCQUFpQjtTQUNsQixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsc0JBQXNCLENBQUMsS0FBb0IsRUFBRSxFQUFVLEVBQUUsUUFBZ0IsRUFBRSxrQkFBMEI7UUFDakgsT0FBTyxJQUFJLHNCQUFzQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDM0MsUUFBUTtZQUNSLGtCQUFrQjtTQUNuQixDQUFDLENBQUM7SUFDTCxDQUFDO0lBa0RPLGdCQUFnQixDQUFDLElBQVk7UUFDbkMsT0FBTztZQUNMLGFBQWEsRUFBRTtnQkFDYixpQkFBaUIsRUFBRSxJQUFJO2FBQ3hCO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFTyxjQUFjLENBQUMsSUFBWTtRQUNqQyxPQUFPO1lBQ0wsV0FBVyxFQUFFO2dCQUNYLGVBQWUsRUFBRSxJQUFJO2FBQ3RCO1NBQ0YsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQW5GRCx3Q0FtRkM7QUE4QkQ7O0dBRUc7QUFDSCxNQUFNLHNCQUF1QixTQUFRLEdBQUcsQ0FBQyxRQUFRO0lBVy9DLFlBQVksS0FBb0IsRUFBRSxFQUFVLEVBQUUsS0FBK0I7UUFDM0UsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLEtBQUssQ0FBQyxpQkFBaUIsRUFBRTtZQUMzQixJQUFJLENBQUMsaUJBQWlCLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDO1lBQ2pELElBQUksQ0FBQyxrQkFBa0IsR0FBRyxHQUFHLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxZQUFhLENBQUMsQ0FBQyxDQUFDO1NBQ3BJO2FBQU0sSUFBSSxLQUFLLENBQUMsa0JBQWtCLElBQUksS0FBSyxDQUFDLFFBQVEsRUFBRTtZQUNyRCxJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDO1lBQ25ELElBQUksQ0FBQyxpQkFBaUIsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7Z0JBQ3BELE9BQU8sRUFBRSxTQUFTO2dCQUNsQixRQUFRLEVBQUUsUUFBUSxLQUFLLENBQUMsUUFBUSxpQkFBaUI7Z0JBQ2pELFlBQVksRUFBRSxJQUFJLENBQUMsa0JBQWtCO2FBQ3RDLENBQUMsQ0FBQztTQUNKO2FBQU07WUFDTCxNQUFNLElBQUksS0FBSyxDQUFDLCtCQUErQixDQUFDLENBQUM7U0FDbEQ7SUFDSCxDQUFDO0NBQ0YiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbmltcG9ydCB7IENmblZpcnR1YWxTZXJ2aWNlIH0gZnJvbSAnLi9hcHBtZXNoLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBJTWVzaCB9IGZyb20gJy4vbWVzaCc7XG5pbXBvcnQgeyBJVmlydHVhbE5vZGUgfSBmcm9tICcuL3ZpcnR1YWwtbm9kZSc7XG5pbXBvcnQgeyBJVmlydHVhbFJvdXRlciB9IGZyb20gJy4vdmlydHVhbC1yb3V0ZXInO1xuXG4vKipcbiAqIFJlcHJlc2VudHMgdGhlIGludGVyZmFjZSB3aGljaCBhbGwgVmlydHVhbFNlcnZpY2UgYmFzZWQgY2xhc3NlcyBNVVNUIGltcGxlbWVudFxuICovXG5leHBvcnQgaW50ZXJmYWNlIElWaXJ0dWFsU2VydmljZSBleHRlbmRzIGNkay5JUmVzb3VyY2Uge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIFZpcnR1YWxTZXJ2aWNlXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IHZpcnR1YWxTZXJ2aWNlTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIFJlc291cmNlIE5hbWUgKEFSTikgZm9yIHRoZSB2aXJ0dWFsIHNlcnZpY2VcbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgdmlydHVhbFNlcnZpY2VBcm46IHN0cmluZztcbn1cblxuLyoqXG4gKiBUaGUgYmFzZSBwcm9wZXJ0aWVzIHdoaWNoIGFsbCBjbGFzc2VzIGluIFZpcnR1YWxTZXJ2aWNlIHdpbGwgaW5oZXJpdCBmcm9tXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVmlydHVhbFNlcnZpY2VCYXNlUHJvcHMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIFZpcnR1YWxTZXJ2aWNlLlxuICAgKlxuICAgKiBJdCBpcyByZWNvbW1lbmRlZCB0aGlzIGZvbGxvd3MgdGhlIGZ1bGx5LXF1YWxpZmllZCBkb21haW4gbmFtZSBmb3JtYXQsXG4gICAqIHN1Y2ggYXMgXCJteS1zZXJ2aWNlLmRlZmF1bHQuc3ZjLmNsdXN0ZXIubG9jYWxcIi5cbiAgICpcbiAgICogQGV4YW1wbGUgc2VydmljZS5kb21haW4ubG9jYWxcbiAgICogQGRlZmF1bHQgLSBBIG5hbWUgaXMgYXV0b21hdGljYWxseSBnZW5lcmF0ZWRcbiAgICovXG4gIHJlYWRvbmx5IHZpcnR1YWxTZXJ2aWNlTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIFZpcnR1YWxSb3V0ZXIgd2hpY2ggdGhlIFZpcnR1YWxTZXJ2aWNlIHVzZXMgYXMgcHJvdmlkZXJcbiAgICpcbiAgICogQGRlZmF1bHQgLSBBdCBtb3N0IG9uZSBvZiB2aXJ0dWFsUm91dGVyIGFuZCB2aXJ0dWFsTm9kZSBpcyBhbGxvd2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgdmlydHVhbFJvdXRlcj86IElWaXJ0dWFsUm91dGVyO1xuXG4gIC8qKlxuICAgKiBUaGUgVmlydHVhbE5vZGUgYXR0YWNoZWQgdG8gdGhlIHZpcnR1YWwgc2VydmljZVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEF0IG1vc3Qgb25lIG9mIHZpcnR1YWxSb3V0ZXIgYW5kIHZpcnR1YWxOb2RlIGlzIGFsbG93ZWQuXG4gICAqL1xuICByZWFkb25seSB2aXJ0dWFsTm9kZT86IElWaXJ0dWFsTm9kZTtcbn1cblxuLyoqXG4gKiBUaGUgcHJvcGVydGllcyBhcHBsaWVkIHRvIHRoZSBWaXJ0dWFsU2VydmljZSBiZWluZyBkZWZpbmVcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBWaXJ0dWFsU2VydmljZVByb3BzIGV4dGVuZHMgVmlydHVhbFNlcnZpY2VCYXNlUHJvcHMge1xuICAvKipcbiAgICogVGhlIEFwcE1lc2ggbWVzaCBuYW1lIGZvciB3aGljaCB0aGUgVmlydHVhbFNlcnZpY2UgYmVsb25ncyB0b1xuICAgKi9cbiAgcmVhZG9ubHkgbWVzaDogSU1lc2g7XG59XG5cbi8qKlxuICogVmlydHVhbFNlcnZpY2UgcmVwcmVzZW50cyBhIHNlcnZpY2UgaW5zaWRlIGFuIEFwcE1lc2hcbiAqXG4gKiBJdCByb3V0ZXMgdHJhZmZpYyBlaXRoZXIgdG8gYSBWaXJ0dWFsIE5vZGUgb3IgdG8gYSBWaXJ0dWFsIFJvdXRlci5cbiAqXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHAtbWVzaC9sYXRlc3QvdXNlcmd1aWRlL3ZpcnR1YWxfc2VydmljZXMuaHRtbFxuICovXG5leHBvcnQgY2xhc3MgVmlydHVhbFNlcnZpY2UgZXh0ZW5kcyBjZGsuUmVzb3VyY2UgaW1wbGVtZW50cyBJVmlydHVhbFNlcnZpY2Uge1xuICAvKipcbiAgICogSW1wb3J0IGFuIGV4aXN0aW5nIFZpcnR1YWxTZXJ2aWNlIGdpdmVuIGFuIEFSTlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tVmlydHVhbFNlcnZpY2VBcm4oc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHZpcnR1YWxTZXJ2aWNlQXJuOiBzdHJpbmcpOiBJVmlydHVhbFNlcnZpY2Uge1xuICAgIHJldHVybiBuZXcgSW1wb3J0ZWRWaXJ0dWFsU2VydmljZShzY29wZSwgaWQsICB7XG4gICAgICB2aXJ0dWFsU2VydmljZUFybixcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJbXBvcnQgYW4gZXhpc3RpbmcgVmlydHVhbFNlcnZpY2UgZ2l2ZW4gbWVzaCBhbmQgc2VydmljZSBuYW1lc1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tVmlydHVhbFNlcnZpY2VOYW1lKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBtZXNoTmFtZTogc3RyaW5nLCB2aXJ0dWFsU2VydmljZU5hbWU6IHN0cmluZyk6IElWaXJ0dWFsU2VydmljZSB7XG4gICAgcmV0dXJuIG5ldyBJbXBvcnRlZFZpcnR1YWxTZXJ2aWNlKHNjb3BlLCBpZCwge1xuICAgICAgbWVzaE5hbWUsXG4gICAgICB2aXJ0dWFsU2VydmljZU5hbWUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIFZpcnR1YWxTZXJ2aWNlLCBpdCBpcyByZWNvbW1lbmRlZCB0aGlzIGZvbGxvd3MgdGhlIGZ1bGx5LXF1YWxpZmllZCBkb21haW4gbmFtZSBmb3JtYXQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdmlydHVhbFNlcnZpY2VOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBbWF6b24gUmVzb3VyY2UgTmFtZSAoQVJOKSBmb3IgdGhlIHZpcnR1YWwgc2VydmljZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHZpcnR1YWxTZXJ2aWNlQXJuOiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSByZWFkb25seSB2aXJ0dWFsU2VydmljZVByb3ZpZGVyPzogQ2ZuVmlydHVhbFNlcnZpY2UuVmlydHVhbFNlcnZpY2VQcm92aWRlclByb3BlcnR5O1xuICBwcml2YXRlIHJlYWRvbmx5IG1lc2g6IElNZXNoO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogVmlydHVhbFNlcnZpY2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgcGh5c2ljYWxOYW1lOiBwcm9wcy52aXJ0dWFsU2VydmljZU5hbWUgfHwgY2RrLkxhenkuc3RyaW5nVmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLm5vZGUudW5pcXVlSWQgfSksXG4gICAgfSk7XG5cbiAgICBpZiAocHJvcHMudmlydHVhbE5vZGUgJiYgcHJvcHMudmlydHVhbFJvdXRlcikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdNdXN0IHByb3ZpZGUgb25seSBvbmUgb2YgdmlydHVhbE5vZGUgb3IgdmlydHVhbFJvdXRlciBmb3IgdGhlIHByb3ZpZGVyJyk7XG4gICAgfVxuXG4gICAgdGhpcy5tZXNoID0gcHJvcHMubWVzaDtcblxuICAgIC8vIENoZWNrIHdoaWNoIHByb3ZpZGVyIHRvIHVzZSBub2RlIG9yIHJvdXRlciAob3IgbmVpdGhlcilcbiAgICBpZiAocHJvcHMudmlydHVhbFJvdXRlcikge1xuICAgICAgdGhpcy52aXJ0dWFsU2VydmljZVByb3ZpZGVyID0gdGhpcy5hZGRWaXJ0dWFsUm91dGVyKHByb3BzLnZpcnR1YWxSb3V0ZXIudmlydHVhbFJvdXRlck5hbWUpO1xuICAgIH1cbiAgICBpZiAocHJvcHMudmlydHVhbE5vZGUpIHtcbiAgICAgIHRoaXMudmlydHVhbFNlcnZpY2VQcm92aWRlciA9IHRoaXMuYWRkVmlydHVhbE5vZGUocHJvcHMudmlydHVhbE5vZGUudmlydHVhbE5vZGVOYW1lKTtcbiAgICB9XG5cbiAgICBjb25zdCBzdmMgPSBuZXcgQ2ZuVmlydHVhbFNlcnZpY2UodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgbWVzaE5hbWU6IHRoaXMubWVzaC5tZXNoTmFtZSxcbiAgICAgIHZpcnR1YWxTZXJ2aWNlTmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgICBzcGVjOiB7XG4gICAgICAgIHByb3ZpZGVyOiB0aGlzLnZpcnR1YWxTZXJ2aWNlUHJvdmlkZXIsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGhpcy52aXJ0dWFsU2VydmljZU5hbWUgPSB0aGlzLmdldFJlc291cmNlTmFtZUF0dHJpYnV0ZShzdmMuYXR0clZpcnR1YWxTZXJ2aWNlTmFtZSk7XG4gICAgdGhpcy52aXJ0dWFsU2VydmljZUFybiA9IHRoaXMuZ2V0UmVzb3VyY2VBcm5BdHRyaWJ1dGUoc3ZjLnJlZiwge1xuICAgICAgc2VydmljZTogJ2FwcG1lc2gnLFxuICAgICAgcmVzb3VyY2U6IGBtZXNoLyR7cHJvcHMubWVzaC5tZXNoTmFtZX0vdmlydHVhbFNlcnZpY2VgLFxuICAgICAgcmVzb3VyY2VOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkVmlydHVhbFJvdXRlcihuYW1lOiBzdHJpbmcpOiBDZm5WaXJ0dWFsU2VydmljZS5WaXJ0dWFsU2VydmljZVByb3ZpZGVyUHJvcGVydHkge1xuICAgIHJldHVybiB7XG4gICAgICB2aXJ0dWFsUm91dGVyOiB7XG4gICAgICAgIHZpcnR1YWxSb3V0ZXJOYW1lOiBuYW1lLFxuICAgICAgfSxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBhZGRWaXJ0dWFsTm9kZShuYW1lOiBzdHJpbmcpOiBDZm5WaXJ0dWFsU2VydmljZS5WaXJ0dWFsU2VydmljZVByb3ZpZGVyUHJvcGVydHkge1xuICAgIHJldHVybiB7XG4gICAgICB2aXJ0dWFsTm9kZToge1xuICAgICAgICB2aXJ0dWFsTm9kZU5hbWU6IG5hbWUsXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBJbnRlcmZhY2Ugd2l0aCBwcm9wZXJ0aWVzIG5jZWNlc3NhcnkgdG8gaW1wb3J0IGEgcmV1c2FibGUgVmlydHVhbFNlcnZpY2VcbiAqL1xuaW50ZXJmYWNlIFZpcnR1YWxTZXJ2aWNlQXR0cmlidXRlcyB7XG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIFJlc291cmNlIE5hbWUgKEFSTikgZm9yIHRoZSB2aXJ0dWFsIHNlcnZpY2VcbiAgICpcbiAgICogQGRlZmF1bHQgLSBSZXF1aXJlZCBpZiB2aXJ0dWFsU2VydmljZU5hbWUgYW5kIHZpcnR1YWxNZXNoTmFtZSBhcmUgbm90IHN1cHBsaWVkLlxuICAgKi9cbiAgcmVhZG9ubHkgdmlydHVhbFNlcnZpY2VBcm4/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBWaXJ0dWFsU2VydmljZSwgaXQgaXMgcmVjb21tZW5kZWQgdGhpcyBmb2xsb3dzIHRoZSBmdWxseS1xdWFsaWZpZWQgZG9tYWluIG5hbWUgZm9ybWF0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFJlcXVpcmVkIGlmIHZpcnR1YWxTZXJ2aWNlQXJuIGlzIG5vdCBzdXBwbGllZC5cbiAgICovXG4gIHJlYWRvbmx5IHZpcnR1YWxTZXJ2aWNlTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIHNlcnZpY2UgbWVzaCB0aGF0IHRoZSB2aXJ0dWFsIHNlcnZpY2UgcmVzaWRlcyBpblxuICAgKlxuICAgKiBVc2VkIHRvIGRlcml2ZSBBUk4gZnJvbSBtZXNoIG5hbWUgaWYgQVJOIG5vdCBwcm92aWRlZFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFJlcXVpcmVkIGlmIHZpcnR1YWxTZXJ2aWNlQXJuIGlzIG5vdCBzdXBwbGllZC5cbiAgICovXG4gIHJlYWRvbmx5IG1lc2hOYW1lPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIFJldHVybnMgcHJvcGVydGllcyB0aGF0IGFsbG93cyBhIFZpcnR1YWxTZXJ2aWNlIHRvIGJlIGltcG9ydGVkXG4gKi9cbmNsYXNzIEltcG9ydGVkVmlydHVhbFNlcnZpY2UgZXh0ZW5kcyBjZGsuUmVzb3VyY2UgaW1wbGVtZW50cyBJVmlydHVhbFNlcnZpY2Uge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIFZpcnR1YWxTZXJ2aWNlLCBpdCBpcyByZWNvbW1lbmRlZCB0aGlzIGZvbGxvd3MgdGhlIGZ1bGx5LXF1YWxpZmllZCBkb21haW4gbmFtZSBmb3JtYXQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdmlydHVhbFNlcnZpY2VOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBbWF6b24gUmVzb3VyY2UgTmFtZSAoQVJOKSBmb3IgdGhlIHZpcnR1YWwgc2VydmljZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHZpcnR1YWxTZXJ2aWNlQXJuOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBWaXJ0dWFsU2VydmljZUF0dHJpYnV0ZXMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgaWYgKHByb3BzLnZpcnR1YWxTZXJ2aWNlQXJuKSB7XG4gICAgICB0aGlzLnZpcnR1YWxTZXJ2aWNlQXJuID0gcHJvcHMudmlydHVhbFNlcnZpY2VBcm47XG4gICAgICB0aGlzLnZpcnR1YWxTZXJ2aWNlTmFtZSA9IGNkay5Gbi5zZWxlY3QoMiwgY2RrLkZuLnNwbGl0KCcvJywgY2RrLlN0YWNrLm9mKHNjb3BlKS5wYXJzZUFybihwcm9wcy52aXJ0dWFsU2VydmljZUFybikucmVzb3VyY2VOYW1lISkpO1xuICAgIH0gZWxzZSBpZiAocHJvcHMudmlydHVhbFNlcnZpY2VOYW1lICYmIHByb3BzLm1lc2hOYW1lKSB7XG4gICAgICB0aGlzLnZpcnR1YWxTZXJ2aWNlTmFtZSA9IHByb3BzLnZpcnR1YWxTZXJ2aWNlTmFtZTtcbiAgICAgIHRoaXMudmlydHVhbFNlcnZpY2VBcm4gPSBjZGsuU3RhY2sub2YodGhpcykuZm9ybWF0QXJuKHtcbiAgICAgICAgc2VydmljZTogJ2FwcG1lc2gnLFxuICAgICAgICByZXNvdXJjZTogYG1lc2gvJHtwcm9wcy5tZXNoTmFtZX0vdmlydHVhbFNlcnZpY2VgLFxuICAgICAgICByZXNvdXJjZU5hbWU6IHRoaXMudmlydHVhbFNlcnZpY2VOYW1lLFxuICAgICAgfSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignTmVlZCBlaXRoZXIgYXJuIG9yIGJvdGggbmFtZXMnKTtcbiAgICB9XG4gIH1cbn1cbiJdfQ==