"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const mesh_1 = require("./mesh");
const route_1 = require("./route");
const shared_interfaces_1 = require("./shared-interfaces");
class VirtualRouterBase extends cdk.Resource {
    /**
     * Add a single route to the router
     */
    addRoute(id, props) {
        const route = new route_1.Route(this, id, {
            ...props,
            routeName: id,
            mesh: this.mesh,
            virtualRouter: this,
        });
        return route;
    }
}
class VirtualRouter extends VirtualRouterBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualRouterName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.listeners = new Array();
        this.mesh = props.mesh;
        this.addListener(props.listener || { portMapping: { port: 8080, protocol: shared_interfaces_1.Protocol.HTTP } });
        const router = new appmesh_generated_1.CfnVirtualRouter(this, 'Resource', {
            virtualRouterName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                listeners: this.listeners,
            },
        });
        this.virtualRouterName = this.getResourceNameAttribute(router.attrVirtualRouterName);
        this.virtualRouterArn = this.getResourceArnAttribute(router.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualRouter`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualRouter given an ARN
     */
    static fromVirtualRouterArn(scope, id, virtualRouterArn) {
        return new ImportedVirtualRouter(scope, id, { virtualRouterArn });
    }
    /**
     * Import an existing VirtualRouter given names
     */
    static fromVirtualRouterName(scope, id, meshName, virtualRouterName) {
        return new ImportedVirtualRouter(scope, id, { meshName, virtualRouterName });
    }
    /**
     * Import an existing virtual router given attributes
     */
    static fromVirtualRouterAttributes(scope, id, attrs) {
        return new ImportedVirtualRouter(scope, id, attrs);
    }
    /**
     * Add port mappings to the router
     */
    addListener(listener) {
        this.listeners.push({
            portMapping: listener.portMapping,
        });
    }
}
exports.VirtualRouter = VirtualRouter;
/**
 * Used to import a VirtualRouter and perform actions or read properties from
 */
class ImportedVirtualRouter extends VirtualRouterBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.mesh) {
            this._mesh = props.mesh;
        }
        if (props.meshName) {
            if (props.mesh) {
                throw new Error('Supply either \'mesh\' or \'meshName\', but not both');
            }
            this._mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', props.meshName);
        }
        if (props.virtualRouterArn) {
            this.virtualRouterArn = props.virtualRouterArn;
            this.virtualRouterName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualRouterArn).resourceName));
        }
        else if (props.virtualRouterName && props.meshName) {
            this.virtualRouterName = props.virtualRouterName;
            this.virtualRouterArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualRouter`,
                resourceName: this.virtualRouterName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
    /**
     * The AppMesh mesh the VirtualRouter belongs to
     */
    get mesh() {
        if (!this._mesh) {
            throw new Error('Please supply either \'mesh\' or \'meshName\' when calling \'fromVirtualRouterAttributes\'');
        }
        return this._mesh;
    }
}
//# sourceMappingURL=data:application/json;base64,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