"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const virtual_node_1 = require("./virtual-node");
const virtual_router_1 = require("./virtual-router");
const virtual_service_1 = require("./virtual-service");
/**
 * A utility enum defined for the egressFilter type property, the default of DROP_ALL,
 * allows traffic only to other resources inside the mesh, or API calls to amazon resources.
 *
 * @default DROP_ALL
 */
var MeshFilterType;
(function (MeshFilterType) {
    /**
     * Allows all outbound traffic
     */
    MeshFilterType["ALLOW_ALL"] = "ALLOW_ALL";
    /**
     * Allows traffic only to other resources inside the mesh, or API calls to amazon resources
     */
    MeshFilterType["DROP_ALL"] = "DROP_ALL";
})(MeshFilterType = exports.MeshFilterType || (exports.MeshFilterType = {}));
/**
 * Represents a new or imported AppMesh mesh
 */
class MeshBase extends cdk.Resource {
    /**
     * Adds a VirtualRouter to the Mesh with the given id and props
     */
    addVirtualRouter(id, props = {}) {
        return new virtual_router_1.VirtualRouter(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * Adds a VirtualService with the given id
     */
    addVirtualService(id, props = {}) {
        return new virtual_service_1.VirtualService(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * Adds a VirtualNode to the Mesh
     */
    addVirtualNode(id, props = {}) {
        return new virtual_node_1.VirtualNode(this, id, {
            ...props,
            mesh: this,
        });
    }
}
/**
 * Define a new AppMesh mesh
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html
 */
class Mesh extends MeshBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.meshName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        const mesh = new appmesh_generated_1.CfnMesh(this, 'Resource', {
            meshName: this.physicalName,
            spec: {
                egressFilter: props.egressFilter ? {
                    type: props.egressFilter,
                } : undefined,
            },
        });
        this.meshName = this.getResourceNameAttribute(mesh.attrMeshName);
        this.meshArn = this.getResourceArnAttribute(mesh.ref, {
            service: 'appmesh',
            resource: 'mesh',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing mesh by arn
     */
    static fromMeshArn(scope, id, meshArn) {
        const parts = cdk.Stack.of(scope).parseArn(meshArn);
        class Import extends MeshBase {
            constructor() {
                super(...arguments);
                this.meshName = parts.resourceName || '';
                this.meshArn = meshArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing mesh by name
     */
    static fromMeshName(scope, id, meshName) {
        const arn = cdk.Stack.of(scope).formatArn({
            service: 'appmesh',
            resource: 'mesh',
            resourceName: meshName,
        });
        class Import extends MeshBase {
            constructor() {
                super(...arguments);
                this.meshName = meshName;
                this.meshArn = arn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Mesh = Mesh;
//# sourceMappingURL=data:application/json;base64,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