import re
import operator as op
from copy import copy
from dataclasses import dataclass
from typing import Any

import click
import yaml
from linkml_runtime.utils.formatutils import camelcase, underscore
from linkml_runtime.utils.schemaview import SchemaView

from linkml_dataops.apiroot import ApiRoot

@dataclass
class ApiGenerator(ApiRoot):
    """
    Generates an CRUD datamodel from an existing Datamodel

    For each class MyClass, will generate:

    .. code:: yaml

      classes:
        AddMyClass:
          slot_usage:
            value: range: MyClass
        RemoveMyClass:
          slot_usage:
            value: range: MyClass
        MyClassQuery:
          slots: [constraints]
        MyClassfetchById:   ## if MyClass has an identifier
          slots: [id_value]

    These generated classes can be used to represent changes and queries respectively

    Instances of these objects can be used as currency in both changers and query engines respectively
    """

    def serialize(self, container_class=None, prefix_uri=None, include_slots_as_params=False):
        """
        Generates and serializes the CRUD API as yaml

        :param container_class:
        :param prefix_uri:
        :param include_slots_as_params: (experimental)
        :return:
        """
        sv = self.schemaview
        cns = sv.all_classes(imports=False).keys()
        if container_class != None:
            cns = self._get_top_level_classes(container_class)

        src = sv.schema
        name = f'{src.name}_api'
        if prefix_uri is None:
            prefix_uri = sv.schema.id.rstrip('/') + f"/{name}/"
        classes = {
            "LocalChange": {
                "mixin": True,
                "slots": [
                    "value",
                    "path"
                ],
            },
            "LocalQuery": {
                "mixin": True,
                "slots": [
                    "target_class",
                    "path",
                ],
            },
            "__Any": {
                "class_uri": "linkml:Any",
                "abstract": True
            }
        }
        schema = {

            "id": "https://w3id.org/linkml/tests/kitchen_sink_api",
            "name": name,
            "description": f"API for querying and manipulating objects from the {src.name} schema",
            "prefixes": {
                name: prefix_uri,
                "linkml": "https://w3id.org/linkml/"
            },
            "default_prefix": name,
            "imports": [
                "linkml:types",
                src.name
            ],
            "default_range": "string",
            "slots": {
                "value": {
                    "inlined": True
                },
                "path": {},
                "constraints": {
                    "range": "__Any"
                },
                "id_value": {},
                "target_class": {}
            },
            "classes": classes
        }

        cmts = ["This is an autogenerated class"]

        for cn in cns:
            c = sv.get_class(cn)
            if c.class_uri == 'linkml:Any':
                continue
            cn_camel = camelcase(cn)
            slot_names = sv.class_slots(cn)
            if not c.abstract and not c.mixin:
                classes[f'Add{cn_camel}'] = {
                    "description": f'A change action that adds a {cn_camel} to a database',
                    "comments": copy(cmts),
                    "mixins": "LocalChange",
                    "slot_usage": {
                        "value": {
                            "range": cn,
                            "inlined": True
                        }
                    }
                }
                classes[f'Remove{cn_camel}'] = {
                    "description": f'A change action that remoaves a {cn_camel} to a database',
                    "comments": copy(cmts),
                    "mixins": "LocalChange",
                    "slot_usage": {
                        "value": {
                            "range": cn,
                            "inlined": True
                        }
                    }
                }
            if include_slots_as_params:
                q_slot_names = slot_names
            else:
                q_slot_names = []
            classes[f'{cn_camel}Query'] = {
                "description": f'A query object for instances of {cn_camel} from a database',
                "comments": copy(cmts),
                "mixins": "LocalQuery",
                "slots": [
                    "constraints"
                ] + q_slot_names,
            }
            classes[f'{cn_camel}FetchById'] = {
                "description": f'A query object for fetching an instance of {cn_camel} from a database by id',
                "comments": copy(cmts),
                "mixins": "LocalChange",
                "slots": [
                    "id_value"
                ],
            }
        return yaml.safe_dump(schema, sort_keys=False)

@click.command()
@click.option('-R', '--container-class', help="name of class that contains top level objects")
@click.argument('schema')
def cli(schema,  **args):
    """
    Generates an API datamodel from an existing datamodel, both query and change classes.

    Example: if a schema contains a class Person, the following classes will be generated:

    Query classes: PersonQuery, PersonFetchById
    Change classes: AddPerson, RemovePerson

    These can be used with changer or query objects
    """
    sv = SchemaView(schema)
    gen = ApiGenerator(schemaview=sv)
    print(gen.serialize(**args))


if __name__ == '__main__':
    cli()
