###############################################################################
# (c) Copyright 2021 CERN for the benefit of the LHCb Collaboration           #
#                                                                             #
# This software is distributed under the terms of the GNU General Public      #
# Licence version 3 (GPL Version 3), copied verbatim in the file "COPYING".   #
#                                                                             #
# In applying this licence, CERN does not waive the privileges and immunities #
# granted to it by virtue of its status as an Intergovernmental Organization  #
# or submit itself to any jurisdiction.                                       #
###############################################################################
from textwrap import dedent

import pytest

import LbAPCommon
from LbAPCommon import checks
from LbAPCommon.checks_utils import checks_to_JSON

pytest.importorskip("XRootD")


def test_num_entries_parsing_to_JSON():
    rendered_yaml = dedent(
        """\
    checks:
        check_num_entries:
            type: num_entries
            count: 1000
            tree_pattern: DecayTree

    job_1:
        application: DaVinci/v45r8
        input:
            bk_query: /bookkeeping/path/ALLSTREAMS.DST
        output: FILETYPE.ROOT
        options:
            - options.py
            - $VAR/a.py
        wg: Charm
        inform: a.b@c.d
        checks:
            - check_num_entries
    """
    )
    jobs_data, checks_data = LbAPCommon.parse_yaml(rendered_yaml)

    job_name = list(jobs_data.keys())[0]
    check_name = list(checks_data.keys())[0]

    result = checks.run_job_checks(
        [check_name],
        checks_data,
        [
            "root://eospublic.cern.ch//eos/opendata/lhcb/AntimatterMatters2017/data/B2HHH_MagnetDown.root"
        ],
    )[check_name]

    check_results_with_job = {
        job_name: {
            check_name: result,
        }
    }

    checks_json = checks_to_JSON(checks_data, check_results_with_job)

    json_expected = dedent(
        """\
    {
      "job_1": {
        "check_num_entries": {
          "passed": true,
          "messages": [
            "Found 5135823 in DecayTree (1000 required)"
          ],
          "input": {
            "type": "num_entries",
            "count": 1000,
            "tree_pattern": "DecayTree"
          },
          "output": {
            "DecayTree": {
              "num_entries": 5135823
            }
          }
        }
      }
    }"""
    )

    assert checks_json == json_expected


def test_range_parsing_to_JSON():
    rendered_yaml = dedent(
        """\
    checks:
        check_range:
            type: range
            expression: H1_PZ
            limits:
                min: 0
                max: 500000
            blind_ranges:
                -
                    min: 80000
                    max: 100000
                -
                    min: 180000
                    max: 200000
            tree_pattern: DecayTree

    job_1:
        application: DaVinci/v45r8
        input:
            bk_query: /bookkeeping/path/ALLSTREAMS.DST
        output: FILETYPE.ROOT
        options:
            - options.py
            - $VAR/a.py
        wg: Charm
        inform: a.b@c.d
        checks:
            - check_range
    """
    )
    jobs_data, checks_data = LbAPCommon.parse_yaml(rendered_yaml)

    job_name = list(jobs_data.keys())[0]
    check_name = list(checks_data.keys())[0]

    result = checks.run_job_checks(
        [check_name],
        checks_data,
        [
            "root://eospublic.cern.ch//eos/opendata/lhcb/AntimatterMatters2017/data/B2HHH_MagnetDown.root"
        ],
    )[check_name]

    check_results_with_job = {
        job_name: {
            check_name: result,
        }
    }

    checks_json = checks_to_JSON(checks_data, check_results_with_job)

    json_expected = dedent(
        """\
{
  "job_1": {
    "check_range": {
      "passed": true,
      "messages": [
        "Histogram of H1_PZ successfully filled from TTree DecayTree (contains 4776546.0 events)"
      ],
      "input": {
        "type": "range",
        "expression": "H1_PZ",
        "limits": {
          "min": 0.0,
          "max": 500000.0
        },
        "blind_ranges": [
          {
            "min": 80000.0,
            "max": 100000.0
          },
          {
            "min": 180000.0,
            "max": 200000.0
          }
        ],
        "tree_pattern": "DecayTree",
        "n_bins": 50
      },
      "output": {
        "DecayTree": {
          "histograms": [
            {
              "version": 1,
              "name": "DecayTree H1_PZ",
              "axes": [
                {
                  "name": "H1_PZ",
                  "nbins": 50,
                  "min": 0.0,
                  "max": 500000.0
                }
              ],
              "contents": [
                0.0,
                1068515.0,
                911171.0,
                630749.0,
                449908.0,
                336446.0,
                265471.0,
                219142.0,
                186119.0,
                0.0,
                0.0,
                122888.0,
                107285.0,
                92548.0,
                78798.0,
                65261.0,
                53400.0,
                44413.0,
                36398.0,
                0.0,
                0.0,
                19816.0,
                16119.0,
                12874.0,
                10512.0,
                8506.0,
                6869.0,
                5591.0,
                4593.0,
                3772.0,
                3049.0,
                2522.0,
                2166.0,
                1754.0,
                1520.0,
                1286.0,
                1033.0,
                932.0,
                822.0,
                720.0,
                610.0,
                474.0,
                443.0,
                391.0,
                347.0,
                310.0,
                262.0,
                210.0,
                186.0,
                171.0,
                174.0,
                0.0
              ],
              "sumw2": [
                0.0,
                1068515.0,
                911171.0,
                630749.0,
                449908.0,
                336446.0,
                265471.0,
                219142.0,
                186119.0,
                0.0,
                0.0,
                122888.0,
                107285.0,
                92548.0,
                78798.0,
                65261.0,
                53400.0,
                44413.0,
                36398.0,
                0.0,
                0.0,
                19816.0,
                16119.0,
                12874.0,
                10512.0,
                8506.0,
                6869.0,
                5591.0,
                4593.0,
                3772.0,
                3049.0,
                2522.0,
                2166.0,
                1754.0,
                1520.0,
                1286.0,
                1033.0,
                932.0,
                822.0,
                720.0,
                610.0,
                474.0,
                443.0,
                391.0,
                347.0,
                310.0,
                262.0,
                210.0,
                186.0,
                171.0,
                174.0,
                0.0
              ]
            }
          ],
          "num_entries": 4776546,
          "mean": 44931.44209225662,
          "variance": 2682154203.3712554,
          "stddev": 51789.51827707278,
          "num_entries_in_mean_window": 0
        }
      }
    }
  }
}"""
    )

    assert checks_json == json_expected


def test_range_2d_parsing_to_JSON():
    rendered_yaml = dedent(
        """\
    checks:
        check_range_nd:
            type: range_nd
            expressions:
                x: H1_PZ
                y: H2_PZ
            limits:
                x:
                    min: 0
                    max: 500000
                y:
                    min: 0
                    max: 500000
            n_bins:
                x: 25
                y: 25
            tree_pattern: DecayTree

    job_1:
        application: DaVinci/v45r8
        input:
            bk_query: /bookkeeping/path/ALLSTREAMS.DST
        output: FILETYPE.ROOT
        options:
            - options.py
            - $VAR/a.py
        wg: Charm
        inform: a.b@c.d
        checks:
            - check_range_nd
    """
    )
    jobs_data, checks_data = LbAPCommon.parse_yaml(rendered_yaml)

    job_name = list(jobs_data.keys())[0]
    check_name = list(checks_data.keys())[0]

    result = checks.run_job_checks(
        [check_name],
        checks_data,
        [
            "root://eospublic.cern.ch//eos/opendata/lhcb/AntimatterMatters2017/data/B2HHH_MagnetDown.root"
        ],
    )[check_name]

    check_results_with_job = {
        job_name: {
            check_name: result,
        }
    }

    checks_json = checks_to_JSON(checks_data, check_results_with_job)

    json_expected = dedent(
        """\
{
  "job_1": {
    "check_range_nd": {
      "passed": true,
      "messages": [
        "Histogram of H1_PZ, H2_PZ successfully filled from TTree DecayTree (contains 5134453.0 events)"
      ],
      "input": {
        "type": "range_nd",
        "expressions": {
          "x": "H1_PZ",
          "y": "H2_PZ"
        },
        "limits": {
          "x": {
            "min": 0.0,
            "max": 500000.0
          },
          "y": {
            "min": 0.0,
            "max": 500000.0
          }
        },
        "n_bins": {
          "x": 25,
          "y": 25
        },
        "tree_pattern": "DecayTree"
      },
      "output": {
        "DecayTree": {
          "histograms": [
            {
              "version": 1,
              "name": "DecayTree H1_PZ/H2_PZ",
              "axes": [
                {
                  "name": "H1_PZ",
                  "nbins": 25,
                  "min": 0.0,
                  "max": 500000.0
                },
                {
                  "name": "H2_PZ",
                  "nbins": 25,
                  "min": 0.0,
                  "max": 500000.0
                }
              ],
              "contents": [
                [
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  1018980.0,
                  500540.0,
                  234750.0,
                  116342.0,
                  58377.0,
                  27836.0,
                  12717.0,
                  5471.0,
                  2326.0,
                  1118.0,
                  582.0,
                  292.0,
                  155.0,
                  89.0,
                  46.0,
                  27.0,
                  16.0,
                  11.0,
                  5.0,
                  2.0,
                  0.0,
                  3.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0
                ],
                [
                  0.0,
                  617269.0,
                  255329.0,
                  113352.0,
                  52786.0,
                  24158.0,
                  10419.0,
                  4186.0,
                  1613.0,
                  715.0,
                  352.0,
                  192.0,
                  121.0,
                  67.0,
                  30.0,
                  23.0,
                  11.0,
                  9.0,
                  5.0,
                  6.0,
                  0.0,
                  4.0,
                  4.0,
                  2.0,
                  1.0,
                  1.0,
                  2.0
                ],
                [
                  0.0,
                  364465.0,
                  132717.0,
                  58847.0,
                  26582.0,
                  11571.0,
                  4716.0,
                  1697.0,
                  609.0,
                  315.0,
                  161.0,
                  91.0,
                  55.0,
                  35.0,
                  17.0,
                  15.0,
                  8.0,
                  5.0,
                  2.0,
                  4.0,
                  3.0,
                  0.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  259905.0,
                  84353.0,
                  35695.0,
                  14985.0,
                  6261.0,
                  2356.0,
                  894.0,
                  386.0,
                  173.0,
                  94.0,
                  65.0,
                  31.0,
                  23.0,
                  17.0,
                  4.0,
                  4.0,
                  7.0,
                  3.0,
                  2.0,
                  0.0,
                  0.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0
                ],
                [
                  0.0,
                  208043.0,
                  57993.0,
                  22481.0,
                  9128.0,
                  3512.0,
                  1292.0,
                  466.0,
                  250.0,
                  97.0,
                  61.0,
                  50.0,
                  22.0,
                  16.0,
                  1.0,
                  7.0,
                  4.0,
                  1.0,
                  2.0,
                  3.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  2.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  167270.0,
                  40397.0,
                  14084.0,
                  5198.0,
                  1931.0,
                  680.0,
                  286.0,
                  157.0,
                  73.0,
                  32.0,
                  25.0,
                  15.0,
                  8.0,
                  6.0,
                  5.0,
                  3.0,
                  1.0,
                  0.0,
                  0.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  130047.0,
                  27584.0,
                  8606.0,
                  3062.0,
                  1149.0,
                  433.0,
                  215.0,
                  101.0,
                  55.0,
                  33.0,
                  23.0,
                  14.0,
                  10.0,
                  3.0,
                  0.0,
                  5.0,
                  1.0,
                  3.0,
                  0.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  92642.0,
                  17926.0,
                  5174.0,
                  1714.0,
                  645.0,
                  277.0,
                  126.0,
                  63.0,
                  36.0,
                  24.0,
                  15.0,
                  8.0,
                  4.0,
                  2.0,
                  1.0,
                  1.0,
                  0.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  64561.0,
                  11441.0,
                  3097.0,
                  953.0,
                  375.0,
                  183.0,
                  79.0,
                  47.0,
                  29.0,
                  22.0,
                  7.0,
                  8.0,
                  4.0,
                  2.0,
                  1.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  44079.0,
                  7366.0,
                  1860.0,
                  675.0,
                  230.0,
                  105.0,
                  72.0,
                  40.0,
                  20.0,
                  18.0,
                  7.0,
                  2.0,
                  3.0,
                  1.0,
                  0.0,
                  1.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  29506.0,
                  4528.0,
                  1150.0,
                  392.0,
                  165.0,
                  88.0,
                  46.0,
                  22.0,
                  15.0,
                  8.0,
                  5.0,
                  6.0,
                  0.0,
                  1.0,
                  3.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  19241.0,
                  2917.0,
                  733.0,
                  246.0,
                  109.0,
                  58.0,
                  30.0,
                  16.0,
                  15.0,
                  10.0,
                  4.0,
                  2.0,
                  2.0,
                  0.0,
                  0.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  12641.0,
                  1890.0,
                  479.0,
                  171.0,
                  85.0,
                  49.0,
                  20.0,
                  10.0,
                  10.0,
                  8.0,
                  4.0,
                  3.0,
                  1.0,
                  2.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  8424.0,
                  1220.0,
                  313.0,
                  104.0,
                  59.0,
                  23.0,
                  15.0,
                  11.0,
                  4.0,
                  4.0,
                  2.0,
                  2.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0
                ],
                [
                  0.0,
                  5641.0,
                  755.0,
                  224.0,
                  94.0,
                  43.0,
                  26.0,
                  18.0,
                  6.0,
                  8.0,
                  2.0,
                  1.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0
                ],
                [
                  0.0,
                  3837.0,
                  560.0,
                  145.0,
                  53.0,
                  47.0,
                  21.0,
                  8.0,
                  9.0,
                  3.0,
                  1.0,
                  2.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  2677.0,
                  365.0,
                  121.0,
                  53.0,
                  27.0,
                  15.0,
                  5.0,
                  2.0,
                  7.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  1871.0,
                  284.0,
                  89.0,
                  24.0,
                  16.0,
                  13.0,
                  4.0,
                  4.0,
                  5.0,
                  1.0,
                  4.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  1447.0,
                  181.0,
                  64.0,
                  22.0,
                  16.0,
                  10.0,
                  2.0,
                  4.0,
                  6.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  1083.0,
                  164.0,
                  41.0,
                  18.0,
                  11.0,
                  1.0,
                  3.0,
                  4.0,
                  2.0,
                  1.0,
                  1.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  750.0,
                  89.0,
                  36.0,
                  18.0,
                  9.0,
                  7.0,
                  1.0,
                  2.0,
                  1.0,
                  1.0,
                  0.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  597.0,
                  89.0,
                  26.0,
                  10.0,
                  5.0,
                  3.0,
                  4.0,
                  1.0,
                  1.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  448.0,
                  80.0,
                  17.0,
                  12.0,
                  7.0,
                  4.0,
                  2.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  316.0,
                  51.0,
                  15.0,
                  5.0,
                  2.0,
                  1.0,
                  0.0,
                  1.0,
                  2.0,
                  1.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  277.0,
                  42.0,
                  18.0,
                  3.0,
                  2.0,
                  1.0,
                  0.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  1012.0,
                  222.0,
                  67.0,
                  24.0,
                  16.0,
                  5.0,
                  7.0,
                  5.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  2.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0
                ]
              ],
              "sumw2": [
                [
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  1018980.0,
                  500540.0,
                  234750.0,
                  116342.0,
                  58377.0,
                  27836.0,
                  12717.0,
                  5471.0,
                  2326.0,
                  1118.0,
                  582.0,
                  292.0,
                  155.0,
                  89.0,
                  46.0,
                  27.0,
                  16.0,
                  11.0,
                  5.0,
                  2.0,
                  0.0,
                  3.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0
                ],
                [
                  0.0,
                  617269.0,
                  255329.0,
                  113352.0,
                  52786.0,
                  24158.0,
                  10419.0,
                  4186.0,
                  1613.0,
                  715.0,
                  352.0,
                  192.0,
                  121.0,
                  67.0,
                  30.0,
                  23.0,
                  11.0,
                  9.0,
                  5.0,
                  6.0,
                  0.0,
                  4.0,
                  4.0,
                  2.0,
                  1.0,
                  1.0,
                  2.0
                ],
                [
                  0.0,
                  364465.0,
                  132717.0,
                  58847.0,
                  26582.0,
                  11571.0,
                  4716.0,
                  1697.0,
                  609.0,
                  315.0,
                  161.0,
                  91.0,
                  55.0,
                  35.0,
                  17.0,
                  15.0,
                  8.0,
                  5.0,
                  2.0,
                  4.0,
                  3.0,
                  0.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  259905.0,
                  84353.0,
                  35695.0,
                  14985.0,
                  6261.0,
                  2356.0,
                  894.0,
                  386.0,
                  173.0,
                  94.0,
                  65.0,
                  31.0,
                  23.0,
                  17.0,
                  4.0,
                  4.0,
                  7.0,
                  3.0,
                  2.0,
                  0.0,
                  0.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0
                ],
                [
                  0.0,
                  208043.0,
                  57993.0,
                  22481.0,
                  9128.0,
                  3512.0,
                  1292.0,
                  466.0,
                  250.0,
                  97.0,
                  61.0,
                  50.0,
                  22.0,
                  16.0,
                  1.0,
                  7.0,
                  4.0,
                  1.0,
                  2.0,
                  3.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  2.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  167270.0,
                  40397.0,
                  14084.0,
                  5198.0,
                  1931.0,
                  680.0,
                  286.0,
                  157.0,
                  73.0,
                  32.0,
                  25.0,
                  15.0,
                  8.0,
                  6.0,
                  5.0,
                  3.0,
                  1.0,
                  0.0,
                  0.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  130047.0,
                  27584.0,
                  8606.0,
                  3062.0,
                  1149.0,
                  433.0,
                  215.0,
                  101.0,
                  55.0,
                  33.0,
                  23.0,
                  14.0,
                  10.0,
                  3.0,
                  0.0,
                  5.0,
                  1.0,
                  3.0,
                  0.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  92642.0,
                  17926.0,
                  5174.0,
                  1714.0,
                  645.0,
                  277.0,
                  126.0,
                  63.0,
                  36.0,
                  24.0,
                  15.0,
                  8.0,
                  4.0,
                  2.0,
                  1.0,
                  1.0,
                  0.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  64561.0,
                  11441.0,
                  3097.0,
                  953.0,
                  375.0,
                  183.0,
                  79.0,
                  47.0,
                  29.0,
                  22.0,
                  7.0,
                  8.0,
                  4.0,
                  2.0,
                  1.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  44079.0,
                  7366.0,
                  1860.0,
                  675.0,
                  230.0,
                  105.0,
                  72.0,
                  40.0,
                  20.0,
                  18.0,
                  7.0,
                  2.0,
                  3.0,
                  1.0,
                  0.0,
                  1.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  29506.0,
                  4528.0,
                  1150.0,
                  392.0,
                  165.0,
                  88.0,
                  46.0,
                  22.0,
                  15.0,
                  8.0,
                  5.0,
                  6.0,
                  0.0,
                  1.0,
                  3.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  19241.0,
                  2917.0,
                  733.0,
                  246.0,
                  109.0,
                  58.0,
                  30.0,
                  16.0,
                  15.0,
                  10.0,
                  4.0,
                  2.0,
                  2.0,
                  0.0,
                  0.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  12641.0,
                  1890.0,
                  479.0,
                  171.0,
                  85.0,
                  49.0,
                  20.0,
                  10.0,
                  10.0,
                  8.0,
                  4.0,
                  3.0,
                  1.0,
                  2.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  8424.0,
                  1220.0,
                  313.0,
                  104.0,
                  59.0,
                  23.0,
                  15.0,
                  11.0,
                  4.0,
                  4.0,
                  2.0,
                  2.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0
                ],
                [
                  0.0,
                  5641.0,
                  755.0,
                  224.0,
                  94.0,
                  43.0,
                  26.0,
                  18.0,
                  6.0,
                  8.0,
                  2.0,
                  1.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0
                ],
                [
                  0.0,
                  3837.0,
                  560.0,
                  145.0,
                  53.0,
                  47.0,
                  21.0,
                  8.0,
                  9.0,
                  3.0,
                  1.0,
                  2.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  2677.0,
                  365.0,
                  121.0,
                  53.0,
                  27.0,
                  15.0,
                  5.0,
                  2.0,
                  7.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  1871.0,
                  284.0,
                  89.0,
                  24.0,
                  16.0,
                  13.0,
                  4.0,
                  4.0,
                  5.0,
                  1.0,
                  4.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  1447.0,
                  181.0,
                  64.0,
                  22.0,
                  16.0,
                  10.0,
                  2.0,
                  4.0,
                  6.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  1083.0,
                  164.0,
                  41.0,
                  18.0,
                  11.0,
                  1.0,
                  3.0,
                  4.0,
                  2.0,
                  1.0,
                  1.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  750.0,
                  89.0,
                  36.0,
                  18.0,
                  9.0,
                  7.0,
                  1.0,
                  2.0,
                  1.0,
                  1.0,
                  0.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  597.0,
                  89.0,
                  26.0,
                  10.0,
                  5.0,
                  3.0,
                  4.0,
                  1.0,
                  1.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  448.0,
                  80.0,
                  17.0,
                  12.0,
                  7.0,
                  4.0,
                  2.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  316.0,
                  51.0,
                  15.0,
                  5.0,
                  2.0,
                  1.0,
                  0.0,
                  1.0,
                  2.0,
                  1.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  277.0,
                  42.0,
                  18.0,
                  3.0,
                  2.0,
                  1.0,
                  0.0,
                  1.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0
                ],
                [
                  0.0,
                  1012.0,
                  222.0,
                  67.0,
                  24.0,
                  16.0,
                  5.0,
                  7.0,
                  5.0,
                  2.0,
                  0.0,
                  0.0,
                  0.0,
                  2.0,
                  1.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  0.0,
                  1.0
                ]
              ]
            }
          ],
          "num_entries": 5135823
        }
      }
    }
  }
}"""
    )

    assert checks_json == json_expected
