"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisStreamsToKinesisFirehoseToS3 = void 0;
const kdfToS3 = require("@aws-solutions-constructs/aws-kinesisfirehose-s3");
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-constructs/core");
const iam = require("@aws-cdk/aws-iam");
const core_2 = require("@aws-solutions-constructs/core");
class KinesisStreamsToKinesisFirehoseToS3 extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the KinesisStreamsToKinesisFirehoseToS3 class.
     * @since 1.68.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the Kinesis Stream
        this.kinesisStream = defaults.buildKinesisStream(this, {
            existingStreamObj: props.existingStreamObj,
            kinesisStreamProps: props.kinesisStreamProps
        });
        const kinesisStreamsRole = new iam.Role(scope, 'KinesisStreamsRole', {
            assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
            inlinePolicies: {
                KinesisStreamsRoleRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                "kinesis:DescribeStream",
                                "kinesis:GetShardIterator",
                                "kinesis:GetRecords",
                                "kinesis:ListShards"
                            ],
                            resources: [this.kinesisStream.streamArn]
                        })]
                })
            }
        });
        // This Construct requires that the deliveryStreamType be overriden regardless of what is specified in the user props
        if (props.kinesisFirehoseProps) {
            if (props.kinesisFirehoseProps.deliveryStreamType !== undefined) {
                defaults.printWarning('Overriding deliveryStreamType type to be KinesisStreamAsSource');
            }
            if (props.kinesisFirehoseProps.kinesisStreamSourceConfiguration !== undefined) {
                defaults.printWarning('Overriding kinesisStreamSourceConfiguration');
            }
        }
        const _kinesisFirehoseProps = {
            deliveryStreamType: 'KinesisStreamAsSource',
            kinesisStreamSourceConfiguration: {
                kinesisStreamArn: this.kinesisStream.streamArn,
                roleArn: kinesisStreamsRole.roleArn
            }
        };
        const kdfToS3Construct = new kdfToS3.KinesisFirehoseToS3(this, 'KinesisFirehoseToS3', {
            kinesisFirehoseProps: core_2.overrideProps(props.kinesisFirehoseProps, _kinesisFirehoseProps),
            existingBucketObj: props.existingBucketObj,
            bucketProps: props.bucketProps
        });
        this.kinesisFirehose = kdfToS3Construct.kinesisFirehose;
        this.kinesisFirehoseRole = kdfToS3Construct.kinesisFirehoseRole;
        this.kinesisFirehoseLogGroup = kdfToS3Construct.kinesisFirehoseLogGroup;
        if (props.createCloudWatchAlarms === undefined || props.createCloudWatchAlarms) {
            // Deploy best practices CW Alarms for Kinesis Stream
            this.cloudwatchAlarms = defaults.buildKinesisStreamCWAlarms(this);
        }
    }
}
exports.KinesisStreamsToKinesisFirehoseToS3 = KinesisStreamsToKinesisFirehoseToS3;
//# sourceMappingURL=data:application/json;base64,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