import importlib
import logging
import json
import yaml
import requests
import urllib3
import io
import sys
import time
import re

try:
    from typing import Union, Dict, List, Any, Literal
except ImportError:
    from typing_extensions import Literal

if sys.version_info[0] == 3:
    unicode = str


def api(location=None, verify=True, logger=None, loglevel=logging.INFO, ext=None):
    """Create an instance of an Api class

    generator.Generator outputs a base Api class with the following:
    - an abstract method for each OpenAPI path item object
    - a concrete properties for each unique OpenAPI path item parameter.

    generator.Generator also outputs an HttpApi class that inherits the base
    Api class, implements the abstract methods and uses the common HttpTransport
    class send_recv method to communicate with a REST based server.

    Args
    ----
    - location (str): The location of an Open Traffic Generator server.
    - verify (bool): Verify the server's TLS certificate, or a string, in which
      case it must be a path to a CA bundle to use. Defaults to `True`.
      When set to `False`, requests will accept any TLS certificate presented by
      the server, and will ignore hostname mismatches and/or expired
      certificates, which will make your application vulnerable to
      man-in-the-middle (MitM) attacks. Setting verify to `False`
      may be useful during local development or testing.
    - logger (logging.Logger): A user defined logging.logger, if none is provided
      then a default logger with a stdout handler will be provided
    - loglevel (logging.loglevel): The logging package log level.
      The default loglevel is logging.INFO
    - ext (str): Name of an extension package
    """
    params = locals()
    if ext is None:
        return HttpApi(**params)
    try:
        lib = importlib.import_module("snappi_{}.snappi_convergence_api".format(ext))
        return lib.Api(**params)
    except ImportError as err:
        msg = "Extension %s is not installed or invalid: %s"
        raise Exception(msg % (ext, err))


class HttpTransport(object):
    def __init__(self, **kwargs):
        """Use args from api() method to instantiate an HTTP transport"""
        self.location = kwargs["location"] if "location" in kwargs else "https://localhost"
        self.verify = kwargs["verify"] if "verify" in kwargs else False
        self.logger = kwargs["logger"] if "logger" in kwargs else None
        self.loglevel = kwargs["loglevel"] if "loglevel" in kwargs else logging.DEBUG
        if self.logger is None:
            stdout_handler = logging.StreamHandler(sys.stdout)
            formatter = logging.Formatter(fmt="%(asctime)s [%(name)s] [%(levelname)s] %(message)s", datefmt="%Y-%m-%d %H:%M:%S")
            formatter.converter = time.gmtime
            stdout_handler.setFormatter(formatter)
            self.logger = logging.Logger(self.__module__, level=self.loglevel)
            self.logger.addHandler(stdout_handler)
        self.logger.debug("HttpTransport args: {}".format(", ".join(["{}={!r}".format(k, v) for k, v in kwargs.items()])))
        if self.verify is False:
            urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
            self.logger.warning("Certificate verification is disabled")
        self._session = requests.Session()

    def send_recv(self, method, relative_url, payload=None, return_object=None):
        url = "%s%s" % (self.location, relative_url)
        data = None
        if payload is not None:
            if isinstance(payload, (str, unicode)):
                data = payload
            elif isinstance(payload, OpenApiBase):
                data = payload.serialize()
            else:
                raise Exception("Type of payload provided is unknown")
        response = self._session.request(
            method=method,
            url=url,
            data=data,
            verify=False,
            allow_redirects=True,
            # TODO: add a timeout here
            headers={"Content-Type": "application/json"},
        )
        if response.ok:
            if "application/json" in response.headers["content-type"]:
                # TODO: we might want to check for utf-8 charset and decode
                # accordingly, but current impl works for now
                response_dict = yaml.safe_load(response.text)
                if return_object is None:
                    # if response type is not provided, return dictionary
                    # instead of python object
                    return response_dict
                else:
                    return return_object.deserialize(response_dict)
            elif "application/octet-stream" in response.headers["content-type"]:
                return io.BytesIO(response.content)
            else:
                # TODO: for now, return bare response object for unknown
                # content types
                return response
        else:
            raise Exception(response.status_code, yaml.safe_load(response.text))


class OpenApiBase(object):
    """Base class for all generated classes"""

    JSON = "json"
    YAML = "yaml"
    DICT = "dict"

    __slots__ = ()

    def __init__(self):
        pass

    def serialize(self, encoding=JSON):
        """Serialize the current object according to a specified encoding.

        Args
        ----
        - encoding (str[json, yaml, dict]): The object will be recursively
            serialized according to the specified encoding.
            The supported encodings are json, yaml and python dict.

        Returns
        -------
        - obj(Union[str, dict]): A str or dict object depending on the specified
            encoding. The json and yaml encodings will return a str object and
            the dict encoding will return a python dict object.
        """
        if encoding == OpenApiBase.JSON:
            return json.dumps(self._encode(), indent=2)
        elif encoding == OpenApiBase.YAML:
            return yaml.safe_dump(self._encode())
        elif encoding == OpenApiBase.DICT:
            return self._encode()
        else:
            raise NotImplementedError("Encoding %s not supported" % encoding)

    def _encode(self):
        raise NotImplementedError()

    def deserialize(self, serialized_object):
        """Deserialize a python object into the current object.

        If the input `serialized_object` does not match the current
        openapi object an exception will be raised.

        Args
        ----
        - serialized_object (Union[str, dict]): The object to deserialize.
            If the serialized_object is of type str then the internal encoding
            of the serialized_object must be json or yaml.

        Returns
        -------
        - obj(OpenApiObject): This object with all the
            serialized_object deserialized within.
        """
        if isinstance(serialized_object, (str, unicode)):
            serialized_object = yaml.safe_load(serialized_object)
        self._decode(serialized_object)
        return self

    def _decode(self, dict_object):
        raise NotImplementedError()


class OpenApiValidator(object):

    _MAC_REGEX = re.compile(
        r"^([\da-fA-F]{2}[:]){5}[\da-fA-F]{2}$")
    _IPV6_REP1 = re.compile(r"^:[\da-fA-F].+")
    _IPV6_REP2 = re.compile(r".+[\da-fA-F]:$")
    _IPV6_REP3 = re.compile(r"^" +
        r"[\da-fA-F]{1,4}:" *7 + r"[\da-fA-F]{1,4}$")
    _HEX_REGEX = re.compile(r"^0?x?[\da-fA-F]+$")

    __slots__ = ()

    def __init__(self):
        pass

    def validate_mac(self, mac):
        if mac is None:
            return False
        if isinstance(mac, list):
            return all([
                True if self._MAC_REGEX.match(m) else False
                for m in mac
            ])
        if self._MAC_REGEX.match(mac):
            return True
        return False

    def validate_ipv4(self, ip):
        if ip is None:
            return False
        try:
            if isinstance(ip, list):
                return all([
                    all([0 <= int(oct) <= 255 for oct in i.split(".", 3)])
                    for i in ip
                ])
            else:
                return all([0 <= int(oct) <= 255 for oct in ip.split(".", 3)])
        except Exception:
            return False

    def _validate_ipv6(self, ip):
        if ip is None:
            return False
        if self._IPV6_REP1.match(ip) or self._IPV6_REP2.match(ip):
            return False
        if ip.count("::") == 0:
            if self._IPV6_REP3.match(ip):
                return True
            else:
                return False
        if ip.count(":") > 7 or ip.count("::") > 1 or ip.count(":::") > 0:
            return False
        return True

    def validate_ipv6(self, ip):
        if isinstance(ip, list):
            return all([
                self._validate_ipv6(i) for i in ip
            ])
        return self._validate_ipv6(ip)

    def validate_hex(self, hex):
        if isinstance(hex, list):
            return all([
                self._HEX_REGEX(h)
                for h in hex
            ])
        if self._HEX_REGEX.match(hex):
            return True
        return False

    def validate_integer(self, value):
        if not isinstance(value, list):
            value = [value]
        return all([isinstance(i, int) for i in value])

    def validate_float(self, value):
        if not isinstance(value, list):
            value = [value]
        return all([
            isinstance(i, (float, int)) for i in value
        ])

    def validate_string(self, value):
        if not isinstance(value, list):
            value = [value]
        return all([
            isinstance(i, (str, unicode)) for i in value
        ])
    
    def validate_bool(self, value):
        if not isinstance(value, list):
            value = [value]
        return all([
            isinstance(i, bool) for i in value
        ])

    def validate_list(self, value):
        # TODO pending item validation
        return isinstance(value, list)
    
    def validate_binary(self, value):
        if not isinstance(value, list):
            value = [value]
        return all([
            all([
                True if int(b) == 0 or int(b) == 1 else False
                for b in binary
            ])
            for binary in value
        ])
    
    def types_validation(self, value, type_, err_msg):
        type_map = {
            int: "integer", str: "string",
            float: "float", bool: "bool",
            list: "list", "int64": "integer",
            "int32": "integer", "double": "float"
        }
        if type_ in type_map:
            type_ = type_map[type_]
        v_obj = getattr(self, "validate_{}".format(type_))
        if v_obj is None:
            msg = "{} is not a valid or unsupported format".format(type_)
            raise TypeError(msg)
        if v_obj(value) is False:
            raise TypeError(err_msg)


class OpenApiObject(OpenApiBase, OpenApiValidator):
    """Base class for any /components/schemas object

    Every OpenApiObject is reuseable within the schema so it can
    exist in multiple locations within the hierarchy.
    That means it can exist in multiple locations as a
    leaf, parent/choice or parent.
    """

    __slots__ = ("_properties", "_parent", "_choice")

    def __init__(self, parent=None, choice=None):
        super(OpenApiObject, self).__init__()
        self._parent = parent
        self._choice = choice
        self._properties = {}

    @property
    def parent(self):
        return self._parent

    def _set_choice(self, name):
        if "choice" in dir(self) and "_TYPES" in dir(self) \
            and "choice" in self._TYPES and name in self._TYPES["choice"]["enum"]:
            for enum in self._TYPES["choice"]["enum"]:
                if enum in self._properties and name != enum:
                    self._properties.pop(enum)
            self._properties["choice"] = name
        
    def _get_property(self, name, default_value=None, parent=None, choice=None):
        if name in self._properties and self._properties[name] is not None:
            return self._properties[name]
        if isinstance(default_value, type) is True:
            self._set_choice(name)
            self._properties[name] = default_value(parent=parent, choice=choice)

            if "_DEFAULTS" in dir(self._properties[name]) and\
                "choice" in self._properties[name]._DEFAULTS:
                getattr(self._properties[name], self._properties[name]._DEFAULTS["choice"])
        else:
            if default_value is None and name in self._DEFAULTS:
                self._set_choice(name)
                self._properties[name] = self._DEFAULTS[name]
            else:
                self._properties[name] = default_value
        return self._properties[name]

    def _set_property(self, name, value, choice=None):
        if name in self._DEFAULTS and value is None:
            self._set_choice(name)
            self._properties[name] = self._DEFAULTS[name]
        else:
            self._set_choice(name)
            self._properties[name] = value
        if self._parent is not None and self._choice is not None and value is not None:
            self._parent._set_property("choice", self._choice)

    def _encode(self):
        """Helper method for serialization"""
        output = {}
        self._validate_required()
        for key, value in self._properties.items():
            self._validate_types(key, value)
            if isinstance(value, (OpenApiObject, OpenApiIter)):
                output[key] = value._encode()
            elif value is not None:
                output[key] = value
        return output

    def _decode(self, obj):
        openapi_names = dir(self)
        dtypes = [list, str, int, float, bool]
        for property_name, property_value in obj.items():
            if property_name in openapi_names:
                if isinstance(property_value, dict):
                    child = self._get_child_class(property_name)
                    if "_choice" in dir(child[1]) and "_parent" in dir(child[1]):
                        property_value = child[1](self, property_name)._decode(property_value)
                    else:
                        property_value = child[1]()._decode(property_value)
                elif isinstance(property_value, list) and \
                    property_name in self._TYPES and \
                        self._TYPES[property_name]["type"] not in dtypes:
                    child = self._get_child_class(property_name, True)
                    openapi_list = child[0]()
                    for item in property_value:
                        item = child[1]()._decode(item)
                        openapi_list._items.append(item)
                    property_value = openapi_list
                elif property_name in self._DEFAULTS and property_value is None:
                    if isinstance(self._DEFAULTS[property_name], tuple(dtypes)):
                        property_value = self._DEFAULTS[property_name]
                self._properties[property_name] = property_value
            self._validate_types(property_name, property_value)
        self._validate_required()
        return self

    def _get_child_class(self, property_name, is_property_list=False):
        list_class = None
        class_name = self._TYPES[property_name]["type"]
        module = importlib.import_module(self.__module__)
        object_class = getattr(module, class_name)
        if is_property_list is True:
            list_class = object_class
            object_class = getattr(module, class_name[0:-4])
        return (list_class, object_class)

    def __str__(self):
        return self.serialize(encoding=self.YAML)

    def __deepcopy__(self, memo):
        """Creates a deep copy of the current object"""
        return self.__class__().deserialize(self.serialize())

    def __copy__(self):
        """Creates a deep copy of the current object"""
        return self.__deepcopy__(None)

    def __eq__(self, other):
        return self.__str__() == other.__str__()

    def clone(self):
        """Creates a deep copy of the current object"""
        return self.__deepcopy__(None)
    
    def _validate_required(self):
        """Validates the required properties are set
        """
        if getattr(self, "_REQUIRED", None) is None:
            return
        for p in self._REQUIRED:
            if p in self._properties and self._properties[p] is not None:
                continue
            msg = "{} is a mandatory property of {}"\
                " and should not be set to None".format(
                p, self.__class__
            )
            raise ValueError(msg)

    def _validate_types(self, property_name, property_value):
        common_data_types = [list, str, int, float, bool]
        if property_name not in self._TYPES:
            raise ValueError("Invalid Property {}".format(property_name))
        details = self._TYPES[property_name]
        if property_value is None and property_name not in self._DEFAULTS and \
            property_name not in self._REQUIRED:
            return
        if "enum" in details and property_value not in details["enum"]:
            msg = "property {} shall be one of these" \
                " {} enum, but got {} at {}"
            raise TypeError(msg.format(
                property_name, details["enum"], property_value, self.__class__
            ))
        if details["type"] in common_data_types and \
                "format" not in details:
            msg = "property {} shall be of type {}, but got {} at {}".format(
                    property_name, details["type"], type(property_value), self.__class__
                )
            self.types_validation(property_value, details["type"], msg)

        if details["type"] not in common_data_types:
            class_name = details["type"]
            # TODO Need to revisit importlib
            module = importlib.import_module(self.__module__)
            object_class = getattr(module, class_name)
            if not isinstance(property_value, object_class):
                msg = "property {} shall be of type {}," \
                    " but got {} at {}"
                raise TypeError(msg.format(
                    property_name, class_name, type(property_value),
                    self.__class__
                ))
        if "format" in details:
            msg = "Invalid {} format, expected {} at {}".format(
                property_value, details["format"], self.__class__
            )
            self.types_validation(property_value, details["format"], msg)

    def validate(self):
        self._validate_required()
        for key, value in self._properties.items():
            self._validate_types(key, value)
    
    def get(self, name, with_default=False):
        """ 
        getattr for openapi object
        """
        if self._properties.get(name) is not None:
            return self._properties[name]
        elif with_default:
            # TODO need to find a way to avoid getattr
            choice = self._properties.get("choice")\
                    if "choice" in dir(self) else None
            getattr(self, name)
            if "choice" in dir(self):
                if choice is None and "choice" in self._properties:
                    self._properties.pop("choice")
                else:
                    self._properties["choice"] = choice
            return self._properties.pop(name)
        return None


class OpenApiIter(OpenApiBase):
    """Container class for OpenApiObject

    Inheriting classes contain 0..n instances of an OpenAPI components/schemas
    object.
    - config.flows.flow(name="1").flow(name="2").flow(name="3")

    The __getitem__ method allows getting an instance using ordinal.
    - config.flows[0]
    - config.flows[1:]
    - config.flows[0:1]
    - f1, f2, f3 = config.flows

    The __iter__ method allows for iterating across the encapsulated contents
    - for flow in config.flows:
    """

    __slots__ = ("_index", "_items")

    def __init__(self):
        super(OpenApiIter, self).__init__()
        self._index = -1
        self._items = []

    def __len__(self):
        return len(self._items)

    def _getitem(self, key):
        found = None
        if isinstance(key, int):
            found = self._items[key]
        elif isinstance(key, slice) is True:
            start, stop, step = key.indices(len(self))
            sliced = self.__class__()
            for i in range(start, stop, step):
                sliced._items.append(self._items[i])
            return sliced
        elif isinstance(key, str):
            for item in self._items:
                if item.name == key:
                    found = item
        if found is None:
            raise IndexError()
        if self._GETITEM_RETURNS_CHOICE_OBJECT is True and found._properties.get("choice") is not None:
            return found._properties[found._properties["choice"]]
        return found

    def _iter(self):
        self._index = -1
        return self

    def _next(self):
        if self._index + 1 >= len(self._items):
            raise StopIteration
        else:
            self._index += 1
        return self.__getitem__(self._index)

    def _add(self, item):
        self._items.append(item)
        self._index = len(self._items) - 1

    def append(self, item):
        """Append an item to the end of OpenApiIter
        TBD: type check, raise error on mismatch
        """
        if isinstance(item, OpenApiObject) is False:
            raise Exception("Item is not an instance of OpenApiObject")
        self._add(item)
        return self

    def clear(self):
        del self._items[:]
        self._index = -1

    def _encode(self):
        return [item._encode() for item in self._items]

    def _decode(self, encoded_list):
        item_class_name = self.__class__.__name__.replace("Iter", "")
        module = importlib.import_module(self.__module__)
        object_class = getattr(module, item_class_name)
        self.clear()
        for item in encoded_list:
            self._add(object_class()._decode(item))

    def __copy__(self):
        raise NotImplementedError("Shallow copy of OpenApiIter objects is not supported")

    def __deepcopy__(self, memo):
        raise NotImplementedError("Deep copy of OpenApiIter objects is not supported")

    def __str__(self):
        return yaml.safe_dump(self._encode())

    def __eq__(self, other):
        return self.__str__() == other.__str__()


class ConvergenceConfig(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'config': {'type': 'Config'},
        'event_location': {'type': str},
        'rx_rate_threshold': {'type': float},
    } # type: Dict[str, str]

    _REQUIRED = ('config',) # type: tuple(str)

    _DEFAULTS = {
        'event_location': '',
        'rx_rate_threshold': 95.0,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, event_location=None, rx_rate_threshold=None):
        super(ConvergenceConfig, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('event_location', event_location)
        self._set_property('rx_rate_threshold', rx_rate_threshold)

    @property
    def config(self):
        # type: () -> Config
        """config getter

        A container for all models that are part of the configuration.The core open traffic generator configuration

        Returns: Config
        """
        return self._get_property('config', Config)

    @property
    def event_location(self):
        # type: () -> str
        """event_location getter

        The location from which events will be generated. 

        Returns: str
        """
        return self._get_property('event_location')

    @event_location.setter
    def event_location(self, value):
        """event_location setter

        The location from which events will be generated. 

        value: str
        """
        self._set_property('event_location', value)

    @property
    def rx_rate_threshold(self):
        # type: () -> float
        """rx_rate_threshold getter

        Convergence will be calculated when the rx rate of a flow passes . below then above the threshold value. 

        Returns: float
        """
        return self._get_property('rx_rate_threshold')

    @rx_rate_threshold.setter
    def rx_rate_threshold(self, value):
        """rx_rate_threshold setter

        Convergence will be calculated when the rx rate of a flow passes . below then above the threshold value. 

        value: float
        """
        self._set_property('rx_rate_threshold', value)


class Config(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'ports': {'type': 'PortIter'},
        'lags': {'type': 'LagIter'},
        'layer1': {'type': 'Layer1Iter'},
        'captures': {'type': 'CaptureIter'},
        'devices': {'type': 'DeviceIter'},
        'flows': {'type': 'FlowIter'},
        'events': {'type': 'Event'},
        'options': {'type': 'ConfigOptions'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(Config, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def ports(self):
        # type: () -> PortIter
        """ports getter

        The ports that will be configured on the traffic generator.

        Returns: PortIter
        """
        return self._get_property('ports', PortIter, self._parent, self._choice)

    @property
    def lags(self):
        # type: () -> LagIter
        """lags getter

        The lags that will be configured on the traffic generator.

        Returns: LagIter
        """
        return self._get_property('lags', LagIter, self._parent, self._choice)

    @property
    def layer1(self):
        # type: () -> Layer1Iter
        """layer1 getter

        The layer1 settings that will be configured on the traffic generator.

        Returns: Layer1Iter
        """
        return self._get_property('layer1', Layer1Iter, self._parent, self._choice)

    @property
    def captures(self):
        # type: () -> CaptureIter
        """captures getter

        The capture settings that will be configured on the traffic generator.

        Returns: CaptureIter
        """
        return self._get_property('captures', CaptureIter, self._parent, self._choice)

    @property
    def devices(self):
        # type: () -> DeviceIter
        """devices getter

        The emulated device settings that will be configured on the traffic generator.

        Returns: DeviceIter
        """
        return self._get_property('devices', DeviceIter, self._parent, self._choice)

    @property
    def flows(self):
        # type: () -> FlowIter
        """flows getter

        The flows that will be configured on the traffic generator.

        Returns: FlowIter
        """
        return self._get_property('flows', FlowIter, self._parent, self._choice)

    @property
    def events(self):
        # type: () -> Event
        """events getter

        The optional container for event configuration.

        Returns: Event
        """
        return self._get_property('events', Event)

    @property
    def options(self):
        # type: () -> ConfigOptions
        """options getter

        Global configuration options.

        Returns: ConfigOptions
        """
        return self._get_property('options', ConfigOptions)


class Port(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'location': {'type': str},
        'name': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED = ('name',) # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, location=None, name=None):
        super(Port, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('location', location)
        self._set_property('name', name)

    @property
    def location(self):
        # type: () -> str
        """location getter

        The location of a test port. It is the endpoint where packets will emit from.. Test port locations can be the following:. - physical appliance with multiple ports. - physical chassis with multiple cards and ports. - local interface. - virtual machine, docker container, kubernetes cluster. . The test port location format is implementation specific. Use the /results/capabilities API to determine what formats an implementation supports for the location property.. Get the configured location state by using the /results/port API.

        Returns: str
        """
        return self._get_property('location')

    @location.setter
    def location(self, value):
        """location setter

        The location of a test port. It is the endpoint where packets will emit from.. Test port locations can be the following:. - physical appliance with multiple ports. - physical chassis with multiple cards and ports. - local interface. - virtual machine, docker container, kubernetes cluster. . The test port location format is implementation specific. Use the /results/capabilities API to determine what formats an implementation supports for the location property.. Get the configured location state by using the /results/port API.

        value: str
        """
        self._set_property('location', value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property('name')

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property('name', value)


class PortIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(PortIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Port]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> PortIter
        return self._iter()

    def __next__(self):
        # type: () -> Port
        return self._next()

    def next(self):
        # type: () -> Port
        return self._next()

    def port(self, location=None, name=None):
        # type: (str,str) -> PortIter
        """Factory method that creates an instance of the Port class

        An abstract test port.

        Returns: PortIter
        """
        item = Port(parent=self._parent, choice=self._choice, location=location, name=name)
        self._add(item)
        return self



class Lag(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'ports': {'type': 'LagPortIter'},
        'name': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED = ('name',) # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, name=None):
        super(Lag, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('name', name)

    @property
    def ports(self):
        # type: () -> LagPortIter
        """ports getter

        TBD

        Returns: LagPortIter
        """
        return self._get_property('ports', LagPortIter, self._parent, self._choice)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property('name')

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property('name', value)


class LagPort(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'port_name': {'type': str},
        'protocol': {'type': 'LagProtocol'},
        'ethernet': {'type': 'DeviceEthernetBase'},
    } # type: Dict[str, str]

    _REQUIRED = ('port_name', 'protocol', 'ethernet') # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, port_name=None):
        super(LagPort, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('port_name', port_name)

    @property
    def port_name(self):
        # type: () -> str
        """port_name getter

        The name of a port object that will be part of the LAG. . . x-constraint:. - /components/schemas/Port/properties/name. 

        Returns: str
        """
        return self._get_property('port_name')

    @port_name.setter
    def port_name(self, value):
        """port_name setter

        The name of a port object that will be part of the LAG. . . x-constraint:. - /components/schemas/Port/properties/name. 

        value: str
        """
        self._set_property('port_name', value)

    @property
    def protocol(self):
        # type: () -> LagProtocol
        """protocol getter

        

        Returns: LagProtocol
        """
        return self._get_property('protocol', LagProtocol)

    @property
    def ethernet(self):
        # type: () -> DeviceEthernetBase
        """ethernet getter

        Base ethernet interface

        Returns: DeviceEthernetBase
        """
        return self._get_property('ethernet', DeviceEthernetBase)


class LagProtocol(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'lacp',
                'static',
            ],
        },
        'lacp': {'type': 'LagLacp'},
        'static': {'type': 'LagStatic'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'lacp',
    } # type: Dict[str, Union(type)]

    LACP = 'lacp' # type: str
    STATIC = 'static' # type: str

    def __init__(self, parent=None, choice=None):
        super(LagProtocol, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def lacp(self):
        # type: () -> LagLacp
        """Factory property that returns an instance of the LagLacp class

        The container for link aggregation control protocol settings.

        Returns: LagLacp
        """
        return self._get_property('lacp', LagLacp, self, 'lacp')

    @property
    def static(self):
        # type: () -> LagStatic
        """Factory property that returns an instance of the LagStatic class

        The container for static link aggregation protocol settings.

        Returns: LagStatic
        """
        return self._get_property('static', LagStatic, self, 'static')

    @property
    def choice(self):
        # type: () -> Union[Literal["lacp"], Literal["static"]]
        """choice getter

        The type of LAG protocol.

        Returns: Union[Literal["lacp"], Literal["static"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of LAG protocol.

        value: Union[Literal["lacp"], Literal["static"]]
        """
        self._set_property('choice', value)


class LagLacp(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'actor_key': {'type': int},
        'actor_port_number': {'type': int},
        'actor_port_priority': {'type': int},
        'actor_system_id': {
            'type': str,
            'format': 'mac',
        },
        'actor_system_priority': {'type': int},
        'lacpdu_periodic_time_interval': {'type': int},
        'lacpdu_timeout': {'type': int},
        'actor_activity': {
            'type': str,
            'enum': [
                'active',
                'passive',
            ],
        },
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'actor_key': 0,
        'actor_port_number': 0,
        'actor_port_priority': 1,
        'actor_system_id': '00:00:00:00:00:00',
        'actor_system_priority': 0,
        'lacpdu_periodic_time_interval': 0,
        'lacpdu_timeout': 0,
        'actor_activity': 'active',
    } # type: Dict[str, Union(type)]

    FAST = 1 #
    SLOW = 30 #
    AUTO = 0 #

    SHORT = 3 #
    LONG = 90 #
    AUTO = 0 #

    ACTIVE = 'active' # type: str
    PASSIVE = 'passive' # type: str

    def __init__(self, parent=None, choice=None, actor_key=None, actor_port_number=None, actor_port_priority=None, actor_system_id=None, actor_system_priority=None, lacpdu_periodic_time_interval=None, lacpdu_timeout=None, actor_activity=None):
        super(LagLacp, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('actor_key', actor_key)
        self._set_property('actor_port_number', actor_port_number)
        self._set_property('actor_port_priority', actor_port_priority)
        self._set_property('actor_system_id', actor_system_id)
        self._set_property('actor_system_priority', actor_system_priority)
        self._set_property('lacpdu_periodic_time_interval', lacpdu_periodic_time_interval)
        self._set_property('lacpdu_timeout', lacpdu_timeout)
        self._set_property('actor_activity', actor_activity)

    @property
    def actor_key(self):
        # type: () -> int
        """actor_key getter

        The actor key

        Returns: int
        """
        return self._get_property('actor_key')

    @actor_key.setter
    def actor_key(self, value):
        """actor_key setter

        The actor key

        value: int
        """
        self._set_property('actor_key', value)

    @property
    def actor_port_number(self):
        # type: () -> int
        """actor_port_number getter

        The actor port number

        Returns: int
        """
        return self._get_property('actor_port_number')

    @actor_port_number.setter
    def actor_port_number(self, value):
        """actor_port_number setter

        The actor port number

        value: int
        """
        self._set_property('actor_port_number', value)

    @property
    def actor_port_priority(self):
        # type: () -> int
        """actor_port_priority getter

        The actor port priority

        Returns: int
        """
        return self._get_property('actor_port_priority')

    @actor_port_priority.setter
    def actor_port_priority(self, value):
        """actor_port_priority setter

        The actor port priority

        value: int
        """
        self._set_property('actor_port_priority', value)

    @property
    def actor_system_id(self):
        # type: () -> str
        """actor_system_id getter

        The actor system id

        Returns: str
        """
        return self._get_property('actor_system_id')

    @actor_system_id.setter
    def actor_system_id(self, value):
        """actor_system_id setter

        The actor system id

        value: str
        """
        self._set_property('actor_system_id', value)

    @property
    def actor_system_priority(self):
        # type: () -> int
        """actor_system_priority getter

        The actor system priority

        Returns: int
        """
        return self._get_property('actor_system_priority')

    @actor_system_priority.setter
    def actor_system_priority(self, value):
        """actor_system_priority setter

        The actor system priority

        value: int
        """
        self._set_property('actor_system_priority', value)

    @property
    def lacpdu_periodic_time_interval(self):
        # type: () -> int
        """lacpdu_periodic_time_interval getter

        This field defines how frequently LACPDUs are sent to the link partner

        Returns: int
        """
        return self._get_property('lacpdu_periodic_time_interval')

    @lacpdu_periodic_time_interval.setter
    def lacpdu_periodic_time_interval(self, value):
        """lacpdu_periodic_time_interval setter

        This field defines how frequently LACPDUs are sent to the link partner

        value: int
        """
        self._set_property('lacpdu_periodic_time_interval', value)

    @property
    def lacpdu_timeout(self):
        # type: () -> int
        """lacpdu_timeout getter

        This timer is used to detect whether received protocol information has expired

        Returns: int
        """
        return self._get_property('lacpdu_timeout')

    @lacpdu_timeout.setter
    def lacpdu_timeout(self, value):
        """lacpdu_timeout setter

        This timer is used to detect whether received protocol information has expired

        value: int
        """
        self._set_property('lacpdu_timeout', value)

    @property
    def actor_activity(self):
        # type: () -> Union[Literal["active"], Literal["passive"]]
        """actor_activity getter

        Sets the value of LACP actor activity as either passive or active. Passive indicates the port's preference for not transmitting LACPDUs unless its partner's control is Active. Active indicates the port's preference to participate in the protocol regardless of the partner's control value

        Returns: Union[Literal["active"], Literal["passive"]]
        """
        return self._get_property('actor_activity')

    @actor_activity.setter
    def actor_activity(self, value):
        """actor_activity setter

        Sets the value of LACP actor activity as either passive or active. Passive indicates the port's preference for not transmitting LACPDUs unless its partner's control is Active. Active indicates the port's preference to participate in the protocol regardless of the partner's control value

        value: Union[Literal["active"], Literal["passive"]]
        """
        self._set_property('actor_activity', value)


class LagStatic(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'lag_id': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'lag_id': 0,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, lag_id=None):
        super(LagStatic, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('lag_id', lag_id)

    @property
    def lag_id(self):
        # type: () -> int
        """lag_id getter

        The static lag id

        Returns: int
        """
        return self._get_property('lag_id')

    @lag_id.setter
    def lag_id(self, value):
        """lag_id setter

        The static lag id

        value: int
        """
        self._set_property('lag_id', value)


class DeviceEthernetBase(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'mac': {
            'type': str,
            'format': 'mac',
        },
        'mtu': {'type': int},
        'vlans': {'type': 'DeviceVlanIter'},
        'name': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED = ('mac', 'name') # type: tuple(str)

    _DEFAULTS = {
        'mtu': 1500,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, mac=None, mtu=None, name=None):
        super(DeviceEthernetBase, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('mac', mac)
        self._set_property('mtu', mtu)
        self._set_property('name', name)

    @property
    def mac(self):
        # type: () -> str
        """mac getter

        Media access control address

        Returns: str
        """
        return self._get_property('mac')

    @mac.setter
    def mac(self, value):
        """mac setter

        Media access control address

        value: str
        """
        self._set_property('mac', value)

    @property
    def mtu(self):
        # type: () -> int
        """mtu getter

        Maximum transmission unit

        Returns: int
        """
        return self._get_property('mtu')

    @mtu.setter
    def mtu(self, value):
        """mtu setter

        Maximum transmission unit

        value: int
        """
        self._set_property('mtu', value)

    @property
    def vlans(self):
        # type: () -> DeviceVlanIter
        """vlans getter

        List of VLANs

        Returns: DeviceVlanIter
        """
        return self._get_property('vlans', DeviceVlanIter, self._parent, self._choice)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property('name')

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property('name', value)


class DeviceVlan(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'tpid': {
            'type': str,
            'enum': [
                'x8100',
                'x88A8',
                'x9100',
                'x9200',
                'x9300',
            ],
        },
        'priority': {'type': int},
        'id': {'type': int},
        'name': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED = ('name',) # type: tuple(str)

    _DEFAULTS = {
        'tpid': 'x8100',
        'priority': 0,
        'id': 1,
    } # type: Dict[str, Union(type)]

    X8100 = 'x8100' # type: str
    X88A8 = 'x88A8' # type: str
    X9100 = 'x9100' # type: str
    X9200 = 'x9200' # type: str
    X9300 = 'x9300' # type: str

    def __init__(self, parent=None, choice=None, tpid=None, priority=None, id=None, name=None):
        super(DeviceVlan, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('tpid', tpid)
        self._set_property('priority', priority)
        self._set_property('id', id)
        self._set_property('name', name)

    @property
    def tpid(self):
        # type: () -> Union[Literal["x8100"], Literal["x88A8"], Literal["x9100"], Literal["x9200"], Literal["x9300"]]
        """tpid getter

        Tag protocol identifier

        Returns: Union[Literal["x8100"], Literal["x88A8"], Literal["x9100"], Literal["x9200"], Literal["x9300"]]
        """
        return self._get_property('tpid')

    @tpid.setter
    def tpid(self, value):
        """tpid setter

        Tag protocol identifier

        value: Union[Literal["x8100"], Literal["x88A8"], Literal["x9100"], Literal["x9200"], Literal["x9300"]]
        """
        self._set_property('tpid', value)

    @property
    def priority(self):
        # type: () -> int
        """priority getter

        Priority code point

        Returns: int
        """
        return self._get_property('priority')

    @priority.setter
    def priority(self, value):
        """priority setter

        Priority code point

        value: int
        """
        self._set_property('priority', value)

    @property
    def id(self):
        # type: () -> int
        """id getter

        VLAN identifier

        Returns: int
        """
        return self._get_property('id')

    @id.setter
    def id(self, value):
        """id setter

        VLAN identifier

        value: int
        """
        self._set_property('id', value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property('name')

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property('name', value)


class DeviceVlanIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceVlanIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceVlan]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceVlanIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceVlan
        return self._next()

    def next(self):
        # type: () -> DeviceVlan
        return self._next()

    def vlan(self, tpid='x8100', priority=0, id=1, name=None):
        # type: (Union[Literal["x8100"], Literal["x88A8"], Literal["x9100"], Literal["x9200"], Literal["x9300"]],int,int,str) -> DeviceVlanIter
        """Factory method that creates an instance of the DeviceVlan class

        Emulated VLAN protocol

        Returns: DeviceVlanIter
        """
        item = DeviceVlan(parent=self._parent, choice=self._choice, tpid=tpid, priority=priority, id=id, name=name)
        self._add(item)
        return self



class LagPortIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(LagPortIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[LagPort]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> LagPortIter
        return self._iter()

    def __next__(self):
        # type: () -> LagPort
        return self._next()

    def next(self):
        # type: () -> LagPort
        return self._next()

    def port(self, port_name=None):
        # type: (str) -> LagPortIter
        """Factory method that creates an instance of the LagPort class

        The container for a port's ethernet interface and LAG protocol settings

        Returns: LagPortIter
        """
        item = LagPort(parent=self._parent, choice=self._choice, port_name=port_name)
        self._add(item)
        return self



class LagIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(LagIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Lag]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> LagIter
        return self._iter()

    def __next__(self):
        # type: () -> Lag
        return self._next()

    def next(self):
        # type: () -> Lag
        return self._next()

    def lag(self, name=None):
        # type: (str) -> LagIter
        """Factory method that creates an instance of the Lag class

        The container for multiple LAG ports

        Returns: LagIter
        """
        item = Lag(parent=self._parent, choice=self._choice, name=name)
        self._add(item)
        return self



class Layer1(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'port_names': {'type': list},
        'speed': {
            'type': str,
            'enum': [
                'speed_10_fd_mbps',
                'speed_10_hd_mbps',
                'speed_100_fd_mbps',
                'speed_100_hd_mbps',
                'speed_1_gbps',
                'speed_10_gbps',
                'speed_25_gbps',
                'speed_40_gbps',
                'speed_100_gbps',
                'speed_200_gbps',
                'speed_400_gbps',
            ],
        },
        'media': {
            'type': str,
            'enum': [
                'copper',
                'fiber',
                'sgmii',
            ],
        },
        'promiscuous': {'type': bool},
        'mtu': {'type': int},
        'ieee_media_defaults': {'type': bool},
        'auto_negotiate': {'type': bool},
        'auto_negotiation': {'type': 'Layer1AutoNegotiation'},
        'flow_control': {'type': 'Layer1FlowControl'},
        'name': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED = ('port_names', 'name') # type: tuple(str)

    _DEFAULTS = {
        'speed': 'speed_10_gbps',
        'media': 'copper',
        'promiscuous': True,
        'mtu': 1500,
        'ieee_media_defaults': True,
        'auto_negotiate': True,
    } # type: Dict[str, Union(type)]

    SPEED_10_FD_MBPS = 'speed_10_fd_mbps' # type: str
    SPEED_10_HD_MBPS = 'speed_10_hd_mbps' # type: str
    SPEED_100_FD_MBPS = 'speed_100_fd_mbps' # type: str
    SPEED_100_HD_MBPS = 'speed_100_hd_mbps' # type: str
    SPEED_1_GBPS = 'speed_1_gbps' # type: str
    SPEED_10_GBPS = 'speed_10_gbps' # type: str
    SPEED_25_GBPS = 'speed_25_gbps' # type: str
    SPEED_40_GBPS = 'speed_40_gbps' # type: str
    SPEED_100_GBPS = 'speed_100_gbps' # type: str
    SPEED_200_GBPS = 'speed_200_gbps' # type: str
    SPEED_400_GBPS = 'speed_400_gbps' # type: str

    COPPER = 'copper' # type: str
    FIBER = 'fiber' # type: str
    SGMII = 'sgmii' # type: str

    def __init__(self, parent=None, choice=None, port_names=None, speed=None, media=None, promiscuous=None, mtu=None, ieee_media_defaults=None, auto_negotiate=None, name=None):
        super(Layer1, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('port_names', port_names)
        self._set_property('speed', speed)
        self._set_property('media', media)
        self._set_property('promiscuous', promiscuous)
        self._set_property('mtu', mtu)
        self._set_property('ieee_media_defaults', ieee_media_defaults)
        self._set_property('auto_negotiate', auto_negotiate)
        self._set_property('name', name)

    @property
    def port_names(self):
        # type: () -> List[str]
        """port_names getter

        A list of unique names of port objects that will share the. choice settings. . . x-constraint:. - /components/schemas/Port/properties/name. 

        Returns: List[str]
        """
        return self._get_property('port_names')

    @port_names.setter
    def port_names(self, value):
        """port_names setter

        A list of unique names of port objects that will share the. choice settings. . . x-constraint:. - /components/schemas/Port/properties/name. 

        value: List[str]
        """
        self._set_property('port_names', value)

    @property
    def speed(self):
        # type: () -> Union[Literal["speed_100_fd_mbps"], Literal["speed_100_gbps"], Literal["speed_100_hd_mbps"], Literal["speed_10_fd_mbps"], Literal["speed_10_gbps"], Literal["speed_10_hd_mbps"], Literal["speed_1_gbps"], Literal["speed_200_gbps"], Literal["speed_25_gbps"], Literal["speed_400_gbps"], Literal["speed_40_gbps"]]
        """speed getter

        Set the speed if supported.

        Returns: Union[Literal["speed_100_fd_mbps"], Literal["speed_100_gbps"], Literal["speed_100_hd_mbps"], Literal["speed_10_fd_mbps"], Literal["speed_10_gbps"], Literal["speed_10_hd_mbps"], Literal["speed_1_gbps"], Literal["speed_200_gbps"], Literal["speed_25_gbps"], Literal["speed_400_gbps"], Literal["speed_40_gbps"]]
        """
        return self._get_property('speed')

    @speed.setter
    def speed(self, value):
        """speed setter

        Set the speed if supported.

        value: Union[Literal["speed_100_fd_mbps"], Literal["speed_100_gbps"], Literal["speed_100_hd_mbps"], Literal["speed_10_fd_mbps"], Literal["speed_10_gbps"], Literal["speed_10_hd_mbps"], Literal["speed_1_gbps"], Literal["speed_200_gbps"], Literal["speed_25_gbps"], Literal["speed_400_gbps"], Literal["speed_40_gbps"]]
        """
        self._set_property('speed', value)

    @property
    def media(self):
        # type: () -> Union[Literal["copper"], Literal["fiber"], Literal["sgmii"]]
        """media getter

        Set the type of media interface if supported.

        Returns: Union[Literal["copper"], Literal["fiber"], Literal["sgmii"]]
        """
        return self._get_property('media')

    @media.setter
    def media(self, value):
        """media setter

        Set the type of media interface if supported.

        value: Union[Literal["copper"], Literal["fiber"], Literal["sgmii"]]
        """
        self._set_property('media', value)

    @property
    def promiscuous(self):
        # type: () -> bool
        """promiscuous getter

        Enable promiscuous mode if supported.

        Returns: bool
        """
        return self._get_property('promiscuous')

    @promiscuous.setter
    def promiscuous(self, value):
        """promiscuous setter

        Enable promiscuous mode if supported.

        value: bool
        """
        self._set_property('promiscuous', value)

    @property
    def mtu(self):
        # type: () -> int
        """mtu getter

        Set the maximum transmission unit size if supported.

        Returns: int
        """
        return self._get_property('mtu')

    @mtu.setter
    def mtu(self, value):
        """mtu setter

        Set the maximum transmission unit size if supported.

        value: int
        """
        self._set_property('mtu', value)

    @property
    def ieee_media_defaults(self):
        # type: () -> bool
        """ieee_media_defaults getter

        Set to true to override the auto_negotiate, link_training. and rs_fec settings for gigabit ethernet interfaces.

        Returns: bool
        """
        return self._get_property('ieee_media_defaults')

    @ieee_media_defaults.setter
    def ieee_media_defaults(self, value):
        """ieee_media_defaults setter

        Set to true to override the auto_negotiate, link_training. and rs_fec settings for gigabit ethernet interfaces.

        value: bool
        """
        self._set_property('ieee_media_defaults', value)

    @property
    def auto_negotiate(self):
        # type: () -> bool
        """auto_negotiate getter

        Enable/disable auto negotiation.

        Returns: bool
        """
        return self._get_property('auto_negotiate')

    @auto_negotiate.setter
    def auto_negotiate(self, value):
        """auto_negotiate setter

        Enable/disable auto negotiation.

        value: bool
        """
        self._set_property('auto_negotiate', value)

    @property
    def auto_negotiation(self):
        # type: () -> Layer1AutoNegotiation
        """auto_negotiation getter

        Container for auto negotiation settings

        Returns: Layer1AutoNegotiation
        """
        return self._get_property('auto_negotiation', Layer1AutoNegotiation)

    @property
    def flow_control(self):
        # type: () -> Layer1FlowControl
        """flow_control getter

        A container for layer1 receive flow control settings.. To enable flow control settings on ports this object must be a valid. object not a null value.

        Returns: Layer1FlowControl
        """
        return self._get_property('flow_control', Layer1FlowControl)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property('name')

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property('name', value)


class Layer1AutoNegotiation(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'advertise_1000_mbps': {'type': bool},
        'advertise_100_fd_mbps': {'type': bool},
        'advertise_100_hd_mbps': {'type': bool},
        'advertise_10_fd_mbps': {'type': bool},
        'advertise_10_hd_mbps': {'type': bool},
        'link_training': {'type': bool},
        'rs_fec': {'type': bool},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'advertise_1000_mbps': True,
        'advertise_100_fd_mbps': True,
        'advertise_100_hd_mbps': True,
        'advertise_10_fd_mbps': True,
        'advertise_10_hd_mbps': True,
        'link_training': False,
        'rs_fec': False,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, advertise_1000_mbps=None, advertise_100_fd_mbps=None, advertise_100_hd_mbps=None, advertise_10_fd_mbps=None, advertise_10_hd_mbps=None, link_training=None, rs_fec=None):
        super(Layer1AutoNegotiation, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('advertise_1000_mbps', advertise_1000_mbps)
        self._set_property('advertise_100_fd_mbps', advertise_100_fd_mbps)
        self._set_property('advertise_100_hd_mbps', advertise_100_hd_mbps)
        self._set_property('advertise_10_fd_mbps', advertise_10_fd_mbps)
        self._set_property('advertise_10_hd_mbps', advertise_10_hd_mbps)
        self._set_property('link_training', link_training)
        self._set_property('rs_fec', rs_fec)

    @property
    def advertise_1000_mbps(self):
        # type: () -> bool
        """advertise_1000_mbps getter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        Returns: bool
        """
        return self._get_property('advertise_1000_mbps')

    @advertise_1000_mbps.setter
    def advertise_1000_mbps(self, value):
        """advertise_1000_mbps setter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        value: bool
        """
        self._set_property('advertise_1000_mbps', value)

    @property
    def advertise_100_fd_mbps(self):
        # type: () -> bool
        """advertise_100_fd_mbps getter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        Returns: bool
        """
        return self._get_property('advertise_100_fd_mbps')

    @advertise_100_fd_mbps.setter
    def advertise_100_fd_mbps(self, value):
        """advertise_100_fd_mbps setter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        value: bool
        """
        self._set_property('advertise_100_fd_mbps', value)

    @property
    def advertise_100_hd_mbps(self):
        # type: () -> bool
        """advertise_100_hd_mbps getter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        Returns: bool
        """
        return self._get_property('advertise_100_hd_mbps')

    @advertise_100_hd_mbps.setter
    def advertise_100_hd_mbps(self, value):
        """advertise_100_hd_mbps setter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        value: bool
        """
        self._set_property('advertise_100_hd_mbps', value)

    @property
    def advertise_10_fd_mbps(self):
        # type: () -> bool
        """advertise_10_fd_mbps getter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        Returns: bool
        """
        return self._get_property('advertise_10_fd_mbps')

    @advertise_10_fd_mbps.setter
    def advertise_10_fd_mbps(self, value):
        """advertise_10_fd_mbps setter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        value: bool
        """
        self._set_property('advertise_10_fd_mbps', value)

    @property
    def advertise_10_hd_mbps(self):
        # type: () -> bool
        """advertise_10_hd_mbps getter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        Returns: bool
        """
        return self._get_property('advertise_10_hd_mbps')

    @advertise_10_hd_mbps.setter
    def advertise_10_hd_mbps(self, value):
        """advertise_10_hd_mbps setter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        value: bool
        """
        self._set_property('advertise_10_hd_mbps', value)

    @property
    def link_training(self):
        # type: () -> bool
        """link_training getter

        Enable/disable gigabit ethernet link training.

        Returns: bool
        """
        return self._get_property('link_training')

    @link_training.setter
    def link_training(self, value):
        """link_training setter

        Enable/disable gigabit ethernet link training.

        value: bool
        """
        self._set_property('link_training', value)

    @property
    def rs_fec(self):
        # type: () -> bool
        """rs_fec getter

        Enable/disable gigabit ethernet reed solomon forward error correction (RS FEC).

        Returns: bool
        """
        return self._get_property('rs_fec')

    @rs_fec.setter
    def rs_fec(self, value):
        """rs_fec setter

        Enable/disable gigabit ethernet reed solomon forward error correction (RS FEC).

        value: bool
        """
        self._set_property('rs_fec', value)


class Layer1FlowControl(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'directed_address': {'type': str},
        'choice': {
            'type': str,
            'enum': [
                'ieee_802_1qbb',
                'ieee_802_3x',
            ],
        },
        'ieee_802_1qbb': {'type': 'Layer1Ieee8021qbb'},
        'ieee_802_3x': {'type': 'Layer1Ieee8023x'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'directed_address': '0180C2000001',
        'choice': 'ieee_802_1qbb',
    } # type: Dict[str, Union(type)]

    IEEE_802_1QBB = 'ieee_802_1qbb' # type: str
    IEEE_802_3X = 'ieee_802_3x' # type: str

    def __init__(self, parent=None, choice=None, directed_address=None):
        super(Layer1FlowControl, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('directed_address', directed_address)

    @property
    def ieee_802_1qbb(self):
        # type: () -> Layer1Ieee8021qbb
        """Factory property that returns an instance of the Layer1Ieee8021qbb class

        These settings enhance the existing 802.3x pause priority capabilities . to enable flow control based on 802.1p priorities (classes of service). 

        Returns: Layer1Ieee8021qbb
        """
        return self._get_property('ieee_802_1qbb', Layer1Ieee8021qbb, self, 'ieee_802_1qbb')

    @property
    def ieee_802_3x(self):
        # type: () -> Layer1Ieee8023x
        """Factory property that returns an instance of the Layer1Ieee8023x class

        A container for ieee 802.3x rx pause settings

        Returns: Layer1Ieee8023x
        """
        return self._get_property('ieee_802_3x', Layer1Ieee8023x, self, 'ieee_802_3x')

    @property
    def directed_address(self):
        # type: () -> str
        """directed_address getter

        The 48bit mac address that the layer1 port names will listen on. for a directed pause. 

        Returns: str
        """
        return self._get_property('directed_address')

    @directed_address.setter
    def directed_address(self, value):
        """directed_address setter

        The 48bit mac address that the layer1 port names will listen on. for a directed pause. 

        value: str
        """
        self._set_property('directed_address', value)

    @property
    def choice(self):
        # type: () -> Union[Literal["ieee_802_1qbb"], Literal["ieee_802_3x"]]
        """choice getter

        The type of priority flow control.

        Returns: Union[Literal["ieee_802_1qbb"], Literal["ieee_802_3x"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of priority flow control.

        value: Union[Literal["ieee_802_1qbb"], Literal["ieee_802_3x"]]
        """
        self._set_property('choice', value)


class Layer1Ieee8021qbb(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'pfc_delay': {'type': int},
        'pfc_class_0': {'type': int},
        'pfc_class_1': {'type': int},
        'pfc_class_2': {'type': int},
        'pfc_class_3': {'type': int},
        'pfc_class_4': {'type': int},
        'pfc_class_5': {'type': int},
        'pfc_class_6': {'type': int},
        'pfc_class_7': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'pfc_delay': 0,
        'pfc_class_0': 0,
        'pfc_class_1': 1,
        'pfc_class_2': 2,
        'pfc_class_3': 3,
        'pfc_class_4': 4,
        'pfc_class_5': 5,
        'pfc_class_6': 6,
        'pfc_class_7': 7,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, pfc_delay=None, pfc_class_0=None, pfc_class_1=None, pfc_class_2=None, pfc_class_3=None, pfc_class_4=None, pfc_class_5=None, pfc_class_6=None, pfc_class_7=None):
        super(Layer1Ieee8021qbb, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('pfc_delay', pfc_delay)
        self._set_property('pfc_class_0', pfc_class_0)
        self._set_property('pfc_class_1', pfc_class_1)
        self._set_property('pfc_class_2', pfc_class_2)
        self._set_property('pfc_class_3', pfc_class_3)
        self._set_property('pfc_class_4', pfc_class_4)
        self._set_property('pfc_class_5', pfc_class_5)
        self._set_property('pfc_class_6', pfc_class_6)
        self._set_property('pfc_class_7', pfc_class_7)

    @property
    def pfc_delay(self):
        # type: () -> int
        """pfc_delay getter

        The upper limit on the transmit time of a queue after receiving a . message to pause a specified priority.. A value of 0 or null indicates that pfc delay will not be enabled. 

        Returns: int
        """
        return self._get_property('pfc_delay')

    @pfc_delay.setter
    def pfc_delay(self, value):
        """pfc_delay setter

        The upper limit on the transmit time of a queue after receiving a . message to pause a specified priority.. A value of 0 or null indicates that pfc delay will not be enabled. 

        value: int
        """
        self._set_property('pfc_delay', value)

    @property
    def pfc_class_0(self):
        # type: () -> int
        """pfc_class_0 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property('pfc_class_0')

    @pfc_class_0.setter
    def pfc_class_0(self, value):
        """pfc_class_0 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property('pfc_class_0', value)

    @property
    def pfc_class_1(self):
        # type: () -> int
        """pfc_class_1 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property('pfc_class_1')

    @pfc_class_1.setter
    def pfc_class_1(self, value):
        """pfc_class_1 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property('pfc_class_1', value)

    @property
    def pfc_class_2(self):
        # type: () -> int
        """pfc_class_2 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property('pfc_class_2')

    @pfc_class_2.setter
    def pfc_class_2(self, value):
        """pfc_class_2 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property('pfc_class_2', value)

    @property
    def pfc_class_3(self):
        # type: () -> int
        """pfc_class_3 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property('pfc_class_3')

    @pfc_class_3.setter
    def pfc_class_3(self, value):
        """pfc_class_3 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property('pfc_class_3', value)

    @property
    def pfc_class_4(self):
        # type: () -> int
        """pfc_class_4 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property('pfc_class_4')

    @pfc_class_4.setter
    def pfc_class_4(self, value):
        """pfc_class_4 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property('pfc_class_4', value)

    @property
    def pfc_class_5(self):
        # type: () -> int
        """pfc_class_5 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property('pfc_class_5')

    @pfc_class_5.setter
    def pfc_class_5(self, value):
        """pfc_class_5 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property('pfc_class_5', value)

    @property
    def pfc_class_6(self):
        # type: () -> int
        """pfc_class_6 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property('pfc_class_6')

    @pfc_class_6.setter
    def pfc_class_6(self, value):
        """pfc_class_6 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property('pfc_class_6', value)

    @property
    def pfc_class_7(self):
        # type: () -> int
        """pfc_class_7 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property('pfc_class_7')

    @pfc_class_7.setter
    def pfc_class_7(self, value):
        """pfc_class_7 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property('pfc_class_7', value)


class Layer1Ieee8023x(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {} # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(Layer1Ieee8023x, self).__init__()
        self._parent = parent
        self._choice = choice


class Layer1Iter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(Layer1Iter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Layer1]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> Layer1Iter
        return self._iter()

    def __next__(self):
        # type: () -> Layer1
        return self._next()

    def next(self):
        # type: () -> Layer1
        return self._next()

    def layer1(self, port_names=None, speed='speed_10_gbps', media='copper', promiscuous=True, mtu=1500, ieee_media_defaults=True, auto_negotiate=True, name=None):
        # type: (List[str],Union[Literal["speed_100_fd_mbps"], Literal["speed_100_gbps"], Literal["speed_100_hd_mbps"], Literal["speed_10_fd_mbps"], Literal["speed_10_gbps"], Literal["speed_10_hd_mbps"], Literal["speed_1_gbps"], Literal["speed_200_gbps"], Literal["speed_25_gbps"], Literal["speed_400_gbps"], Literal["speed_40_gbps"]],Union[Literal["copper"], Literal["fiber"], Literal["sgmii"]],bool,int,bool,bool,str) -> Layer1Iter
        """Factory method that creates an instance of the Layer1 class

        A container for layer1 settings.

        Returns: Layer1Iter
        """
        item = Layer1(parent=self._parent, choice=self._choice, port_names=port_names, speed=speed, media=media, promiscuous=promiscuous, mtu=mtu, ieee_media_defaults=ieee_media_defaults, auto_negotiate=auto_negotiate, name=name)
        self._add(item)
        return self



class Capture(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'port_names': {'type': list},
        'filters': {'type': 'CaptureFilterIter'},
        'overwrite': {'type': bool},
        'packet_size': {'type': int},
        'format': {
            'type': str,
            'enum': [
                'pcap',
                'pcapng',
            ],
        },
        'name': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED = ('port_names', 'name') # type: tuple(str)

    _DEFAULTS = {
        'overwrite': False,
        'format': 'pcap',
    } # type: Dict[str, Union(type)]

    PCAP = 'pcap' # type: str
    PCAPNG = 'pcapng' # type: str

    def __init__(self, parent=None, choice=None, port_names=None, overwrite=None, packet_size=None, format=None, name=None):
        super(Capture, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('port_names', port_names)
        self._set_property('overwrite', overwrite)
        self._set_property('packet_size', packet_size)
        self._set_property('format', format)
        self._set_property('name', name)

    @property
    def port_names(self):
        # type: () -> List[str]
        """port_names getter

        The unique names of ports that the capture settings will apply to. Port_names cannot be duplicated between capture objects.. . x-constraint:. - /components/schemas/Port/properties/name. 

        Returns: List[str]
        """
        return self._get_property('port_names')

    @port_names.setter
    def port_names(self, value):
        """port_names setter

        The unique names of ports that the capture settings will apply to. Port_names cannot be duplicated between capture objects.. . x-constraint:. - /components/schemas/Port/properties/name. 

        value: List[str]
        """
        self._set_property('port_names', value)

    @property
    def filters(self):
        # type: () -> CaptureFilterIter
        """filters getter

        A list of filters to apply to the capturing ports. If no filters are specified then all packets will be captured. A capture can have multiple filters. The number of filters supported is determined by the implementation which can be retrieved using the capabilities API.. When multiple filters are specified the capture implementation must && (and) all the filters.

        Returns: CaptureFilterIter
        """
        return self._get_property('filters', CaptureFilterIter, self._parent, self._choice)

    @property
    def overwrite(self):
        # type: () -> bool
        """overwrite getter

        Overwrite the capture buffer.

        Returns: bool
        """
        return self._get_property('overwrite')

    @overwrite.setter
    def overwrite(self, value):
        """overwrite setter

        Overwrite the capture buffer.

        value: bool
        """
        self._set_property('overwrite', value)

    @property
    def packet_size(self):
        # type: () -> int
        """packet_size getter

        The maximum size of each captured packet. If no value is specified or it is null then the entire packet will be captured.

        Returns: int
        """
        return self._get_property('packet_size')

    @packet_size.setter
    def packet_size(self, value):
        """packet_size setter

        The maximum size of each captured packet. If no value is specified or it is null then the entire packet will be captured.

        value: int
        """
        self._set_property('packet_size', value)

    @property
    def format(self):
        # type: () -> Union[Literal["pcap"], Literal["pcapng"]]
        """format getter

        The format of the capture file.

        Returns: Union[Literal["pcap"], Literal["pcapng"]]
        """
        return self._get_property('format')

    @format.setter
    def format(self, value):
        """format setter

        The format of the capture file.

        value: Union[Literal["pcap"], Literal["pcapng"]]
        """
        self._set_property('format', value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property('name')

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property('name', value)


class CaptureFilter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'custom',
                'ethernet',
                'vlan',
                'ipv4',
            ],
        },
        'custom': {'type': 'CaptureCustom'},
        'ethernet': {'type': 'CaptureEthernet'},
        'vlan': {'type': 'CaptureVlan'},
        'ipv4': {'type': 'CaptureIpv4'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'custom',
    } # type: Dict[str, Union(type)]

    CUSTOM = 'custom' # type: str
    ETHERNET = 'ethernet' # type: str
    VLAN = 'vlan' # type: str
    IPV4 = 'ipv4' # type: str

    def __init__(self, parent=None, choice=None):
        super(CaptureFilter, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def custom(self):
        # type: () -> CaptureCustom
        """Factory property that returns an instance of the CaptureCustom class

        TBD

        Returns: CaptureCustom
        """
        return self._get_property('custom', CaptureCustom, self, 'custom')

    @property
    def ethernet(self):
        # type: () -> CaptureEthernet
        """Factory property that returns an instance of the CaptureEthernet class

        TBD

        Returns: CaptureEthernet
        """
        return self._get_property('ethernet', CaptureEthernet, self, 'ethernet')

    @property
    def vlan(self):
        # type: () -> CaptureVlan
        """Factory property that returns an instance of the CaptureVlan class

        TBD

        Returns: CaptureVlan
        """
        return self._get_property('vlan', CaptureVlan, self, 'vlan')

    @property
    def ipv4(self):
        # type: () -> CaptureIpv4
        """Factory property that returns an instance of the CaptureIpv4 class

        TBD

        Returns: CaptureIpv4
        """
        return self._get_property('ipv4', CaptureIpv4, self, 'ipv4')

    @property
    def choice(self):
        # type: () -> Union[Literal["custom"], Literal["ethernet"], Literal["ipv4"], Literal["vlan"]]
        """choice getter

        The type of capture filter.

        Returns: Union[Literal["custom"], Literal["ethernet"], Literal["ipv4"], Literal["vlan"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of capture filter.

        value: Union[Literal["custom"], Literal["ethernet"], Literal["ipv4"], Literal["vlan"]]
        """
        self._set_property('choice', value)


class CaptureCustom(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'offset': {'type': int},
        'bit_length': {'type': int},
        'value': {
            'type': str,
            'format': 'hex',
        },
        'mask': {
            'type': str,
            'format': 'hex',
        },
        'negate': {'type': bool},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'offset': 0,
        'bit_length': 8,
        'value': '00',
        'mask': '00',
        'negate': False,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, offset=None, bit_length=None, value=None, mask=None, negate=None):
        super(CaptureCustom, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('offset', offset)
        self._set_property('bit_length', bit_length)
        self._set_property('value', value)
        self._set_property('mask', mask)
        self._set_property('negate', negate)

    @property
    def offset(self):
        # type: () -> int
        """offset getter

        The byte offset of field to filter on

        Returns: int
        """
        return self._get_property('offset')

    @offset.setter
    def offset(self, value):
        """offset setter

        The byte offset of field to filter on

        value: int
        """
        self._set_property('offset', value)

    @property
    def bit_length(self):
        # type: () -> int
        """bit_length getter

        The bit length of field to filter on

        Returns: int
        """
        return self._get_property('bit_length')

    @bit_length.setter
    def bit_length(self, value):
        """bit_length setter

        The bit length of field to filter on

        value: int
        """
        self._set_property('bit_length', value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property('value', value)

    @property
    def mask(self):
        # type: () -> str
        """mask getter

        TBD

        Returns: str
        """
        return self._get_property('mask')

    @mask.setter
    def mask(self, value):
        """mask setter

        TBD

        value: str
        """
        self._set_property('mask', value)

    @property
    def negate(self):
        # type: () -> bool
        """negate getter

        TBD

        Returns: bool
        """
        return self._get_property('negate')

    @negate.setter
    def negate(self, value):
        """negate setter

        TBD

        value: bool
        """
        self._set_property('negate', value)


class CaptureEthernet(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'src': {'type': 'CaptureField'},
        'dst': {'type': 'CaptureField'},
        'ether_type': {'type': 'CaptureField'},
        'pfc_queue': {'type': 'CaptureField'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(CaptureEthernet, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def src(self):
        # type: () -> CaptureField
        """src getter

        

        Returns: CaptureField
        """
        return self._get_property('src', CaptureField)

    @property
    def dst(self):
        # type: () -> CaptureField
        """dst getter

        

        Returns: CaptureField
        """
        return self._get_property('dst', CaptureField)

    @property
    def ether_type(self):
        # type: () -> CaptureField
        """ether_type getter

        

        Returns: CaptureField
        """
        return self._get_property('ether_type', CaptureField)

    @property
    def pfc_queue(self):
        # type: () -> CaptureField
        """pfc_queue getter

        

        Returns: CaptureField
        """
        return self._get_property('pfc_queue', CaptureField)


class CaptureField(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'value': {
            'type': str,
            'format': 'hex',
        },
        'mask': {
            'type': str,
            'format': 'hex',
        },
        'negate': {'type': bool},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'value': '00',
        'mask': '00',
        'negate': False,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, value=None, mask=None, negate=None):
        super(CaptureField, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('value', value)
        self._set_property('mask', mask)
        self._set_property('negate', negate)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property('value', value)

    @property
    def mask(self):
        # type: () -> str
        """mask getter

        TBD

        Returns: str
        """
        return self._get_property('mask')

    @mask.setter
    def mask(self, value):
        """mask setter

        TBD

        value: str
        """
        self._set_property('mask', value)

    @property
    def negate(self):
        # type: () -> bool
        """negate getter

        TBD

        Returns: bool
        """
        return self._get_property('negate')

    @negate.setter
    def negate(self, value):
        """negate setter

        TBD

        value: bool
        """
        self._set_property('negate', value)


class CaptureVlan(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'priority': {'type': 'CaptureField'},
        'cfi': {'type': 'CaptureField'},
        'id': {'type': 'CaptureField'},
        'protocol': {'type': 'CaptureField'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(CaptureVlan, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def priority(self):
        # type: () -> CaptureField
        """priority getter

        

        Returns: CaptureField
        """
        return self._get_property('priority', CaptureField)

    @property
    def cfi(self):
        # type: () -> CaptureField
        """cfi getter

        

        Returns: CaptureField
        """
        return self._get_property('cfi', CaptureField)

    @property
    def id(self):
        # type: () -> CaptureField
        """id getter

        

        Returns: CaptureField
        """
        return self._get_property('id', CaptureField)

    @property
    def protocol(self):
        # type: () -> CaptureField
        """protocol getter

        

        Returns: CaptureField
        """
        return self._get_property('protocol', CaptureField)


class CaptureIpv4(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'version': {'type': 'CaptureField'},
        'header_length': {'type': 'CaptureField'},
        'priority': {'type': 'CaptureField'},
        'total_length': {'type': 'CaptureField'},
        'identification': {'type': 'CaptureField'},
        'reserved': {'type': 'CaptureField'},
        'dont_fragment': {'type': 'CaptureField'},
        'more_fragments': {'type': 'CaptureField'},
        'fragment_offset': {'type': 'CaptureField'},
        'time_to_live': {'type': 'CaptureField'},
        'protocol': {'type': 'CaptureField'},
        'header_checksum': {'type': 'CaptureField'},
        'src': {'type': 'CaptureField'},
        'dst': {'type': 'CaptureField'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(CaptureIpv4, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def version(self):
        # type: () -> CaptureField
        """version getter

        

        Returns: CaptureField
        """
        return self._get_property('version', CaptureField)

    @property
    def header_length(self):
        # type: () -> CaptureField
        """header_length getter

        

        Returns: CaptureField
        """
        return self._get_property('header_length', CaptureField)

    @property
    def priority(self):
        # type: () -> CaptureField
        """priority getter

        

        Returns: CaptureField
        """
        return self._get_property('priority', CaptureField)

    @property
    def total_length(self):
        # type: () -> CaptureField
        """total_length getter

        

        Returns: CaptureField
        """
        return self._get_property('total_length', CaptureField)

    @property
    def identification(self):
        # type: () -> CaptureField
        """identification getter

        

        Returns: CaptureField
        """
        return self._get_property('identification', CaptureField)

    @property
    def reserved(self):
        # type: () -> CaptureField
        """reserved getter

        

        Returns: CaptureField
        """
        return self._get_property('reserved', CaptureField)

    @property
    def dont_fragment(self):
        # type: () -> CaptureField
        """dont_fragment getter

        

        Returns: CaptureField
        """
        return self._get_property('dont_fragment', CaptureField)

    @property
    def more_fragments(self):
        # type: () -> CaptureField
        """more_fragments getter

        

        Returns: CaptureField
        """
        return self._get_property('more_fragments', CaptureField)

    @property
    def fragment_offset(self):
        # type: () -> CaptureField
        """fragment_offset getter

        

        Returns: CaptureField
        """
        return self._get_property('fragment_offset', CaptureField)

    @property
    def time_to_live(self):
        # type: () -> CaptureField
        """time_to_live getter

        

        Returns: CaptureField
        """
        return self._get_property('time_to_live', CaptureField)

    @property
    def protocol(self):
        # type: () -> CaptureField
        """protocol getter

        

        Returns: CaptureField
        """
        return self._get_property('protocol', CaptureField)

    @property
    def header_checksum(self):
        # type: () -> CaptureField
        """header_checksum getter

        

        Returns: CaptureField
        """
        return self._get_property('header_checksum', CaptureField)

    @property
    def src(self):
        # type: () -> CaptureField
        """src getter

        

        Returns: CaptureField
        """
        return self._get_property('src', CaptureField)

    @property
    def dst(self):
        # type: () -> CaptureField
        """dst getter

        

        Returns: CaptureField
        """
        return self._get_property('dst', CaptureField)


class CaptureFilterIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = True

    def __init__(self, parent=None, choice=None):
        super(CaptureFilterIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[CaptureCustom, CaptureEthernet, CaptureFilter, CaptureIpv4, CaptureVlan]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> CaptureFilterIter
        return self._iter()

    def __next__(self):
        # type: () -> CaptureFilter
        return self._next()

    def next(self):
        # type: () -> CaptureFilter
        return self._next()

    def filter(self):
        # type: () -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureFilter class

        Container for capture filters

        Returns: CaptureFilterIter
        """
        item = CaptureFilter(parent=self._parent, choice=self._choice)
        self._add(item)
        return self


    def custom(self, offset=0, bit_length=8, value='00', mask='00', negate=False):
        # type: (int,int,str,str,bool) -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureCustom class

        TBD

        Returns: CaptureFilterIter
        """
        item = CaptureFilter()
        item.custom
        item.choice = 'custom'
        self._add(item)
        return self


    def ethernet(self):
        # type: () -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureEthernet class

        TBD

        Returns: CaptureFilterIter
        """
        item = CaptureFilter()
        item.ethernet
        item.choice = 'ethernet'
        self._add(item)
        return self


    def vlan(self):
        # type: () -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureVlan class

        TBD

        Returns: CaptureFilterIter
        """
        item = CaptureFilter()
        item.vlan
        item.choice = 'vlan'
        self._add(item)
        return self


    def ipv4(self):
        # type: () -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureIpv4 class

        TBD

        Returns: CaptureFilterIter
        """
        item = CaptureFilter()
        item.ipv4
        item.choice = 'ipv4'
        self._add(item)
        return self



class CaptureIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(CaptureIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Capture]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> CaptureIter
        return self._iter()

    def __next__(self):
        # type: () -> Capture
        return self._next()

    def next(self):
        # type: () -> Capture
        return self._next()

    def capture(self, port_names=None, overwrite=False, packet_size=None, format='pcap', name=None):
        # type: (List[str],bool,int,Union[Literal["pcap"], Literal["pcapng"]],str) -> CaptureIter
        """Factory method that creates an instance of the Capture class

        Container for capture settings.

        Returns: CaptureIter
        """
        item = Capture(parent=self._parent, choice=self._choice, port_names=port_names, overwrite=overwrite, packet_size=packet_size, format=format, name=name)
        self._add(item)
        return self



class Device(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'container_name': {'type': str},
        'ethernet': {'type': 'DeviceEthernet'},
        'name': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED = ('container_name', 'ethernet', 'name') # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, container_name=None, name=None):
        super(Device, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('container_name', container_name)
        self._set_property('name', name)

    @property
    def container_name(self):
        # type: () -> str
        """container_name getter

        The unique name of a Port or Lag object that will contain the emulated interfaces and/or protocol devices.. . x-constraint:. - /components/schemas/Port/properties/name. - /components/schemas/Lag/properties/name. 

        Returns: str
        """
        return self._get_property('container_name')

    @container_name.setter
    def container_name(self, value):
        """container_name setter

        The unique name of a Port or Lag object that will contain the emulated interfaces and/or protocol devices.. . x-constraint:. - /components/schemas/Port/properties/name. - /components/schemas/Lag/properties/name. 

        value: str
        """
        self._set_property('container_name', value)

    @property
    def ethernet(self):
        # type: () -> DeviceEthernet
        """ethernet getter

        An ethernet interface with an IPv4 and IPv6 interface stack. Base ethernet interfaceThe ethernet stack.

        Returns: DeviceEthernet
        """
        return self._get_property('ethernet', DeviceEthernet)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property('name')

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property('name', value)


class DeviceEthernet(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'ipv4': {'type': 'DeviceIpv4'},
        'ipv6': {'type': 'DeviceIpv6'},
        'mac': {
            'type': str,
            'format': 'mac',
        },
        'mtu': {'type': int},
        'vlans': {'type': 'DeviceVlanIter'},
        'name': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED = ('mac', 'name') # type: tuple(str)

    _DEFAULTS = {
        'mtu': 1500,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, mac=None, mtu=None, name=None):
        super(DeviceEthernet, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('mac', mac)
        self._set_property('mtu', mtu)
        self._set_property('name', name)

    @property
    def ipv4(self):
        # type: () -> DeviceIpv4
        """ipv4 getter

        An IPv4 interface with gateway. A base IPv4 interface

        Returns: DeviceIpv4
        """
        return self._get_property('ipv4', DeviceIpv4)

    @property
    def ipv6(self):
        # type: () -> DeviceIpv6
        """ipv6 getter

        An IPv6 interface with gateway. A base IPv6 interface

        Returns: DeviceIpv6
        """
        return self._get_property('ipv6', DeviceIpv6)

    @property
    def mac(self):
        # type: () -> str
        """mac getter

        Media access control address

        Returns: str
        """
        return self._get_property('mac')

    @mac.setter
    def mac(self, value):
        """mac setter

        Media access control address

        value: str
        """
        self._set_property('mac', value)

    @property
    def mtu(self):
        # type: () -> int
        """mtu getter

        Maximum transmission unit

        Returns: int
        """
        return self._get_property('mtu')

    @mtu.setter
    def mtu(self, value):
        """mtu setter

        Maximum transmission unit

        value: int
        """
        self._set_property('mtu', value)

    @property
    def vlans(self):
        # type: () -> DeviceVlanIter
        """vlans getter

        List of VLANs

        Returns: DeviceVlanIter
        """
        return self._get_property('vlans', DeviceVlanIter, self._parent, self._choice)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property('name')

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property('name', value)


class DeviceIpv4(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'gateway': {
            'type': str,
            'format': 'ipv4',
        },
        'address': {
            'type': str,
            'format': 'ipv4',
        },
        'prefix': {'type': int},
        'bgpv4': {'type': 'DeviceBgpv4'},
        'name': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED = ('gateway', 'address', 'name') # type: tuple(str)

    _DEFAULTS = {
        'prefix': 24,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, gateway=None, address=None, prefix=None, name=None):
        super(DeviceIpv4, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('gateway', gateway)
        self._set_property('address', address)
        self._set_property('prefix', prefix)
        self._set_property('name', name)

    @property
    def gateway(self):
        # type: () -> str
        """gateway getter

        The IPv4 address of the gateway

        Returns: str
        """
        return self._get_property('gateway')

    @gateway.setter
    def gateway(self, value):
        """gateway setter

        The IPv4 address of the gateway

        value: str
        """
        self._set_property('gateway', value)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The IPv4 address

        Returns: str
        """
        return self._get_property('address')

    @address.setter
    def address(self, value):
        """address setter

        The IPv4 address

        value: str
        """
        self._set_property('address', value)

    @property
    def prefix(self):
        # type: () -> int
        """prefix getter

        The prefix of the IPv4 address

        Returns: int
        """
        return self._get_property('prefix')

    @prefix.setter
    def prefix(self, value):
        """prefix setter

        The prefix of the IPv4 address

        value: int
        """
        self._set_property('prefix', value)

    @property
    def bgpv4(self):
        # type: () -> DeviceBgpv4
        """bgpv4 getter

        Container for emulated BGPv4 peers and routes.. Container for basic emulated BGP peer settings.

        Returns: DeviceBgpv4
        """
        return self._get_property('bgpv4', DeviceBgpv4)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property('name')

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property('name', value)


class DeviceBgpv4(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'local_address': {
            'type': str,
            'format': 'ipv4',
        },
        'dut_address': {
            'type': str,
            'format': 'ipv4',
        },
        'router_id': {
            'type': str,
            'format': 'ipv4',
        },
        'as_type': {
            'type': str,
            'enum': [
                'ibgp',
                'ebgp',
            ],
        },
        'as_number': {'type': int},
        'as_number_width': {
            'type': str,
            'enum': [
                'two',
                'four',
            ],
        },
        'as_number_set_mode': {
            'type': str,
            'enum': [
                'do_not_include_as',
                'include_as_seq',
                'include_as_set',
                'include_as_seq_confed',
                'include_as_set_confed',
                'prepend_as_to_first_segment',
            ],
        },
        'advanced': {'type': 'DeviceBgpAdvanced'},
        'capability': {'type': 'DeviceBgpCapability'},
        'sr_te_policies': {'type': 'DeviceBgpSrTePolicyIter'},
        'bgpv4_routes': {'type': 'DeviceBgpv4RouteIter'},
        'bgpv6_routes': {'type': 'DeviceBgpv6RouteIter'},
        'name': {'type': str},
        'active': {'type': bool},
    } # type: Dict[str, str]

    _REQUIRED = ('local_address', 'dut_address', 'as_type', 'as_number', 'name') # type: tuple(str)

    _DEFAULTS = {
        'router_id': '0.0.0.0',
        'as_number_width': 'four',
        'as_number_set_mode': 'do_not_include_as',
        'active': True,
    } # type: Dict[str, Union(type)]

    IBGP = 'ibgp' # type: str
    EBGP = 'ebgp' # type: str

    TWO = 'two' # type: str
    FOUR = 'four' # type: str

    DO_NOT_INCLUDE_AS = 'do_not_include_as' # type: str
    INCLUDE_AS_SEQ = 'include_as_seq' # type: str
    INCLUDE_AS_SET = 'include_as_set' # type: str
    INCLUDE_AS_SEQ_CONFED = 'include_as_seq_confed' # type: str
    INCLUDE_AS_SET_CONFED = 'include_as_set_confed' # type: str
    PREPEND_AS_TO_FIRST_SEGMENT = 'prepend_as_to_first_segment' # type: str

    def __init__(self, parent=None, choice=None, local_address=None, dut_address=None, router_id=None, as_type=None, as_number=None, as_number_width=None, as_number_set_mode=None, name=None, active=None):
        super(DeviceBgpv4, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('local_address', local_address)
        self._set_property('dut_address', dut_address)
        self._set_property('router_id', router_id)
        self._set_property('as_type', as_type)
        self._set_property('as_number', as_number)
        self._set_property('as_number_width', as_number_width)
        self._set_property('as_number_set_mode', as_number_set_mode)
        self._set_property('name', name)
        self._set_property('active', active)

    @property
    def local_address(self):
        # type: () -> str
        """local_address getter

        Local IPv4 address of the emulated router

        Returns: str
        """
        return self._get_property('local_address')

    @local_address.setter
    def local_address(self, value):
        """local_address setter

        Local IPv4 address of the emulated router

        value: str
        """
        self._set_property('local_address', value)

    @property
    def dut_address(self):
        # type: () -> str
        """dut_address getter

        IPv4 address of the BGP peer for the session

        Returns: str
        """
        return self._get_property('dut_address')

    @dut_address.setter
    def dut_address(self, value):
        """dut_address setter

        IPv4 address of the BGP peer for the session

        value: str
        """
        self._set_property('dut_address', value)

    @property
    def router_id(self):
        # type: () -> str
        """router_id getter

        The BGP router ID is a unique identifier used by routing protocols. It is a 32-bit value that is often represented by an IPv4 address.

        Returns: str
        """
        return self._get_property('router_id')

    @router_id.setter
    def router_id(self, value):
        """router_id setter

        The BGP router ID is a unique identifier used by routing protocols. It is a 32-bit value that is often represented by an IPv4 address.

        value: str
        """
        self._set_property('router_id', value)

    @property
    def as_type(self):
        # type: () -> Union[Literal["ebgp"], Literal["ibgp"]]
        """as_type getter

        The type of BGP autonomous system. External BGP is used for BGP links between two or more autonomous systems (ebgp) Internal BGP is used within a single autonomous system (ibgp). BGP property defaults are aligned with this object defined as an internal BGP peer. If the as_type is specified as ebgp then other properties will need to be specified according an external BGP peer.

        Returns: Union[Literal["ebgp"], Literal["ibgp"]]
        """
        return self._get_property('as_type')

    @as_type.setter
    def as_type(self, value):
        """as_type setter

        The type of BGP autonomous system. External BGP is used for BGP links between two or more autonomous systems (ebgp) Internal BGP is used within a single autonomous system (ibgp). BGP property defaults are aligned with this object defined as an internal BGP peer. If the as_type is specified as ebgp then other properties will need to be specified according an external BGP peer.

        value: Union[Literal["ebgp"], Literal["ibgp"]]
        """
        self._set_property('as_type', value)

    @property
    def as_number(self):
        # type: () -> int
        """as_number getter

        Autonomous System Number (AS number or ASN)

        Returns: int
        """
        return self._get_property('as_number')

    @as_number.setter
    def as_number(self, value):
        """as_number setter

        Autonomous System Number (AS number or ASN)

        value: int
        """
        self._set_property('as_number', value)

    @property
    def as_number_width(self):
        # type: () -> Union[Literal["four"], Literal["two"]]
        """as_number_width getter

        The width in bytes of the as_number values. Any as_number values that exceed the width MUST result in an error

        Returns: Union[Literal["four"], Literal["two"]]
        """
        return self._get_property('as_number_width')

    @as_number_width.setter
    def as_number_width(self, value):
        """as_number_width setter

        The width in bytes of the as_number values. Any as_number values that exceed the width MUST result in an error

        value: Union[Literal["four"], Literal["two"]]
        """
        self._set_property('as_number_width', value)

    @property
    def as_number_set_mode(self):
        # type: () -> Union[Literal["do_not_include_as"], Literal["include_as_seq"], Literal["include_as_seq_confed"], Literal["include_as_set"], Literal["include_as_set_confed"], Literal["prepend_as_to_first_segment"]]
        """as_number_set_mode getter

        The AS number set mode

        Returns: Union[Literal["do_not_include_as"], Literal["include_as_seq"], Literal["include_as_seq_confed"], Literal["include_as_set"], Literal["include_as_set_confed"], Literal["prepend_as_to_first_segment"]]
        """
        return self._get_property('as_number_set_mode')

    @as_number_set_mode.setter
    def as_number_set_mode(self, value):
        """as_number_set_mode setter

        The AS number set mode

        value: Union[Literal["do_not_include_as"], Literal["include_as_seq"], Literal["include_as_seq_confed"], Literal["include_as_set"], Literal["include_as_set_confed"], Literal["prepend_as_to_first_segment"]]
        """
        self._set_property('as_number_set_mode', value)

    @property
    def advanced(self):
        # type: () -> DeviceBgpAdvanced
        """advanced getter

        Container for BGP advanced settings.

        Returns: DeviceBgpAdvanced
        """
        return self._get_property('advanced', DeviceBgpAdvanced)

    @property
    def capability(self):
        # type: () -> DeviceBgpCapability
        """capability getter

        Container for BGP capability settings.

        Returns: DeviceBgpCapability
        """
        return self._get_property('capability', DeviceBgpCapability)

    @property
    def sr_te_policies(self):
        # type: () -> DeviceBgpSrTePolicyIter
        """sr_te_policies getter

        Segment routing/traffic engineering policies

        Returns: DeviceBgpSrTePolicyIter
        """
        return self._get_property('sr_te_policies', DeviceBgpSrTePolicyIter, self._parent, self._choice)

    @property
    def bgpv4_routes(self):
        # type: () -> DeviceBgpv4RouteIter
        """bgpv4_routes getter

        Emulated BGPv4 routes

        Returns: DeviceBgpv4RouteIter
        """
        return self._get_property('bgpv4_routes', DeviceBgpv4RouteIter, self._parent, self._choice)

    @property
    def bgpv6_routes(self):
        # type: () -> DeviceBgpv6RouteIter
        """bgpv6_routes getter

        Emulated BGPv6 routes

        Returns: DeviceBgpv6RouteIter
        """
        return self._get_property('bgpv6_routes', DeviceBgpv6RouteIter, self._parent, self._choice)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property('name')

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property('name', value)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property('active')

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property('active', value)


class DeviceBgpAdvanced(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'hold_time_interval': {'type': int},
        'keep_alive_interval': {'type': int},
        'update_interval': {'type': int},
        'time_to_live': {'type': int},
        'md5_key': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'hold_time_interval': 90,
        'keep_alive_interval': 30,
        'update_interval': 0,
        'time_to_live': 64,
        'md5_key': '',
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, hold_time_interval=None, keep_alive_interval=None, update_interval=None, time_to_live=None, md5_key=None):
        super(DeviceBgpAdvanced, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('hold_time_interval', hold_time_interval)
        self._set_property('keep_alive_interval', keep_alive_interval)
        self._set_property('update_interval', update_interval)
        self._set_property('time_to_live', time_to_live)
        self._set_property('md5_key', md5_key)

    @property
    def hold_time_interval(self):
        # type: () -> int
        """hold_time_interval getter

        Number of seconds the sender proposes for the value of the Hold Timer

        Returns: int
        """
        return self._get_property('hold_time_interval')

    @hold_time_interval.setter
    def hold_time_interval(self, value):
        """hold_time_interval setter

        Number of seconds the sender proposes for the value of the Hold Timer

        value: int
        """
        self._set_property('hold_time_interval', value)

    @property
    def keep_alive_interval(self):
        # type: () -> int
        """keep_alive_interval getter

        Number of seconds between transmissions of Keep Alive messages by router

        Returns: int
        """
        return self._get_property('keep_alive_interval')

    @keep_alive_interval.setter
    def keep_alive_interval(self, value):
        """keep_alive_interval setter

        Number of seconds between transmissions of Keep Alive messages by router

        value: int
        """
        self._set_property('keep_alive_interval', value)

    @property
    def update_interval(self):
        # type: () -> int
        """update_interval getter

        The time interval at which UPDATE messages are sent to the DUT, expressed as the number of milliseconds between UPDATE messages.

        Returns: int
        """
        return self._get_property('update_interval')

    @update_interval.setter
    def update_interval(self, value):
        """update_interval setter

        The time interval at which UPDATE messages are sent to the DUT, expressed as the number of milliseconds between UPDATE messages.

        value: int
        """
        self._set_property('update_interval', value)

    @property
    def time_to_live(self):
        # type: () -> int
        """time_to_live getter

        The limited number of iterations that a unit of data can experience before the data is discarded. This is placed in the TTL field in the IP header of the transmitted packets.

        Returns: int
        """
        return self._get_property('time_to_live')

    @time_to_live.setter
    def time_to_live(self, value):
        """time_to_live setter

        The limited number of iterations that a unit of data can experience before the data is discarded. This is placed in the TTL field in the IP header of the transmitted packets.

        value: int
        """
        self._set_property('time_to_live', value)

    @property
    def md5_key(self):
        # type: () -> str
        """md5_key getter

        The value to be used as a secret MD5 key for authentication. If null or an empty string then authentication will not be enabled.

        Returns: str
        """
        return self._get_property('md5_key')

    @md5_key.setter
    def md5_key(self, value):
        """md5_key setter

        The value to be used as a secret MD5 key for authentication. If null or an empty string then authentication will not be enabled.

        value: str
        """
        self._set_property('md5_key', value)


class DeviceBgpCapability(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'vpls': {'type': bool},
        'route_refresh': {'type': bool},
        'route_constraint': {'type': bool},
        'link_state_non_vpn': {'type': bool},
        'link_state_vpn': {'type': bool},
        'evpn': {'type': bool},
        'extended_next_hop_encoding': {'type': bool},
        'ipv4_unicast': {'type': bool},
        'ipv4_multicast': {'type': bool},
        'ipv4_multicast_vpn': {'type': bool},
        'ipv4_mpls_vpn': {'type': bool},
        'ipv4_mdt': {'type': bool},
        'ipv4_multicast_mpls_vpn': {'type': bool},
        'ipv4_unicast_flow_spec': {'type': bool},
        'ipv4_sr_te_policy': {'type': bool},
        'ipv4_unicast_add_path': {'type': bool},
        'ipv6_unicast': {'type': bool},
        'ipv6_multicast': {'type': bool},
        'ipv6_multicast_vpn': {'type': bool},
        'ipv6_mpls_vpn': {'type': bool},
        'ipv6_mdt': {'type': bool},
        'ipv6_multicast_mpls_vpn': {'type': bool},
        'ipv6_unicast_flow_spec': {'type': bool},
        'ipv6_sr_te_policy': {'type': bool},
        'ipv6_unicast_add_path': {'type': bool},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'vpls': True,
        'route_refresh': True,
        'route_constraint': False,
        'link_state_non_vpn': False,
        'link_state_vpn': False,
        'evpn': False,
        'extended_next_hop_encoding': False,
        'ipv4_unicast': True,
        'ipv4_multicast': True,
        'ipv4_multicast_vpn': False,
        'ipv4_mpls_vpn': True,
        'ipv4_mdt': False,
        'ipv4_multicast_mpls_vpn': False,
        'ipv4_unicast_flow_spec': False,
        'ipv4_sr_te_policy': False,
        'ipv4_unicast_add_path': False,
        'ipv6_unicast': True,
        'ipv6_multicast': True,
        'ipv6_multicast_vpn': False,
        'ipv6_mpls_vpn': True,
        'ipv6_mdt': False,
        'ipv6_multicast_mpls_vpn': False,
        'ipv6_unicast_flow_spec': False,
        'ipv6_sr_te_policy': False,
        'ipv6_unicast_add_path': False,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, vpls=None, route_refresh=None, route_constraint=None, link_state_non_vpn=None, link_state_vpn=None, evpn=None, extended_next_hop_encoding=None, ipv4_unicast=None, ipv4_multicast=None, ipv4_multicast_vpn=None, ipv4_mpls_vpn=None, ipv4_mdt=None, ipv4_multicast_mpls_vpn=None, ipv4_unicast_flow_spec=None, ipv4_sr_te_policy=None, ipv4_unicast_add_path=None, ipv6_unicast=None, ipv6_multicast=None, ipv6_multicast_vpn=None, ipv6_mpls_vpn=None, ipv6_mdt=None, ipv6_multicast_mpls_vpn=None, ipv6_unicast_flow_spec=None, ipv6_sr_te_policy=None, ipv6_unicast_add_path=None):
        super(DeviceBgpCapability, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('vpls', vpls)
        self._set_property('route_refresh', route_refresh)
        self._set_property('route_constraint', route_constraint)
        self._set_property('link_state_non_vpn', link_state_non_vpn)
        self._set_property('link_state_vpn', link_state_vpn)
        self._set_property('evpn', evpn)
        self._set_property('extended_next_hop_encoding', extended_next_hop_encoding)
        self._set_property('ipv4_unicast', ipv4_unicast)
        self._set_property('ipv4_multicast', ipv4_multicast)
        self._set_property('ipv4_multicast_vpn', ipv4_multicast_vpn)
        self._set_property('ipv4_mpls_vpn', ipv4_mpls_vpn)
        self._set_property('ipv4_mdt', ipv4_mdt)
        self._set_property('ipv4_multicast_mpls_vpn', ipv4_multicast_mpls_vpn)
        self._set_property('ipv4_unicast_flow_spec', ipv4_unicast_flow_spec)
        self._set_property('ipv4_sr_te_policy', ipv4_sr_te_policy)
        self._set_property('ipv4_unicast_add_path', ipv4_unicast_add_path)
        self._set_property('ipv6_unicast', ipv6_unicast)
        self._set_property('ipv6_multicast', ipv6_multicast)
        self._set_property('ipv6_multicast_vpn', ipv6_multicast_vpn)
        self._set_property('ipv6_mpls_vpn', ipv6_mpls_vpn)
        self._set_property('ipv6_mdt', ipv6_mdt)
        self._set_property('ipv6_multicast_mpls_vpn', ipv6_multicast_mpls_vpn)
        self._set_property('ipv6_unicast_flow_spec', ipv6_unicast_flow_spec)
        self._set_property('ipv6_sr_te_policy', ipv6_sr_te_policy)
        self._set_property('ipv6_unicast_add_path', ipv6_unicast_add_path)

    @property
    def vpls(self):
        # type: () -> bool
        """vpls getter

        TBD

        Returns: bool
        """
        return self._get_property('vpls')

    @vpls.setter
    def vpls(self, value):
        """vpls setter

        TBD

        value: bool
        """
        self._set_property('vpls', value)

    @property
    def route_refresh(self):
        # type: () -> bool
        """route_refresh getter

        TBD

        Returns: bool
        """
        return self._get_property('route_refresh')

    @route_refresh.setter
    def route_refresh(self, value):
        """route_refresh setter

        TBD

        value: bool
        """
        self._set_property('route_refresh', value)

    @property
    def route_constraint(self):
        # type: () -> bool
        """route_constraint getter

        TBD

        Returns: bool
        """
        return self._get_property('route_constraint')

    @route_constraint.setter
    def route_constraint(self, value):
        """route_constraint setter

        TBD

        value: bool
        """
        self._set_property('route_constraint', value)

    @property
    def link_state_non_vpn(self):
        # type: () -> bool
        """link_state_non_vpn getter

        TBD

        Returns: bool
        """
        return self._get_property('link_state_non_vpn')

    @link_state_non_vpn.setter
    def link_state_non_vpn(self, value):
        """link_state_non_vpn setter

        TBD

        value: bool
        """
        self._set_property('link_state_non_vpn', value)

    @property
    def link_state_vpn(self):
        # type: () -> bool
        """link_state_vpn getter

        TBD

        Returns: bool
        """
        return self._get_property('link_state_vpn')

    @link_state_vpn.setter
    def link_state_vpn(self, value):
        """link_state_vpn setter

        TBD

        value: bool
        """
        self._set_property('link_state_vpn', value)

    @property
    def evpn(self):
        # type: () -> bool
        """evpn getter

        TBD

        Returns: bool
        """
        return self._get_property('evpn')

    @evpn.setter
    def evpn(self, value):
        """evpn setter

        TBD

        value: bool
        """
        self._set_property('evpn', value)

    @property
    def extended_next_hop_encoding(self):
        # type: () -> bool
        """extended_next_hop_encoding getter

        TBD

        Returns: bool
        """
        return self._get_property('extended_next_hop_encoding')

    @extended_next_hop_encoding.setter
    def extended_next_hop_encoding(self, value):
        """extended_next_hop_encoding setter

        TBD

        value: bool
        """
        self._set_property('extended_next_hop_encoding', value)

    @property
    def ipv4_unicast(self):
        # type: () -> bool
        """ipv4_unicast getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv4_unicast')

    @ipv4_unicast.setter
    def ipv4_unicast(self, value):
        """ipv4_unicast setter

        TBD

        value: bool
        """
        self._set_property('ipv4_unicast', value)

    @property
    def ipv4_multicast(self):
        # type: () -> bool
        """ipv4_multicast getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv4_multicast')

    @ipv4_multicast.setter
    def ipv4_multicast(self, value):
        """ipv4_multicast setter

        TBD

        value: bool
        """
        self._set_property('ipv4_multicast', value)

    @property
    def ipv4_multicast_vpn(self):
        # type: () -> bool
        """ipv4_multicast_vpn getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv4_multicast_vpn')

    @ipv4_multicast_vpn.setter
    def ipv4_multicast_vpn(self, value):
        """ipv4_multicast_vpn setter

        TBD

        value: bool
        """
        self._set_property('ipv4_multicast_vpn', value)

    @property
    def ipv4_mpls_vpn(self):
        # type: () -> bool
        """ipv4_mpls_vpn getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv4_mpls_vpn')

    @ipv4_mpls_vpn.setter
    def ipv4_mpls_vpn(self, value):
        """ipv4_mpls_vpn setter

        TBD

        value: bool
        """
        self._set_property('ipv4_mpls_vpn', value)

    @property
    def ipv4_mdt(self):
        # type: () -> bool
        """ipv4_mdt getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv4_mdt')

    @ipv4_mdt.setter
    def ipv4_mdt(self, value):
        """ipv4_mdt setter

        TBD

        value: bool
        """
        self._set_property('ipv4_mdt', value)

    @property
    def ipv4_multicast_mpls_vpn(self):
        # type: () -> bool
        """ipv4_multicast_mpls_vpn getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv4_multicast_mpls_vpn')

    @ipv4_multicast_mpls_vpn.setter
    def ipv4_multicast_mpls_vpn(self, value):
        """ipv4_multicast_mpls_vpn setter

        TBD

        value: bool
        """
        self._set_property('ipv4_multicast_mpls_vpn', value)

    @property
    def ipv4_unicast_flow_spec(self):
        # type: () -> bool
        """ipv4_unicast_flow_spec getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv4_unicast_flow_spec')

    @ipv4_unicast_flow_spec.setter
    def ipv4_unicast_flow_spec(self, value):
        """ipv4_unicast_flow_spec setter

        TBD

        value: bool
        """
        self._set_property('ipv4_unicast_flow_spec', value)

    @property
    def ipv4_sr_te_policy(self):
        # type: () -> bool
        """ipv4_sr_te_policy getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv4_sr_te_policy')

    @ipv4_sr_te_policy.setter
    def ipv4_sr_te_policy(self, value):
        """ipv4_sr_te_policy setter

        TBD

        value: bool
        """
        self._set_property('ipv4_sr_te_policy', value)

    @property
    def ipv4_unicast_add_path(self):
        # type: () -> bool
        """ipv4_unicast_add_path getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv4_unicast_add_path')

    @ipv4_unicast_add_path.setter
    def ipv4_unicast_add_path(self, value):
        """ipv4_unicast_add_path setter

        TBD

        value: bool
        """
        self._set_property('ipv4_unicast_add_path', value)

    @property
    def ipv6_unicast(self):
        # type: () -> bool
        """ipv6_unicast getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv6_unicast')

    @ipv6_unicast.setter
    def ipv6_unicast(self, value):
        """ipv6_unicast setter

        TBD

        value: bool
        """
        self._set_property('ipv6_unicast', value)

    @property
    def ipv6_multicast(self):
        # type: () -> bool
        """ipv6_multicast getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv6_multicast')

    @ipv6_multicast.setter
    def ipv6_multicast(self, value):
        """ipv6_multicast setter

        TBD

        value: bool
        """
        self._set_property('ipv6_multicast', value)

    @property
    def ipv6_multicast_vpn(self):
        # type: () -> bool
        """ipv6_multicast_vpn getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv6_multicast_vpn')

    @ipv6_multicast_vpn.setter
    def ipv6_multicast_vpn(self, value):
        """ipv6_multicast_vpn setter

        TBD

        value: bool
        """
        self._set_property('ipv6_multicast_vpn', value)

    @property
    def ipv6_mpls_vpn(self):
        # type: () -> bool
        """ipv6_mpls_vpn getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv6_mpls_vpn')

    @ipv6_mpls_vpn.setter
    def ipv6_mpls_vpn(self, value):
        """ipv6_mpls_vpn setter

        TBD

        value: bool
        """
        self._set_property('ipv6_mpls_vpn', value)

    @property
    def ipv6_mdt(self):
        # type: () -> bool
        """ipv6_mdt getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv6_mdt')

    @ipv6_mdt.setter
    def ipv6_mdt(self, value):
        """ipv6_mdt setter

        TBD

        value: bool
        """
        self._set_property('ipv6_mdt', value)

    @property
    def ipv6_multicast_mpls_vpn(self):
        # type: () -> bool
        """ipv6_multicast_mpls_vpn getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv6_multicast_mpls_vpn')

    @ipv6_multicast_mpls_vpn.setter
    def ipv6_multicast_mpls_vpn(self, value):
        """ipv6_multicast_mpls_vpn setter

        TBD

        value: bool
        """
        self._set_property('ipv6_multicast_mpls_vpn', value)

    @property
    def ipv6_unicast_flow_spec(self):
        # type: () -> bool
        """ipv6_unicast_flow_spec getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv6_unicast_flow_spec')

    @ipv6_unicast_flow_spec.setter
    def ipv6_unicast_flow_spec(self, value):
        """ipv6_unicast_flow_spec setter

        TBD

        value: bool
        """
        self._set_property('ipv6_unicast_flow_spec', value)

    @property
    def ipv6_sr_te_policy(self):
        # type: () -> bool
        """ipv6_sr_te_policy getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv6_sr_te_policy')

    @ipv6_sr_te_policy.setter
    def ipv6_sr_te_policy(self, value):
        """ipv6_sr_te_policy setter

        TBD

        value: bool
        """
        self._set_property('ipv6_sr_te_policy', value)

    @property
    def ipv6_unicast_add_path(self):
        # type: () -> bool
        """ipv6_unicast_add_path getter

        TBD

        Returns: bool
        """
        return self._get_property('ipv6_unicast_add_path')

    @ipv6_unicast_add_path.setter
    def ipv6_unicast_add_path(self, value):
        """ipv6_unicast_add_path setter

        TBD

        value: bool
        """
        self._set_property('ipv6_unicast_add_path', value)


class DeviceBgpSrTePolicy(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'policy_type': {
            'type': str,
            'enum': [
                'ipv4',
                'ipv6',
            ],
        },
        'distinguisher': {'type': int},
        'color': {'type': int},
        'ipv4_endpoint': {
            'type': str,
            'format': 'ipv4',
        },
        'ipv6_endpoint': {
            'type': str,
            'format': 'ipv6',
        },
        'next_hop': {'type': 'DeviceBgpSrTePolicyNextHop'},
        'add_path': {'type': 'DeviceBgpAddPath'},
        'as_path': {'type': 'DeviceBgpAsPath'},
        'tunnel_tlvs': {'type': 'DeviceBgpTunnelTlvIter'},
        'communities': {'type': 'DeviceBgpCommunityIter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'policy_type': 'ipv4',
        'distinguisher': 1,
        'color': 100,
        'ipv4_endpoint': '0.0.0.0',
        'ipv6_endpoint': '::0',
    } # type: Dict[str, Union(type)]

    IPV4 = 'ipv4' # type: str
    IPV6 = 'ipv6' # type: str

    def __init__(self, parent=None, choice=None, policy_type=None, distinguisher=None, color=None, ipv4_endpoint=None, ipv6_endpoint=None):
        super(DeviceBgpSrTePolicy, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('policy_type', policy_type)
        self._set_property('distinguisher', distinguisher)
        self._set_property('color', color)
        self._set_property('ipv4_endpoint', ipv4_endpoint)
        self._set_property('ipv6_endpoint', ipv6_endpoint)

    @property
    def policy_type(self):
        # type: () -> Union[Literal["ipv4"], Literal["ipv6"]]
        """policy_type getter

        Address family indicator (AFI).

        Returns: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        return self._get_property('policy_type')

    @policy_type.setter
    def policy_type(self, value):
        """policy_type setter

        Address family indicator (AFI).

        value: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        self._set_property('policy_type', value)

    @property
    def distinguisher(self):
        # type: () -> int
        """distinguisher getter

        Identifies the policy in the context of (color and endpoint) tuple. It is used by the SR Policy originator to make unique multiple occurrences of the same SR Policy.

        Returns: int
        """
        return self._get_property('distinguisher')

    @distinguisher.setter
    def distinguisher(self, value):
        """distinguisher setter

        Identifies the policy in the context of (color and endpoint) tuple. It is used by the SR Policy originator to make unique multiple occurrences of the same SR Policy.

        value: int
        """
        self._set_property('distinguisher', value)

    @property
    def color(self):
        # type: () -> int
        """color getter

        Identifies the policy. It is used to match the color of the destination prefixes to steer traffic into the SR Policy. 

        Returns: int
        """
        return self._get_property('color')

    @color.setter
    def color(self, value):
        """color setter

        Identifies the policy. It is used to match the color of the destination prefixes to steer traffic into the SR Policy. 

        value: int
        """
        self._set_property('color', value)

    @property
    def ipv4_endpoint(self):
        # type: () -> str
        """ipv4_endpoint getter

        Specifies a single node or a set of nodes. It is selected on the basis of the policy_type (AFI).

        Returns: str
        """
        return self._get_property('ipv4_endpoint')

    @ipv4_endpoint.setter
    def ipv4_endpoint(self, value):
        """ipv4_endpoint setter

        Specifies a single node or a set of nodes. It is selected on the basis of the policy_type (AFI).

        value: str
        """
        self._set_property('ipv4_endpoint', value)

    @property
    def ipv6_endpoint(self):
        # type: () -> str
        """ipv6_endpoint getter

        Specifies a single node or a set of nodes. It is selected on the basis of the policy_type (AFI).

        Returns: str
        """
        return self._get_property('ipv6_endpoint')

    @ipv6_endpoint.setter
    def ipv6_endpoint(self, value):
        """ipv6_endpoint setter

        Specifies a single node or a set of nodes. It is selected on the basis of the policy_type (AFI).

        value: str
        """
        self._set_property('ipv6_endpoint', value)

    @property
    def next_hop(self):
        # type: () -> DeviceBgpSrTePolicyNextHop
        """next_hop getter

        Optional container for BGP SR TE Policy next hop settings.

        Returns: DeviceBgpSrTePolicyNextHop
        """
        return self._get_property('next_hop', DeviceBgpSrTePolicyNextHop)

    @property
    def add_path(self):
        # type: () -> DeviceBgpAddPath
        """add_path getter

        The BGP Additional Paths feature is a BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.

        Returns: DeviceBgpAddPath
        """
        return self._get_property('add_path', DeviceBgpAddPath)

    @property
    def as_path(self):
        # type: () -> DeviceBgpAsPath
        """as_path getter

        Autonomous Systems (AS) numbers that a route passes through to reach the destination

        Returns: DeviceBgpAsPath
        """
        return self._get_property('as_path', DeviceBgpAsPath)

    @property
    def tunnel_tlvs(self):
        # type: () -> DeviceBgpTunnelTlvIter
        """tunnel_tlvs getter

        Optional tunnel TLV settings

        Returns: DeviceBgpTunnelTlvIter
        """
        return self._get_property('tunnel_tlvs', DeviceBgpTunnelTlvIter, self._parent, self._choice)

    @property
    def communities(self):
        # type: () -> DeviceBgpCommunityIter
        """communities getter

        Optional community settings

        Returns: DeviceBgpCommunityIter
        """
        return self._get_property('communities', DeviceBgpCommunityIter, self._parent, self._choice)


class DeviceBgpSrTePolicyNextHop(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'next_hop_mode': {
            'type': str,
            'enum': [
                'local_ip',
                'manual',
            ],
        },
        'next_hop_address_type': {
            'type': str,
            'enum': [
                'ipv4',
                'ipv6',
            ],
        },
        'ipv4_address': {
            'type': str,
            'format': 'ipv4',
        },
        'ipv6_address': {
            'type': str,
            'format': 'ipv6',
        },
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'next_hop_mode': 'local_ip',
        'next_hop_address_type': 'ipv4',
        'ipv4_address': '0.0.0.0',
        'ipv6_address': '::0',
    } # type: Dict[str, Union(type)]

    LOCAL_IP = 'local_ip' # type: str
    MANUAL = 'manual' # type: str

    IPV4 = 'ipv4' # type: str
    IPV6 = 'ipv6' # type: str

    def __init__(self, parent=None, choice=None, next_hop_mode=None, next_hop_address_type=None, ipv4_address=None, ipv6_address=None):
        super(DeviceBgpSrTePolicyNextHop, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('next_hop_mode', next_hop_mode)
        self._set_property('next_hop_address_type', next_hop_address_type)
        self._set_property('ipv4_address', ipv4_address)
        self._set_property('ipv6_address', ipv6_address)

    @property
    def next_hop_mode(self):
        # type: () -> Union[Literal["local_ip"], Literal["manual"]]
        """next_hop_mode getter

        TBD

        Returns: Union[Literal["local_ip"], Literal["manual"]]
        """
        return self._get_property('next_hop_mode')

    @next_hop_mode.setter
    def next_hop_mode(self, value):
        """next_hop_mode setter

        TBD

        value: Union[Literal["local_ip"], Literal["manual"]]
        """
        self._set_property('next_hop_mode', value)

    @property
    def next_hop_address_type(self):
        # type: () -> Union[Literal["ipv4"], Literal["ipv6"]]
        """next_hop_address_type getter

        TBD

        Returns: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        return self._get_property('next_hop_address_type')

    @next_hop_address_type.setter
    def next_hop_address_type(self, value):
        """next_hop_address_type setter

        TBD

        value: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        self._set_property('next_hop_address_type', value)

    @property
    def ipv4_address(self):
        # type: () -> str
        """ipv4_address getter

        The IPv4 address of the next hop if the next_hop_mode is manual and the next_hop_address_type is IPv4.

        Returns: str
        """
        return self._get_property('ipv4_address')

    @ipv4_address.setter
    def ipv4_address(self, value):
        """ipv4_address setter

        The IPv4 address of the next hop if the next_hop_mode is manual and the next_hop_address_type is IPv4.

        value: str
        """
        self._set_property('ipv4_address', value)

    @property
    def ipv6_address(self):
        # type: () -> str
        """ipv6_address getter

        The IPv6 address of the next hop if the next_hop_mode is manual and the next_hop_address_type is IPv6.

        Returns: str
        """
        return self._get_property('ipv6_address')

    @ipv6_address.setter
    def ipv6_address(self, value):
        """ipv6_address setter

        The IPv6 address of the next hop if the next_hop_mode is manual and the next_hop_address_type is IPv6.

        value: str
        """
        self._set_property('ipv6_address', value)


class DeviceBgpAddPath(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'path_id': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'path_id': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, path_id=None):
        super(DeviceBgpAddPath, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('path_id', path_id)

    @property
    def path_id(self):
        # type: () -> int
        """path_id getter

        The id of the additional path.

        Returns: int
        """
        return self._get_property('path_id')

    @path_id.setter
    def path_id(self, value):
        """path_id setter

        The id of the additional path.

        value: int
        """
        self._set_property('path_id', value)


class DeviceBgpAsPath(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'override_peer_as_set_mode': {'type': bool},
        'as_set_mode': {
            'type': str,
            'enum': [
                'do_not_include_local_as',
                'include_as_seq',
                'include_as_set',
                'include_as_confed_seq',
                'include_as_confed_set',
                'prepend_to_first_segment',
            ],
        },
        'as_path_segments': {'type': 'DeviceBgpAsPathSegmentIter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'override_peer_as_set_mode': True,
        'as_set_mode': 'do_not_include_local_as',
    } # type: Dict[str, Union(type)]

    DO_NOT_INCLUDE_LOCAL_AS = 'do_not_include_local_as' # type: str
    INCLUDE_AS_SEQ = 'include_as_seq' # type: str
    INCLUDE_AS_SET = 'include_as_set' # type: str
    INCLUDE_AS_CONFED_SEQ = 'include_as_confed_seq' # type: str
    INCLUDE_AS_CONFED_SET = 'include_as_confed_set' # type: str
    PREPEND_TO_FIRST_SEGMENT = 'prepend_to_first_segment' # type: str

    def __init__(self, parent=None, choice=None, override_peer_as_set_mode=None, as_set_mode=None):
        super(DeviceBgpAsPath, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('override_peer_as_set_mode', override_peer_as_set_mode)
        self._set_property('as_set_mode', as_set_mode)

    @property
    def override_peer_as_set_mode(self):
        # type: () -> bool
        """override_peer_as_set_mode getter

        TBD

        Returns: bool
        """
        return self._get_property('override_peer_as_set_mode')

    @override_peer_as_set_mode.setter
    def override_peer_as_set_mode(self, value):
        """override_peer_as_set_mode setter

        TBD

        value: bool
        """
        self._set_property('override_peer_as_set_mode', value)

    @property
    def as_set_mode(self):
        # type: () -> Union[Literal["do_not_include_local_as"], Literal["include_as_confed_seq"], Literal["include_as_confed_set"], Literal["include_as_seq"], Literal["include_as_set"], Literal["prepend_to_first_segment"]]
        """as_set_mode getter

        TBD

        Returns: Union[Literal["do_not_include_local_as"], Literal["include_as_confed_seq"], Literal["include_as_confed_set"], Literal["include_as_seq"], Literal["include_as_set"], Literal["prepend_to_first_segment"]]
        """
        return self._get_property('as_set_mode')

    @as_set_mode.setter
    def as_set_mode(self, value):
        """as_set_mode setter

        TBD

        value: Union[Literal["do_not_include_local_as"], Literal["include_as_confed_seq"], Literal["include_as_confed_set"], Literal["include_as_seq"], Literal["include_as_set"], Literal["prepend_to_first_segment"]]
        """
        self._set_property('as_set_mode', value)

    @property
    def as_path_segments(self):
        # type: () -> DeviceBgpAsPathSegmentIter
        """as_path_segments getter

        The AS path segments (non random) per route range

        Returns: DeviceBgpAsPathSegmentIter
        """
        return self._get_property('as_path_segments', DeviceBgpAsPathSegmentIter, self._parent, self._choice)


class DeviceBgpAsPathSegment(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'segment_type': {
            'type': str,
            'enum': [
                'as_seq',
                'as_set',
                'as_confed_seq',
                'as_confed_set',
            ],
        },
        'as_numbers': {'type': list},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'segment_type': 'as_seq',
    } # type: Dict[str, Union(type)]

    AS_SEQ = 'as_seq' # type: str
    AS_SET = 'as_set' # type: str
    AS_CONFED_SEQ = 'as_confed_seq' # type: str
    AS_CONFED_SET = 'as_confed_set' # type: str

    def __init__(self, parent=None, choice=None, segment_type=None, as_numbers=None):
        super(DeviceBgpAsPathSegment, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('segment_type', segment_type)
        self._set_property('as_numbers', as_numbers)

    @property
    def segment_type(self):
        # type: () -> Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]]
        """segment_type getter

        AS sequence is the most common type of AS_PATH, it contains the list of ASNs starting with the most recent ASN being added read from left to right.. The other three AS_PATH types are used for Confederations - AS_SET is the type of AS_PATH attribute that summarizes routes using using the aggregate-address command, allowing AS_PATHs to be summarized in the update as well. - AS_CONFED_SEQ gives the list of ASNs in the path starting with the most recent ASN to be added reading left to right - AS_CONFED_SET will allow summarization of multiple AS PATHs to be sent in BGP Updates.

        Returns: Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]]
        """
        return self._get_property('segment_type')

    @segment_type.setter
    def segment_type(self, value):
        """segment_type setter

        AS sequence is the most common type of AS_PATH, it contains the list of ASNs starting with the most recent ASN being added read from left to right.. The other three AS_PATH types are used for Confederations - AS_SET is the type of AS_PATH attribute that summarizes routes using using the aggregate-address command, allowing AS_PATHs to be summarized in the update as well. - AS_CONFED_SEQ gives the list of ASNs in the path starting with the most recent ASN to be added reading left to right - AS_CONFED_SET will allow summarization of multiple AS PATHs to be sent in BGP Updates.

        value: Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]]
        """
        self._set_property('segment_type', value)

    @property
    def as_numbers(self):
        # type: () -> List[int]
        """as_numbers getter

        The AS numbers in this AS path segment.

        Returns: List[int]
        """
        return self._get_property('as_numbers')

    @as_numbers.setter
    def as_numbers(self, value):
        """as_numbers setter

        The AS numbers in this AS path segment.

        value: List[int]
        """
        self._set_property('as_numbers', value)


class DeviceBgpAsPathSegmentIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceBgpAsPathSegmentIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceBgpAsPathSegment]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceBgpAsPathSegmentIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceBgpAsPathSegment
        return self._next()

    def next(self):
        # type: () -> DeviceBgpAsPathSegment
        return self._next()

    def bgpaspathsegment(self, segment_type='as_seq', as_numbers=None):
        # type: (Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]],List[int]) -> DeviceBgpAsPathSegmentIter
        """Factory method that creates an instance of the DeviceBgpAsPathSegment class

        Container for a single BGP AS path segment

        Returns: DeviceBgpAsPathSegmentIter
        """
        item = DeviceBgpAsPathSegment(parent=self._parent, choice=self._choice, segment_type=segment_type, as_numbers=as_numbers)
        self._add(item)
        return self



class DeviceBgpTunnelTlv(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'segment_lists': {'type': 'DeviceBgpSegmentListIter'},
        'remote_endpoint_sub_tlv': {'type': 'DeviceBgpRemoteEndpointSubTlv'},
        'preference_sub_tlv': {'type': 'DeviceBgpPreferenceSubTlv'},
        'binding_sub_tlv': {'type': 'DeviceBgpBindingSubTlv'},
        'explicit_null_label_policy_sub_tlv': {'type': 'DeviceBgpExplicitNullLabelPolicySubTlv'},
        'active': {'type': bool},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'active': True,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, active=None):
        super(DeviceBgpTunnelTlv, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('active', active)

    @property
    def segment_lists(self):
        # type: () -> DeviceBgpSegmentListIter
        """segment_lists getter

        TBD

        Returns: DeviceBgpSegmentListIter
        """
        return self._get_property('segment_lists', DeviceBgpSegmentListIter, self._parent, self._choice)

    @property
    def remote_endpoint_sub_tlv(self):
        # type: () -> DeviceBgpRemoteEndpointSubTlv
        """remote_endpoint_sub_tlv getter

        Container for BGP remote endpoint sub TLV settings.

        Returns: DeviceBgpRemoteEndpointSubTlv
        """
        return self._get_property('remote_endpoint_sub_tlv', DeviceBgpRemoteEndpointSubTlv)

    @property
    def preference_sub_tlv(self):
        # type: () -> DeviceBgpPreferenceSubTlv
        """preference_sub_tlv getter

        Container for BGP preference sub TLV settings.

        Returns: DeviceBgpPreferenceSubTlv
        """
        return self._get_property('preference_sub_tlv', DeviceBgpPreferenceSubTlv)

    @property
    def binding_sub_tlv(self):
        # type: () -> DeviceBgpBindingSubTlv
        """binding_sub_tlv getter

        Container for BGP binding sub TLV settings.

        Returns: DeviceBgpBindingSubTlv
        """
        return self._get_property('binding_sub_tlv', DeviceBgpBindingSubTlv)

    @property
    def explicit_null_label_policy_sub_tlv(self):
        # type: () -> DeviceBgpExplicitNullLabelPolicySubTlv
        """explicit_null_label_policy_sub_tlv getter

        Container for BGP explicit null label policy sub TLV settings.

        Returns: DeviceBgpExplicitNullLabelPolicySubTlv
        """
        return self._get_property('explicit_null_label_policy_sub_tlv', DeviceBgpExplicitNullLabelPolicySubTlv)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property('active')

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property('active', value)


class DeviceBgpSegmentList(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'segment_weight': {'type': int},
        'segments': {'type': 'DeviceBgpSegmentIter'},
        'active': {'type': bool},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'segment_weight': 0,
        'active': True,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, segment_weight=None, active=None):
        super(DeviceBgpSegmentList, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('segment_weight', segment_weight)
        self._set_property('active', active)

    @property
    def segment_weight(self):
        # type: () -> int
        """segment_weight getter

        The weight associated with a given path.

        Returns: int
        """
        return self._get_property('segment_weight')

    @segment_weight.setter
    def segment_weight(self, value):
        """segment_weight setter

        The weight associated with a given path.

        value: int
        """
        self._set_property('segment_weight', value)

    @property
    def segments(self):
        # type: () -> DeviceBgpSegmentIter
        """segments getter

        TBD

        Returns: DeviceBgpSegmentIter
        """
        return self._get_property('segments', DeviceBgpSegmentIter, self._parent, self._choice)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property('active')

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property('active', value)


class DeviceBgpSegment(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'segment_type': {
            'type': str,
            'enum': [
                'mpls_sid',
                'ipv6_sid',
            ],
        },
        'mpls_label': {'type': int},
        'mpls_tc': {'type': int},
        'mpls_ttl': {'type': int},
        'v_flag': {'type': bool},
        'ipv6_sid': {
            'type': str,
            'format': 'ipv6',
        },
        'remaining_flag_bits': {'type': int},
        'active': {'type': bool},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'segment_type': 'mpls_sid',
        'mpls_label': 0,
        'mpls_tc': 0,
        'mpls_ttl': 0,
        'v_flag': False,
        'ipv6_sid': '::0',
        'remaining_flag_bits': 0,
        'active': True,
    } # type: Dict[str, Union(type)]

    MPLS_SID = 'mpls_sid' # type: str
    IPV6_SID = 'ipv6_sid' # type: str

    def __init__(self, parent=None, choice=None, segment_type=None, mpls_label=None, mpls_tc=None, mpls_ttl=None, v_flag=None, ipv6_sid=None, remaining_flag_bits=None, active=None):
        super(DeviceBgpSegment, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('segment_type', segment_type)
        self._set_property('mpls_label', mpls_label)
        self._set_property('mpls_tc', mpls_tc)
        self._set_property('mpls_ttl', mpls_ttl)
        self._set_property('v_flag', v_flag)
        self._set_property('ipv6_sid', ipv6_sid)
        self._set_property('remaining_flag_bits', remaining_flag_bits)
        self._set_property('active', active)

    @property
    def segment_type(self):
        # type: () -> Union[Literal["ipv6_sid"], Literal["mpls_sid"]]
        """segment_type getter

        TBD

        Returns: Union[Literal["ipv6_sid"], Literal["mpls_sid"]]
        """
        return self._get_property('segment_type')

    @segment_type.setter
    def segment_type(self, value):
        """segment_type setter

        TBD

        value: Union[Literal["ipv6_sid"], Literal["mpls_sid"]]
        """
        self._set_property('segment_type', value)

    @property
    def mpls_label(self):
        # type: () -> int
        """mpls_label getter

        MPLS label

        Returns: int
        """
        return self._get_property('mpls_label')

    @mpls_label.setter
    def mpls_label(self, value):
        """mpls_label setter

        MPLS label

        value: int
        """
        self._set_property('mpls_label', value)

    @property
    def mpls_tc(self):
        # type: () -> int
        """mpls_tc getter

        TBD

        Returns: int
        """
        return self._get_property('mpls_tc')

    @mpls_tc.setter
    def mpls_tc(self, value):
        """mpls_tc setter

        TBD

        value: int
        """
        self._set_property('mpls_tc', value)

    @property
    def mpls_ttl(self):
        # type: () -> int
        """mpls_ttl getter

        TBD

        Returns: int
        """
        return self._get_property('mpls_ttl')

    @mpls_ttl.setter
    def mpls_ttl(self, value):
        """mpls_ttl setter

        TBD

        value: int
        """
        self._set_property('mpls_ttl', value)

    @property
    def v_flag(self):
        # type: () -> bool
        """v_flag getter

        TBD

        Returns: bool
        """
        return self._get_property('v_flag')

    @v_flag.setter
    def v_flag(self, value):
        """v_flag setter

        TBD

        value: bool
        """
        self._set_property('v_flag', value)

    @property
    def ipv6_sid(self):
        # type: () -> str
        """ipv6_sid getter

        TBD

        Returns: str
        """
        return self._get_property('ipv6_sid')

    @ipv6_sid.setter
    def ipv6_sid(self, value):
        """ipv6_sid setter

        TBD

        value: str
        """
        self._set_property('ipv6_sid', value)

    @property
    def remaining_flag_bits(self):
        # type: () -> int
        """remaining_flag_bits getter

        TBD

        Returns: int
        """
        return self._get_property('remaining_flag_bits')

    @remaining_flag_bits.setter
    def remaining_flag_bits(self, value):
        """remaining_flag_bits setter

        TBD

        value: int
        """
        self._set_property('remaining_flag_bits', value)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property('active')

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property('active', value)


class DeviceBgpSegmentIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceBgpSegmentIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceBgpSegment]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceBgpSegmentIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceBgpSegment
        return self._next()

    def next(self):
        # type: () -> DeviceBgpSegment
        return self._next()

    def bgpsegment(self, segment_type='mpls_sid', mpls_label=0, mpls_tc=0, mpls_ttl=0, v_flag=False, ipv6_sid='::0', remaining_flag_bits=0, active=True):
        # type: (Union[Literal["ipv6_sid"], Literal["mpls_sid"]],int,int,int,bool,str,int,bool) -> DeviceBgpSegmentIter
        """Factory method that creates an instance of the DeviceBgpSegment class

        Optional container for BGP SR TE Policy segment settings.

        Returns: DeviceBgpSegmentIter
        """
        item = DeviceBgpSegment(parent=self._parent, choice=self._choice, segment_type=segment_type, mpls_label=mpls_label, mpls_tc=mpls_tc, mpls_ttl=mpls_ttl, v_flag=v_flag, ipv6_sid=ipv6_sid, remaining_flag_bits=remaining_flag_bits, active=active)
        self._add(item)
        return self



class DeviceBgpSegmentListIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceBgpSegmentListIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceBgpSegmentList]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceBgpSegmentListIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceBgpSegmentList
        return self._next()

    def next(self):
        # type: () -> DeviceBgpSegmentList
        return self._next()

    def bgpsegmentlist(self, segment_weight=0, active=True):
        # type: (int,bool) -> DeviceBgpSegmentListIter
        """Factory method that creates an instance of the DeviceBgpSegmentList class

        Optional container for BGP SR TE Policy segment list settings.

        Returns: DeviceBgpSegmentListIter
        """
        item = DeviceBgpSegmentList(parent=self._parent, choice=self._choice, segment_weight=segment_weight, active=active)
        self._add(item)
        return self



class DeviceBgpRemoteEndpointSubTlv(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'as_number': {'type': int},
        'address_family': {
            'type': str,
            'enum': [
                'ipv4',
                'ipv6',
            ],
        },
        'ipv4_address': {
            'type': str,
            'format': 'ipv4',
        },
        'ipv6_address': {
            'type': str,
            'format': 'ipv6',
        },
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'as_number': 0,
        'address_family': 'ipv4',
        'ipv4_address': '0.0.0.0',
        'ipv6_address': '::0',
    } # type: Dict[str, Union(type)]

    IPV4 = 'ipv4' # type: str
    IPV6 = 'ipv6' # type: str

    def __init__(self, parent=None, choice=None, as_number=None, address_family=None, ipv4_address=None, ipv6_address=None):
        super(DeviceBgpRemoteEndpointSubTlv, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('as_number', as_number)
        self._set_property('address_family', address_family)
        self._set_property('ipv4_address', ipv4_address)
        self._set_property('ipv6_address', ipv6_address)

    @property
    def as_number(self):
        # type: () -> int
        """as_number getter

        Autonomous system (AS) number

        Returns: int
        """
        return self._get_property('as_number')

    @as_number.setter
    def as_number(self, value):
        """as_number setter

        Autonomous system (AS) number

        value: int
        """
        self._set_property('as_number', value)

    @property
    def address_family(self):
        # type: () -> Union[Literal["ipv4"], Literal["ipv6"]]
        """address_family getter

        Determines the address type

        Returns: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        return self._get_property('address_family')

    @address_family.setter
    def address_family(self, value):
        """address_family setter

        Determines the address type

        value: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        self._set_property('address_family', value)

    @property
    def ipv4_address(self):
        # type: () -> str
        """ipv4_address getter

        The IPv4 address

        Returns: str
        """
        return self._get_property('ipv4_address')

    @ipv4_address.setter
    def ipv4_address(self, value):
        """ipv4_address setter

        The IPv4 address

        value: str
        """
        self._set_property('ipv4_address', value)

    @property
    def ipv6_address(self):
        # type: () -> str
        """ipv6_address getter

        The IPv6 address

        Returns: str
        """
        return self._get_property('ipv6_address')

    @ipv6_address.setter
    def ipv6_address(self, value):
        """ipv6_address setter

        The IPv6 address

        value: str
        """
        self._set_property('ipv6_address', value)


class DeviceBgpPreferenceSubTlv(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'preference': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'preference': 0,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, preference=None):
        super(DeviceBgpPreferenceSubTlv, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('preference', preference)

    @property
    def preference(self):
        # type: () -> int
        """preference getter

        TBD

        Returns: int
        """
        return self._get_property('preference')

    @preference.setter
    def preference(self, value):
        """preference setter

        TBD

        value: int
        """
        self._set_property('preference', value)


class DeviceBgpBindingSubTlv(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'binding_sid_type': {
            'type': str,
            'enum': [
                'no_binding',
                'four_octet_sid',
                'ipv6_sid',
            ],
        },
        'four_octet_sid': {'type': int},
        'bsid_as_mpls_label': {'type': bool},
        'ipv6_sid': {
            'type': str,
            'format': 'ipv6',
        },
        's_flag': {'type': bool},
        'i_flag': {'type': bool},
        'remaining_flag_bits': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'binding_sid_type': 'no_binding',
        'four_octet_sid': 0,
        'bsid_as_mpls_label': False,
        'ipv6_sid': '::0',
        's_flag': False,
        'i_flag': False,
        'remaining_flag_bits': 0,
    } # type: Dict[str, Union(type)]

    NO_BINDING = 'no_binding' # type: str
    FOUR_OCTET_SID = 'four_octet_sid' # type: str
    IPV6_SID = 'ipv6_sid' # type: str

    def __init__(self, parent=None, choice=None, binding_sid_type=None, four_octet_sid=None, bsid_as_mpls_label=None, ipv6_sid=None, s_flag=None, i_flag=None, remaining_flag_bits=None):
        super(DeviceBgpBindingSubTlv, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('binding_sid_type', binding_sid_type)
        self._set_property('four_octet_sid', four_octet_sid)
        self._set_property('bsid_as_mpls_label', bsid_as_mpls_label)
        self._set_property('ipv6_sid', ipv6_sid)
        self._set_property('s_flag', s_flag)
        self._set_property('i_flag', i_flag)
        self._set_property('remaining_flag_bits', remaining_flag_bits)

    @property
    def binding_sid_type(self):
        # type: () -> Union[Literal["four_octet_sid"], Literal["ipv6_sid"], Literal["no_binding"]]
        """binding_sid_type getter

        TBD

        Returns: Union[Literal["four_octet_sid"], Literal["ipv6_sid"], Literal["no_binding"]]
        """
        return self._get_property('binding_sid_type')

    @binding_sid_type.setter
    def binding_sid_type(self, value):
        """binding_sid_type setter

        TBD

        value: Union[Literal["four_octet_sid"], Literal["ipv6_sid"], Literal["no_binding"]]
        """
        self._set_property('binding_sid_type', value)

    @property
    def four_octet_sid(self):
        # type: () -> int
        """four_octet_sid getter

        TBD

        Returns: int
        """
        return self._get_property('four_octet_sid')

    @four_octet_sid.setter
    def four_octet_sid(self, value):
        """four_octet_sid setter

        TBD

        value: int
        """
        self._set_property('four_octet_sid', value)

    @property
    def bsid_as_mpls_label(self):
        # type: () -> bool
        """bsid_as_mpls_label getter

        Only valid if binding_sid_type is four_octet_sid

        Returns: bool
        """
        return self._get_property('bsid_as_mpls_label')

    @bsid_as_mpls_label.setter
    def bsid_as_mpls_label(self, value):
        """bsid_as_mpls_label setter

        Only valid if binding_sid_type is four_octet_sid

        value: bool
        """
        self._set_property('bsid_as_mpls_label', value)

    @property
    def ipv6_sid(self):
        # type: () -> str
        """ipv6_sid getter

        Only valid if binding_sid_type is ipv6_sid

        Returns: str
        """
        return self._get_property('ipv6_sid')

    @ipv6_sid.setter
    def ipv6_sid(self, value):
        """ipv6_sid setter

        Only valid if binding_sid_type is ipv6_sid

        value: str
        """
        self._set_property('ipv6_sid', value)

    @property
    def s_flag(self):
        # type: () -> bool
        """s_flag getter

        TBD

        Returns: bool
        """
        return self._get_property('s_flag')

    @s_flag.setter
    def s_flag(self, value):
        """s_flag setter

        TBD

        value: bool
        """
        self._set_property('s_flag', value)

    @property
    def i_flag(self):
        # type: () -> bool
        """i_flag getter

        TBD

        Returns: bool
        """
        return self._get_property('i_flag')

    @i_flag.setter
    def i_flag(self, value):
        """i_flag setter

        TBD

        value: bool
        """
        self._set_property('i_flag', value)

    @property
    def remaining_flag_bits(self):
        # type: () -> int
        """remaining_flag_bits getter

        TBD

        Returns: int
        """
        return self._get_property('remaining_flag_bits')

    @remaining_flag_bits.setter
    def remaining_flag_bits(self, value):
        """remaining_flag_bits setter

        TBD

        value: int
        """
        self._set_property('remaining_flag_bits', value)


class DeviceBgpExplicitNullLabelPolicySubTlv(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'explicit_null_label_policy': {
            'type': str,
            'enum': [
                'reserved_enlp',
                'push_ipv4_enlp',
                'push_ipv6_enlp',
                'push_ipv4_ipv6_enlp',
                'do_not_push_enlp',
            ],
        },
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'explicit_null_label_policy': 'do_not_push_enlp',
    } # type: Dict[str, Union(type)]

    RESERVED_ENLP = 'reserved_enlp' # type: str
    PUSH_IPV4_ENLP = 'push_ipv4_enlp' # type: str
    PUSH_IPV6_ENLP = 'push_ipv6_enlp' # type: str
    PUSH_IPV4_IPV6_ENLP = 'push_ipv4_ipv6_enlp' # type: str
    DO_NOT_PUSH_ENLP = 'do_not_push_enlp' # type: str

    def __init__(self, parent=None, choice=None, explicit_null_label_policy=None):
        super(DeviceBgpExplicitNullLabelPolicySubTlv, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('explicit_null_label_policy', explicit_null_label_policy)

    @property
    def explicit_null_label_policy(self):
        # type: () -> Union[Literal["do_not_push_enlp"], Literal["push_ipv4_enlp"], Literal["push_ipv4_ipv6_enlp"], Literal["push_ipv6_enlp"], Literal["reserved_enlp"]]
        """explicit_null_label_policy getter

        The value of the explicit null label policy 

        Returns: Union[Literal["do_not_push_enlp"], Literal["push_ipv4_enlp"], Literal["push_ipv4_ipv6_enlp"], Literal["push_ipv6_enlp"], Literal["reserved_enlp"]]
        """
        return self._get_property('explicit_null_label_policy')

    @explicit_null_label_policy.setter
    def explicit_null_label_policy(self, value):
        """explicit_null_label_policy setter

        The value of the explicit null label policy 

        value: Union[Literal["do_not_push_enlp"], Literal["push_ipv4_enlp"], Literal["push_ipv4_ipv6_enlp"], Literal["push_ipv6_enlp"], Literal["reserved_enlp"]]
        """
        self._set_property('explicit_null_label_policy', value)


class DeviceBgpTunnelTlvIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceBgpTunnelTlvIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceBgpTunnelTlv]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceBgpTunnelTlvIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceBgpTunnelTlv
        return self._next()

    def next(self):
        # type: () -> DeviceBgpTunnelTlv
        return self._next()

    def bgptunneltlv(self, active=True):
        # type: (bool) -> DeviceBgpTunnelTlvIter
        """Factory method that creates an instance of the DeviceBgpTunnelTlv class

        Container for BGP tunnel TLV settings.

        Returns: DeviceBgpTunnelTlvIter
        """
        item = DeviceBgpTunnelTlv(parent=self._parent, choice=self._choice, active=active)
        self._add(item)
        return self



class DeviceBgpCommunity(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'community_type': {
            'type': str,
            'enum': [
                'manual_as_number',
                'no_export',
                'no_advertised',
                'no_export_subconfed',
                'llgr_stale',
                'no_llgr',
            ],
        },
        'as_number': {'type': int},
        'as_custom': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'as_number': 0,
        'as_custom': 0,
    } # type: Dict[str, Union(type)]

    MANUAL_AS_NUMBER = 'manual_as_number' # type: str
    NO_EXPORT = 'no_export' # type: str
    NO_ADVERTISED = 'no_advertised' # type: str
    NO_EXPORT_SUBCONFED = 'no_export_subconfed' # type: str
    LLGR_STALE = 'llgr_stale' # type: str
    NO_LLGR = 'no_llgr' # type: str

    def __init__(self, parent=None, choice=None, community_type=None, as_number=None, as_custom=None):
        super(DeviceBgpCommunity, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('community_type', community_type)
        self._set_property('as_number', as_number)
        self._set_property('as_custom', as_custom)

    @property
    def community_type(self):
        # type: () -> Union[Literal["llgr_stale"], Literal["manual_as_number"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]]
        """community_type getter

        The type of community AS number.

        Returns: Union[Literal["llgr_stale"], Literal["manual_as_number"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]]
        """
        return self._get_property('community_type')

    @community_type.setter
    def community_type(self, value):
        """community_type setter

        The type of community AS number.

        value: Union[Literal["llgr_stale"], Literal["manual_as_number"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]]
        """
        self._set_property('community_type', value)

    @property
    def as_number(self):
        # type: () -> int
        """as_number getter

        First two octets of 32 bit community AS number

        Returns: int
        """
        return self._get_property('as_number')

    @as_number.setter
    def as_number(self, value):
        """as_number setter

        First two octets of 32 bit community AS number

        value: int
        """
        self._set_property('as_number', value)

    @property
    def as_custom(self):
        # type: () -> int
        """as_custom getter

        Last two octets of the community AS number 

        Returns: int
        """
        return self._get_property('as_custom')

    @as_custom.setter
    def as_custom(self, value):
        """as_custom setter

        Last two octets of the community AS number 

        value: int
        """
        self._set_property('as_custom', value)


class DeviceBgpCommunityIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceBgpCommunityIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceBgpCommunity]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceBgpCommunityIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceBgpCommunity
        return self._next()

    def next(self):
        # type: () -> DeviceBgpCommunity
        return self._next()

    def bgpcommunity(self, community_type=None, as_number=0, as_custom=0):
        # type: (Union[Literal["llgr_stale"], Literal["manual_as_number"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]],int,int) -> DeviceBgpCommunityIter
        """Factory method that creates an instance of the DeviceBgpCommunity class

        BGP communities provide additional capability for tagging routes and for modifying BGP routing policy on upstream and downstream routers BGP community is a 32-bit number which broken into 16-bit AS number and a 16-bit custom value

        Returns: DeviceBgpCommunityIter
        """
        item = DeviceBgpCommunity(parent=self._parent, choice=self._choice, community_type=community_type, as_number=as_number, as_custom=as_custom)
        self._add(item)
        return self



class DeviceBgpSrTePolicyIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceBgpSrTePolicyIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceBgpSrTePolicy]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceBgpSrTePolicyIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceBgpSrTePolicy
        return self._next()

    def next(self):
        # type: () -> DeviceBgpSrTePolicy
        return self._next()

    def bgpsrtepolicy(self, policy_type='ipv4', distinguisher=1, color=100, ipv4_endpoint='0.0.0.0', ipv6_endpoint='::0'):
        # type: (Union[Literal["ipv4"], Literal["ipv6"]],int,int,str,str) -> DeviceBgpSrTePolicyIter
        """Factory method that creates an instance of the DeviceBgpSrTePolicy class

        Container for BGP basic segment routing traffic engineering policy . settings.. 

        Returns: DeviceBgpSrTePolicyIter
        """
        item = DeviceBgpSrTePolicy(parent=self._parent, choice=self._choice, policy_type=policy_type, distinguisher=distinguisher, color=color, ipv4_endpoint=ipv4_endpoint, ipv6_endpoint=ipv6_endpoint)
        self._add(item)
        return self



class DeviceBgpv4Route(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'addresses': {'type': 'DeviceBgpv4RouteAddressIter'},
        'next_hop_address': {
            'type': str,
            'format': 'ipv4',
        },
        'advanced': {'type': 'DeviceBgpRouteAdvanced'},
        'communities': {'type': 'DeviceBgpCommunityIter'},
        'as_path': {'type': 'DeviceBgpAsPath'},
        'add_path': {'type': 'DeviceBgpAddPath'},
        'name': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED = ('name',) # type: tuple(str)

    _DEFAULTS = {
        'next_hop_address': '0.0.0.0',
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, next_hop_address=None, name=None):
        super(DeviceBgpv4Route, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('next_hop_address', next_hop_address)
        self._set_property('name', name)

    @property
    def addresses(self):
        # type: () -> DeviceBgpv4RouteAddressIter
        """addresses getter

        A list of symmetrical or asymmetrical route addresses

        Returns: DeviceBgpv4RouteAddressIter
        """
        return self._get_property('addresses', DeviceBgpv4RouteAddressIter, self._parent, self._choice)

    @property
    def next_hop_address(self):
        # type: () -> str
        """next_hop_address getter

        IP Address of next router to forward a packet to its final destination

        Returns: str
        """
        return self._get_property('next_hop_address')

    @next_hop_address.setter
    def next_hop_address(self, value):
        """next_hop_address setter

        IP Address of next router to forward a packet to its final destination

        value: str
        """
        self._set_property('next_hop_address', value)

    @property
    def advanced(self):
        # type: () -> DeviceBgpRouteAdvanced
        """advanced getter

        Container for advanced BGP route range settings

        Returns: DeviceBgpRouteAdvanced
        """
        return self._get_property('advanced', DeviceBgpRouteAdvanced)

    @property
    def communities(self):
        # type: () -> DeviceBgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: DeviceBgpCommunityIter
        """
        return self._get_property('communities', DeviceBgpCommunityIter, self._parent, self._choice)

    @property
    def as_path(self):
        # type: () -> DeviceBgpAsPath
        """as_path getter

        Autonomous Systems (AS) numbers that a route passes through to reach the destination

        Returns: DeviceBgpAsPath
        """
        return self._get_property('as_path', DeviceBgpAsPath)

    @property
    def add_path(self):
        # type: () -> DeviceBgpAddPath
        """add_path getter

        The BGP Additional Paths feature is a BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.

        Returns: DeviceBgpAddPath
        """
        return self._get_property('add_path', DeviceBgpAddPath)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property('name')

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property('name', value)


class DeviceBgpv4RouteAddress(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'address': {
            'type': str,
            'format': 'ipv4',
        },
        'prefix': {'type': int},
        'count': {'type': int},
        'step': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED = ('address',) # type: tuple(str)

    _DEFAULTS = {
        'prefix': 24,
        'count': 1,
        'step': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, address=None, prefix=None, count=None, step=None):
        super(DeviceBgpv4RouteAddress, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('address', address)
        self._set_property('prefix', prefix)
        self._set_property('count', count)
        self._set_property('step', step)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The starting address of the network

        Returns: str
        """
        return self._get_property('address')

    @address.setter
    def address(self, value):
        """address setter

        The starting address of the network

        value: str
        """
        self._set_property('address', value)

    @property
    def prefix(self):
        # type: () -> int
        """prefix getter

        The IPv4 network prefix to be applied to the address. 

        Returns: int
        """
        return self._get_property('prefix')

    @prefix.setter
    def prefix(self, value):
        """prefix setter

        The IPv4 network prefix to be applied to the address. 

        value: int
        """
        self._set_property('prefix', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        The total number of addresses in the range

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        The total number of addresses in the range

        value: int
        """
        self._set_property('count', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        The amount to increase each address by

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        The amount to increase each address by

        value: int
        """
        self._set_property('step', value)


class DeviceBgpv4RouteAddressIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceBgpv4RouteAddressIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceBgpv4RouteAddress]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceBgpv4RouteAddressIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceBgpv4RouteAddress
        return self._next()

    def next(self):
        # type: () -> DeviceBgpv4RouteAddress
        return self._next()

    def bgpv4routeaddress(self, address=None, prefix=24, count=1, step=1):
        # type: (str,int,int,int) -> DeviceBgpv4RouteAddressIter
        """Factory method that creates an instance of the DeviceBgpv4RouteAddress class

        A container for BGPv4 route addresses

        Returns: DeviceBgpv4RouteAddressIter
        """
        item = DeviceBgpv4RouteAddress(parent=self._parent, choice=self._choice, address=address, prefix=prefix, count=count, step=step)
        self._add(item)
        return self



class DeviceBgpRouteAdvanced(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'multi_exit_discriminator': {'type': int},
        'origin': {
            'type': str,
            'enum': [
                'igp',
                'egp',
            ],
        },
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'multi_exit_discriminator': 0,
        'origin': 'igp',
    } # type: Dict[str, Union(type)]

    IGP = 'igp' # type: str
    EGP = 'egp' # type: str

    def __init__(self, parent=None, choice=None, multi_exit_discriminator=None, origin=None):
        super(DeviceBgpRouteAdvanced, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('multi_exit_discriminator', multi_exit_discriminator)
        self._set_property('origin', origin)

    @property
    def multi_exit_discriminator(self):
        # type: () -> int
        """multi_exit_discriminator getter

        The multi exit discriminator (MED) value. A null value means the MED feature is not enabled.

        Returns: int
        """
        return self._get_property('multi_exit_discriminator')

    @multi_exit_discriminator.setter
    def multi_exit_discriminator(self, value):
        """multi_exit_discriminator setter

        The multi exit discriminator (MED) value. A null value means the MED feature is not enabled.

        value: int
        """
        self._set_property('multi_exit_discriminator', value)

    @property
    def origin(self):
        # type: () -> Union[Literal["egp"], Literal["igp"]]
        """origin getter

        The origin value. A null value means the origin feature is not enabled.

        Returns: Union[Literal["egp"], Literal["igp"]]
        """
        return self._get_property('origin')

    @origin.setter
    def origin(self, value):
        """origin setter

        The origin value. A null value means the origin feature is not enabled.

        value: Union[Literal["egp"], Literal["igp"]]
        """
        self._set_property('origin', value)


class DeviceBgpv4RouteIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceBgpv4RouteIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceBgpv4Route]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceBgpv4RouteIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceBgpv4Route
        return self._next()

    def next(self):
        # type: () -> DeviceBgpv4Route
        return self._next()

    def bgpv4route(self, next_hop_address='0.0.0.0', name=None):
        # type: (str,str) -> DeviceBgpv4RouteIter
        """Factory method that creates an instance of the DeviceBgpv4Route class

        Emulated BGPv4 route. Container for BGP route ranges.

        Returns: DeviceBgpv4RouteIter
        """
        item = DeviceBgpv4Route(parent=self._parent, choice=self._choice, next_hop_address=next_hop_address, name=name)
        self._add(item)
        return self



class DeviceBgpv6Route(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'addresses': {'type': 'DeviceBgpv6RouteAddressIter'},
        'next_hop_address': {
            'type': str,
            'format': 'ipv6',
        },
        'advanced': {'type': 'DeviceBgpRouteAdvanced'},
        'communities': {'type': 'DeviceBgpCommunityIter'},
        'as_path': {'type': 'DeviceBgpAsPath'},
        'add_path': {'type': 'DeviceBgpAddPath'},
        'name': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED = ('name',) # type: tuple(str)

    _DEFAULTS = {
        'next_hop_address': '::0',
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, next_hop_address=None, name=None):
        super(DeviceBgpv6Route, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('next_hop_address', next_hop_address)
        self._set_property('name', name)

    @property
    def addresses(self):
        # type: () -> DeviceBgpv6RouteAddressIter
        """addresses getter

        A list of symmetrical or asymmetrical route addresses

        Returns: DeviceBgpv6RouteAddressIter
        """
        return self._get_property('addresses', DeviceBgpv6RouteAddressIter, self._parent, self._choice)

    @property
    def next_hop_address(self):
        # type: () -> str
        """next_hop_address getter

        IP Address of next router to forward a packet to its final destination

        Returns: str
        """
        return self._get_property('next_hop_address')

    @next_hop_address.setter
    def next_hop_address(self, value):
        """next_hop_address setter

        IP Address of next router to forward a packet to its final destination

        value: str
        """
        self._set_property('next_hop_address', value)

    @property
    def advanced(self):
        # type: () -> DeviceBgpRouteAdvanced
        """advanced getter

        Container for advanced BGP route range settings

        Returns: DeviceBgpRouteAdvanced
        """
        return self._get_property('advanced', DeviceBgpRouteAdvanced)

    @property
    def communities(self):
        # type: () -> DeviceBgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: DeviceBgpCommunityIter
        """
        return self._get_property('communities', DeviceBgpCommunityIter, self._parent, self._choice)

    @property
    def as_path(self):
        # type: () -> DeviceBgpAsPath
        """as_path getter

        Autonomous Systems (AS) numbers that a route passes through to reach the destination

        Returns: DeviceBgpAsPath
        """
        return self._get_property('as_path', DeviceBgpAsPath)

    @property
    def add_path(self):
        # type: () -> DeviceBgpAddPath
        """add_path getter

        The BGP Additional Paths feature is a BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.

        Returns: DeviceBgpAddPath
        """
        return self._get_property('add_path', DeviceBgpAddPath)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property('name')

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property('name', value)


class DeviceBgpv6RouteAddress(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'address': {
            'type': str,
            'format': 'ipv6',
        },
        'prefix': {'type': int},
        'count': {'type': int},
        'step': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED = ('address',) # type: tuple(str)

    _DEFAULTS = {
        'prefix': 64,
        'count': 1,
        'step': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, address=None, prefix=None, count=None, step=None):
        super(DeviceBgpv6RouteAddress, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('address', address)
        self._set_property('prefix', prefix)
        self._set_property('count', count)
        self._set_property('step', step)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The starting address of the network

        Returns: str
        """
        return self._get_property('address')

    @address.setter
    def address(self, value):
        """address setter

        The starting address of the network

        value: str
        """
        self._set_property('address', value)

    @property
    def prefix(self):
        # type: () -> int
        """prefix getter

        The IPv6 network prefix to be applied to the address

        Returns: int
        """
        return self._get_property('prefix')

    @prefix.setter
    def prefix(self, value):
        """prefix setter

        The IPv6 network prefix to be applied to the address

        value: int
        """
        self._set_property('prefix', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        The total number of addresses in the range

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        The total number of addresses in the range

        value: int
        """
        self._set_property('count', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        The amount to increase each address by

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        The amount to increase each address by

        value: int
        """
        self._set_property('step', value)


class DeviceBgpv6RouteAddressIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceBgpv6RouteAddressIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceBgpv6RouteAddress]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceBgpv6RouteAddressIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceBgpv6RouteAddress
        return self._next()

    def next(self):
        # type: () -> DeviceBgpv6RouteAddress
        return self._next()

    def bgpv6routeaddress(self, address=None, prefix=64, count=1, step=1):
        # type: (str,int,int,int) -> DeviceBgpv6RouteAddressIter
        """Factory method that creates an instance of the DeviceBgpv6RouteAddress class

        A container for BGPv6 route addressses

        Returns: DeviceBgpv6RouteAddressIter
        """
        item = DeviceBgpv6RouteAddress(parent=self._parent, choice=self._choice, address=address, prefix=prefix, count=count, step=step)
        self._add(item)
        return self



class DeviceBgpv6RouteIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceBgpv6RouteIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceBgpv6Route]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceBgpv6RouteIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceBgpv6Route
        return self._next()

    def next(self):
        # type: () -> DeviceBgpv6Route
        return self._next()

    def bgpv6route(self, next_hop_address='::0', name=None):
        # type: (str,str) -> DeviceBgpv6RouteIter
        """Factory method that creates an instance of the DeviceBgpv6Route class

        Emulated BGPv6 route. Container for BGP route ranges.

        Returns: DeviceBgpv6RouteIter
        """
        item = DeviceBgpv6Route(parent=self._parent, choice=self._choice, next_hop_address=next_hop_address, name=name)
        self._add(item)
        return self



class DeviceIpv6(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'gateway': {
            'type': str,
            'format': 'ipv6',
        },
        'address': {
            'type': str,
            'format': 'ipv6',
        },
        'prefix': {'type': int},
        'bgpv6': {'type': 'DeviceBgpv6'},
        'name': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED = ('gateway', 'address', 'name') # type: tuple(str)

    _DEFAULTS = {
        'prefix': 64,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, gateway=None, address=None, prefix=None, name=None):
        super(DeviceIpv6, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('gateway', gateway)
        self._set_property('address', address)
        self._set_property('prefix', prefix)
        self._set_property('name', name)

    @property
    def gateway(self):
        # type: () -> str
        """gateway getter

        The IPv6 gateway address

        Returns: str
        """
        return self._get_property('gateway')

    @gateway.setter
    def gateway(self, value):
        """gateway setter

        The IPv6 gateway address

        value: str
        """
        self._set_property('gateway', value)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The IPv6 address.

        Returns: str
        """
        return self._get_property('address')

    @address.setter
    def address(self, value):
        """address setter

        The IPv6 address.

        value: str
        """
        self._set_property('address', value)

    @property
    def prefix(self):
        # type: () -> int
        """prefix getter

        The network prefix.

        Returns: int
        """
        return self._get_property('prefix')

    @prefix.setter
    def prefix(self, value):
        """prefix setter

        The network prefix.

        value: int
        """
        self._set_property('prefix', value)

    @property
    def bgpv6(self):
        # type: () -> DeviceBgpv6
        """bgpv6 getter

        Container for BGPv6 peer settings and routes.. Container for basic emulated BGP peer settings.

        Returns: DeviceBgpv6
        """
        return self._get_property('bgpv6', DeviceBgpv6)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property('name')

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property('name', value)


class DeviceBgpv6(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'local_address': {
            'type': str,
            'format': 'ipv6',
        },
        'dut_address': {
            'type': str,
            'format': 'ipv6',
        },
        'segment_routing': {'type': 'DeviceBgpv6SegmentRouting'},
        'router_id': {
            'type': str,
            'format': 'ipv4',
        },
        'as_type': {
            'type': str,
            'enum': [
                'ibgp',
                'ebgp',
            ],
        },
        'as_number': {'type': int},
        'as_number_width': {
            'type': str,
            'enum': [
                'two',
                'four',
            ],
        },
        'as_number_set_mode': {
            'type': str,
            'enum': [
                'do_not_include_as',
                'include_as_seq',
                'include_as_set',
                'include_as_seq_confed',
                'include_as_set_confed',
                'prepend_as_to_first_segment',
            ],
        },
        'advanced': {'type': 'DeviceBgpAdvanced'},
        'capability': {'type': 'DeviceBgpCapability'},
        'sr_te_policies': {'type': 'DeviceBgpSrTePolicyIter'},
        'bgpv4_routes': {'type': 'DeviceBgpv4RouteIter'},
        'bgpv6_routes': {'type': 'DeviceBgpv6RouteIter'},
        'name': {'type': str},
        'active': {'type': bool},
    } # type: Dict[str, str]

    _REQUIRED = ('local_address', 'dut_address', 'as_type', 'as_number', 'name') # type: tuple(str)

    _DEFAULTS = {
        'router_id': '0.0.0.0',
        'as_number_width': 'four',
        'as_number_set_mode': 'do_not_include_as',
        'active': True,
    } # type: Dict[str, Union(type)]

    IBGP = 'ibgp' # type: str
    EBGP = 'ebgp' # type: str

    TWO = 'two' # type: str
    FOUR = 'four' # type: str

    DO_NOT_INCLUDE_AS = 'do_not_include_as' # type: str
    INCLUDE_AS_SEQ = 'include_as_seq' # type: str
    INCLUDE_AS_SET = 'include_as_set' # type: str
    INCLUDE_AS_SEQ_CONFED = 'include_as_seq_confed' # type: str
    INCLUDE_AS_SET_CONFED = 'include_as_set_confed' # type: str
    PREPEND_AS_TO_FIRST_SEGMENT = 'prepend_as_to_first_segment' # type: str

    def __init__(self, parent=None, choice=None, local_address=None, dut_address=None, router_id=None, as_type=None, as_number=None, as_number_width=None, as_number_set_mode=None, name=None, active=None):
        super(DeviceBgpv6, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('local_address', local_address)
        self._set_property('dut_address', dut_address)
        self._set_property('router_id', router_id)
        self._set_property('as_type', as_type)
        self._set_property('as_number', as_number)
        self._set_property('as_number_width', as_number_width)
        self._set_property('as_number_set_mode', as_number_set_mode)
        self._set_property('name', name)
        self._set_property('active', active)

    @property
    def local_address(self):
        # type: () -> str
        """local_address getter

        Local IPv6 address of the emulated router

        Returns: str
        """
        return self._get_property('local_address')

    @local_address.setter
    def local_address(self, value):
        """local_address setter

        Local IPv6 address of the emulated router

        value: str
        """
        self._set_property('local_address', value)

    @property
    def dut_address(self):
        # type: () -> str
        """dut_address getter

        IPv6 address of the BGP peer for the session

        Returns: str
        """
        return self._get_property('dut_address')

    @dut_address.setter
    def dut_address(self, value):
        """dut_address setter

        IPv6 address of the BGP peer for the session

        value: str
        """
        self._set_property('dut_address', value)

    @property
    def segment_routing(self):
        # type: () -> DeviceBgpv6SegmentRouting
        """segment_routing getter

        Container for BGPv6 segment routing settings.

        Returns: DeviceBgpv6SegmentRouting
        """
        return self._get_property('segment_routing', DeviceBgpv6SegmentRouting)

    @property
    def router_id(self):
        # type: () -> str
        """router_id getter

        The BGP router ID is a unique identifier used by routing protocols. It is a 32-bit value that is often represented by an IPv4 address.

        Returns: str
        """
        return self._get_property('router_id')

    @router_id.setter
    def router_id(self, value):
        """router_id setter

        The BGP router ID is a unique identifier used by routing protocols. It is a 32-bit value that is often represented by an IPv4 address.

        value: str
        """
        self._set_property('router_id', value)

    @property
    def as_type(self):
        # type: () -> Union[Literal["ebgp"], Literal["ibgp"]]
        """as_type getter

        The type of BGP autonomous system. External BGP is used for BGP links between two or more autonomous systems (ebgp) Internal BGP is used within a single autonomous system (ibgp). BGP property defaults are aligned with this object defined as an internal BGP peer. If the as_type is specified as ebgp then other properties will need to be specified according an external BGP peer.

        Returns: Union[Literal["ebgp"], Literal["ibgp"]]
        """
        return self._get_property('as_type')

    @as_type.setter
    def as_type(self, value):
        """as_type setter

        The type of BGP autonomous system. External BGP is used for BGP links between two or more autonomous systems (ebgp) Internal BGP is used within a single autonomous system (ibgp). BGP property defaults are aligned with this object defined as an internal BGP peer. If the as_type is specified as ebgp then other properties will need to be specified according an external BGP peer.

        value: Union[Literal["ebgp"], Literal["ibgp"]]
        """
        self._set_property('as_type', value)

    @property
    def as_number(self):
        # type: () -> int
        """as_number getter

        Autonomous System Number (AS number or ASN)

        Returns: int
        """
        return self._get_property('as_number')

    @as_number.setter
    def as_number(self, value):
        """as_number setter

        Autonomous System Number (AS number or ASN)

        value: int
        """
        self._set_property('as_number', value)

    @property
    def as_number_width(self):
        # type: () -> Union[Literal["four"], Literal["two"]]
        """as_number_width getter

        The width in bytes of the as_number values. Any as_number values that exceed the width MUST result in an error

        Returns: Union[Literal["four"], Literal["two"]]
        """
        return self._get_property('as_number_width')

    @as_number_width.setter
    def as_number_width(self, value):
        """as_number_width setter

        The width in bytes of the as_number values. Any as_number values that exceed the width MUST result in an error

        value: Union[Literal["four"], Literal["two"]]
        """
        self._set_property('as_number_width', value)

    @property
    def as_number_set_mode(self):
        # type: () -> Union[Literal["do_not_include_as"], Literal["include_as_seq"], Literal["include_as_seq_confed"], Literal["include_as_set"], Literal["include_as_set_confed"], Literal["prepend_as_to_first_segment"]]
        """as_number_set_mode getter

        The AS number set mode

        Returns: Union[Literal["do_not_include_as"], Literal["include_as_seq"], Literal["include_as_seq_confed"], Literal["include_as_set"], Literal["include_as_set_confed"], Literal["prepend_as_to_first_segment"]]
        """
        return self._get_property('as_number_set_mode')

    @as_number_set_mode.setter
    def as_number_set_mode(self, value):
        """as_number_set_mode setter

        The AS number set mode

        value: Union[Literal["do_not_include_as"], Literal["include_as_seq"], Literal["include_as_seq_confed"], Literal["include_as_set"], Literal["include_as_set_confed"], Literal["prepend_as_to_first_segment"]]
        """
        self._set_property('as_number_set_mode', value)

    @property
    def advanced(self):
        # type: () -> DeviceBgpAdvanced
        """advanced getter

        Container for BGP advanced settings.

        Returns: DeviceBgpAdvanced
        """
        return self._get_property('advanced', DeviceBgpAdvanced)

    @property
    def capability(self):
        # type: () -> DeviceBgpCapability
        """capability getter

        Container for BGP capability settings.

        Returns: DeviceBgpCapability
        """
        return self._get_property('capability', DeviceBgpCapability)

    @property
    def sr_te_policies(self):
        # type: () -> DeviceBgpSrTePolicyIter
        """sr_te_policies getter

        Segment routing/traffic engineering policies

        Returns: DeviceBgpSrTePolicyIter
        """
        return self._get_property('sr_te_policies', DeviceBgpSrTePolicyIter, self._parent, self._choice)

    @property
    def bgpv4_routes(self):
        # type: () -> DeviceBgpv4RouteIter
        """bgpv4_routes getter

        Emulated BGPv4 routes

        Returns: DeviceBgpv4RouteIter
        """
        return self._get_property('bgpv4_routes', DeviceBgpv4RouteIter, self._parent, self._choice)

    @property
    def bgpv6_routes(self):
        # type: () -> DeviceBgpv6RouteIter
        """bgpv6_routes getter

        Emulated BGPv6 routes

        Returns: DeviceBgpv6RouteIter
        """
        return self._get_property('bgpv6_routes', DeviceBgpv6RouteIter, self._parent, self._choice)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property('name')

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property('name', value)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property('active')

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property('active', value)


class DeviceBgpv6SegmentRouting(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'ingress_supports_vpn': {'type': bool},
        'reduced_encapsulation': {'type': bool},
        'copy_time_to_live': {'type': bool},
        'time_to_live': {'type': int},
        'max_sids_per_srh': {'type': int},
        'auto_generate_segment_left_value': {'type': bool},
        'segment_left_value': {'type': int},
        'advertise_sr_te_policy': {'type': bool},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'ingress_supports_vpn': False,
        'reduced_encapsulation': False,
        'copy_time_to_live': False,
        'time_to_live': 0,
        'max_sids_per_srh': 0,
        'auto_generate_segment_left_value': False,
        'segment_left_value': 0,
        'advertise_sr_te_policy': False,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, ingress_supports_vpn=None, reduced_encapsulation=None, copy_time_to_live=None, time_to_live=None, max_sids_per_srh=None, auto_generate_segment_left_value=None, segment_left_value=None, advertise_sr_te_policy=None):
        super(DeviceBgpv6SegmentRouting, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('ingress_supports_vpn', ingress_supports_vpn)
        self._set_property('reduced_encapsulation', reduced_encapsulation)
        self._set_property('copy_time_to_live', copy_time_to_live)
        self._set_property('time_to_live', time_to_live)
        self._set_property('max_sids_per_srh', max_sids_per_srh)
        self._set_property('auto_generate_segment_left_value', auto_generate_segment_left_value)
        self._set_property('segment_left_value', segment_left_value)
        self._set_property('advertise_sr_te_policy', advertise_sr_te_policy)

    @property
    def ingress_supports_vpn(self):
        # type: () -> bool
        """ingress_supports_vpn getter

        TBD

        Returns: bool
        """
        return self._get_property('ingress_supports_vpn')

    @ingress_supports_vpn.setter
    def ingress_supports_vpn(self, value):
        """ingress_supports_vpn setter

        TBD

        value: bool
        """
        self._set_property('ingress_supports_vpn', value)

    @property
    def reduced_encapsulation(self):
        # type: () -> bool
        """reduced_encapsulation getter

        TBD

        Returns: bool
        """
        return self._get_property('reduced_encapsulation')

    @reduced_encapsulation.setter
    def reduced_encapsulation(self, value):
        """reduced_encapsulation setter

        TBD

        value: bool
        """
        self._set_property('reduced_encapsulation', value)

    @property
    def copy_time_to_live(self):
        # type: () -> bool
        """copy_time_to_live getter

        TBD

        Returns: bool
        """
        return self._get_property('copy_time_to_live')

    @copy_time_to_live.setter
    def copy_time_to_live(self, value):
        """copy_time_to_live setter

        TBD

        value: bool
        """
        self._set_property('copy_time_to_live', value)

    @property
    def time_to_live(self):
        # type: () -> int
        """time_to_live getter

        TBD

        Returns: int
        """
        return self._get_property('time_to_live')

    @time_to_live.setter
    def time_to_live(self, value):
        """time_to_live setter

        TBD

        value: int
        """
        self._set_property('time_to_live', value)

    @property
    def max_sids_per_srh(self):
        # type: () -> int
        """max_sids_per_srh getter

        TBD

        Returns: int
        """
        return self._get_property('max_sids_per_srh')

    @max_sids_per_srh.setter
    def max_sids_per_srh(self, value):
        """max_sids_per_srh setter

        TBD

        value: int
        """
        self._set_property('max_sids_per_srh', value)

    @property
    def auto_generate_segment_left_value(self):
        # type: () -> bool
        """auto_generate_segment_left_value getter

        TBD

        Returns: bool
        """
        return self._get_property('auto_generate_segment_left_value')

    @auto_generate_segment_left_value.setter
    def auto_generate_segment_left_value(self, value):
        """auto_generate_segment_left_value setter

        TBD

        value: bool
        """
        self._set_property('auto_generate_segment_left_value', value)

    @property
    def segment_left_value(self):
        # type: () -> int
        """segment_left_value getter

        TBD

        Returns: int
        """
        return self._get_property('segment_left_value')

    @segment_left_value.setter
    def segment_left_value(self, value):
        """segment_left_value setter

        TBD

        value: int
        """
        self._set_property('segment_left_value', value)

    @property
    def advertise_sr_te_policy(self):
        # type: () -> bool
        """advertise_sr_te_policy getter

        TBD

        Returns: bool
        """
        return self._get_property('advertise_sr_te_policy')

    @advertise_sr_te_policy.setter
    def advertise_sr_te_policy(self, value):
        """advertise_sr_te_policy setter

        TBD

        value: bool
        """
        self._set_property('advertise_sr_te_policy', value)


class DeviceIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Device]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceIter
        return self._iter()

    def __next__(self):
        # type: () -> Device
        return self._next()

    def next(self):
        # type: () -> Device
        return self._next()

    def device(self, container_name=None, name=None):
        # type: (str,str) -> DeviceIter
        """Factory method that creates an instance of the Device class

        A container for emulated interfaces and protocol devices.

        Returns: DeviceIter
        """
        item = Device(parent=self._parent, choice=self._choice, container_name=container_name, name=name)
        self._add(item)
        return self



class Flow(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'tx_rx': {'type': 'FlowTxRx'},
        'packet': {'type': 'FlowHeaderIter'},
        'size': {'type': 'FlowSize'},
        'rate': {'type': 'FlowRate'},
        'duration': {'type': 'FlowDuration'},
        'metrics': {'type': 'FlowMetrics'},
        'name': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED = ('tx_rx', 'name') # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, name=None):
        super(Flow, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('name', name)

    @property
    def tx_rx(self):
        # type: () -> FlowTxRx
        """tx_rx getter

        A container for different types of transmit and receive . endpoint containers.The transmit and receive endpoints.

        Returns: FlowTxRx
        """
        return self._get_property('tx_rx', FlowTxRx)

    @property
    def packet(self):
        # type: () -> FlowHeaderIter
        """packet getter

        The header is a list of traffic protocol headers.. . The order of traffic protocol headers assigned to the list is the. order they will appear on the wire.. . In the case of an empty list the keyword/value of minItems: 1 . indicates that an implementation MUST provide at least one . Flow.Header object.. . The default value for the Flow.Header choice property is ethernet . which will result in an implementation by default providing at least . one ethernet packet header.

        Returns: FlowHeaderIter
        """
        return self._get_property('packet', FlowHeaderIter, self._parent, self._choice)

    @property
    def size(self):
        # type: () -> FlowSize
        """size getter

        The frame size which overrides the total length of the packetThe size of the packets.

        Returns: FlowSize
        """
        return self._get_property('size', FlowSize)

    @property
    def rate(self):
        # type: () -> FlowRate
        """rate getter

        The rate of packet transmissionThe transmit rate of the packets.

        Returns: FlowRate
        """
        return self._get_property('rate', FlowRate)

    @property
    def duration(self):
        # type: () -> FlowDuration
        """duration getter

        A container for different transmit durations. The transmit duration of the packets.

        Returns: FlowDuration
        """
        return self._get_property('duration', FlowDuration)

    @property
    def metrics(self):
        # type: () -> FlowMetrics
        """metrics getter

        The optional container for configuring flow metrics.Flow metrics. 

        Returns: FlowMetrics
        """
        return self._get_property('metrics', FlowMetrics)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property('name')

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property('name', value)


class FlowTxRx(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'port',
                'device',
            ],
        },
        'port': {'type': 'FlowPort'},
        'device': {'type': 'FlowDevice'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'port',
    } # type: Dict[str, Union(type)]

    PORT = 'port' # type: str
    DEVICE = 'device' # type: str

    def __init__(self, parent=None, choice=None):
        super(FlowTxRx, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def port(self):
        # type: () -> FlowPort
        """Factory property that returns an instance of the FlowPort class

        A container for a transmit port and 0..n intended receive ports.. When assigning this container to a flow the flows's . packet headers will not be populated with any address resolution . information such as source and/or destination addresses. . For example Flow.Ethernet dst mac address values will be defaulted to 0. . For full control over the Flow.properties.packet header contents use this . container. 

        Returns: FlowPort
        """
        return self._get_property('port', FlowPort, self, 'port')

    @property
    def device(self):
        # type: () -> FlowDevice
        """Factory property that returns an instance of the FlowDevice class

        A container for declaring a map of 1..n transmit devices. to 1..n receive devices. This allows for a single flow to have . different tx to rx device flows such as a single one to one map or a . many to many map.

        Returns: FlowDevice
        """
        return self._get_property('device', FlowDevice, self, 'device')

    @property
    def choice(self):
        # type: () -> Union[Literal["device"], Literal["port"]]
        """choice getter

        The type of transmit and receive container used by the flow.

        Returns: Union[Literal["device"], Literal["port"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of transmit and receive container used by the flow.

        value: Union[Literal["device"], Literal["port"]]
        """
        self._set_property('choice', value)


class FlowPort(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'tx_name': {'type': str},
        'rx_name': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED = ('tx_name',) # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, tx_name=None, rx_name=None):
        super(FlowPort, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('tx_name', tx_name)
        self._set_property('rx_name', rx_name)

    @property
    def tx_name(self):
        # type: () -> str
        """tx_name getter

        The unique name of a port that is the transmit port.. . x-constraint:. - /components/schemas/Port/properties/name. - /components/schemas/Lag/properties/name. 

        Returns: str
        """
        return self._get_property('tx_name')

    @tx_name.setter
    def tx_name(self, value):
        """tx_name setter

        The unique name of a port that is the transmit port.. . x-constraint:. - /components/schemas/Port/properties/name. - /components/schemas/Lag/properties/name. 

        value: str
        """
        self._set_property('tx_name', value)

    @property
    def rx_name(self):
        # type: () -> str
        """rx_name getter

        The unique name of a port that is the intended receive port.. . x-constraint:. - /components/schemas/Port/properties/name. - /components/schemas/Lag/properties/name. 

        Returns: str
        """
        return self._get_property('rx_name')

    @rx_name.setter
    def rx_name(self, value):
        """rx_name setter

        The unique name of a port that is the intended receive port.. . x-constraint:. - /components/schemas/Port/properties/name. - /components/schemas/Lag/properties/name. 

        value: str
        """
        self._set_property('rx_name', value)


class FlowDevice(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'mode': {
            'type': str,
            'enum': [
                'mesh',
                'one_to_one',
            ],
        },
        'tx_names': {'type': list},
        'rx_names': {'type': list},
    } # type: Dict[str, str]

    _REQUIRED = ('tx_names', 'rx_names') # type: tuple(str)

    _DEFAULTS = {
        'mode': 'mesh',
    } # type: Dict[str, Union(type)]

    MESH = 'mesh' # type: str
    ONE_TO_ONE = 'one_to_one' # type: str

    def __init__(self, parent=None, choice=None, mode=None, tx_names=None, rx_names=None):
        super(FlowDevice, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('mode', mode)
        self._set_property('tx_names', tx_names)
        self._set_property('rx_names', rx_names)

    @property
    def mode(self):
        # type: () -> Union[Literal["mesh"], Literal["one_to_one"]]
        """mode getter

        Determines the mapping of tx_names to rx_names.. . The mesh mode creates traffic between each value in tx_names to. every value in rx_names except in the case where the individual . tx_names value is equal to rx_names value which will be ignored.. Identical values in tx_names or rx_names MUST raise an error.. . The one_to_one mode creates traffic between each value pair in . of tx_names and rx_names by index.. The length of tx_names and rx_name MUST be the same.. Identical values in tx_names or rx_names MUST raise an error.

        Returns: Union[Literal["mesh"], Literal["one_to_one"]]
        """
        return self._get_property('mode')

    @mode.setter
    def mode(self, value):
        """mode setter

        Determines the mapping of tx_names to rx_names.. . The mesh mode creates traffic between each value in tx_names to. every value in rx_names except in the case where the individual . tx_names value is equal to rx_names value which will be ignored.. Identical values in tx_names or rx_names MUST raise an error.. . The one_to_one mode creates traffic between each value pair in . of tx_names and rx_names by index.. The length of tx_names and rx_name MUST be the same.. Identical values in tx_names or rx_names MUST raise an error.

        value: Union[Literal["mesh"], Literal["one_to_one"]]
        """
        self._set_property('mode', value)

    @property
    def tx_names(self):
        # type: () -> List[str]
        """tx_names getter

        TBD. . x-constraint:. - /components/schemas/Device.Ethernet/properties/name. - /components/schemas/Device.Ipv4/properties/name. - /components/schemas/Device.Ipv6/properties/name. - /components/schemas/Device.Bgpv4RouteRange/properties/name. - /components/schemas/Device.Bgpv6RouteRange/properties/name. 

        Returns: List[str]
        """
        return self._get_property('tx_names')

    @tx_names.setter
    def tx_names(self, value):
        """tx_names setter

        TBD. . x-constraint:. - /components/schemas/Device.Ethernet/properties/name. - /components/schemas/Device.Ipv4/properties/name. - /components/schemas/Device.Ipv6/properties/name. - /components/schemas/Device.Bgpv4RouteRange/properties/name. - /components/schemas/Device.Bgpv6RouteRange/properties/name. 

        value: List[str]
        """
        self._set_property('tx_names', value)

    @property
    def rx_names(self):
        # type: () -> List[str]
        """rx_names getter

        TBD. . x-constraint:. - /components/schemas/Device.Ethernet/properties/name. - /components/schemas/Device.Ipv4/properties/name. - /components/schemas/Device.Ipv6/properties/name. - /components/schemas/Device.Bgpv4RouteRange/properties/name. - /components/schemas/Device.Bgpv6RouteRange/properties/name. 

        Returns: List[str]
        """
        return self._get_property('rx_names')

    @rx_names.setter
    def rx_names(self, value):
        """rx_names setter

        TBD. . x-constraint:. - /components/schemas/Device.Ethernet/properties/name. - /components/schemas/Device.Ipv4/properties/name. - /components/schemas/Device.Ipv6/properties/name. - /components/schemas/Device.Bgpv4RouteRange/properties/name. - /components/schemas/Device.Bgpv6RouteRange/properties/name. 

        value: List[str]
        """
        self._set_property('rx_names', value)


class FlowHeader(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'custom',
                'ethernet',
                'vlan',
                'vxlan',
                'ipv4',
                'ipv6',
                'pfcpause',
                'ethernetpause',
                'tcp',
                'udp',
                'gre',
                'gtpv1',
                'gtpv2',
                'arp',
                'icmp',
                'icmpv6',
                'ppp',
                'igmpv1',
            ],
        },
        'custom': {'type': 'FlowCustom'},
        'ethernet': {'type': 'FlowEthernet'},
        'vlan': {'type': 'FlowVlan'},
        'vxlan': {'type': 'FlowVxlan'},
        'ipv4': {'type': 'FlowIpv4'},
        'ipv6': {'type': 'FlowIpv6'},
        'pfcpause': {'type': 'FlowPfcPause'},
        'ethernetpause': {'type': 'FlowEthernetPause'},
        'tcp': {'type': 'FlowTcp'},
        'udp': {'type': 'FlowUdp'},
        'gre': {'type': 'FlowGre'},
        'gtpv1': {'type': 'FlowGtpv1'},
        'gtpv2': {'type': 'FlowGtpv2'},
        'arp': {'type': 'FlowArp'},
        'icmp': {'type': 'FlowIcmp'},
        'icmpv6': {'type': 'FlowIcmpv6'},
        'ppp': {'type': 'FlowPpp'},
        'igmpv1': {'type': 'FlowIgmpv1'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'ethernet',
    } # type: Dict[str, Union(type)]

    CUSTOM = 'custom' # type: str
    ETHERNET = 'ethernet' # type: str
    VLAN = 'vlan' # type: str
    VXLAN = 'vxlan' # type: str
    IPV4 = 'ipv4' # type: str
    IPV6 = 'ipv6' # type: str
    PFCPAUSE = 'pfcpause' # type: str
    ETHERNETPAUSE = 'ethernetpause' # type: str
    TCP = 'tcp' # type: str
    UDP = 'udp' # type: str
    GRE = 'gre' # type: str
    GTPV1 = 'gtpv1' # type: str
    GTPV2 = 'gtpv2' # type: str
    ARP = 'arp' # type: str
    ICMP = 'icmp' # type: str
    ICMPV6 = 'icmpv6' # type: str
    PPP = 'ppp' # type: str
    IGMPV1 = 'igmpv1' # type: str

    def __init__(self, parent=None, choice=None):
        super(FlowHeader, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def custom(self):
        # type: () -> FlowCustom
        """Factory property that returns an instance of the FlowCustom class

        Custom packet header

        Returns: FlowCustom
        """
        return self._get_property('custom', FlowCustom, self, 'custom')

    @property
    def ethernet(self):
        # type: () -> FlowEthernet
        """Factory property that returns an instance of the FlowEthernet class

        Ethernet packet header

        Returns: FlowEthernet
        """
        return self._get_property('ethernet', FlowEthernet, self, 'ethernet')

    @property
    def vlan(self):
        # type: () -> FlowVlan
        """Factory property that returns an instance of the FlowVlan class

        VLAN packet header

        Returns: FlowVlan
        """
        return self._get_property('vlan', FlowVlan, self, 'vlan')

    @property
    def vxlan(self):
        # type: () -> FlowVxlan
        """Factory property that returns an instance of the FlowVxlan class

        VXLAN packet header

        Returns: FlowVxlan
        """
        return self._get_property('vxlan', FlowVxlan, self, 'vxlan')

    @property
    def ipv4(self):
        # type: () -> FlowIpv4
        """Factory property that returns an instance of the FlowIpv4 class

        IPv4 packet header

        Returns: FlowIpv4
        """
        return self._get_property('ipv4', FlowIpv4, self, 'ipv4')

    @property
    def ipv6(self):
        # type: () -> FlowIpv6
        """Factory property that returns an instance of the FlowIpv6 class

        IPv6 packet header

        Returns: FlowIpv6
        """
        return self._get_property('ipv6', FlowIpv6, self, 'ipv6')

    @property
    def pfcpause(self):
        # type: () -> FlowPfcPause
        """Factory property that returns an instance of the FlowPfcPause class

        IEEE 802.1Qbb PFC Pause packet header.

        Returns: FlowPfcPause
        """
        return self._get_property('pfcpause', FlowPfcPause, self, 'pfcpause')

    @property
    def ethernetpause(self):
        # type: () -> FlowEthernetPause
        """Factory property that returns an instance of the FlowEthernetPause class

        IEEE 802.3x global ethernet pause packet header

        Returns: FlowEthernetPause
        """
        return self._get_property('ethernetpause', FlowEthernetPause, self, 'ethernetpause')

    @property
    def tcp(self):
        # type: () -> FlowTcp
        """Factory property that returns an instance of the FlowTcp class

        TCP packet header

        Returns: FlowTcp
        """
        return self._get_property('tcp', FlowTcp, self, 'tcp')

    @property
    def udp(self):
        # type: () -> FlowUdp
        """Factory property that returns an instance of the FlowUdp class

        UDP packet header

        Returns: FlowUdp
        """
        return self._get_property('udp', FlowUdp, self, 'udp')

    @property
    def gre(self):
        # type: () -> FlowGre
        """Factory property that returns an instance of the FlowGre class

        Standard GRE packet header (RFC2784)

        Returns: FlowGre
        """
        return self._get_property('gre', FlowGre, self, 'gre')

    @property
    def gtpv1(self):
        # type: () -> FlowGtpv1
        """Factory property that returns an instance of the FlowGtpv1 class

        GTPv1 packet header

        Returns: FlowGtpv1
        """
        return self._get_property('gtpv1', FlowGtpv1, self, 'gtpv1')

    @property
    def gtpv2(self):
        # type: () -> FlowGtpv2
        """Factory property that returns an instance of the FlowGtpv2 class

        GTPv2 packet header

        Returns: FlowGtpv2
        """
        return self._get_property('gtpv2', FlowGtpv2, self, 'gtpv2')

    @property
    def arp(self):
        # type: () -> FlowArp
        """Factory property that returns an instance of the FlowArp class

        ARP packet header

        Returns: FlowArp
        """
        return self._get_property('arp', FlowArp, self, 'arp')

    @property
    def icmp(self):
        # type: () -> FlowIcmp
        """Factory property that returns an instance of the FlowIcmp class

        ICMP packet header

        Returns: FlowIcmp
        """
        return self._get_property('icmp', FlowIcmp, self, 'icmp')

    @property
    def icmpv6(self):
        # type: () -> FlowIcmpv6
        """Factory property that returns an instance of the FlowIcmpv6 class

        ICMPv6 packet header

        Returns: FlowIcmpv6
        """
        return self._get_property('icmpv6', FlowIcmpv6, self, 'icmpv6')

    @property
    def ppp(self):
        # type: () -> FlowPpp
        """Factory property that returns an instance of the FlowPpp class

        PPP packet header

        Returns: FlowPpp
        """
        return self._get_property('ppp', FlowPpp, self, 'ppp')

    @property
    def igmpv1(self):
        # type: () -> FlowIgmpv1
        """Factory property that returns an instance of the FlowIgmpv1 class

        IGMPv1 packet header

        Returns: FlowIgmpv1
        """
        return self._get_property('igmpv1', FlowIgmpv1, self, 'igmpv1')

    @property
    def choice(self):
        # type: () -> Union[Literal["arp"], Literal["custom"], Literal["ethernet"], Literal["ethernetpause"], Literal["gre"], Literal["gtpv1"], Literal["gtpv2"], Literal["icmp"], Literal["icmpv6"], Literal["igmpv1"], Literal["ipv4"], Literal["ipv6"], Literal["pfcpause"], Literal["ppp"], Literal["tcp"], Literal["udp"], Literal["vlan"], Literal["vxlan"]]
        """choice getter

        The available types of flow headers. If one is not provided the . default ethernet packet header MUST be provided.

        Returns: Union[Literal["arp"], Literal["custom"], Literal["ethernet"], Literal["ethernetpause"], Literal["gre"], Literal["gtpv1"], Literal["gtpv2"], Literal["icmp"], Literal["icmpv6"], Literal["igmpv1"], Literal["ipv4"], Literal["ipv6"], Literal["pfcpause"], Literal["ppp"], Literal["tcp"], Literal["udp"], Literal["vlan"], Literal["vxlan"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        The available types of flow headers. If one is not provided the . default ethernet packet header MUST be provided.

        value: Union[Literal["arp"], Literal["custom"], Literal["ethernet"], Literal["ethernetpause"], Literal["gre"], Literal["gtpv1"], Literal["gtpv2"], Literal["icmp"], Literal["icmpv6"], Literal["igmpv1"], Literal["ipv4"], Literal["ipv6"], Literal["pfcpause"], Literal["ppp"], Literal["tcp"], Literal["udp"], Literal["vlan"], Literal["vxlan"]]
        """
        self._set_property('choice', value)


class FlowCustom(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'bytes': {'type': str},
    } # type: Dict[str, str]

    _REQUIRED = ('bytes',) # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, bytes=None):
        super(FlowCustom, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('bytes', bytes)

    @property
    def bytes(self):
        # type: () -> str
        """bytes getter

        A custom packet header defined as a string of hex bytes. The string MUST contain sequence of valid hex bytes. Spaces or colons can be part of the bytes but will be discarded. This packet header can be used in multiple places in the packet.

        Returns: str
        """
        return self._get_property('bytes')

    @bytes.setter
    def bytes(self, value):
        """bytes setter

        A custom packet header defined as a string of hex bytes. The string MUST contain sequence of valid hex bytes. Spaces or colons can be part of the bytes but will be discarded. This packet header can be used in multiple places in the packet.

        value: str
        """
        self._set_property('bytes', value)


class FlowEthernet(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'dst': {'type': 'PatternFlowEthernetDst'},
        'src': {'type': 'PatternFlowEthernetSrc'},
        'ether_type': {'type': 'PatternFlowEthernetEtherType'},
        'pfc_queue': {'type': 'PatternFlowEthernetPfcQueue'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowEthernet, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def dst(self):
        # type: () -> PatternFlowEthernetDst
        """dst getter

        Destination MAC addressDestination MAC address

        Returns: PatternFlowEthernetDst
        """
        return self._get_property('dst', PatternFlowEthernetDst)

    @property
    def src(self):
        # type: () -> PatternFlowEthernetSrc
        """src getter

        Source MAC addressSource MAC address

        Returns: PatternFlowEthernetSrc
        """
        return self._get_property('src', PatternFlowEthernetSrc)

    @property
    def ether_type(self):
        # type: () -> PatternFlowEthernetEtherType
        """ether_type getter

        Ethernet typeEthernet type

        Returns: PatternFlowEthernetEtherType
        """
        return self._get_property('ether_type', PatternFlowEthernetEtherType)

    @property
    def pfc_queue(self):
        # type: () -> PatternFlowEthernetPfcQueue
        """pfc_queue getter

        Priority flow control queuePriority flow control queue

        Returns: PatternFlowEthernetPfcQueue
        """
        return self._get_property('pfc_queue', PatternFlowEthernetPfcQueue)


class PatternFlowEthernetDst(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {
            'type': str,
            'format': 'mac',
        },
        'values': {
            'type': list,
            'format': 'mac',
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowEthernetDstCounter'},
        'decrement': {'type': 'PatternFlowEthernetDstCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': '00:00:00:00:00:00',
        'values': ['00:00:00:00:00:00'],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowEthernetDst, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowEthernetDstCounter
        """Factory property that returns an instance of the PatternFlowEthernetDstCounter class

        mac counter pattern

        Returns: PatternFlowEthernetDstCounter
        """
        return self._get_property('increment', PatternFlowEthernetDstCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowEthernetDstCounter
        """Factory property that returns an instance of the PatternFlowEthernetDstCounter class

        mac counter pattern

        Returns: PatternFlowEthernetDstCounter
        """
        return self._get_property('decrement', PatternFlowEthernetDstCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowEthernetDstCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {
            'type': str,
            'format': 'mac',
        },
        'step': {
            'type': str,
            'format': 'mac',
        },
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': '00:00:00:00:00:00',
        'step': '00:00:00:00:00:01',
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowEthernetDstCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowEthernetSrc(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {
            'type': str,
            'format': 'mac',
        },
        'values': {
            'type': list,
            'format': 'mac',
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowEthernetSrcCounter'},
        'decrement': {'type': 'PatternFlowEthernetSrcCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': '00:00:00:00:00:00',
        'values': ['00:00:00:00:00:00'],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowEthernetSrc, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowEthernetSrcCounter
        """Factory property that returns an instance of the PatternFlowEthernetSrcCounter class

        mac counter pattern

        Returns: PatternFlowEthernetSrcCounter
        """
        return self._get_property('increment', PatternFlowEthernetSrcCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowEthernetSrcCounter
        """Factory property that returns an instance of the PatternFlowEthernetSrcCounter class

        mac counter pattern

        Returns: PatternFlowEthernetSrcCounter
        """
        return self._get_property('decrement', PatternFlowEthernetSrcCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowEthernetSrcCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {
            'type': str,
            'format': 'mac',
        },
        'step': {
            'type': str,
            'format': 'mac',
        },
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': '00:00:00:00:00:00',
        'step': '00:00:00:00:00:01',
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowEthernetSrcCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowEthernetEtherType(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'auto',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'auto': {
            'type': str,
            'enum': [
                'auto',
            ],
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowEthernetEtherTypeCounter'},
        'decrement': {'type': 'PatternFlowEthernetEtherTypeCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'auto',
        'value': 65535,
        'values': [65535],
        'auto': 'auto',
    } # type: Dict[str, Union(type)]

    IPV4 = 2048 #
    IPV6 = 34525 #
    ARP = 2054 #
    VLAN_802_1_Q = 33024 #
    RESERVED = 65535 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    AUTO = 'auto' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    AUTO = 'auto' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowEthernetEtherType, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowEthernetEtherTypeCounter
        """Factory property that returns an instance of the PatternFlowEthernetEtherTypeCounter class

        integer counter pattern

        Returns: PatternFlowEthernetEtherTypeCounter
        """
        return self._get_property('increment', PatternFlowEthernetEtherTypeCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowEthernetEtherTypeCounter
        """Factory property that returns an instance of the PatternFlowEthernetEtherTypeCounter class

        integer counter pattern

        Returns: PatternFlowEthernetEtherTypeCounter
        """
        return self._get_property('decrement', PatternFlowEthernetEtherTypeCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def auto(self):
        # type: () -> Union[Literal["auto"]]
        """auto getter

        TBD

        Returns: Union[Literal["auto"]]
        """
        return self._get_property('auto')

    @auto.setter
    def auto(self, value):
        """auto setter

        TBD

        value: Union[Literal["auto"]]
        """
        self._set_property('auto', value, 'auto')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowEthernetEtherTypeCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 65535,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    IPV4 = 2048 #
    IPV6 = 34525 #
    ARP = 2054 #
    VLAN_802_1_Q = 33024 #
    RESERVED = 65535 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowEthernetEtherTypeCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowEthernetPfcQueue(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowEthernetPfcQueueCounter'},
        'decrement': {'type': 'PatternFlowEthernetPfcQueueCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowEthernetPfcQueue, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowEthernetPfcQueueCounter
        """Factory property that returns an instance of the PatternFlowEthernetPfcQueueCounter class

        integer counter pattern

        Returns: PatternFlowEthernetPfcQueueCounter
        """
        return self._get_property('increment', PatternFlowEthernetPfcQueueCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowEthernetPfcQueueCounter
        """Factory property that returns an instance of the PatternFlowEthernetPfcQueueCounter class

        integer counter pattern

        Returns: PatternFlowEthernetPfcQueueCounter
        """
        return self._get_property('decrement', PatternFlowEthernetPfcQueueCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowEthernetPfcQueueCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowEthernetPfcQueueCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class FlowVlan(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'priority': {'type': 'PatternFlowVlanPriority'},
        'cfi': {'type': 'PatternFlowVlanCfi'},
        'id': {'type': 'PatternFlowVlanId'},
        'tpid': {'type': 'PatternFlowVlanTpid'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowVlan, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def priority(self):
        # type: () -> PatternFlowVlanPriority
        """priority getter

        Priority code pointPriority code point

        Returns: PatternFlowVlanPriority
        """
        return self._get_property('priority', PatternFlowVlanPriority)

    @property
    def cfi(self):
        # type: () -> PatternFlowVlanCfi
        """cfi getter

        Canonical format indicator or drop elegible indicatorCanonical format indicator or drop elegible indicator

        Returns: PatternFlowVlanCfi
        """
        return self._get_property('cfi', PatternFlowVlanCfi)

    @property
    def id(self):
        # type: () -> PatternFlowVlanId
        """id getter

        Vlan identifierVlan identifier

        Returns: PatternFlowVlanId
        """
        return self._get_property('id', PatternFlowVlanId)

    @property
    def tpid(self):
        # type: () -> PatternFlowVlanTpid
        """tpid getter

        Protocol identifierProtocol identifier

        Returns: PatternFlowVlanTpid
        """
        return self._get_property('tpid', PatternFlowVlanTpid)


class PatternFlowVlanPriority(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowVlanPriorityCounter'},
        'decrement': {'type': 'PatternFlowVlanPriorityCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowVlanPriority, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowVlanPriorityCounter
        """Factory property that returns an instance of the PatternFlowVlanPriorityCounter class

        integer counter pattern

        Returns: PatternFlowVlanPriorityCounter
        """
        return self._get_property('increment', PatternFlowVlanPriorityCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowVlanPriorityCounter
        """Factory property that returns an instance of the PatternFlowVlanPriorityCounter class

        integer counter pattern

        Returns: PatternFlowVlanPriorityCounter
        """
        return self._get_property('decrement', PatternFlowVlanPriorityCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowVlanPriorityCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowVlanPriorityCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowVlanCfi(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowVlanCfiCounter'},
        'decrement': {'type': 'PatternFlowVlanCfiCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowVlanCfi, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowVlanCfiCounter
        """Factory property that returns an instance of the PatternFlowVlanCfiCounter class

        integer counter pattern

        Returns: PatternFlowVlanCfiCounter
        """
        return self._get_property('increment', PatternFlowVlanCfiCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowVlanCfiCounter
        """Factory property that returns an instance of the PatternFlowVlanCfiCounter class

        integer counter pattern

        Returns: PatternFlowVlanCfiCounter
        """
        return self._get_property('decrement', PatternFlowVlanCfiCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowVlanCfiCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowVlanCfiCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowVlanId(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowVlanIdCounter'},
        'decrement': {'type': 'PatternFlowVlanIdCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowVlanId, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowVlanIdCounter
        """Factory property that returns an instance of the PatternFlowVlanIdCounter class

        integer counter pattern

        Returns: PatternFlowVlanIdCounter
        """
        return self._get_property('increment', PatternFlowVlanIdCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowVlanIdCounter
        """Factory property that returns an instance of the PatternFlowVlanIdCounter class

        integer counter pattern

        Returns: PatternFlowVlanIdCounter
        """
        return self._get_property('decrement', PatternFlowVlanIdCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowVlanIdCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowVlanIdCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowVlanTpid(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowVlanTpidCounter'},
        'decrement': {'type': 'PatternFlowVlanTpidCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 33024,
        'values': [33024],
    } # type: Dict[str, Union(type)]

    X8100 = 33024 #
    X88A8 = 34984 #
    X9100 = 37120 #
    X9200 = 37376 #
    X9300 = 37632 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowVlanTpid, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowVlanTpidCounter
        """Factory property that returns an instance of the PatternFlowVlanTpidCounter class

        integer counter pattern

        Returns: PatternFlowVlanTpidCounter
        """
        return self._get_property('increment', PatternFlowVlanTpidCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowVlanTpidCounter
        """Factory property that returns an instance of the PatternFlowVlanTpidCounter class

        integer counter pattern

        Returns: PatternFlowVlanTpidCounter
        """
        return self._get_property('decrement', PatternFlowVlanTpidCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowVlanTpidCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 33024,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    X8100 = 33024 #
    X88A8 = 34984 #
    X9100 = 37120 #
    X9200 = 37376 #
    X9300 = 37632 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowVlanTpidCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class FlowVxlan(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'flags': {'type': 'PatternFlowVxlanFlags'},
        'reserved0': {'type': 'PatternFlowVxlanReserved0'},
        'vni': {'type': 'PatternFlowVxlanVni'},
        'reserved1': {'type': 'PatternFlowVxlanReserved1'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowVxlan, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def flags(self):
        # type: () -> PatternFlowVxlanFlags
        """flags getter

        Flags field with a bit format of RRRRIRRR. The I flag MUST be set to 1 for a valid vxlan network id (VNI). The other 7 bits (designated "R") are reserved fields and MUST be set to zero on transmission and ignored on receipt.Flags field with a bit format of RRRRIRRR. The I flag MUST be set to 1 for a valid vxlan network id (VNI). The other 7 bits (designated "R") are reserved fields and MUST be set to zero on transmission and ignored on receipt.

        Returns: PatternFlowVxlanFlags
        """
        return self._get_property('flags', PatternFlowVxlanFlags)

    @property
    def reserved0(self):
        # type: () -> PatternFlowVxlanReserved0
        """reserved0 getter

        Reserved fieldReserved field

        Returns: PatternFlowVxlanReserved0
        """
        return self._get_property('reserved0', PatternFlowVxlanReserved0)

    @property
    def vni(self):
        # type: () -> PatternFlowVxlanVni
        """vni getter

        VXLAN network idVXLAN network id

        Returns: PatternFlowVxlanVni
        """
        return self._get_property('vni', PatternFlowVxlanVni)

    @property
    def reserved1(self):
        # type: () -> PatternFlowVxlanReserved1
        """reserved1 getter

        Reserved fieldReserved field

        Returns: PatternFlowVxlanReserved1
        """
        return self._get_property('reserved1', PatternFlowVxlanReserved1)


class PatternFlowVxlanFlags(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowVxlanFlagsCounter'},
        'decrement': {'type': 'PatternFlowVxlanFlagsCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 8,
        'values': [8],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowVxlanFlags, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowVxlanFlagsCounter
        """Factory property that returns an instance of the PatternFlowVxlanFlagsCounter class

        integer counter pattern

        Returns: PatternFlowVxlanFlagsCounter
        """
        return self._get_property('increment', PatternFlowVxlanFlagsCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowVxlanFlagsCounter
        """Factory property that returns an instance of the PatternFlowVxlanFlagsCounter class

        integer counter pattern

        Returns: PatternFlowVxlanFlagsCounter
        """
        return self._get_property('decrement', PatternFlowVxlanFlagsCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowVxlanFlagsCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 8,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowVxlanFlagsCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowVxlanReserved0(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowVxlanReserved0Counter'},
        'decrement': {'type': 'PatternFlowVxlanReserved0Counter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowVxlanReserved0, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowVxlanReserved0Counter
        """Factory property that returns an instance of the PatternFlowVxlanReserved0Counter class

        integer counter pattern

        Returns: PatternFlowVxlanReserved0Counter
        """
        return self._get_property('increment', PatternFlowVxlanReserved0Counter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowVxlanReserved0Counter
        """Factory property that returns an instance of the PatternFlowVxlanReserved0Counter class

        integer counter pattern

        Returns: PatternFlowVxlanReserved0Counter
        """
        return self._get_property('decrement', PatternFlowVxlanReserved0Counter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowVxlanReserved0Counter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowVxlanReserved0Counter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowVxlanVni(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowVxlanVniCounter'},
        'decrement': {'type': 'PatternFlowVxlanVniCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowVxlanVni, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowVxlanVniCounter
        """Factory property that returns an instance of the PatternFlowVxlanVniCounter class

        integer counter pattern

        Returns: PatternFlowVxlanVniCounter
        """
        return self._get_property('increment', PatternFlowVxlanVniCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowVxlanVniCounter
        """Factory property that returns an instance of the PatternFlowVxlanVniCounter class

        integer counter pattern

        Returns: PatternFlowVxlanVniCounter
        """
        return self._get_property('decrement', PatternFlowVxlanVniCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowVxlanVniCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowVxlanVniCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowVxlanReserved1(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowVxlanReserved1Counter'},
        'decrement': {'type': 'PatternFlowVxlanReserved1Counter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowVxlanReserved1, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowVxlanReserved1Counter
        """Factory property that returns an instance of the PatternFlowVxlanReserved1Counter class

        integer counter pattern

        Returns: PatternFlowVxlanReserved1Counter
        """
        return self._get_property('increment', PatternFlowVxlanReserved1Counter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowVxlanReserved1Counter
        """Factory property that returns an instance of the PatternFlowVxlanReserved1Counter class

        integer counter pattern

        Returns: PatternFlowVxlanReserved1Counter
        """
        return self._get_property('decrement', PatternFlowVxlanReserved1Counter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowVxlanReserved1Counter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowVxlanReserved1Counter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class FlowIpv4(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'version': {'type': 'PatternFlowIpv4Version'},
        'header_length': {'type': 'PatternFlowIpv4HeaderLength'},
        'priority': {'type': 'FlowIpv4Priority'},
        'total_length': {'type': 'PatternFlowIpv4TotalLength'},
        'identification': {'type': 'PatternFlowIpv4Identification'},
        'reserved': {'type': 'PatternFlowIpv4Reserved'},
        'dont_fragment': {'type': 'PatternFlowIpv4DontFragment'},
        'more_fragments': {'type': 'PatternFlowIpv4MoreFragments'},
        'fragment_offset': {'type': 'PatternFlowIpv4FragmentOffset'},
        'time_to_live': {'type': 'PatternFlowIpv4TimeToLive'},
        'protocol': {'type': 'PatternFlowIpv4Protocol'},
        'header_checksum': {'type': 'PatternFlowIpv4HeaderChecksum'},
        'src': {'type': 'PatternFlowIpv4Src'},
        'dst': {'type': 'PatternFlowIpv4Dst'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowIpv4, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def version(self):
        # type: () -> PatternFlowIpv4Version
        """version getter

        VersionVersion

        Returns: PatternFlowIpv4Version
        """
        return self._get_property('version', PatternFlowIpv4Version)

    @property
    def header_length(self):
        # type: () -> PatternFlowIpv4HeaderLength
        """header_length getter

        Header lengthHeader length

        Returns: PatternFlowIpv4HeaderLength
        """
        return self._get_property('header_length', PatternFlowIpv4HeaderLength)

    @property
    def priority(self):
        # type: () -> FlowIpv4Priority
        """priority getter

        A container for ipv4 raw, tos, dscp ip priorities.A container for ipv4 raw, tos, dscp ip priorities.

        Returns: FlowIpv4Priority
        """
        return self._get_property('priority', FlowIpv4Priority)

    @property
    def total_length(self):
        # type: () -> PatternFlowIpv4TotalLength
        """total_length getter

        Total lengthTotal length

        Returns: PatternFlowIpv4TotalLength
        """
        return self._get_property('total_length', PatternFlowIpv4TotalLength)

    @property
    def identification(self):
        # type: () -> PatternFlowIpv4Identification
        """identification getter

        IdentificationIdentification

        Returns: PatternFlowIpv4Identification
        """
        return self._get_property('identification', PatternFlowIpv4Identification)

    @property
    def reserved(self):
        # type: () -> PatternFlowIpv4Reserved
        """reserved getter

        Reserved flag.Reserved flag.

        Returns: PatternFlowIpv4Reserved
        """
        return self._get_property('reserved', PatternFlowIpv4Reserved)

    @property
    def dont_fragment(self):
        # type: () -> PatternFlowIpv4DontFragment
        """dont_fragment getter

        Dont fragment flag If the dont_fragment flag is set and fragmentation is required to route the packet then the packet is dropped.Dont fragment flag If the dont_fragment flag is set and fragmentation is required to route the packet then the packet is dropped.

        Returns: PatternFlowIpv4DontFragment
        """
        return self._get_property('dont_fragment', PatternFlowIpv4DontFragment)

    @property
    def more_fragments(self):
        # type: () -> PatternFlowIpv4MoreFragments
        """more_fragments getter

        More fragments flagMore fragments flag

        Returns: PatternFlowIpv4MoreFragments
        """
        return self._get_property('more_fragments', PatternFlowIpv4MoreFragments)

    @property
    def fragment_offset(self):
        # type: () -> PatternFlowIpv4FragmentOffset
        """fragment_offset getter

        Fragment offsetFragment offset

        Returns: PatternFlowIpv4FragmentOffset
        """
        return self._get_property('fragment_offset', PatternFlowIpv4FragmentOffset)

    @property
    def time_to_live(self):
        # type: () -> PatternFlowIpv4TimeToLive
        """time_to_live getter

        Time to liveTime to live

        Returns: PatternFlowIpv4TimeToLive
        """
        return self._get_property('time_to_live', PatternFlowIpv4TimeToLive)

    @property
    def protocol(self):
        # type: () -> PatternFlowIpv4Protocol
        """protocol getter

        Protocol, default is 61 any host internal protocolProtocol, default is 61 any host internal protocol

        Returns: PatternFlowIpv4Protocol
        """
        return self._get_property('protocol', PatternFlowIpv4Protocol)

    @property
    def header_checksum(self):
        # type: () -> PatternFlowIpv4HeaderChecksum
        """header_checksum getter

        Header checksumHeader checksum

        Returns: PatternFlowIpv4HeaderChecksum
        """
        return self._get_property('header_checksum', PatternFlowIpv4HeaderChecksum)

    @property
    def src(self):
        # type: () -> PatternFlowIpv4Src
        """src getter

        Source addressSource address

        Returns: PatternFlowIpv4Src
        """
        return self._get_property('src', PatternFlowIpv4Src)

    @property
    def dst(self):
        # type: () -> PatternFlowIpv4Dst
        """dst getter

        Destination addressDestination address

        Returns: PatternFlowIpv4Dst
        """
        return self._get_property('dst', PatternFlowIpv4Dst)


class PatternFlowIpv4Version(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4VersionCounter'},
        'decrement': {'type': 'PatternFlowIpv4VersionCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 4,
        'values': [4],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4Version, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4VersionCounter
        """Factory property that returns an instance of the PatternFlowIpv4VersionCounter class

        integer counter pattern

        Returns: PatternFlowIpv4VersionCounter
        """
        return self._get_property('increment', PatternFlowIpv4VersionCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4VersionCounter
        """Factory property that returns an instance of the PatternFlowIpv4VersionCounter class

        integer counter pattern

        Returns: PatternFlowIpv4VersionCounter
        """
        return self._get_property('decrement', PatternFlowIpv4VersionCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4VersionCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 4,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4VersionCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4HeaderLength(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'auto',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'auto': {
            'type': str,
            'enum': [
                'auto',
            ],
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4HeaderLengthCounter'},
        'decrement': {'type': 'PatternFlowIpv4HeaderLengthCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'auto',
        'value': 5,
        'values': [5],
        'auto': 'auto',
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    AUTO = 'auto' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    AUTO = 'auto' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4HeaderLength, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4HeaderLengthCounter
        """Factory property that returns an instance of the PatternFlowIpv4HeaderLengthCounter class

        integer counter pattern

        Returns: PatternFlowIpv4HeaderLengthCounter
        """
        return self._get_property('increment', PatternFlowIpv4HeaderLengthCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4HeaderLengthCounter
        """Factory property that returns an instance of the PatternFlowIpv4HeaderLengthCounter class

        integer counter pattern

        Returns: PatternFlowIpv4HeaderLengthCounter
        """
        return self._get_property('decrement', PatternFlowIpv4HeaderLengthCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def auto(self):
        # type: () -> Union[Literal["auto"]]
        """auto getter

        TBD

        Returns: Union[Literal["auto"]]
        """
        return self._get_property('auto')

    @auto.setter
    def auto(self, value):
        """auto setter

        TBD

        value: Union[Literal["auto"]]
        """
        self._set_property('auto', value, 'auto')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4HeaderLengthCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 5,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4HeaderLengthCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class FlowIpv4Priority(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'raw',
                'tos',
                'dscp',
            ],
        },
        'raw': {'type': 'PatternFlowIpv4PriorityRaw'},
        'tos': {'type': 'FlowIpv4Tos'},
        'dscp': {'type': 'FlowIpv4Dscp'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'dscp',
    } # type: Dict[str, Union(type)]

    RAW = 'raw' # type: str
    TOS = 'tos' # type: str
    DSCP = 'dscp' # type: str

    def __init__(self, parent=None, choice=None):
        super(FlowIpv4Priority, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def raw(self):
        # type: () -> PatternFlowIpv4PriorityRaw
        """Factory property that returns an instance of the PatternFlowIpv4PriorityRaw class

        Raw priority

        Returns: PatternFlowIpv4PriorityRaw
        """
        return self._get_property('raw', PatternFlowIpv4PriorityRaw, self, 'raw')

    @property
    def tos(self):
        # type: () -> FlowIpv4Tos
        """Factory property that returns an instance of the FlowIpv4Tos class

        Type of service (TOS) packet field.

        Returns: FlowIpv4Tos
        """
        return self._get_property('tos', FlowIpv4Tos, self, 'tos')

    @property
    def dscp(self):
        # type: () -> FlowIpv4Dscp
        """Factory property that returns an instance of the FlowIpv4Dscp class

        Differentiated services code point (DSCP) packet field.

        Returns: FlowIpv4Dscp
        """
        return self._get_property('dscp', FlowIpv4Dscp, self, 'dscp')

    @property
    def choice(self):
        # type: () -> Union[Literal["dscp"], Literal["raw"], Literal["tos"]]
        """choice getter

        TBD

        Returns: Union[Literal["dscp"], Literal["raw"], Literal["tos"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["dscp"], Literal["raw"], Literal["tos"]]
        """
        self._set_property('choice', value)


class PatternFlowIpv4PriorityRaw(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4PriorityRawCounter'},
        'decrement': {'type': 'PatternFlowIpv4PriorityRawCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4PriorityRaw, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4PriorityRawCounter
        """Factory property that returns an instance of the PatternFlowIpv4PriorityRawCounter class

        integer counter pattern

        Returns: PatternFlowIpv4PriorityRawCounter
        """
        return self._get_property('increment', PatternFlowIpv4PriorityRawCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4PriorityRawCounter
        """Factory property that returns an instance of the PatternFlowIpv4PriorityRawCounter class

        integer counter pattern

        Returns: PatternFlowIpv4PriorityRawCounter
        """
        return self._get_property('decrement', PatternFlowIpv4PriorityRawCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4PriorityRawCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4PriorityRawCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class FlowIpv4Tos(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'precedence': {'type': 'PatternFlowIpv4TosPrecedence'},
        'delay': {'type': 'PatternFlowIpv4TosDelay'},
        'throughput': {'type': 'PatternFlowIpv4TosThroughput'},
        'reliability': {'type': 'PatternFlowIpv4TosReliability'},
        'monetary': {'type': 'PatternFlowIpv4TosMonetary'},
        'unused': {'type': 'PatternFlowIpv4TosUnused'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowIpv4Tos, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def precedence(self):
        # type: () -> PatternFlowIpv4TosPrecedence
        """precedence getter

        PrecedencePrecedence

        Returns: PatternFlowIpv4TosPrecedence
        """
        return self._get_property('precedence', PatternFlowIpv4TosPrecedence)

    @property
    def delay(self):
        # type: () -> PatternFlowIpv4TosDelay
        """delay getter

        DelayDelay

        Returns: PatternFlowIpv4TosDelay
        """
        return self._get_property('delay', PatternFlowIpv4TosDelay)

    @property
    def throughput(self):
        # type: () -> PatternFlowIpv4TosThroughput
        """throughput getter

        ThroughputThroughput

        Returns: PatternFlowIpv4TosThroughput
        """
        return self._get_property('throughput', PatternFlowIpv4TosThroughput)

    @property
    def reliability(self):
        # type: () -> PatternFlowIpv4TosReliability
        """reliability getter

        ReliabilityReliability

        Returns: PatternFlowIpv4TosReliability
        """
        return self._get_property('reliability', PatternFlowIpv4TosReliability)

    @property
    def monetary(self):
        # type: () -> PatternFlowIpv4TosMonetary
        """monetary getter

        MonetaryMonetary

        Returns: PatternFlowIpv4TosMonetary
        """
        return self._get_property('monetary', PatternFlowIpv4TosMonetary)

    @property
    def unused(self):
        # type: () -> PatternFlowIpv4TosUnused
        """unused getter

        UnusedUnused

        Returns: PatternFlowIpv4TosUnused
        """
        return self._get_property('unused', PatternFlowIpv4TosUnused)


class PatternFlowIpv4TosPrecedence(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4TosPrecedenceCounter'},
        'decrement': {'type': 'PatternFlowIpv4TosPrecedenceCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    ROUTINE = 0 #
    PRIORITY = 1 #
    IMMEDIATE = 2 #
    FLASH = 3 #
    FLASH_OVERRIDE = 4 #
    CRITIC_ECP = 5 #
    INTERNETWORK_CONTROL = 6 #
    NETWORK_CONTROL = 7 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4TosPrecedence, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TosPrecedenceCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosPrecedenceCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosPrecedenceCounter
        """
        return self._get_property('increment', PatternFlowIpv4TosPrecedenceCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TosPrecedenceCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosPrecedenceCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosPrecedenceCounter
        """
        return self._get_property('decrement', PatternFlowIpv4TosPrecedenceCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4TosPrecedenceCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    ROUTINE = 0 #
    PRIORITY = 1 #
    IMMEDIATE = 2 #
    FLASH = 3 #
    FLASH_OVERRIDE = 4 #
    CRITIC_ECP = 5 #
    INTERNETWORK_CONTROL = 6 #
    NETWORK_CONTROL = 7 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4TosPrecedenceCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4TosDelay(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4TosDelayCounter'},
        'decrement': {'type': 'PatternFlowIpv4TosDelayCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    NORMAL = 0 #
    LOW = 1 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4TosDelay, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TosDelayCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosDelayCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosDelayCounter
        """
        return self._get_property('increment', PatternFlowIpv4TosDelayCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TosDelayCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosDelayCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosDelayCounter
        """
        return self._get_property('decrement', PatternFlowIpv4TosDelayCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4TosDelayCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    NORMAL = 0 #
    LOW = 1 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4TosDelayCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4TosThroughput(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4TosThroughputCounter'},
        'decrement': {'type': 'PatternFlowIpv4TosThroughputCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    NORMAL = 0 #
    LOW = 1 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4TosThroughput, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TosThroughputCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosThroughputCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosThroughputCounter
        """
        return self._get_property('increment', PatternFlowIpv4TosThroughputCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TosThroughputCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosThroughputCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosThroughputCounter
        """
        return self._get_property('decrement', PatternFlowIpv4TosThroughputCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4TosThroughputCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    NORMAL = 0 #
    LOW = 1 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4TosThroughputCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4TosReliability(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4TosReliabilityCounter'},
        'decrement': {'type': 'PatternFlowIpv4TosReliabilityCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    NORMAL = 0 #
    LOW = 1 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4TosReliability, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TosReliabilityCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosReliabilityCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosReliabilityCounter
        """
        return self._get_property('increment', PatternFlowIpv4TosReliabilityCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TosReliabilityCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosReliabilityCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosReliabilityCounter
        """
        return self._get_property('decrement', PatternFlowIpv4TosReliabilityCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4TosReliabilityCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    NORMAL = 0 #
    LOW = 1 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4TosReliabilityCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4TosMonetary(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4TosMonetaryCounter'},
        'decrement': {'type': 'PatternFlowIpv4TosMonetaryCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    NORMAL = 0 #
    LOW = 1 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4TosMonetary, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TosMonetaryCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosMonetaryCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosMonetaryCounter
        """
        return self._get_property('increment', PatternFlowIpv4TosMonetaryCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TosMonetaryCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosMonetaryCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosMonetaryCounter
        """
        return self._get_property('decrement', PatternFlowIpv4TosMonetaryCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4TosMonetaryCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    NORMAL = 0 #
    LOW = 1 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4TosMonetaryCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4TosUnused(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4TosUnusedCounter'},
        'decrement': {'type': 'PatternFlowIpv4TosUnusedCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4TosUnused, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TosUnusedCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosUnusedCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosUnusedCounter
        """
        return self._get_property('increment', PatternFlowIpv4TosUnusedCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TosUnusedCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosUnusedCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosUnusedCounter
        """
        return self._get_property('decrement', PatternFlowIpv4TosUnusedCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4TosUnusedCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4TosUnusedCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class FlowIpv4Dscp(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'phb': {'type': 'PatternFlowIpv4DscpPhb'},
        'ecn': {'type': 'PatternFlowIpv4DscpEcn'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowIpv4Dscp, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def phb(self):
        # type: () -> PatternFlowIpv4DscpPhb
        """phb getter

        Per hop behaviorPer hop behavior

        Returns: PatternFlowIpv4DscpPhb
        """
        return self._get_property('phb', PatternFlowIpv4DscpPhb)

    @property
    def ecn(self):
        # type: () -> PatternFlowIpv4DscpEcn
        """ecn getter

        Explicit congestion notificationExplicit congestion notification

        Returns: PatternFlowIpv4DscpEcn
        """
        return self._get_property('ecn', PatternFlowIpv4DscpEcn)


class PatternFlowIpv4DscpPhb(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4DscpPhbCounter'},
        'decrement': {'type': 'PatternFlowIpv4DscpPhbCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    DEFAULT = 0 #
    CS1 = 8 #
    CS2 = 16 #
    CS3 = 24 #
    CS4 = 32 #
    CS5 = 40 #
    CS6 = 48 #
    CS7 = 56 #
    AF11 = 10 #
    AF12 = 12 #
    AF13 = 14 #
    AF21 = 18 #
    AF22 = 20 #
    AF23 = 22 #
    AF31 = 26 #
    AF32 = 28 #
    AF33 = 30 #
    AF41 = 34 #
    AF42 = 36 #
    AF43 = 38 #
    EF46 = 46 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4DscpPhb, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4DscpPhbCounter
        """Factory property that returns an instance of the PatternFlowIpv4DscpPhbCounter class

        integer counter pattern

        Returns: PatternFlowIpv4DscpPhbCounter
        """
        return self._get_property('increment', PatternFlowIpv4DscpPhbCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4DscpPhbCounter
        """Factory property that returns an instance of the PatternFlowIpv4DscpPhbCounter class

        integer counter pattern

        Returns: PatternFlowIpv4DscpPhbCounter
        """
        return self._get_property('decrement', PatternFlowIpv4DscpPhbCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4DscpPhbCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    DEFAULT = 0 #
    CS1 = 8 #
    CS2 = 16 #
    CS3 = 24 #
    CS4 = 32 #
    CS5 = 40 #
    CS6 = 48 #
    CS7 = 56 #
    AF11 = 10 #
    AF12 = 12 #
    AF13 = 14 #
    AF21 = 18 #
    AF22 = 20 #
    AF23 = 22 #
    AF31 = 26 #
    AF32 = 28 #
    AF33 = 30 #
    AF41 = 34 #
    AF42 = 36 #
    AF43 = 38 #
    EF46 = 46 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4DscpPhbCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4DscpEcn(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4DscpEcnCounter'},
        'decrement': {'type': 'PatternFlowIpv4DscpEcnCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    NON_CAPABLE = 0 #
    CAPABLE_TRANSPORT_0 = 1 #
    CAPABLE_TRANSPORT_1 = 2 #
    CONGESTION_ENCOUNTERED = 3 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4DscpEcn, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4DscpEcnCounter
        """Factory property that returns an instance of the PatternFlowIpv4DscpEcnCounter class

        integer counter pattern

        Returns: PatternFlowIpv4DscpEcnCounter
        """
        return self._get_property('increment', PatternFlowIpv4DscpEcnCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4DscpEcnCounter
        """Factory property that returns an instance of the PatternFlowIpv4DscpEcnCounter class

        integer counter pattern

        Returns: PatternFlowIpv4DscpEcnCounter
        """
        return self._get_property('decrement', PatternFlowIpv4DscpEcnCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4DscpEcnCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    NON_CAPABLE = 0 #
    CAPABLE_TRANSPORT_0 = 1 #
    CAPABLE_TRANSPORT_1 = 2 #
    CONGESTION_ENCOUNTERED = 3 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4DscpEcnCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4TotalLength(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'auto',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'auto': {
            'type': str,
            'enum': [
                'auto',
            ],
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4TotalLengthCounter'},
        'decrement': {'type': 'PatternFlowIpv4TotalLengthCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'auto',
        'value': 46,
        'values': [46],
        'auto': 'auto',
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    AUTO = 'auto' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    AUTO = 'auto' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4TotalLength, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TotalLengthCounter
        """Factory property that returns an instance of the PatternFlowIpv4TotalLengthCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TotalLengthCounter
        """
        return self._get_property('increment', PatternFlowIpv4TotalLengthCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TotalLengthCounter
        """Factory property that returns an instance of the PatternFlowIpv4TotalLengthCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TotalLengthCounter
        """
        return self._get_property('decrement', PatternFlowIpv4TotalLengthCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def auto(self):
        # type: () -> Union[Literal["auto"]]
        """auto getter

        TBD

        Returns: Union[Literal["auto"]]
        """
        return self._get_property('auto')

    @auto.setter
    def auto(self, value):
        """auto setter

        TBD

        value: Union[Literal["auto"]]
        """
        self._set_property('auto', value, 'auto')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4TotalLengthCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 46,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4TotalLengthCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4Identification(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4IdentificationCounter'},
        'decrement': {'type': 'PatternFlowIpv4IdentificationCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4Identification, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4IdentificationCounter
        """Factory property that returns an instance of the PatternFlowIpv4IdentificationCounter class

        integer counter pattern

        Returns: PatternFlowIpv4IdentificationCounter
        """
        return self._get_property('increment', PatternFlowIpv4IdentificationCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4IdentificationCounter
        """Factory property that returns an instance of the PatternFlowIpv4IdentificationCounter class

        integer counter pattern

        Returns: PatternFlowIpv4IdentificationCounter
        """
        return self._get_property('decrement', PatternFlowIpv4IdentificationCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4IdentificationCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4IdentificationCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4Reserved(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4ReservedCounter'},
        'decrement': {'type': 'PatternFlowIpv4ReservedCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4Reserved, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4ReservedCounter
        """Factory property that returns an instance of the PatternFlowIpv4ReservedCounter class

        integer counter pattern

        Returns: PatternFlowIpv4ReservedCounter
        """
        return self._get_property('increment', PatternFlowIpv4ReservedCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4ReservedCounter
        """Factory property that returns an instance of the PatternFlowIpv4ReservedCounter class

        integer counter pattern

        Returns: PatternFlowIpv4ReservedCounter
        """
        return self._get_property('decrement', PatternFlowIpv4ReservedCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4ReservedCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4ReservedCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4DontFragment(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4DontFragmentCounter'},
        'decrement': {'type': 'PatternFlowIpv4DontFragmentCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4DontFragment, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4DontFragmentCounter
        """Factory property that returns an instance of the PatternFlowIpv4DontFragmentCounter class

        integer counter pattern

        Returns: PatternFlowIpv4DontFragmentCounter
        """
        return self._get_property('increment', PatternFlowIpv4DontFragmentCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4DontFragmentCounter
        """Factory property that returns an instance of the PatternFlowIpv4DontFragmentCounter class

        integer counter pattern

        Returns: PatternFlowIpv4DontFragmentCounter
        """
        return self._get_property('decrement', PatternFlowIpv4DontFragmentCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4DontFragmentCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4DontFragmentCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4MoreFragments(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4MoreFragmentsCounter'},
        'decrement': {'type': 'PatternFlowIpv4MoreFragmentsCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4MoreFragments, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4MoreFragmentsCounter
        """Factory property that returns an instance of the PatternFlowIpv4MoreFragmentsCounter class

        integer counter pattern

        Returns: PatternFlowIpv4MoreFragmentsCounter
        """
        return self._get_property('increment', PatternFlowIpv4MoreFragmentsCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4MoreFragmentsCounter
        """Factory property that returns an instance of the PatternFlowIpv4MoreFragmentsCounter class

        integer counter pattern

        Returns: PatternFlowIpv4MoreFragmentsCounter
        """
        return self._get_property('decrement', PatternFlowIpv4MoreFragmentsCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4MoreFragmentsCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4MoreFragmentsCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4FragmentOffset(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4FragmentOffsetCounter'},
        'decrement': {'type': 'PatternFlowIpv4FragmentOffsetCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4FragmentOffset, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4FragmentOffsetCounter
        """Factory property that returns an instance of the PatternFlowIpv4FragmentOffsetCounter class

        integer counter pattern

        Returns: PatternFlowIpv4FragmentOffsetCounter
        """
        return self._get_property('increment', PatternFlowIpv4FragmentOffsetCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4FragmentOffsetCounter
        """Factory property that returns an instance of the PatternFlowIpv4FragmentOffsetCounter class

        integer counter pattern

        Returns: PatternFlowIpv4FragmentOffsetCounter
        """
        return self._get_property('decrement', PatternFlowIpv4FragmentOffsetCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4FragmentOffsetCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4FragmentOffsetCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4TimeToLive(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4TimeToLiveCounter'},
        'decrement': {'type': 'PatternFlowIpv4TimeToLiveCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 64,
        'values': [64],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4TimeToLive, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TimeToLiveCounter
        """Factory property that returns an instance of the PatternFlowIpv4TimeToLiveCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TimeToLiveCounter
        """
        return self._get_property('increment', PatternFlowIpv4TimeToLiveCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TimeToLiveCounter
        """Factory property that returns an instance of the PatternFlowIpv4TimeToLiveCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TimeToLiveCounter
        """
        return self._get_property('decrement', PatternFlowIpv4TimeToLiveCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4TimeToLiveCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 64,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4TimeToLiveCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4Protocol(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4ProtocolCounter'},
        'decrement': {'type': 'PatternFlowIpv4ProtocolCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 61,
        'values': [61],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4Protocol, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4ProtocolCounter
        """Factory property that returns an instance of the PatternFlowIpv4ProtocolCounter class

        integer counter pattern

        Returns: PatternFlowIpv4ProtocolCounter
        """
        return self._get_property('increment', PatternFlowIpv4ProtocolCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4ProtocolCounter
        """Factory property that returns an instance of the PatternFlowIpv4ProtocolCounter class

        integer counter pattern

        Returns: PatternFlowIpv4ProtocolCounter
        """
        return self._get_property('decrement', PatternFlowIpv4ProtocolCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4ProtocolCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 61,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4ProtocolCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4HeaderChecksum(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'generated',
                'custom',
            ],
        },
        'generated': {
            'type': str,
            'enum': [
                'good',
                'bad',
            ],
        },
        'custom': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED = ('choice',) # type: tuple(str)

    _DEFAULTS = {
        'choice': 'generated',
        'generated': 'good',
    } # type: Dict[str, Union(type)]

    GENERATED = 'generated' # type: str
    CUSTOM = 'custom' # type: str

    GOOD = 'good' # type: str
    BAD = 'bad' # type: str

    def __init__(self, parent=None, choice=None):
        super(PatternFlowIpv4HeaderChecksum, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def choice(self):
        # type: () -> Union[Literal["custom"], Literal["generated"]]
        """choice getter

        The type of checksum

        Returns: Union[Literal["custom"], Literal["generated"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of checksum

        value: Union[Literal["custom"], Literal["generated"]]
        """
        self._set_property('choice', value)

    @property
    def generated(self):
        # type: () -> Union[Literal["bad"], Literal["good"]]
        """generated getter

        A system generated checksum value

        Returns: Union[Literal["bad"], Literal["good"]]
        """
        return self._get_property('generated')

    @generated.setter
    def generated(self, value):
        """generated setter

        A system generated checksum value

        value: Union[Literal["bad"], Literal["good"]]
        """
        self._set_property('generated', value, 'generated')

    @property
    def custom(self):
        # type: () -> int
        """custom getter

        A custom checksum value

        Returns: int
        """
        return self._get_property('custom')

    @custom.setter
    def custom(self, value):
        """custom setter

        A custom checksum value

        value: int
        """
        self._set_property('custom', value, 'custom')


class PatternFlowIpv4Src(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {
            'type': str,
            'format': 'ipv4',
        },
        'values': {
            'type': list,
            'format': 'ipv4',
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4SrcCounter'},
        'decrement': {'type': 'PatternFlowIpv4SrcCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': '0.0.0.0',
        'values': ['0.0.0.0'],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4Src, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4SrcCounter
        """Factory property that returns an instance of the PatternFlowIpv4SrcCounter class

        ipv4 counter pattern

        Returns: PatternFlowIpv4SrcCounter
        """
        return self._get_property('increment', PatternFlowIpv4SrcCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4SrcCounter
        """Factory property that returns an instance of the PatternFlowIpv4SrcCounter class

        ipv4 counter pattern

        Returns: PatternFlowIpv4SrcCounter
        """
        return self._get_property('decrement', PatternFlowIpv4SrcCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4SrcCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {
            'type': str,
            'format': 'ipv4',
        },
        'step': {
            'type': str,
            'format': 'ipv4',
        },
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': '0.0.0.0',
        'step': '0.0.0.1',
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4SrcCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv4Dst(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {
            'type': str,
            'format': 'ipv4',
        },
        'values': {
            'type': list,
            'format': 'ipv4',
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv4DstCounter'},
        'decrement': {'type': 'PatternFlowIpv4DstCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': '0.0.0.0',
        'values': ['0.0.0.0'],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv4Dst, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4DstCounter
        """Factory property that returns an instance of the PatternFlowIpv4DstCounter class

        ipv4 counter pattern

        Returns: PatternFlowIpv4DstCounter
        """
        return self._get_property('increment', PatternFlowIpv4DstCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4DstCounter
        """Factory property that returns an instance of the PatternFlowIpv4DstCounter class

        ipv4 counter pattern

        Returns: PatternFlowIpv4DstCounter
        """
        return self._get_property('decrement', PatternFlowIpv4DstCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv4DstCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {
            'type': str,
            'format': 'ipv4',
        },
        'step': {
            'type': str,
            'format': 'ipv4',
        },
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': '0.0.0.0',
        'step': '0.0.0.1',
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv4DstCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class FlowIpv6(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'version': {'type': 'PatternFlowIpv6Version'},
        'traffic_class': {'type': 'PatternFlowIpv6TrafficClass'},
        'flow_label': {'type': 'PatternFlowIpv6FlowLabel'},
        'payload_length': {'type': 'PatternFlowIpv6PayloadLength'},
        'next_header': {'type': 'PatternFlowIpv6NextHeader'},
        'hop_limit': {'type': 'PatternFlowIpv6HopLimit'},
        'src': {'type': 'PatternFlowIpv6Src'},
        'dst': {'type': 'PatternFlowIpv6Dst'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowIpv6, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def version(self):
        # type: () -> PatternFlowIpv6Version
        """version getter

        Version numberVersion number

        Returns: PatternFlowIpv6Version
        """
        return self._get_property('version', PatternFlowIpv6Version)

    @property
    def traffic_class(self):
        # type: () -> PatternFlowIpv6TrafficClass
        """traffic_class getter

        Traffic classTraffic class

        Returns: PatternFlowIpv6TrafficClass
        """
        return self._get_property('traffic_class', PatternFlowIpv6TrafficClass)

    @property
    def flow_label(self):
        # type: () -> PatternFlowIpv6FlowLabel
        """flow_label getter

        Flow labelFlow label

        Returns: PatternFlowIpv6FlowLabel
        """
        return self._get_property('flow_label', PatternFlowIpv6FlowLabel)

    @property
    def payload_length(self):
        # type: () -> PatternFlowIpv6PayloadLength
        """payload_length getter

        Payload lengthPayload length

        Returns: PatternFlowIpv6PayloadLength
        """
        return self._get_property('payload_length', PatternFlowIpv6PayloadLength)

    @property
    def next_header(self):
        # type: () -> PatternFlowIpv6NextHeader
        """next_header getter

        Next headerNext header

        Returns: PatternFlowIpv6NextHeader
        """
        return self._get_property('next_header', PatternFlowIpv6NextHeader)

    @property
    def hop_limit(self):
        # type: () -> PatternFlowIpv6HopLimit
        """hop_limit getter

        Hop limitHop limit

        Returns: PatternFlowIpv6HopLimit
        """
        return self._get_property('hop_limit', PatternFlowIpv6HopLimit)

    @property
    def src(self):
        # type: () -> PatternFlowIpv6Src
        """src getter

        Source addressSource address

        Returns: PatternFlowIpv6Src
        """
        return self._get_property('src', PatternFlowIpv6Src)

    @property
    def dst(self):
        # type: () -> PatternFlowIpv6Dst
        """dst getter

        Destination addressDestination address

        Returns: PatternFlowIpv6Dst
        """
        return self._get_property('dst', PatternFlowIpv6Dst)


class PatternFlowIpv6Version(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv6VersionCounter'},
        'decrement': {'type': 'PatternFlowIpv6VersionCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 6,
        'values': [6],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv6Version, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6VersionCounter
        """Factory property that returns an instance of the PatternFlowIpv6VersionCounter class

        integer counter pattern

        Returns: PatternFlowIpv6VersionCounter
        """
        return self._get_property('increment', PatternFlowIpv6VersionCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6VersionCounter
        """Factory property that returns an instance of the PatternFlowIpv6VersionCounter class

        integer counter pattern

        Returns: PatternFlowIpv6VersionCounter
        """
        return self._get_property('decrement', PatternFlowIpv6VersionCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv6VersionCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 6,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv6VersionCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv6TrafficClass(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv6TrafficClassCounter'},
        'decrement': {'type': 'PatternFlowIpv6TrafficClassCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv6TrafficClass, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6TrafficClassCounter
        """Factory property that returns an instance of the PatternFlowIpv6TrafficClassCounter class

        integer counter pattern

        Returns: PatternFlowIpv6TrafficClassCounter
        """
        return self._get_property('increment', PatternFlowIpv6TrafficClassCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6TrafficClassCounter
        """Factory property that returns an instance of the PatternFlowIpv6TrafficClassCounter class

        integer counter pattern

        Returns: PatternFlowIpv6TrafficClassCounter
        """
        return self._get_property('decrement', PatternFlowIpv6TrafficClassCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv6TrafficClassCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv6TrafficClassCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv6FlowLabel(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv6FlowLabelCounter'},
        'decrement': {'type': 'PatternFlowIpv6FlowLabelCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv6FlowLabel, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6FlowLabelCounter
        """Factory property that returns an instance of the PatternFlowIpv6FlowLabelCounter class

        integer counter pattern

        Returns: PatternFlowIpv6FlowLabelCounter
        """
        return self._get_property('increment', PatternFlowIpv6FlowLabelCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6FlowLabelCounter
        """Factory property that returns an instance of the PatternFlowIpv6FlowLabelCounter class

        integer counter pattern

        Returns: PatternFlowIpv6FlowLabelCounter
        """
        return self._get_property('decrement', PatternFlowIpv6FlowLabelCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv6FlowLabelCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv6FlowLabelCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv6PayloadLength(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'auto',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'auto': {
            'type': str,
            'enum': [
                'auto',
            ],
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv6PayloadLengthCounter'},
        'decrement': {'type': 'PatternFlowIpv6PayloadLengthCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'auto',
        'value': 0,
        'values': [0],
        'auto': 'auto',
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    AUTO = 'auto' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    AUTO = 'auto' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv6PayloadLength, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6PayloadLengthCounter
        """Factory property that returns an instance of the PatternFlowIpv6PayloadLengthCounter class

        integer counter pattern

        Returns: PatternFlowIpv6PayloadLengthCounter
        """
        return self._get_property('increment', PatternFlowIpv6PayloadLengthCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6PayloadLengthCounter
        """Factory property that returns an instance of the PatternFlowIpv6PayloadLengthCounter class

        integer counter pattern

        Returns: PatternFlowIpv6PayloadLengthCounter
        """
        return self._get_property('decrement', PatternFlowIpv6PayloadLengthCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def auto(self):
        # type: () -> Union[Literal["auto"]]
        """auto getter

        TBD

        Returns: Union[Literal["auto"]]
        """
        return self._get_property('auto')

    @auto.setter
    def auto(self, value):
        """auto setter

        TBD

        value: Union[Literal["auto"]]
        """
        self._set_property('auto', value, 'auto')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv6PayloadLengthCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv6PayloadLengthCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv6NextHeader(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv6NextHeaderCounter'},
        'decrement': {'type': 'PatternFlowIpv6NextHeaderCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 59,
        'values': [59],
    } # type: Dict[str, Union(type)]

    HOPOPT = 0 #
    ICMP = 1 #
    IGMP = 2 #
    GGP = 3 #
    IP_IN_IP = 4 #
    ST = 5 #
    TCP = 6 #
    CPT = 7 #
    EGP = 8 #
    IGP = 9 #
    NO_NEXT_HEADER = 59 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv6NextHeader, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6NextHeaderCounter
        """Factory property that returns an instance of the PatternFlowIpv6NextHeaderCounter class

        integer counter pattern

        Returns: PatternFlowIpv6NextHeaderCounter
        """
        return self._get_property('increment', PatternFlowIpv6NextHeaderCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6NextHeaderCounter
        """Factory property that returns an instance of the PatternFlowIpv6NextHeaderCounter class

        integer counter pattern

        Returns: PatternFlowIpv6NextHeaderCounter
        """
        return self._get_property('decrement', PatternFlowIpv6NextHeaderCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv6NextHeaderCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 59,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    HOPOPT = 0 #
    ICMP = 1 #
    IGMP = 2 #
    GGP = 3 #
    IP_IN_IP = 4 #
    ST = 5 #
    TCP = 6 #
    CPT = 7 #
    EGP = 8 #
    IGP = 9 #
    NO_NEXT_HEADER = 59 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv6NextHeaderCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv6HopLimit(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv6HopLimitCounter'},
        'decrement': {'type': 'PatternFlowIpv6HopLimitCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 64,
        'values': [64],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv6HopLimit, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6HopLimitCounter
        """Factory property that returns an instance of the PatternFlowIpv6HopLimitCounter class

        integer counter pattern

        Returns: PatternFlowIpv6HopLimitCounter
        """
        return self._get_property('increment', PatternFlowIpv6HopLimitCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6HopLimitCounter
        """Factory property that returns an instance of the PatternFlowIpv6HopLimitCounter class

        integer counter pattern

        Returns: PatternFlowIpv6HopLimitCounter
        """
        return self._get_property('decrement', PatternFlowIpv6HopLimitCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv6HopLimitCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 64,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv6HopLimitCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv6Src(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {
            'type': str,
            'format': 'ipv6',
        },
        'values': {
            'type': list,
            'format': 'ipv6',
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv6SrcCounter'},
        'decrement': {'type': 'PatternFlowIpv6SrcCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': '::0',
        'values': ['::0'],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv6Src, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6SrcCounter
        """Factory property that returns an instance of the PatternFlowIpv6SrcCounter class

        ipv6 counter pattern

        Returns: PatternFlowIpv6SrcCounter
        """
        return self._get_property('increment', PatternFlowIpv6SrcCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6SrcCounter
        """Factory property that returns an instance of the PatternFlowIpv6SrcCounter class

        ipv6 counter pattern

        Returns: PatternFlowIpv6SrcCounter
        """
        return self._get_property('decrement', PatternFlowIpv6SrcCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv6SrcCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {
            'type': str,
            'format': 'ipv6',
        },
        'step': {
            'type': str,
            'format': 'ipv6',
        },
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': '::0',
        'step': '::1',
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv6SrcCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowIpv6Dst(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {
            'type': str,
            'format': 'ipv6',
        },
        'values': {
            'type': list,
            'format': 'ipv6',
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowIpv6DstCounter'},
        'decrement': {'type': 'PatternFlowIpv6DstCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': '::0',
        'values': ['::0'],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowIpv6Dst, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6DstCounter
        """Factory property that returns an instance of the PatternFlowIpv6DstCounter class

        ipv6 counter pattern

        Returns: PatternFlowIpv6DstCounter
        """
        return self._get_property('increment', PatternFlowIpv6DstCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6DstCounter
        """Factory property that returns an instance of the PatternFlowIpv6DstCounter class

        ipv6 counter pattern

        Returns: PatternFlowIpv6DstCounter
        """
        return self._get_property('decrement', PatternFlowIpv6DstCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowIpv6DstCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {
            'type': str,
            'format': 'ipv6',
        },
        'step': {
            'type': str,
            'format': 'ipv6',
        },
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': '::0',
        'step': '::1',
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowIpv6DstCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class FlowPfcPause(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'dst': {'type': 'PatternFlowPfcPauseDst'},
        'src': {'type': 'PatternFlowPfcPauseSrc'},
        'ether_type': {'type': 'PatternFlowPfcPauseEtherType'},
        'control_op_code': {'type': 'PatternFlowPfcPauseControlOpCode'},
        'class_enable_vector': {'type': 'PatternFlowPfcPauseClassEnableVector'},
        'pause_class_0': {'type': 'PatternFlowPfcPausePauseClass0'},
        'pause_class_1': {'type': 'PatternFlowPfcPausePauseClass1'},
        'pause_class_2': {'type': 'PatternFlowPfcPausePauseClass2'},
        'pause_class_3': {'type': 'PatternFlowPfcPausePauseClass3'},
        'pause_class_4': {'type': 'PatternFlowPfcPausePauseClass4'},
        'pause_class_5': {'type': 'PatternFlowPfcPausePauseClass5'},
        'pause_class_6': {'type': 'PatternFlowPfcPausePauseClass6'},
        'pause_class_7': {'type': 'PatternFlowPfcPausePauseClass7'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowPfcPause, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def dst(self):
        # type: () -> PatternFlowPfcPauseDst
        """dst getter

        Destination MAC addressDestination MAC address

        Returns: PatternFlowPfcPauseDst
        """
        return self._get_property('dst', PatternFlowPfcPauseDst)

    @property
    def src(self):
        # type: () -> PatternFlowPfcPauseSrc
        """src getter

        Source MAC addressSource MAC address

        Returns: PatternFlowPfcPauseSrc
        """
        return self._get_property('src', PatternFlowPfcPauseSrc)

    @property
    def ether_type(self):
        # type: () -> PatternFlowPfcPauseEtherType
        """ether_type getter

        Ethernet typeEthernet type

        Returns: PatternFlowPfcPauseEtherType
        """
        return self._get_property('ether_type', PatternFlowPfcPauseEtherType)

    @property
    def control_op_code(self):
        # type: () -> PatternFlowPfcPauseControlOpCode
        """control_op_code getter

        Control operation codeControl operation code

        Returns: PatternFlowPfcPauseControlOpCode
        """
        return self._get_property('control_op_code', PatternFlowPfcPauseControlOpCode)

    @property
    def class_enable_vector(self):
        # type: () -> PatternFlowPfcPauseClassEnableVector
        """class_enable_vector getter

        DestinationDestination

        Returns: PatternFlowPfcPauseClassEnableVector
        """
        return self._get_property('class_enable_vector', PatternFlowPfcPauseClassEnableVector)

    @property
    def pause_class_0(self):
        # type: () -> PatternFlowPfcPausePauseClass0
        """pause_class_0 getter

        Pause class 0Pause class 0

        Returns: PatternFlowPfcPausePauseClass0
        """
        return self._get_property('pause_class_0', PatternFlowPfcPausePauseClass0)

    @property
    def pause_class_1(self):
        # type: () -> PatternFlowPfcPausePauseClass1
        """pause_class_1 getter

        Pause class 1Pause class 1

        Returns: PatternFlowPfcPausePauseClass1
        """
        return self._get_property('pause_class_1', PatternFlowPfcPausePauseClass1)

    @property
    def pause_class_2(self):
        # type: () -> PatternFlowPfcPausePauseClass2
        """pause_class_2 getter

        Pause class 2Pause class 2

        Returns: PatternFlowPfcPausePauseClass2
        """
        return self._get_property('pause_class_2', PatternFlowPfcPausePauseClass2)

    @property
    def pause_class_3(self):
        # type: () -> PatternFlowPfcPausePauseClass3
        """pause_class_3 getter

        Pause class 3Pause class 3

        Returns: PatternFlowPfcPausePauseClass3
        """
        return self._get_property('pause_class_3', PatternFlowPfcPausePauseClass3)

    @property
    def pause_class_4(self):
        # type: () -> PatternFlowPfcPausePauseClass4
        """pause_class_4 getter

        Pause class 4Pause class 4

        Returns: PatternFlowPfcPausePauseClass4
        """
        return self._get_property('pause_class_4', PatternFlowPfcPausePauseClass4)

    @property
    def pause_class_5(self):
        # type: () -> PatternFlowPfcPausePauseClass5
        """pause_class_5 getter

        Pause class 5Pause class 5

        Returns: PatternFlowPfcPausePauseClass5
        """
        return self._get_property('pause_class_5', PatternFlowPfcPausePauseClass5)

    @property
    def pause_class_6(self):
        # type: () -> PatternFlowPfcPausePauseClass6
        """pause_class_6 getter

        Pause class 6Pause class 6

        Returns: PatternFlowPfcPausePauseClass6
        """
        return self._get_property('pause_class_6', PatternFlowPfcPausePauseClass6)

    @property
    def pause_class_7(self):
        # type: () -> PatternFlowPfcPausePauseClass7
        """pause_class_7 getter

        Pause class 7Pause class 7

        Returns: PatternFlowPfcPausePauseClass7
        """
        return self._get_property('pause_class_7', PatternFlowPfcPausePauseClass7)


class PatternFlowPfcPauseDst(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {
            'type': str,
            'format': 'mac',
        },
        'values': {
            'type': list,
            'format': 'mac',
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowPfcPauseDstCounter'},
        'decrement': {'type': 'PatternFlowPfcPauseDstCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': '00:00:00:00:00:00',
        'values': ['00:00:00:00:00:00'],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowPfcPauseDst, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPauseDstCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseDstCounter class

        mac counter pattern

        Returns: PatternFlowPfcPauseDstCounter
        """
        return self._get_property('increment', PatternFlowPfcPauseDstCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPauseDstCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseDstCounter class

        mac counter pattern

        Returns: PatternFlowPfcPauseDstCounter
        """
        return self._get_property('decrement', PatternFlowPfcPauseDstCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowPfcPauseDstCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {
            'type': str,
            'format': 'mac',
        },
        'step': {
            'type': str,
            'format': 'mac',
        },
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': '00:00:00:00:00:00',
        'step': '00:00:00:00:00:01',
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowPfcPauseDstCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowPfcPauseSrc(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {
            'type': str,
            'format': 'mac',
        },
        'values': {
            'type': list,
            'format': 'mac',
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowPfcPauseSrcCounter'},
        'decrement': {'type': 'PatternFlowPfcPauseSrcCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': '00:00:00:00:00:00',
        'values': ['00:00:00:00:00:00'],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowPfcPauseSrc, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPauseSrcCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseSrcCounter class

        mac counter pattern

        Returns: PatternFlowPfcPauseSrcCounter
        """
        return self._get_property('increment', PatternFlowPfcPauseSrcCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPauseSrcCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseSrcCounter class

        mac counter pattern

        Returns: PatternFlowPfcPauseSrcCounter
        """
        return self._get_property('decrement', PatternFlowPfcPauseSrcCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowPfcPauseSrcCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {
            'type': str,
            'format': 'mac',
        },
        'step': {
            'type': str,
            'format': 'mac',
        },
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': '00:00:00:00:00:00',
        'step': '00:00:00:00:00:01',
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowPfcPauseSrcCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowPfcPauseEtherType(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowPfcPauseEtherTypeCounter'},
        'decrement': {'type': 'PatternFlowPfcPauseEtherTypeCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 34824,
        'values': [34824],
    } # type: Dict[str, Union(type)]

    X8808 = 34824 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowPfcPauseEtherType, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPauseEtherTypeCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseEtherTypeCounter class

        integer counter pattern

        Returns: PatternFlowPfcPauseEtherTypeCounter
        """
        return self._get_property('increment', PatternFlowPfcPauseEtherTypeCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPauseEtherTypeCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseEtherTypeCounter class

        integer counter pattern

        Returns: PatternFlowPfcPauseEtherTypeCounter
        """
        return self._get_property('decrement', PatternFlowPfcPauseEtherTypeCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowPfcPauseEtherTypeCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 34824,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    X8808 = 34824 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowPfcPauseEtherTypeCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowPfcPauseControlOpCode(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowPfcPauseControlOpCodeCounter'},
        'decrement': {'type': 'PatternFlowPfcPauseControlOpCodeCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 257,
        'values': [257],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowPfcPauseControlOpCode, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPauseControlOpCodeCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseControlOpCodeCounter class

        integer counter pattern

        Returns: PatternFlowPfcPauseControlOpCodeCounter
        """
        return self._get_property('increment', PatternFlowPfcPauseControlOpCodeCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPauseControlOpCodeCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseControlOpCodeCounter class

        integer counter pattern

        Returns: PatternFlowPfcPauseControlOpCodeCounter
        """
        return self._get_property('decrement', PatternFlowPfcPauseControlOpCodeCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowPfcPauseControlOpCodeCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 257,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowPfcPauseControlOpCodeCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowPfcPauseClassEnableVector(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowPfcPauseClassEnableVectorCounter'},
        'decrement': {'type': 'PatternFlowPfcPauseClassEnableVectorCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowPfcPauseClassEnableVector, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPauseClassEnableVectorCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseClassEnableVectorCounter class

        integer counter pattern

        Returns: PatternFlowPfcPauseClassEnableVectorCounter
        """
        return self._get_property('increment', PatternFlowPfcPauseClassEnableVectorCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPauseClassEnableVectorCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseClassEnableVectorCounter class

        integer counter pattern

        Returns: PatternFlowPfcPauseClassEnableVectorCounter
        """
        return self._get_property('decrement', PatternFlowPfcPauseClassEnableVectorCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowPfcPauseClassEnableVectorCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowPfcPauseClassEnableVectorCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowPfcPausePauseClass0(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowPfcPausePauseClass0Counter'},
        'decrement': {'type': 'PatternFlowPfcPausePauseClass0Counter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowPfcPausePauseClass0, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass0Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass0Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass0Counter
        """
        return self._get_property('increment', PatternFlowPfcPausePauseClass0Counter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass0Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass0Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass0Counter
        """
        return self._get_property('decrement', PatternFlowPfcPausePauseClass0Counter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowPfcPausePauseClass0Counter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowPfcPausePauseClass0Counter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowPfcPausePauseClass1(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowPfcPausePauseClass1Counter'},
        'decrement': {'type': 'PatternFlowPfcPausePauseClass1Counter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowPfcPausePauseClass1, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass1Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass1Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass1Counter
        """
        return self._get_property('increment', PatternFlowPfcPausePauseClass1Counter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass1Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass1Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass1Counter
        """
        return self._get_property('decrement', PatternFlowPfcPausePauseClass1Counter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowPfcPausePauseClass1Counter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowPfcPausePauseClass1Counter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowPfcPausePauseClass2(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowPfcPausePauseClass2Counter'},
        'decrement': {'type': 'PatternFlowPfcPausePauseClass2Counter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowPfcPausePauseClass2, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass2Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass2Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass2Counter
        """
        return self._get_property('increment', PatternFlowPfcPausePauseClass2Counter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass2Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass2Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass2Counter
        """
        return self._get_property('decrement', PatternFlowPfcPausePauseClass2Counter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowPfcPausePauseClass2Counter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowPfcPausePauseClass2Counter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowPfcPausePauseClass3(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowPfcPausePauseClass3Counter'},
        'decrement': {'type': 'PatternFlowPfcPausePauseClass3Counter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowPfcPausePauseClass3, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass3Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass3Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass3Counter
        """
        return self._get_property('increment', PatternFlowPfcPausePauseClass3Counter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass3Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass3Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass3Counter
        """
        return self._get_property('decrement', PatternFlowPfcPausePauseClass3Counter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowPfcPausePauseClass3Counter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowPfcPausePauseClass3Counter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowPfcPausePauseClass4(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowPfcPausePauseClass4Counter'},
        'decrement': {'type': 'PatternFlowPfcPausePauseClass4Counter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowPfcPausePauseClass4, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass4Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass4Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass4Counter
        """
        return self._get_property('increment', PatternFlowPfcPausePauseClass4Counter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass4Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass4Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass4Counter
        """
        return self._get_property('decrement', PatternFlowPfcPausePauseClass4Counter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowPfcPausePauseClass4Counter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowPfcPausePauseClass4Counter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowPfcPausePauseClass5(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowPfcPausePauseClass5Counter'},
        'decrement': {'type': 'PatternFlowPfcPausePauseClass5Counter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowPfcPausePauseClass5, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass5Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass5Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass5Counter
        """
        return self._get_property('increment', PatternFlowPfcPausePauseClass5Counter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass5Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass5Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass5Counter
        """
        return self._get_property('decrement', PatternFlowPfcPausePauseClass5Counter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowPfcPausePauseClass5Counter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowPfcPausePauseClass5Counter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowPfcPausePauseClass6(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowPfcPausePauseClass6Counter'},
        'decrement': {'type': 'PatternFlowPfcPausePauseClass6Counter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowPfcPausePauseClass6, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass6Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass6Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass6Counter
        """
        return self._get_property('increment', PatternFlowPfcPausePauseClass6Counter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass6Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass6Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass6Counter
        """
        return self._get_property('decrement', PatternFlowPfcPausePauseClass6Counter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowPfcPausePauseClass6Counter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowPfcPausePauseClass6Counter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowPfcPausePauseClass7(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowPfcPausePauseClass7Counter'},
        'decrement': {'type': 'PatternFlowPfcPausePauseClass7Counter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowPfcPausePauseClass7, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass7Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass7Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass7Counter
        """
        return self._get_property('increment', PatternFlowPfcPausePauseClass7Counter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass7Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass7Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass7Counter
        """
        return self._get_property('decrement', PatternFlowPfcPausePauseClass7Counter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowPfcPausePauseClass7Counter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowPfcPausePauseClass7Counter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class FlowEthernetPause(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'dst': {'type': 'PatternFlowEthernetPauseDst'},
        'src': {'type': 'PatternFlowEthernetPauseSrc'},
        'ether_type': {'type': 'PatternFlowEthernetPauseEtherType'},
        'control_op_code': {'type': 'PatternFlowEthernetPauseControlOpCode'},
        'time': {'type': 'PatternFlowEthernetPauseTime'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowEthernetPause, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def dst(self):
        # type: () -> PatternFlowEthernetPauseDst
        """dst getter

        Destination MAC addressDestination MAC address

        Returns: PatternFlowEthernetPauseDst
        """
        return self._get_property('dst', PatternFlowEthernetPauseDst)

    @property
    def src(self):
        # type: () -> PatternFlowEthernetPauseSrc
        """src getter

        Source MAC addressSource MAC address

        Returns: PatternFlowEthernetPauseSrc
        """
        return self._get_property('src', PatternFlowEthernetPauseSrc)

    @property
    def ether_type(self):
        # type: () -> PatternFlowEthernetPauseEtherType
        """ether_type getter

        Ethernet typeEthernet type

        Returns: PatternFlowEthernetPauseEtherType
        """
        return self._get_property('ether_type', PatternFlowEthernetPauseEtherType)

    @property
    def control_op_code(self):
        # type: () -> PatternFlowEthernetPauseControlOpCode
        """control_op_code getter

        Control operation codeControl operation code

        Returns: PatternFlowEthernetPauseControlOpCode
        """
        return self._get_property('control_op_code', PatternFlowEthernetPauseControlOpCode)

    @property
    def time(self):
        # type: () -> PatternFlowEthernetPauseTime
        """time getter

        TimeTime

        Returns: PatternFlowEthernetPauseTime
        """
        return self._get_property('time', PatternFlowEthernetPauseTime)


class PatternFlowEthernetPauseDst(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {
            'type': str,
            'format': 'mac',
        },
        'values': {
            'type': list,
            'format': 'mac',
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowEthernetPauseDstCounter'},
        'decrement': {'type': 'PatternFlowEthernetPauseDstCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': '00:00:00:00:00:00',
        'values': ['00:00:00:00:00:00'],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowEthernetPauseDst, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowEthernetPauseDstCounter
        """Factory property that returns an instance of the PatternFlowEthernetPauseDstCounter class

        mac counter pattern

        Returns: PatternFlowEthernetPauseDstCounter
        """
        return self._get_property('increment', PatternFlowEthernetPauseDstCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowEthernetPauseDstCounter
        """Factory property that returns an instance of the PatternFlowEthernetPauseDstCounter class

        mac counter pattern

        Returns: PatternFlowEthernetPauseDstCounter
        """
        return self._get_property('decrement', PatternFlowEthernetPauseDstCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowEthernetPauseDstCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {
            'type': str,
            'format': 'mac',
        },
        'step': {
            'type': str,
            'format': 'mac',
        },
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': '00:00:00:00:00:00',
        'step': '00:00:00:00:00:01',
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowEthernetPauseDstCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowEthernetPauseSrc(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {
            'type': str,
            'format': 'mac',
        },
        'values': {
            'type': list,
            'format': 'mac',
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowEthernetPauseSrcCounter'},
        'decrement': {'type': 'PatternFlowEthernetPauseSrcCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': '00:00:00:00:00:00',
        'values': ['00:00:00:00:00:00'],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowEthernetPauseSrc, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowEthernetPauseSrcCounter
        """Factory property that returns an instance of the PatternFlowEthernetPauseSrcCounter class

        mac counter pattern

        Returns: PatternFlowEthernetPauseSrcCounter
        """
        return self._get_property('increment', PatternFlowEthernetPauseSrcCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowEthernetPauseSrcCounter
        """Factory property that returns an instance of the PatternFlowEthernetPauseSrcCounter class

        mac counter pattern

        Returns: PatternFlowEthernetPauseSrcCounter
        """
        return self._get_property('decrement', PatternFlowEthernetPauseSrcCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowEthernetPauseSrcCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {
            'type': str,
            'format': 'mac',
        },
        'step': {
            'type': str,
            'format': 'mac',
        },
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': '00:00:00:00:00:00',
        'step': '00:00:00:00:00:01',
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowEthernetPauseSrcCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowEthernetPauseEtherType(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowEthernetPauseEtherTypeCounter'},
        'decrement': {'type': 'PatternFlowEthernetPauseEtherTypeCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 34824,
        'values': [34824],
    } # type: Dict[str, Union(type)]

    FLOW_CONTROL = 34824 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowEthernetPauseEtherType, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowEthernetPauseEtherTypeCounter
        """Factory property that returns an instance of the PatternFlowEthernetPauseEtherTypeCounter class

        integer counter pattern

        Returns: PatternFlowEthernetPauseEtherTypeCounter
        """
        return self._get_property('increment', PatternFlowEthernetPauseEtherTypeCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowEthernetPauseEtherTypeCounter
        """Factory property that returns an instance of the PatternFlowEthernetPauseEtherTypeCounter class

        integer counter pattern

        Returns: PatternFlowEthernetPauseEtherTypeCounter
        """
        return self._get_property('decrement', PatternFlowEthernetPauseEtherTypeCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowEthernetPauseEtherTypeCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 34824,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    FLOW_CONTROL = 34824 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowEthernetPauseEtherTypeCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowEthernetPauseControlOpCode(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowEthernetPauseControlOpCodeCounter'},
        'decrement': {'type': 'PatternFlowEthernetPauseControlOpCodeCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 1,
        'values': [1],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowEthernetPauseControlOpCode, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowEthernetPauseControlOpCodeCounter
        """Factory property that returns an instance of the PatternFlowEthernetPauseControlOpCodeCounter class

        integer counter pattern

        Returns: PatternFlowEthernetPauseControlOpCodeCounter
        """
        return self._get_property('increment', PatternFlowEthernetPauseControlOpCodeCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowEthernetPauseControlOpCodeCounter
        """Factory property that returns an instance of the PatternFlowEthernetPauseControlOpCodeCounter class

        integer counter pattern

        Returns: PatternFlowEthernetPauseControlOpCodeCounter
        """
        return self._get_property('decrement', PatternFlowEthernetPauseControlOpCodeCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowEthernetPauseControlOpCodeCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 1,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowEthernetPauseControlOpCodeCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowEthernetPauseTime(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowEthernetPauseTimeCounter'},
        'decrement': {'type': 'PatternFlowEthernetPauseTimeCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowEthernetPauseTime, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowEthernetPauseTimeCounter
        """Factory property that returns an instance of the PatternFlowEthernetPauseTimeCounter class

        integer counter pattern

        Returns: PatternFlowEthernetPauseTimeCounter
        """
        return self._get_property('increment', PatternFlowEthernetPauseTimeCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowEthernetPauseTimeCounter
        """Factory property that returns an instance of the PatternFlowEthernetPauseTimeCounter class

        integer counter pattern

        Returns: PatternFlowEthernetPauseTimeCounter
        """
        return self._get_property('decrement', PatternFlowEthernetPauseTimeCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowEthernetPauseTimeCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowEthernetPauseTimeCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class FlowTcp(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'src_port': {'type': 'PatternFlowTcpSrcPort'},
        'dst_port': {'type': 'PatternFlowTcpDstPort'},
        'seq_num': {'type': 'PatternFlowTcpSeqNum'},
        'ack_num': {'type': 'PatternFlowTcpAckNum'},
        'data_offset': {'type': 'PatternFlowTcpDataOffset'},
        'ecn_ns': {'type': 'PatternFlowTcpEcnNs'},
        'ecn_cwr': {'type': 'PatternFlowTcpEcnCwr'},
        'ecn_echo': {'type': 'PatternFlowTcpEcnEcho'},
        'ctl_urg': {'type': 'PatternFlowTcpCtlUrg'},
        'ctl_ack': {'type': 'PatternFlowTcpCtlAck'},
        'ctl_psh': {'type': 'PatternFlowTcpCtlPsh'},
        'ctl_rst': {'type': 'PatternFlowTcpCtlRst'},
        'ctl_syn': {'type': 'PatternFlowTcpCtlSyn'},
        'ctl_fin': {'type': 'PatternFlowTcpCtlFin'},
        'window': {'type': 'PatternFlowTcpWindow'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowTcp, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def src_port(self):
        # type: () -> PatternFlowTcpSrcPort
        """src_port getter

        Source portSource port

        Returns: PatternFlowTcpSrcPort
        """
        return self._get_property('src_port', PatternFlowTcpSrcPort)

    @property
    def dst_port(self):
        # type: () -> PatternFlowTcpDstPort
        """dst_port getter

        Destination portDestination port

        Returns: PatternFlowTcpDstPort
        """
        return self._get_property('dst_port', PatternFlowTcpDstPort)

    @property
    def seq_num(self):
        # type: () -> PatternFlowTcpSeqNum
        """seq_num getter

        Sequence numberSequence number

        Returns: PatternFlowTcpSeqNum
        """
        return self._get_property('seq_num', PatternFlowTcpSeqNum)

    @property
    def ack_num(self):
        # type: () -> PatternFlowTcpAckNum
        """ack_num getter

        Acknowledgement numberAcknowledgement number

        Returns: PatternFlowTcpAckNum
        """
        return self._get_property('ack_num', PatternFlowTcpAckNum)

    @property
    def data_offset(self):
        # type: () -> PatternFlowTcpDataOffset
        """data_offset getter

        The number of 32 bit words in the TCP header. This indicates where the data begins.The number of 32 bit words in the TCP header. This indicates where the data begins.

        Returns: PatternFlowTcpDataOffset
        """
        return self._get_property('data_offset', PatternFlowTcpDataOffset)

    @property
    def ecn_ns(self):
        # type: () -> PatternFlowTcpEcnNs
        """ecn_ns getter

        Explicit congestion notification, concealment protection.Explicit congestion notification, concealment protection.

        Returns: PatternFlowTcpEcnNs
        """
        return self._get_property('ecn_ns', PatternFlowTcpEcnNs)

    @property
    def ecn_cwr(self):
        # type: () -> PatternFlowTcpEcnCwr
        """ecn_cwr getter

        Explicit congestion notification, congestion window reduced.Explicit congestion notification, congestion window reduced.

        Returns: PatternFlowTcpEcnCwr
        """
        return self._get_property('ecn_cwr', PatternFlowTcpEcnCwr)

    @property
    def ecn_echo(self):
        # type: () -> PatternFlowTcpEcnEcho
        """ecn_echo getter

        Explicit congestion notification, echo. 1 indicates the peer is ecn capable. 0 indicates that a packet with ipv4.ecn = 11 in the ip header was received during normal transmission.Explicit congestion notification, echo. 1 indicates the peer is ecn capable. 0 indicates that a packet with ipv4.ecn = 11 in the ip header was received during normal transmission.

        Returns: PatternFlowTcpEcnEcho
        """
        return self._get_property('ecn_echo', PatternFlowTcpEcnEcho)

    @property
    def ctl_urg(self):
        # type: () -> PatternFlowTcpCtlUrg
        """ctl_urg getter

        A value of 1 indicates that the urgent pointer field is significant.A value of 1 indicates that the urgent pointer field is significant.

        Returns: PatternFlowTcpCtlUrg
        """
        return self._get_property('ctl_urg', PatternFlowTcpCtlUrg)

    @property
    def ctl_ack(self):
        # type: () -> PatternFlowTcpCtlAck
        """ctl_ack getter

        A value of 1 indicates that the ackknowledgment field is significant.A value of 1 indicates that the ackknowledgment field is significant.

        Returns: PatternFlowTcpCtlAck
        """
        return self._get_property('ctl_ack', PatternFlowTcpCtlAck)

    @property
    def ctl_psh(self):
        # type: () -> PatternFlowTcpCtlPsh
        """ctl_psh getter

        Asks to push the buffered data to the receiving application. Asks to push the buffered data to the receiving application. 

        Returns: PatternFlowTcpCtlPsh
        """
        return self._get_property('ctl_psh', PatternFlowTcpCtlPsh)

    @property
    def ctl_rst(self):
        # type: () -> PatternFlowTcpCtlRst
        """ctl_rst getter

        Reset the connection. Reset the connection. 

        Returns: PatternFlowTcpCtlRst
        """
        return self._get_property('ctl_rst', PatternFlowTcpCtlRst)

    @property
    def ctl_syn(self):
        # type: () -> PatternFlowTcpCtlSyn
        """ctl_syn getter

        Synchronize sequenece numbers. Synchronize sequenece numbers. 

        Returns: PatternFlowTcpCtlSyn
        """
        return self._get_property('ctl_syn', PatternFlowTcpCtlSyn)

    @property
    def ctl_fin(self):
        # type: () -> PatternFlowTcpCtlFin
        """ctl_fin getter

        Last packet from the sender. Last packet from the sender. 

        Returns: PatternFlowTcpCtlFin
        """
        return self._get_property('ctl_fin', PatternFlowTcpCtlFin)

    @property
    def window(self):
        # type: () -> PatternFlowTcpWindow
        """window getter

        Tcp connection window.Tcp connection window.

        Returns: PatternFlowTcpWindow
        """
        return self._get_property('window', PatternFlowTcpWindow)


class PatternFlowTcpSrcPort(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowTcpSrcPortCounter'},
        'decrement': {'type': 'PatternFlowTcpSrcPortCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowTcpSrcPort, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowTcpSrcPortCounter
        """Factory property that returns an instance of the PatternFlowTcpSrcPortCounter class

        integer counter pattern

        Returns: PatternFlowTcpSrcPortCounter
        """
        return self._get_property('increment', PatternFlowTcpSrcPortCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowTcpSrcPortCounter
        """Factory property that returns an instance of the PatternFlowTcpSrcPortCounter class

        integer counter pattern

        Returns: PatternFlowTcpSrcPortCounter
        """
        return self._get_property('decrement', PatternFlowTcpSrcPortCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowTcpSrcPortCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowTcpSrcPortCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowTcpDstPort(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowTcpDstPortCounter'},
        'decrement': {'type': 'PatternFlowTcpDstPortCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowTcpDstPort, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowTcpDstPortCounter
        """Factory property that returns an instance of the PatternFlowTcpDstPortCounter class

        integer counter pattern

        Returns: PatternFlowTcpDstPortCounter
        """
        return self._get_property('increment', PatternFlowTcpDstPortCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowTcpDstPortCounter
        """Factory property that returns an instance of the PatternFlowTcpDstPortCounter class

        integer counter pattern

        Returns: PatternFlowTcpDstPortCounter
        """
        return self._get_property('decrement', PatternFlowTcpDstPortCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowTcpDstPortCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowTcpDstPortCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowTcpSeqNum(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowTcpSeqNumCounter'},
        'decrement': {'type': 'PatternFlowTcpSeqNumCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowTcpSeqNum, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowTcpSeqNumCounter
        """Factory property that returns an instance of the PatternFlowTcpSeqNumCounter class

        integer counter pattern

        Returns: PatternFlowTcpSeqNumCounter
        """
        return self._get_property('increment', PatternFlowTcpSeqNumCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowTcpSeqNumCounter
        """Factory property that returns an instance of the PatternFlowTcpSeqNumCounter class

        integer counter pattern

        Returns: PatternFlowTcpSeqNumCounter
        """
        return self._get_property('decrement', PatternFlowTcpSeqNumCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowTcpSeqNumCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowTcpSeqNumCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowTcpAckNum(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowTcpAckNumCounter'},
        'decrement': {'type': 'PatternFlowTcpAckNumCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowTcpAckNum, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowTcpAckNumCounter
        """Factory property that returns an instance of the PatternFlowTcpAckNumCounter class

        integer counter pattern

        Returns: PatternFlowTcpAckNumCounter
        """
        return self._get_property('increment', PatternFlowTcpAckNumCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowTcpAckNumCounter
        """Factory property that returns an instance of the PatternFlowTcpAckNumCounter class

        integer counter pattern

        Returns: PatternFlowTcpAckNumCounter
        """
        return self._get_property('decrement', PatternFlowTcpAckNumCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowTcpAckNumCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowTcpAckNumCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowTcpDataOffset(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowTcpDataOffsetCounter'},
        'decrement': {'type': 'PatternFlowTcpDataOffsetCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowTcpDataOffset, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowTcpDataOffsetCounter
        """Factory property that returns an instance of the PatternFlowTcpDataOffsetCounter class

        integer counter pattern

        Returns: PatternFlowTcpDataOffsetCounter
        """
        return self._get_property('increment', PatternFlowTcpDataOffsetCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowTcpDataOffsetCounter
        """Factory property that returns an instance of the PatternFlowTcpDataOffsetCounter class

        integer counter pattern

        Returns: PatternFlowTcpDataOffsetCounter
        """
        return self._get_property('decrement', PatternFlowTcpDataOffsetCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowTcpDataOffsetCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowTcpDataOffsetCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowTcpEcnNs(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowTcpEcnNsCounter'},
        'decrement': {'type': 'PatternFlowTcpEcnNsCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowTcpEcnNs, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowTcpEcnNsCounter
        """Factory property that returns an instance of the PatternFlowTcpEcnNsCounter class

        integer counter pattern

        Returns: PatternFlowTcpEcnNsCounter
        """
        return self._get_property('increment', PatternFlowTcpEcnNsCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowTcpEcnNsCounter
        """Factory property that returns an instance of the PatternFlowTcpEcnNsCounter class

        integer counter pattern

        Returns: PatternFlowTcpEcnNsCounter
        """
        return self._get_property('decrement', PatternFlowTcpEcnNsCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowTcpEcnNsCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowTcpEcnNsCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowTcpEcnCwr(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowTcpEcnCwrCounter'},
        'decrement': {'type': 'PatternFlowTcpEcnCwrCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowTcpEcnCwr, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowTcpEcnCwrCounter
        """Factory property that returns an instance of the PatternFlowTcpEcnCwrCounter class

        integer counter pattern

        Returns: PatternFlowTcpEcnCwrCounter
        """
        return self._get_property('increment', PatternFlowTcpEcnCwrCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowTcpEcnCwrCounter
        """Factory property that returns an instance of the PatternFlowTcpEcnCwrCounter class

        integer counter pattern

        Returns: PatternFlowTcpEcnCwrCounter
        """
        return self._get_property('decrement', PatternFlowTcpEcnCwrCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowTcpEcnCwrCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowTcpEcnCwrCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowTcpEcnEcho(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowTcpEcnEchoCounter'},
        'decrement': {'type': 'PatternFlowTcpEcnEchoCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowTcpEcnEcho, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowTcpEcnEchoCounter
        """Factory property that returns an instance of the PatternFlowTcpEcnEchoCounter class

        integer counter pattern

        Returns: PatternFlowTcpEcnEchoCounter
        """
        return self._get_property('increment', PatternFlowTcpEcnEchoCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowTcpEcnEchoCounter
        """Factory property that returns an instance of the PatternFlowTcpEcnEchoCounter class

        integer counter pattern

        Returns: PatternFlowTcpEcnEchoCounter
        """
        return self._get_property('decrement', PatternFlowTcpEcnEchoCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowTcpEcnEchoCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowTcpEcnEchoCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowTcpCtlUrg(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowTcpCtlUrgCounter'},
        'decrement': {'type': 'PatternFlowTcpCtlUrgCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowTcpCtlUrg, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowTcpCtlUrgCounter
        """Factory property that returns an instance of the PatternFlowTcpCtlUrgCounter class

        integer counter pattern

        Returns: PatternFlowTcpCtlUrgCounter
        """
        return self._get_property('increment', PatternFlowTcpCtlUrgCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowTcpCtlUrgCounter
        """Factory property that returns an instance of the PatternFlowTcpCtlUrgCounter class

        integer counter pattern

        Returns: PatternFlowTcpCtlUrgCounter
        """
        return self._get_property('decrement', PatternFlowTcpCtlUrgCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowTcpCtlUrgCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowTcpCtlUrgCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowTcpCtlAck(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowTcpCtlAckCounter'},
        'decrement': {'type': 'PatternFlowTcpCtlAckCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowTcpCtlAck, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowTcpCtlAckCounter
        """Factory property that returns an instance of the PatternFlowTcpCtlAckCounter class

        integer counter pattern

        Returns: PatternFlowTcpCtlAckCounter
        """
        return self._get_property('increment', PatternFlowTcpCtlAckCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowTcpCtlAckCounter
        """Factory property that returns an instance of the PatternFlowTcpCtlAckCounter class

        integer counter pattern

        Returns: PatternFlowTcpCtlAckCounter
        """
        return self._get_property('decrement', PatternFlowTcpCtlAckCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowTcpCtlAckCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowTcpCtlAckCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowTcpCtlPsh(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowTcpCtlPshCounter'},
        'decrement': {'type': 'PatternFlowTcpCtlPshCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowTcpCtlPsh, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowTcpCtlPshCounter
        """Factory property that returns an instance of the PatternFlowTcpCtlPshCounter class

        integer counter pattern

        Returns: PatternFlowTcpCtlPshCounter
        """
        return self._get_property('increment', PatternFlowTcpCtlPshCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowTcpCtlPshCounter
        """Factory property that returns an instance of the PatternFlowTcpCtlPshCounter class

        integer counter pattern

        Returns: PatternFlowTcpCtlPshCounter
        """
        return self._get_property('decrement', PatternFlowTcpCtlPshCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowTcpCtlPshCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowTcpCtlPshCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowTcpCtlRst(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowTcpCtlRstCounter'},
        'decrement': {'type': 'PatternFlowTcpCtlRstCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowTcpCtlRst, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowTcpCtlRstCounter
        """Factory property that returns an instance of the PatternFlowTcpCtlRstCounter class

        integer counter pattern

        Returns: PatternFlowTcpCtlRstCounter
        """
        return self._get_property('increment', PatternFlowTcpCtlRstCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowTcpCtlRstCounter
        """Factory property that returns an instance of the PatternFlowTcpCtlRstCounter class

        integer counter pattern

        Returns: PatternFlowTcpCtlRstCounter
        """
        return self._get_property('decrement', PatternFlowTcpCtlRstCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowTcpCtlRstCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowTcpCtlRstCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowTcpCtlSyn(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowTcpCtlSynCounter'},
        'decrement': {'type': 'PatternFlowTcpCtlSynCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowTcpCtlSyn, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowTcpCtlSynCounter
        """Factory property that returns an instance of the PatternFlowTcpCtlSynCounter class

        integer counter pattern

        Returns: PatternFlowTcpCtlSynCounter
        """
        return self._get_property('increment', PatternFlowTcpCtlSynCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowTcpCtlSynCounter
        """Factory property that returns an instance of the PatternFlowTcpCtlSynCounter class

        integer counter pattern

        Returns: PatternFlowTcpCtlSynCounter
        """
        return self._get_property('decrement', PatternFlowTcpCtlSynCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowTcpCtlSynCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowTcpCtlSynCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowTcpCtlFin(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowTcpCtlFinCounter'},
        'decrement': {'type': 'PatternFlowTcpCtlFinCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowTcpCtlFin, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowTcpCtlFinCounter
        """Factory property that returns an instance of the PatternFlowTcpCtlFinCounter class

        integer counter pattern

        Returns: PatternFlowTcpCtlFinCounter
        """
        return self._get_property('increment', PatternFlowTcpCtlFinCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowTcpCtlFinCounter
        """Factory property that returns an instance of the PatternFlowTcpCtlFinCounter class

        integer counter pattern

        Returns: PatternFlowTcpCtlFinCounter
        """
        return self._get_property('decrement', PatternFlowTcpCtlFinCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowTcpCtlFinCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowTcpCtlFinCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowTcpWindow(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowTcpWindowCounter'},
        'decrement': {'type': 'PatternFlowTcpWindowCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowTcpWindow, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowTcpWindowCounter
        """Factory property that returns an instance of the PatternFlowTcpWindowCounter class

        integer counter pattern

        Returns: PatternFlowTcpWindowCounter
        """
        return self._get_property('increment', PatternFlowTcpWindowCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowTcpWindowCounter
        """Factory property that returns an instance of the PatternFlowTcpWindowCounter class

        integer counter pattern

        Returns: PatternFlowTcpWindowCounter
        """
        return self._get_property('decrement', PatternFlowTcpWindowCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowTcpWindowCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowTcpWindowCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class FlowUdp(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'src_port': {'type': 'PatternFlowUdpSrcPort'},
        'dst_port': {'type': 'PatternFlowUdpDstPort'},
        'length': {'type': 'PatternFlowUdpLength'},
        'checksum': {'type': 'PatternFlowUdpChecksum'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowUdp, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def src_port(self):
        # type: () -> PatternFlowUdpSrcPort
        """src_port getter

        Source portSource port

        Returns: PatternFlowUdpSrcPort
        """
        return self._get_property('src_port', PatternFlowUdpSrcPort)

    @property
    def dst_port(self):
        # type: () -> PatternFlowUdpDstPort
        """dst_port getter

        Destination portDestination port

        Returns: PatternFlowUdpDstPort
        """
        return self._get_property('dst_port', PatternFlowUdpDstPort)

    @property
    def length(self):
        # type: () -> PatternFlowUdpLength
        """length getter

        LengthLength

        Returns: PatternFlowUdpLength
        """
        return self._get_property('length', PatternFlowUdpLength)

    @property
    def checksum(self):
        # type: () -> PatternFlowUdpChecksum
        """checksum getter

        UDP checksumUDP checksum

        Returns: PatternFlowUdpChecksum
        """
        return self._get_property('checksum', PatternFlowUdpChecksum)


class PatternFlowUdpSrcPort(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowUdpSrcPortCounter'},
        'decrement': {'type': 'PatternFlowUdpSrcPortCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowUdpSrcPort, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowUdpSrcPortCounter
        """Factory property that returns an instance of the PatternFlowUdpSrcPortCounter class

        integer counter pattern

        Returns: PatternFlowUdpSrcPortCounter
        """
        return self._get_property('increment', PatternFlowUdpSrcPortCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowUdpSrcPortCounter
        """Factory property that returns an instance of the PatternFlowUdpSrcPortCounter class

        integer counter pattern

        Returns: PatternFlowUdpSrcPortCounter
        """
        return self._get_property('decrement', PatternFlowUdpSrcPortCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowUdpSrcPortCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowUdpSrcPortCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowUdpDstPort(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowUdpDstPortCounter'},
        'decrement': {'type': 'PatternFlowUdpDstPortCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowUdpDstPort, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowUdpDstPortCounter
        """Factory property that returns an instance of the PatternFlowUdpDstPortCounter class

        integer counter pattern

        Returns: PatternFlowUdpDstPortCounter
        """
        return self._get_property('increment', PatternFlowUdpDstPortCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowUdpDstPortCounter
        """Factory property that returns an instance of the PatternFlowUdpDstPortCounter class

        integer counter pattern

        Returns: PatternFlowUdpDstPortCounter
        """
        return self._get_property('decrement', PatternFlowUdpDstPortCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowUdpDstPortCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowUdpDstPortCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowUdpLength(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowUdpLengthCounter'},
        'decrement': {'type': 'PatternFlowUdpLengthCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowUdpLength, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowUdpLengthCounter
        """Factory property that returns an instance of the PatternFlowUdpLengthCounter class

        integer counter pattern

        Returns: PatternFlowUdpLengthCounter
        """
        return self._get_property('increment', PatternFlowUdpLengthCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowUdpLengthCounter
        """Factory property that returns an instance of the PatternFlowUdpLengthCounter class

        integer counter pattern

        Returns: PatternFlowUdpLengthCounter
        """
        return self._get_property('decrement', PatternFlowUdpLengthCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowUdpLengthCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowUdpLengthCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowUdpChecksum(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'generated',
                'custom',
            ],
        },
        'generated': {
            'type': str,
            'enum': [
                'good',
                'bad',
            ],
        },
        'custom': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED = ('choice',) # type: tuple(str)

    _DEFAULTS = {
        'choice': 'generated',
        'generated': 'good',
    } # type: Dict[str, Union(type)]

    GENERATED = 'generated' # type: str
    CUSTOM = 'custom' # type: str

    GOOD = 'good' # type: str
    BAD = 'bad' # type: str

    def __init__(self, parent=None, choice=None):
        super(PatternFlowUdpChecksum, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def choice(self):
        # type: () -> Union[Literal["custom"], Literal["generated"]]
        """choice getter

        The type of checksum

        Returns: Union[Literal["custom"], Literal["generated"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of checksum

        value: Union[Literal["custom"], Literal["generated"]]
        """
        self._set_property('choice', value)

    @property
    def generated(self):
        # type: () -> Union[Literal["bad"], Literal["good"]]
        """generated getter

        A system generated checksum value

        Returns: Union[Literal["bad"], Literal["good"]]
        """
        return self._get_property('generated')

    @generated.setter
    def generated(self, value):
        """generated setter

        A system generated checksum value

        value: Union[Literal["bad"], Literal["good"]]
        """
        self._set_property('generated', value, 'generated')

    @property
    def custom(self):
        # type: () -> int
        """custom getter

        A custom checksum value

        Returns: int
        """
        return self._get_property('custom')

    @custom.setter
    def custom(self, value):
        """custom setter

        A custom checksum value

        value: int
        """
        self._set_property('custom', value, 'custom')


class FlowGre(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'checksum_present': {'type': 'PatternFlowGreChecksumPresent'},
        'reserved0': {'type': 'PatternFlowGreReserved0'},
        'version': {'type': 'PatternFlowGreVersion'},
        'protocol': {'type': 'PatternFlowGreProtocol'},
        'checksum': {'type': 'PatternFlowGreChecksum'},
        'reserved1': {'type': 'PatternFlowGreReserved1'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowGre, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def checksum_present(self):
        # type: () -> PatternFlowGreChecksumPresent
        """checksum_present getter

        Checksum present bitChecksum present bit

        Returns: PatternFlowGreChecksumPresent
        """
        return self._get_property('checksum_present', PatternFlowGreChecksumPresent)

    @property
    def reserved0(self):
        # type: () -> PatternFlowGreReserved0
        """reserved0 getter

        Reserved bitsReserved bits

        Returns: PatternFlowGreReserved0
        """
        return self._get_property('reserved0', PatternFlowGreReserved0)

    @property
    def version(self):
        # type: () -> PatternFlowGreVersion
        """version getter

        GRE version numberGRE version number

        Returns: PatternFlowGreVersion
        """
        return self._get_property('version', PatternFlowGreVersion)

    @property
    def protocol(self):
        # type: () -> PatternFlowGreProtocol
        """protocol getter

        Protocol type of encapsulated payloadProtocol type of encapsulated payload

        Returns: PatternFlowGreProtocol
        """
        return self._get_property('protocol', PatternFlowGreProtocol)

    @property
    def checksum(self):
        # type: () -> PatternFlowGreChecksum
        """checksum getter

        Optional checksum of GRE header and payload. Only present if the checksum_present bit is set.Optional checksum of GRE header and payload. Only present if the checksum_present bit is set.

        Returns: PatternFlowGreChecksum
        """
        return self._get_property('checksum', PatternFlowGreChecksum)

    @property
    def reserved1(self):
        # type: () -> PatternFlowGreReserved1
        """reserved1 getter

        Optional reserved field. Only present if the checksum_present bit is set.Optional reserved field. Only present if the checksum_present bit is set.

        Returns: PatternFlowGreReserved1
        """
        return self._get_property('reserved1', PatternFlowGreReserved1)


class PatternFlowGreChecksumPresent(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGreChecksumPresentCounter'},
        'decrement': {'type': 'PatternFlowGreChecksumPresentCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGreChecksumPresent, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGreChecksumPresentCounter
        """Factory property that returns an instance of the PatternFlowGreChecksumPresentCounter class

        integer counter pattern

        Returns: PatternFlowGreChecksumPresentCounter
        """
        return self._get_property('increment', PatternFlowGreChecksumPresentCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGreChecksumPresentCounter
        """Factory property that returns an instance of the PatternFlowGreChecksumPresentCounter class

        integer counter pattern

        Returns: PatternFlowGreChecksumPresentCounter
        """
        return self._get_property('decrement', PatternFlowGreChecksumPresentCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGreChecksumPresentCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGreChecksumPresentCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGreReserved0(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGreReserved0Counter'},
        'decrement': {'type': 'PatternFlowGreReserved0Counter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGreReserved0, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGreReserved0Counter
        """Factory property that returns an instance of the PatternFlowGreReserved0Counter class

        integer counter pattern

        Returns: PatternFlowGreReserved0Counter
        """
        return self._get_property('increment', PatternFlowGreReserved0Counter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGreReserved0Counter
        """Factory property that returns an instance of the PatternFlowGreReserved0Counter class

        integer counter pattern

        Returns: PatternFlowGreReserved0Counter
        """
        return self._get_property('decrement', PatternFlowGreReserved0Counter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGreReserved0Counter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGreReserved0Counter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGreVersion(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGreVersionCounter'},
        'decrement': {'type': 'PatternFlowGreVersionCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGreVersion, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGreVersionCounter
        """Factory property that returns an instance of the PatternFlowGreVersionCounter class

        integer counter pattern

        Returns: PatternFlowGreVersionCounter
        """
        return self._get_property('increment', PatternFlowGreVersionCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGreVersionCounter
        """Factory property that returns an instance of the PatternFlowGreVersionCounter class

        integer counter pattern

        Returns: PatternFlowGreVersionCounter
        """
        return self._get_property('decrement', PatternFlowGreVersionCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGreVersionCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGreVersionCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGreProtocol(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGreProtocolCounter'},
        'decrement': {'type': 'PatternFlowGreProtocolCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 2048,
        'values': [2048],
    } # type: Dict[str, Union(type)]

    IPV4 = 2048 #
    IPV6 = 34525 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGreProtocol, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGreProtocolCounter
        """Factory property that returns an instance of the PatternFlowGreProtocolCounter class

        integer counter pattern

        Returns: PatternFlowGreProtocolCounter
        """
        return self._get_property('increment', PatternFlowGreProtocolCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGreProtocolCounter
        """Factory property that returns an instance of the PatternFlowGreProtocolCounter class

        integer counter pattern

        Returns: PatternFlowGreProtocolCounter
        """
        return self._get_property('decrement', PatternFlowGreProtocolCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGreProtocolCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 2048,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    IPV4 = 2048 #
    IPV6 = 34525 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGreProtocolCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGreChecksum(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'generated',
                'custom',
            ],
        },
        'generated': {
            'type': str,
            'enum': [
                'good',
                'bad',
            ],
        },
        'custom': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED = ('choice',) # type: tuple(str)

    _DEFAULTS = {
        'choice': 'generated',
        'generated': 'good',
    } # type: Dict[str, Union(type)]

    GENERATED = 'generated' # type: str
    CUSTOM = 'custom' # type: str

    GOOD = 'good' # type: str
    BAD = 'bad' # type: str

    def __init__(self, parent=None, choice=None):
        super(PatternFlowGreChecksum, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def choice(self):
        # type: () -> Union[Literal["custom"], Literal["generated"]]
        """choice getter

        The type of checksum

        Returns: Union[Literal["custom"], Literal["generated"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of checksum

        value: Union[Literal["custom"], Literal["generated"]]
        """
        self._set_property('choice', value)

    @property
    def generated(self):
        # type: () -> Union[Literal["bad"], Literal["good"]]
        """generated getter

        A system generated checksum value

        Returns: Union[Literal["bad"], Literal["good"]]
        """
        return self._get_property('generated')

    @generated.setter
    def generated(self, value):
        """generated setter

        A system generated checksum value

        value: Union[Literal["bad"], Literal["good"]]
        """
        self._set_property('generated', value, 'generated')

    @property
    def custom(self):
        # type: () -> int
        """custom getter

        A custom checksum value

        Returns: int
        """
        return self._get_property('custom')

    @custom.setter
    def custom(self, value):
        """custom setter

        A custom checksum value

        value: int
        """
        self._set_property('custom', value, 'custom')


class PatternFlowGreReserved1(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGreReserved1Counter'},
        'decrement': {'type': 'PatternFlowGreReserved1Counter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGreReserved1, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGreReserved1Counter
        """Factory property that returns an instance of the PatternFlowGreReserved1Counter class

        integer counter pattern

        Returns: PatternFlowGreReserved1Counter
        """
        return self._get_property('increment', PatternFlowGreReserved1Counter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGreReserved1Counter
        """Factory property that returns an instance of the PatternFlowGreReserved1Counter class

        integer counter pattern

        Returns: PatternFlowGreReserved1Counter
        """
        return self._get_property('decrement', PatternFlowGreReserved1Counter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGreReserved1Counter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGreReserved1Counter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class FlowGtpv1(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'version': {'type': 'PatternFlowGtpv1Version'},
        'protocol_type': {'type': 'PatternFlowGtpv1ProtocolType'},
        'reserved': {'type': 'PatternFlowGtpv1Reserved'},
        'e_flag': {'type': 'PatternFlowGtpv1EFlag'},
        's_flag': {'type': 'PatternFlowGtpv1SFlag'},
        'pn_flag': {'type': 'PatternFlowGtpv1PnFlag'},
        'message_type': {'type': 'PatternFlowGtpv1MessageType'},
        'message_length': {'type': 'PatternFlowGtpv1MessageLength'},
        'teid': {'type': 'PatternFlowGtpv1Teid'},
        'squence_number': {'type': 'PatternFlowGtpv1SquenceNumber'},
        'n_pdu_number': {'type': 'PatternFlowGtpv1NPduNumber'},
        'next_extension_header_type': {'type': 'PatternFlowGtpv1NextExtensionHeaderType'},
        'extension_headers': {'type': 'FlowGtpExtensionIter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowGtpv1, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def version(self):
        # type: () -> PatternFlowGtpv1Version
        """version getter

        GTPv1 versionGTPv1 version

        Returns: PatternFlowGtpv1Version
        """
        return self._get_property('version', PatternFlowGtpv1Version)

    @property
    def protocol_type(self):
        # type: () -> PatternFlowGtpv1ProtocolType
        """protocol_type getter

        Protocol type, GTP is 1, GTP' is 0Protocol type, GTP is 1, GTP' is 0

        Returns: PatternFlowGtpv1ProtocolType
        """
        return self._get_property('protocol_type', PatternFlowGtpv1ProtocolType)

    @property
    def reserved(self):
        # type: () -> PatternFlowGtpv1Reserved
        """reserved getter

        Reserved fieldReserved field

        Returns: PatternFlowGtpv1Reserved
        """
        return self._get_property('reserved', PatternFlowGtpv1Reserved)

    @property
    def e_flag(self):
        # type: () -> PatternFlowGtpv1EFlag
        """e_flag getter

        Extension header field presentExtension header field present

        Returns: PatternFlowGtpv1EFlag
        """
        return self._get_property('e_flag', PatternFlowGtpv1EFlag)

    @property
    def s_flag(self):
        # type: () -> PatternFlowGtpv1SFlag
        """s_flag getter

        Sequence number field presentSequence number field present

        Returns: PatternFlowGtpv1SFlag
        """
        return self._get_property('s_flag', PatternFlowGtpv1SFlag)

    @property
    def pn_flag(self):
        # type: () -> PatternFlowGtpv1PnFlag
        """pn_flag getter

        N-PDU field presentN-PDU field present

        Returns: PatternFlowGtpv1PnFlag
        """
        return self._get_property('pn_flag', PatternFlowGtpv1PnFlag)

    @property
    def message_type(self):
        # type: () -> PatternFlowGtpv1MessageType
        """message_type getter

        The type of GTP message Different types of messages are defined in 3GPP TS 29.060 section 7.1The type of GTP message Different types of messages are defined in 3GPP TS 29.060 section 7.1

        Returns: PatternFlowGtpv1MessageType
        """
        return self._get_property('message_type', PatternFlowGtpv1MessageType)

    @property
    def message_length(self):
        # type: () -> PatternFlowGtpv1MessageLength
        """message_length getter

        The length of the payload (the bytes following the mandatory 8-byte GTP header) in bytes that includes any optional fieldsThe length of the payload (the bytes following the mandatory 8-byte GTP header) in bytes that includes any optional fields

        Returns: PatternFlowGtpv1MessageLength
        """
        return self._get_property('message_length', PatternFlowGtpv1MessageLength)

    @property
    def teid(self):
        # type: () -> PatternFlowGtpv1Teid
        """teid getter

        Tunnel endpoint identifier (TEID) used to multiplex connections in the same GTP tunnelTunnel endpoint identifier (TEID) used to multiplex connections in the same GTP tunnel

        Returns: PatternFlowGtpv1Teid
        """
        return self._get_property('teid', PatternFlowGtpv1Teid)

    @property
    def squence_number(self):
        # type: () -> PatternFlowGtpv1SquenceNumber
        """squence_number getter

        Sequence number. Exists if any of the e_flag, s_flag, or pn_flag bits are on. Must be interpreted only if the s_flag bit is on.Sequence number. Exists if any of the e_flag, s_flag, or pn_flag bits are on. Must be interpreted only if the s_flag bit is on.

        Returns: PatternFlowGtpv1SquenceNumber
        """
        return self._get_property('squence_number', PatternFlowGtpv1SquenceNumber)

    @property
    def n_pdu_number(self):
        # type: () -> PatternFlowGtpv1NPduNumber
        """n_pdu_number getter

        N-PDU number. Exists if any of the e_flag, s_flag, or pn_flag bits are on. Must be interpreted only if the pn_flag bit is on.N-PDU number. Exists if any of the e_flag, s_flag, or pn_flag bits are on. Must be interpreted only if the pn_flag bit is on.

        Returns: PatternFlowGtpv1NPduNumber
        """
        return self._get_property('n_pdu_number', PatternFlowGtpv1NPduNumber)

    @property
    def next_extension_header_type(self):
        # type: () -> PatternFlowGtpv1NextExtensionHeaderType
        """next_extension_header_type getter

        Next extension header. Exists if any of the e_flag, s_flag, or pn_flag bits are on. Must be interpreted only if the e_flag bit is on.Next extension header. Exists if any of the e_flag, s_flag, or pn_flag bits are on. Must be interpreted only if the e_flag bit is on.

        Returns: PatternFlowGtpv1NextExtensionHeaderType
        """
        return self._get_property('next_extension_header_type', PatternFlowGtpv1NextExtensionHeaderType)

    @property
    def extension_headers(self):
        # type: () -> FlowGtpExtensionIter
        """extension_headers getter

        A list of optional extension headers.

        Returns: FlowGtpExtensionIter
        """
        return self._get_property('extension_headers', FlowGtpExtensionIter, self._parent, self._choice)


class PatternFlowGtpv1Version(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv1VersionCounter'},
        'decrement': {'type': 'PatternFlowGtpv1VersionCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 1,
        'values': [1],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv1Version, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv1VersionCounter
        """Factory property that returns an instance of the PatternFlowGtpv1VersionCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1VersionCounter
        """
        return self._get_property('increment', PatternFlowGtpv1VersionCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv1VersionCounter
        """Factory property that returns an instance of the PatternFlowGtpv1VersionCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1VersionCounter
        """
        return self._get_property('decrement', PatternFlowGtpv1VersionCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv1VersionCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 1,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv1VersionCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv1ProtocolType(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv1ProtocolTypeCounter'},
        'decrement': {'type': 'PatternFlowGtpv1ProtocolTypeCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 1,
        'values': [1],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv1ProtocolType, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv1ProtocolTypeCounter
        """Factory property that returns an instance of the PatternFlowGtpv1ProtocolTypeCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1ProtocolTypeCounter
        """
        return self._get_property('increment', PatternFlowGtpv1ProtocolTypeCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv1ProtocolTypeCounter
        """Factory property that returns an instance of the PatternFlowGtpv1ProtocolTypeCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1ProtocolTypeCounter
        """
        return self._get_property('decrement', PatternFlowGtpv1ProtocolTypeCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv1ProtocolTypeCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 1,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv1ProtocolTypeCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv1Reserved(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv1ReservedCounter'},
        'decrement': {'type': 'PatternFlowGtpv1ReservedCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv1Reserved, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv1ReservedCounter
        """Factory property that returns an instance of the PatternFlowGtpv1ReservedCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1ReservedCounter
        """
        return self._get_property('increment', PatternFlowGtpv1ReservedCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv1ReservedCounter
        """Factory property that returns an instance of the PatternFlowGtpv1ReservedCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1ReservedCounter
        """
        return self._get_property('decrement', PatternFlowGtpv1ReservedCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv1ReservedCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv1ReservedCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv1EFlag(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv1EFlagCounter'},
        'decrement': {'type': 'PatternFlowGtpv1EFlagCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv1EFlag, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv1EFlagCounter
        """Factory property that returns an instance of the PatternFlowGtpv1EFlagCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1EFlagCounter
        """
        return self._get_property('increment', PatternFlowGtpv1EFlagCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv1EFlagCounter
        """Factory property that returns an instance of the PatternFlowGtpv1EFlagCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1EFlagCounter
        """
        return self._get_property('decrement', PatternFlowGtpv1EFlagCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv1EFlagCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv1EFlagCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv1SFlag(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv1SFlagCounter'},
        'decrement': {'type': 'PatternFlowGtpv1SFlagCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv1SFlag, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv1SFlagCounter
        """Factory property that returns an instance of the PatternFlowGtpv1SFlagCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1SFlagCounter
        """
        return self._get_property('increment', PatternFlowGtpv1SFlagCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv1SFlagCounter
        """Factory property that returns an instance of the PatternFlowGtpv1SFlagCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1SFlagCounter
        """
        return self._get_property('decrement', PatternFlowGtpv1SFlagCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv1SFlagCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv1SFlagCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv1PnFlag(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv1PnFlagCounter'},
        'decrement': {'type': 'PatternFlowGtpv1PnFlagCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv1PnFlag, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv1PnFlagCounter
        """Factory property that returns an instance of the PatternFlowGtpv1PnFlagCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1PnFlagCounter
        """
        return self._get_property('increment', PatternFlowGtpv1PnFlagCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv1PnFlagCounter
        """Factory property that returns an instance of the PatternFlowGtpv1PnFlagCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1PnFlagCounter
        """
        return self._get_property('decrement', PatternFlowGtpv1PnFlagCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv1PnFlagCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv1PnFlagCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv1MessageType(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv1MessageTypeCounter'},
        'decrement': {'type': 'PatternFlowGtpv1MessageTypeCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv1MessageType, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv1MessageTypeCounter
        """Factory property that returns an instance of the PatternFlowGtpv1MessageTypeCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1MessageTypeCounter
        """
        return self._get_property('increment', PatternFlowGtpv1MessageTypeCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv1MessageTypeCounter
        """Factory property that returns an instance of the PatternFlowGtpv1MessageTypeCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1MessageTypeCounter
        """
        return self._get_property('decrement', PatternFlowGtpv1MessageTypeCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv1MessageTypeCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv1MessageTypeCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv1MessageLength(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv1MessageLengthCounter'},
        'decrement': {'type': 'PatternFlowGtpv1MessageLengthCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv1MessageLength, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv1MessageLengthCounter
        """Factory property that returns an instance of the PatternFlowGtpv1MessageLengthCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1MessageLengthCounter
        """
        return self._get_property('increment', PatternFlowGtpv1MessageLengthCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv1MessageLengthCounter
        """Factory property that returns an instance of the PatternFlowGtpv1MessageLengthCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1MessageLengthCounter
        """
        return self._get_property('decrement', PatternFlowGtpv1MessageLengthCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv1MessageLengthCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv1MessageLengthCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv1Teid(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv1TeidCounter'},
        'decrement': {'type': 'PatternFlowGtpv1TeidCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv1Teid, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv1TeidCounter
        """Factory property that returns an instance of the PatternFlowGtpv1TeidCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1TeidCounter
        """
        return self._get_property('increment', PatternFlowGtpv1TeidCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv1TeidCounter
        """Factory property that returns an instance of the PatternFlowGtpv1TeidCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1TeidCounter
        """
        return self._get_property('decrement', PatternFlowGtpv1TeidCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv1TeidCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv1TeidCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv1SquenceNumber(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv1SquenceNumberCounter'},
        'decrement': {'type': 'PatternFlowGtpv1SquenceNumberCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv1SquenceNumber, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv1SquenceNumberCounter
        """Factory property that returns an instance of the PatternFlowGtpv1SquenceNumberCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1SquenceNumberCounter
        """
        return self._get_property('increment', PatternFlowGtpv1SquenceNumberCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv1SquenceNumberCounter
        """Factory property that returns an instance of the PatternFlowGtpv1SquenceNumberCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1SquenceNumberCounter
        """
        return self._get_property('decrement', PatternFlowGtpv1SquenceNumberCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv1SquenceNumberCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv1SquenceNumberCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv1NPduNumber(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv1NPduNumberCounter'},
        'decrement': {'type': 'PatternFlowGtpv1NPduNumberCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv1NPduNumber, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv1NPduNumberCounter
        """Factory property that returns an instance of the PatternFlowGtpv1NPduNumberCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1NPduNumberCounter
        """
        return self._get_property('increment', PatternFlowGtpv1NPduNumberCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv1NPduNumberCounter
        """Factory property that returns an instance of the PatternFlowGtpv1NPduNumberCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1NPduNumberCounter
        """
        return self._get_property('decrement', PatternFlowGtpv1NPduNumberCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv1NPduNumberCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv1NPduNumberCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv1NextExtensionHeaderType(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv1NextExtensionHeaderTypeCounter'},
        'decrement': {'type': 'PatternFlowGtpv1NextExtensionHeaderTypeCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv1NextExtensionHeaderType, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv1NextExtensionHeaderTypeCounter
        """Factory property that returns an instance of the PatternFlowGtpv1NextExtensionHeaderTypeCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1NextExtensionHeaderTypeCounter
        """
        return self._get_property('increment', PatternFlowGtpv1NextExtensionHeaderTypeCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv1NextExtensionHeaderTypeCounter
        """Factory property that returns an instance of the PatternFlowGtpv1NextExtensionHeaderTypeCounter class

        integer counter pattern

        Returns: PatternFlowGtpv1NextExtensionHeaderTypeCounter
        """
        return self._get_property('decrement', PatternFlowGtpv1NextExtensionHeaderTypeCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv1NextExtensionHeaderTypeCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv1NextExtensionHeaderTypeCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class FlowGtpExtension(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'extension_length': {'type': 'PatternFlowGtpExtensionExtensionLength'},
        'contents': {'type': 'PatternFlowGtpExtensionContents'},
        'next_extension_header': {'type': 'PatternFlowGtpExtensionNextExtensionHeader'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowGtpExtension, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def extension_length(self):
        # type: () -> PatternFlowGtpExtensionExtensionLength
        """extension_length getter

        This field states the length of this extension header, including the length, the contents, and the next extension header field, in 4-octet units, so the length of the extension must always be a multiple of 4.

        Returns: PatternFlowGtpExtensionExtensionLength
        """
        return self._get_property('extension_length', PatternFlowGtpExtensionExtensionLength)

    @property
    def contents(self):
        # type: () -> PatternFlowGtpExtensionContents
        """contents getter

        The extension header contents

        Returns: PatternFlowGtpExtensionContents
        """
        return self._get_property('contents', PatternFlowGtpExtensionContents)

    @property
    def next_extension_header(self):
        # type: () -> PatternFlowGtpExtensionNextExtensionHeader
        """next_extension_header getter

        It states the type of the next extension, or 0 if no next extension exists. This permits chaining several next extension headers.

        Returns: PatternFlowGtpExtensionNextExtensionHeader
        """
        return self._get_property('next_extension_header', PatternFlowGtpExtensionNextExtensionHeader)


class PatternFlowGtpExtensionExtensionLength(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpExtensionExtensionLengthCounter'},
        'decrement': {'type': 'PatternFlowGtpExtensionExtensionLengthCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpExtensionExtensionLength, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpExtensionExtensionLengthCounter
        """Factory property that returns an instance of the PatternFlowGtpExtensionExtensionLengthCounter class

        integer counter pattern

        Returns: PatternFlowGtpExtensionExtensionLengthCounter
        """
        return self._get_property('increment', PatternFlowGtpExtensionExtensionLengthCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpExtensionExtensionLengthCounter
        """Factory property that returns an instance of the PatternFlowGtpExtensionExtensionLengthCounter class

        integer counter pattern

        Returns: PatternFlowGtpExtensionExtensionLengthCounter
        """
        return self._get_property('decrement', PatternFlowGtpExtensionExtensionLengthCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpExtensionExtensionLengthCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpExtensionExtensionLengthCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpExtensionContents(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpExtensionContentsCounter'},
        'decrement': {'type': 'PatternFlowGtpExtensionContentsCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpExtensionContents, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpExtensionContentsCounter
        """Factory property that returns an instance of the PatternFlowGtpExtensionContentsCounter class

        integer counter pattern

        Returns: PatternFlowGtpExtensionContentsCounter
        """
        return self._get_property('increment', PatternFlowGtpExtensionContentsCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpExtensionContentsCounter
        """Factory property that returns an instance of the PatternFlowGtpExtensionContentsCounter class

        integer counter pattern

        Returns: PatternFlowGtpExtensionContentsCounter
        """
        return self._get_property('decrement', PatternFlowGtpExtensionContentsCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpExtensionContentsCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpExtensionContentsCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpExtensionNextExtensionHeader(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpExtensionNextExtensionHeaderCounter'},
        'decrement': {'type': 'PatternFlowGtpExtensionNextExtensionHeaderCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpExtensionNextExtensionHeader, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpExtensionNextExtensionHeaderCounter
        """Factory property that returns an instance of the PatternFlowGtpExtensionNextExtensionHeaderCounter class

        integer counter pattern

        Returns: PatternFlowGtpExtensionNextExtensionHeaderCounter
        """
        return self._get_property('increment', PatternFlowGtpExtensionNextExtensionHeaderCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpExtensionNextExtensionHeaderCounter
        """Factory property that returns an instance of the PatternFlowGtpExtensionNextExtensionHeaderCounter class

        integer counter pattern

        Returns: PatternFlowGtpExtensionNextExtensionHeaderCounter
        """
        return self._get_property('decrement', PatternFlowGtpExtensionNextExtensionHeaderCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpExtensionNextExtensionHeaderCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpExtensionNextExtensionHeaderCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class FlowGtpExtensionIter(OpenApiIter):
    __slots__ = ('_parent', '_choice')

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(FlowGtpExtensionIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[FlowGtpExtension]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> FlowGtpExtensionIter
        return self._iter()

    def __next__(self):
        # type: () -> FlowGtpExtension
        return self._next()

    def next(self):
        # type: () -> FlowGtpExtension
        return self._next()

    def gtpextension(self):
        # type: () -> FlowGtpExtensionIter
        """Factory method that creates an instance of the FlowGtpExtension class

        TBD

        Returns: FlowGtpExtensionIter
        """
        item = FlowGtpExtension(parent=self._parent, choice=self._choice)
        self._add(item)
        return self



class FlowGtpv2(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'version': {'type': 'PatternFlowGtpv2Version'},
        'piggybacking_flag': {'type': 'PatternFlowGtpv2PiggybackingFlag'},
        'teid_flag': {'type': 'PatternFlowGtpv2TeidFlag'},
        'spare1': {'type': 'PatternFlowGtpv2Spare1'},
        'message_type': {'type': 'PatternFlowGtpv2MessageType'},
        'message_length': {'type': 'PatternFlowGtpv2MessageLength'},
        'teid': {'type': 'PatternFlowGtpv2Teid'},
        'sequence_number': {'type': 'PatternFlowGtpv2SequenceNumber'},
        'spare2': {'type': 'PatternFlowGtpv2Spare2'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowGtpv2, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def version(self):
        # type: () -> PatternFlowGtpv2Version
        """version getter

        Version numberVersion number

        Returns: PatternFlowGtpv2Version
        """
        return self._get_property('version', PatternFlowGtpv2Version)

    @property
    def piggybacking_flag(self):
        # type: () -> PatternFlowGtpv2PiggybackingFlag
        """piggybacking_flag getter

        If piggybacking_flag is set to 1 then another GTP-C message with its own header shall be present at the end of the current messageIf piggybacking_flag is set to 1 then another GTP-C message with its own header shall be present at the end of the current message

        Returns: PatternFlowGtpv2PiggybackingFlag
        """
        return self._get_property('piggybacking_flag', PatternFlowGtpv2PiggybackingFlag)

    @property
    def teid_flag(self):
        # type: () -> PatternFlowGtpv2TeidFlag
        """teid_flag getter

        If teid_flag is set to 1 then the TEID field will be present between the message length and the sequence number. All messages except Echo and Echo reply require TEID to be presentIf teid_flag is set to 1 then the TEID field will be present between the message length and the sequence number. All messages except Echo and Echo reply require TEID to be present

        Returns: PatternFlowGtpv2TeidFlag
        """
        return self._get_property('teid_flag', PatternFlowGtpv2TeidFlag)

    @property
    def spare1(self):
        # type: () -> PatternFlowGtpv2Spare1
        """spare1 getter

        A 3-bit reserved field (must be 0).A 3-bit reserved field (must be 0).

        Returns: PatternFlowGtpv2Spare1
        """
        return self._get_property('spare1', PatternFlowGtpv2Spare1)

    @property
    def message_type(self):
        # type: () -> PatternFlowGtpv2MessageType
        """message_type getter

        An 8-bit field that indicates the type of GTP message. Different types of messages are defined in 3GPP TS 29.060 section 7.1An 8-bit field that indicates the type of GTP message. Different types of messages are defined in 3GPP TS 29.060 section 7.1

        Returns: PatternFlowGtpv2MessageType
        """
        return self._get_property('message_type', PatternFlowGtpv2MessageType)

    @property
    def message_length(self):
        # type: () -> PatternFlowGtpv2MessageLength
        """message_length getter

        A 16-bit field that indicates the length of the payload in bytes, excluding the mandatory GTP-c header (first 4 bytes). Includes the TEID and sequence_number if they are present.A 16-bit field that indicates the length of the payload in bytes, excluding the mandatory GTP-c header (first 4 bytes). Includes the TEID and sequence_number if they are present.

        Returns: PatternFlowGtpv2MessageLength
        """
        return self._get_property('message_length', PatternFlowGtpv2MessageLength)

    @property
    def teid(self):
        # type: () -> PatternFlowGtpv2Teid
        """teid getter

        Tunnel endpoint identifier. A 32-bit (4-octet) field used to multiplex different connections in the same GTP tunnel. Is present only if the teid_flag is set.Tunnel endpoint identifier. A 32-bit (4-octet) field used to multiplex different connections in the same GTP tunnel. Is present only if the teid_flag is set.

        Returns: PatternFlowGtpv2Teid
        """
        return self._get_property('teid', PatternFlowGtpv2Teid)

    @property
    def sequence_number(self):
        # type: () -> PatternFlowGtpv2SequenceNumber
        """sequence_number getter

        The sequence numberThe sequence number

        Returns: PatternFlowGtpv2SequenceNumber
        """
        return self._get_property('sequence_number', PatternFlowGtpv2SequenceNumber)

    @property
    def spare2(self):
        # type: () -> PatternFlowGtpv2Spare2
        """spare2 getter

        Reserved fieldReserved field

        Returns: PatternFlowGtpv2Spare2
        """
        return self._get_property('spare2', PatternFlowGtpv2Spare2)


class PatternFlowGtpv2Version(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv2VersionCounter'},
        'decrement': {'type': 'PatternFlowGtpv2VersionCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 2,
        'values': [2],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv2Version, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv2VersionCounter
        """Factory property that returns an instance of the PatternFlowGtpv2VersionCounter class

        integer counter pattern

        Returns: PatternFlowGtpv2VersionCounter
        """
        return self._get_property('increment', PatternFlowGtpv2VersionCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv2VersionCounter
        """Factory property that returns an instance of the PatternFlowGtpv2VersionCounter class

        integer counter pattern

        Returns: PatternFlowGtpv2VersionCounter
        """
        return self._get_property('decrement', PatternFlowGtpv2VersionCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv2VersionCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 2,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv2VersionCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv2PiggybackingFlag(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv2PiggybackingFlagCounter'},
        'decrement': {'type': 'PatternFlowGtpv2PiggybackingFlagCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv2PiggybackingFlag, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv2PiggybackingFlagCounter
        """Factory property that returns an instance of the PatternFlowGtpv2PiggybackingFlagCounter class

        integer counter pattern

        Returns: PatternFlowGtpv2PiggybackingFlagCounter
        """
        return self._get_property('increment', PatternFlowGtpv2PiggybackingFlagCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv2PiggybackingFlagCounter
        """Factory property that returns an instance of the PatternFlowGtpv2PiggybackingFlagCounter class

        integer counter pattern

        Returns: PatternFlowGtpv2PiggybackingFlagCounter
        """
        return self._get_property('decrement', PatternFlowGtpv2PiggybackingFlagCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv2PiggybackingFlagCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv2PiggybackingFlagCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv2TeidFlag(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv2TeidFlagCounter'},
        'decrement': {'type': 'PatternFlowGtpv2TeidFlagCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv2TeidFlag, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv2TeidFlagCounter
        """Factory property that returns an instance of the PatternFlowGtpv2TeidFlagCounter class

        integer counter pattern

        Returns: PatternFlowGtpv2TeidFlagCounter
        """
        return self._get_property('increment', PatternFlowGtpv2TeidFlagCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv2TeidFlagCounter
        """Factory property that returns an instance of the PatternFlowGtpv2TeidFlagCounter class

        integer counter pattern

        Returns: PatternFlowGtpv2TeidFlagCounter
        """
        return self._get_property('decrement', PatternFlowGtpv2TeidFlagCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv2TeidFlagCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv2TeidFlagCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv2Spare1(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv2Spare1Counter'},
        'decrement': {'type': 'PatternFlowGtpv2Spare1Counter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv2Spare1, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv2Spare1Counter
        """Factory property that returns an instance of the PatternFlowGtpv2Spare1Counter class

        integer counter pattern

        Returns: PatternFlowGtpv2Spare1Counter
        """
        return self._get_property('increment', PatternFlowGtpv2Spare1Counter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv2Spare1Counter
        """Factory property that returns an instance of the PatternFlowGtpv2Spare1Counter class

        integer counter pattern

        Returns: PatternFlowGtpv2Spare1Counter
        """
        return self._get_property('decrement', PatternFlowGtpv2Spare1Counter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv2Spare1Counter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv2Spare1Counter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv2MessageType(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv2MessageTypeCounter'},
        'decrement': {'type': 'PatternFlowGtpv2MessageTypeCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv2MessageType, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv2MessageTypeCounter
        """Factory property that returns an instance of the PatternFlowGtpv2MessageTypeCounter class

        integer counter pattern

        Returns: PatternFlowGtpv2MessageTypeCounter
        """
        return self._get_property('increment', PatternFlowGtpv2MessageTypeCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv2MessageTypeCounter
        """Factory property that returns an instance of the PatternFlowGtpv2MessageTypeCounter class

        integer counter pattern

        Returns: PatternFlowGtpv2MessageTypeCounter
        """
        return self._get_property('decrement', PatternFlowGtpv2MessageTypeCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv2MessageTypeCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv2MessageTypeCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv2MessageLength(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv2MessageLengthCounter'},
        'decrement': {'type': 'PatternFlowGtpv2MessageLengthCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv2MessageLength, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv2MessageLengthCounter
        """Factory property that returns an instance of the PatternFlowGtpv2MessageLengthCounter class

        integer counter pattern

        Returns: PatternFlowGtpv2MessageLengthCounter
        """
        return self._get_property('increment', PatternFlowGtpv2MessageLengthCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv2MessageLengthCounter
        """Factory property that returns an instance of the PatternFlowGtpv2MessageLengthCounter class

        integer counter pattern

        Returns: PatternFlowGtpv2MessageLengthCounter
        """
        return self._get_property('decrement', PatternFlowGtpv2MessageLengthCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv2MessageLengthCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv2MessageLengthCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv2Teid(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv2TeidCounter'},
        'decrement': {'type': 'PatternFlowGtpv2TeidCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv2Teid, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv2TeidCounter
        """Factory property that returns an instance of the PatternFlowGtpv2TeidCounter class

        integer counter pattern

        Returns: PatternFlowGtpv2TeidCounter
        """
        return self._get_property('increment', PatternFlowGtpv2TeidCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv2TeidCounter
        """Factory property that returns an instance of the PatternFlowGtpv2TeidCounter class

        integer counter pattern

        Returns: PatternFlowGtpv2TeidCounter
        """
        return self._get_property('decrement', PatternFlowGtpv2TeidCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv2TeidCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv2TeidCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv2SequenceNumber(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv2SequenceNumberCounter'},
        'decrement': {'type': 'PatternFlowGtpv2SequenceNumberCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv2SequenceNumber, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv2SequenceNumberCounter
        """Factory property that returns an instance of the PatternFlowGtpv2SequenceNumberCounter class

        integer counter pattern

        Returns: PatternFlowGtpv2SequenceNumberCounter
        """
        return self._get_property('increment', PatternFlowGtpv2SequenceNumberCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv2SequenceNumberCounter
        """Factory property that returns an instance of the PatternFlowGtpv2SequenceNumberCounter class

        integer counter pattern

        Returns: PatternFlowGtpv2SequenceNumberCounter
        """
        return self._get_property('decrement', PatternFlowGtpv2SequenceNumberCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv2SequenceNumberCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv2SequenceNumberCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowGtpv2Spare2(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowGtpv2Spare2Counter'},
        'decrement': {'type': 'PatternFlowGtpv2Spare2Counter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 0,
        'values': [0],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowGtpv2Spare2, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowGtpv2Spare2Counter
        """Factory property that returns an instance of the PatternFlowGtpv2Spare2Counter class

        integer counter pattern

        Returns: PatternFlowGtpv2Spare2Counter
        """
        return self._get_property('increment', PatternFlowGtpv2Spare2Counter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowGtpv2Spare2Counter
        """Factory property that returns an instance of the PatternFlowGtpv2Spare2Counter class

        integer counter pattern

        Returns: PatternFlowGtpv2Spare2Counter
        """
        return self._get_property('decrement', PatternFlowGtpv2Spare2Counter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowGtpv2Spare2Counter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 0,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowGtpv2Spare2Counter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class FlowArp(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'hardware_type': {'type': 'PatternFlowArpHardwareType'},
        'protocol_type': {'type': 'PatternFlowArpProtocolType'},
        'hardware_length': {'type': 'PatternFlowArpHardwareLength'},
        'protocol_length': {'type': 'PatternFlowArpProtocolLength'},
        'operation': {'type': 'PatternFlowArpOperation'},
        'sender_hardware_addr': {'type': 'PatternFlowArpSenderHardwareAddr'},
        'sender_protocol_addr': {'type': 'PatternFlowArpSenderProtocolAddr'},
        'target_hardware_addr': {'type': 'PatternFlowArpTargetHardwareAddr'},
        'target_protocol_addr': {'type': 'PatternFlowArpTargetProtocolAddr'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS= {} # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(FlowArp, self).__init__()
        self._parent = parent
        self._choice = choice

    @property
    def hardware_type(self):
        # type: () -> PatternFlowArpHardwareType
        """hardware_type getter

        Network link protocol typeNetwork link protocol type

        Returns: PatternFlowArpHardwareType
        """
        return self._get_property('hardware_type', PatternFlowArpHardwareType)

    @property
    def protocol_type(self):
        # type: () -> PatternFlowArpProtocolType
        """protocol_type getter

        The internetwork protocol for which the ARP request is intendedThe internetwork protocol for which the ARP request is intended

        Returns: PatternFlowArpProtocolType
        """
        return self._get_property('protocol_type', PatternFlowArpProtocolType)

    @property
    def hardware_length(self):
        # type: () -> PatternFlowArpHardwareLength
        """hardware_length getter

        Length (in octets) of a hardware addressLength (in octets) of a hardware address

        Returns: PatternFlowArpHardwareLength
        """
        return self._get_property('hardware_length', PatternFlowArpHardwareLength)

    @property
    def protocol_length(self):
        # type: () -> PatternFlowArpProtocolLength
        """protocol_length getter

        Length (in octets) of internetwork addressesLength (in octets) of internetwork addresses

        Returns: PatternFlowArpProtocolLength
        """
        return self._get_property('protocol_length', PatternFlowArpProtocolLength)

    @property
    def operation(self):
        # type: () -> PatternFlowArpOperation
        """operation getter

        The operation that the sender is performingThe operation that the sender is performing

        Returns: PatternFlowArpOperation
        """
        return self._get_property('operation', PatternFlowArpOperation)

    @property
    def sender_hardware_addr(self):
        # type: () -> PatternFlowArpSenderHardwareAddr
        """sender_hardware_addr getter

        Media address of the senderMedia address of the sender

        Returns: PatternFlowArpSenderHardwareAddr
        """
        return self._get_property('sender_hardware_addr', PatternFlowArpSenderHardwareAddr)

    @property
    def sender_protocol_addr(self):
        # type: () -> PatternFlowArpSenderProtocolAddr
        """sender_protocol_addr getter

        Internetwork address of the senderInternetwork address of the sender

        Returns: PatternFlowArpSenderProtocolAddr
        """
        return self._get_property('sender_protocol_addr', PatternFlowArpSenderProtocolAddr)

    @property
    def target_hardware_addr(self):
        # type: () -> PatternFlowArpTargetHardwareAddr
        """target_hardware_addr getter

        Media address of the targetMedia address of the target

        Returns: PatternFlowArpTargetHardwareAddr
        """
        return self._get_property('target_hardware_addr', PatternFlowArpTargetHardwareAddr)

    @property
    def target_protocol_addr(self):
        # type: () -> PatternFlowArpTargetProtocolAddr
        """target_protocol_addr getter

        Internetwork address of the targetInternetwork address of the target

        Returns: PatternFlowArpTargetProtocolAddr
        """
        return self._get_property('target_protocol_addr', PatternFlowArpTargetProtocolAddr)


class PatternFlowArpHardwareType(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowArpHardwareTypeCounter'},
        'decrement': {'type': 'PatternFlowArpHardwareTypeCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 1,
        'values': [1],
    } # type: Dict[str, Union(type)]

    ETHERNET = 1 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowArpHardwareType, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowArpHardwareTypeCounter
        """Factory property that returns an instance of the PatternFlowArpHardwareTypeCounter class

        integer counter pattern

        Returns: PatternFlowArpHardwareTypeCounter
        """
        return self._get_property('increment', PatternFlowArpHardwareTypeCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowArpHardwareTypeCounter
        """Factory property that returns an instance of the PatternFlowArpHardwareTypeCounter class

        integer counter pattern

        Returns: PatternFlowArpHardwareTypeCounter
        """
        return self._get_property('decrement', PatternFlowArpHardwareTypeCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowArpHardwareTypeCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 1,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    ETHERNET = 1 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowArpHardwareTypeCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowArpProtocolType(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowArpProtocolTypeCounter'},
        'decrement': {'type': 'PatternFlowArpProtocolTypeCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 2048,
        'values': [2048],
    } # type: Dict[str, Union(type)]

    IPV4 = 2048 #
    IPV6 = 34525 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowArpProtocolType, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowArpProtocolTypeCounter
        """Factory property that returns an instance of the PatternFlowArpProtocolTypeCounter class

        integer counter pattern

        Returns: PatternFlowArpProtocolTypeCounter
        """
        return self._get_property('increment', PatternFlowArpProtocolTypeCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowArpProtocolTypeCounter
        """Factory property that returns an instance of the PatternFlowArpProtocolTypeCounter class

        integer counter pattern

        Returns: PatternFlowArpProtocolTypeCounter
        """
        return self._get_property('decrement', PatternFlowArpProtocolTypeCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowArpProtocolTypeCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 2048,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    IPV4 = 2048 #
    IPV6 = 34525 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowArpProtocolTypeCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowArpHardwareLength(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowArpHardwareLengthCounter'},
        'decrement': {'type': 'PatternFlowArpHardwareLengthCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 6,
        'values': [6],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowArpHardwareLength, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowArpHardwareLengthCounter
        """Factory property that returns an instance of the PatternFlowArpHardwareLengthCounter class

        integer counter pattern

        Returns: PatternFlowArpHardwareLengthCounter
        """
        return self._get_property('increment', PatternFlowArpHardwareLengthCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowArpHardwareLengthCounter
        """Factory property that returns an instance of the PatternFlowArpHardwareLengthCounter class

        integer counter pattern

        Returns: PatternFlowArpHardwareLengthCounter
        """
        return self._get_property('decrement', PatternFlowArpHardwareLengthCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowArpHardwareLengthCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 6,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowArpHardwareLengthCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowArpProtocolLength(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowArpProtocolLengthCounter'},
        'decrement': {'type': 'PatternFlowArpProtocolLengthCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 4,
        'values': [4],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowArpProtocolLength, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowArpProtocolLengthCounter
        """Factory property that returns an instance of the PatternFlowArpProtocolLengthCounter class

        integer counter pattern

        Returns: PatternFlowArpProtocolLengthCounter
        """
        return self._get_property('increment', PatternFlowArpProtocolLengthCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowArpProtocolLengthCounter
        """Factory property that returns an instance of the PatternFlowArpProtocolLengthCounter class

        integer counter pattern

        Returns: PatternFlowArpProtocolLengthCounter
        """
        return self._get_property('decrement', PatternFlowArpProtocolLengthCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowArpProtocolLengthCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 4,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowArpProtocolLengthCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowArpOperation(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {'type': int},
        'values': {'type': list},
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowArpOperationCounter'},
        'decrement': {'type': 'PatternFlowArpOperationCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': 1,
        'values': [1],
    } # type: Dict[str, Union(type)]

    REQUEST = 1 #
    REPLY = 2 #

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowArpOperation, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowArpOperationCounter
        """Factory property that returns an instance of the PatternFlowArpOperationCounter class

        integer counter pattern

        Returns: PatternFlowArpOperationCounter
        """
        return self._get_property('increment', PatternFlowArpOperationCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowArpOperationCounter
        """Factory property that returns an instance of the PatternFlowArpOperationCounter class

        integer counter pattern

        Returns: PatternFlowArpOperationCounter
        """
        return self._get_property('decrement', PatternFlowArpOperationCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowArpOperationCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {'type': int},
        'step': {'type': int},
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': 1,
        'step': 1,
        'count': 1,
    } # type: Dict[str, Union(type)]

    REQUEST = 1 #
    REPLY = 2 #

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowArpOperationCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowArpSenderHardwareAddr(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {
            'type': str,
            'format': 'mac',
        },
        'values': {
            'type': list,
            'format': 'mac',
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowArpSenderHardwareAddrCounter'},
        'decrement': {'type': 'PatternFlowArpSenderHardwareAddrCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': '00:00:00:00:00:00',
        'values': ['00:00:00:00:00:00'],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowArpSenderHardwareAddr, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowArpSenderHardwareAddrCounter
        """Factory property that returns an instance of the PatternFlowArpSenderHardwareAddrCounter class

        mac counter pattern

        Returns: PatternFlowArpSenderHardwareAddrCounter
        """
        return self._get_property('increment', PatternFlowArpSenderHardwareAddrCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowArpSenderHardwareAddrCounter
        """Factory property that returns an instance of the PatternFlowArpSenderHardwareAddrCounter class

        mac counter pattern

        Returns: PatternFlowArpSenderHardwareAddrCounter
        """
        return self._get_property('decrement', PatternFlowArpSenderHardwareAddrCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowArpSenderHardwareAddrCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {
            'type': str,
            'format': 'mac',
        },
        'step': {
            'type': str,
            'format': 'mac',
        },
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': '00:00:00:00:00:00',
        'step': '00:00:00:00:00:01',
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowArpSenderHardwareAddrCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowArpSenderProtocolAddr(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {
            'type': str,
            'format': 'ipv4',
        },
        'values': {
            'type': list,
            'format': 'ipv4',
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowArpSenderProtocolAddrCounter'},
        'decrement': {'type': 'PatternFlowArpSenderProtocolAddrCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': '0.0.0.0',
        'values': ['0.0.0.0'],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowArpSenderProtocolAddr, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowArpSenderProtocolAddrCounter
        """Factory property that returns an instance of the PatternFlowArpSenderProtocolAddrCounter class

        ipv4 counter pattern

        Returns: PatternFlowArpSenderProtocolAddrCounter
        """
        return self._get_property('increment', PatternFlowArpSenderProtocolAddrCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowArpSenderProtocolAddrCounter
        """Factory property that returns an instance of the PatternFlowArpSenderProtocolAddrCounter class

        ipv4 counter pattern

        Returns: PatternFlowArpSenderProtocolAddrCounter
        """
        return self._get_property('decrement', PatternFlowArpSenderProtocolAddrCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowArpSenderProtocolAddrCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {
            'type': str,
            'format': 'ipv4',
        },
        'step': {
            'type': str,
            'format': 'ipv4',
        },
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': '0.0.0.0',
        'step': '0.0.0.1',
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowArpSenderProtocolAddrCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property('count', value)


class PatternFlowArpTargetHardwareAddr(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'choice': {
            'type': str,
            'enum': [
                'value',
                'values',
                'increment',
                'decrement',
            ],
        },
        'value': {
            'type': str,
            'format': 'mac',
        },
        'values': {
            'type': list,
            'format': 'mac',
        },
        'metric_group': {'type': str},
        'increment': {'type': 'PatternFlowArpTargetHardwareAddrCounter'},
        'decrement': {'type': 'PatternFlowArpTargetHardwareAddrCounter'},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'choice': 'value',
        'value': '00:00:00:00:00:00',
        'values': ['00:00:00:00:00:00'],
    } # type: Dict[str, Union(type)]

    VALUE = 'value' # type: str
    VALUES = 'values' # type: str
    INCREMENT = 'increment' # type: str
    DECREMENT = 'decrement' # type: str

    def __init__(self, parent=None, choice=None, metric_group=None):
        super(PatternFlowArpTargetHardwareAddr, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('metric_group', metric_group)

    @property
    def increment(self):
        # type: () -> PatternFlowArpTargetHardwareAddrCounter
        """Factory property that returns an instance of the PatternFlowArpTargetHardwareAddrCounter class

        mac counter pattern

        Returns: PatternFlowArpTargetHardwareAddrCounter
        """
        return self._get_property('increment', PatternFlowArpTargetHardwareAddrCounter, self, 'increment')

    @property
    def decrement(self):
        # type: () -> PatternFlowArpTargetHardwareAddrCounter
        """Factory property that returns an instance of the PatternFlowArpTargetHardwareAddrCounter class

        mac counter pattern

        Returns: PatternFlowArpTargetHardwareAddrCounter
        """
        return self._get_property('decrement', PatternFlowArpTargetHardwareAddrCounter, self, 'decrement')

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property('choice')

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property('choice', value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property('value')

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property('value', value, 'value')

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property('values')

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property('values', value, 'values')

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property('metric_group')

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property('metric_group', value)


class PatternFlowArpTargetHardwareAddrCounter(OpenApiObject):
    __slots__ = ('_parent', '_choice')

    _TYPES = {
        'start': {
            'type': str,
            'format': 'mac',
        },
        'step': {
            'type': str,
            'format': 'mac',
        },
        'count': {'type': int},
    } # type: Dict[str, str]

    _REQUIRED= () # type: tuple(str)

    _DEFAULTS = {
        'start': '00:00:00:00:00:00',
        'step': '00:00:00:00:00:01',
        'count': 1,
    } # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, start=None, step=None, count=None):
        super(PatternFlowArpTargetHardwareAddrCounter, self).__init__()
        self._parent = parent
        self._choice = choice
        self._set_property('start', start)
        self._set_property('step', step)
        self._set_property('count', count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property('start')

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property('start', value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property('step')

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property('step', value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property('count')

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._