"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiPipelineLib = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const codepipeline_actions = require("@aws-cdk/aws-codepipeline-actions");
const ec2 = require("@aws-cdk/aws-ec2");
const events = require("@aws-cdk/aws-events");
const event_targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const imagebuilder = require("@aws-cdk/aws-imagebuilder");
const kms = require("@aws-cdk/aws-kms");
const sns = require("@aws-cdk/aws-sns");
const cdk = require("@aws-cdk/core");
const component_builder_1 = require("./component_builder");
const image_recipe_version_bump_1 = require("./image_recipe_version_bump");
const slack_notification_1 = require("./slack_notification");
const source_action_builder_1 = require("./source_action_builder");
const ssmupdate = require("./ssm_update");
const utils = require("./utils");
/**
 * Construct for creating a Codepipeline, EC2 Image builder pipeline from 1 pipeline configuration.
 *
 * @stability stable
 */
class AmiPipelineLib extends cdk.Construct {
    /**
     * Constructor.
     *
     * @stability stable
     */
    constructor(scope, id, pipelineConfig, componentDepsConfig, slackConfig) {
        super(scope, id);
        this.id = id;
        this.pipelineConfig = pipelineConfig;
        this.pipelineConfig.name = this.id;
        this.componentDepsConfig = componentDepsConfig;
        this.slackConfig = slackConfig;
        this.componentBuilder = new component_builder_1.ComponentBuilder(this, componentDepsConfig, this.id, this.pipelineConfig.parent_image);
        this.sourceActionBuilder = new source_action_builder_1.SourceActionBuilder(this, pipelineConfig.sources, this.id);
        this.createImagebuilderPipeline();
        this.createCodepipelineProject();
        this.createScheduledTask();
        if (this.topic) {
            new ssmupdate.SsmUpdateConstruct(this, 'SSMUpdate', this.topic, this.pipelineConfig);
            new slack_notification_1.SlackNotification(this, 'SlackNotification', this.topic, slackConfig, `${this.id}Recipe`);
        }
    }
    /**
     * @stability stable
     */
    createScheduledTask() {
        if (this.codepipeline && this.pipelineConfig.schedule) {
            const pipelineTarget = new event_targets.CodePipeline(this.codepipeline);
            new events.Rule(this, 'ScheduleRule', {
                schedule: events.Schedule.expression(this.pipelineConfig.schedule),
                targets: [pipelineTarget],
            });
        }
    }
    /**
     * @stability stable
     */
    createImagebuilderPipeline() {
        let compMap = this.componentBuilder.createComponentDependenciesMap();
        let image_id;
        let parentImage;
        if ('disk_size' in this.pipelineConfig && this.pipelineConfig.disk_size) {
            this.diskSize = this.pipelineConfig.disk_size;
        }
        if ('image_id' in this.pipelineConfig && this.pipelineConfig.image_id) {
            image_id = this.pipelineConfig.image_id;
        }
        else {
            switch (this.pipelineConfig.parent_image) {
                case 'AmazonLinux2':
                    parentImage = ec2.MachineImage.latestAmazonLinux({
                        generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                        edition: ec2.AmazonLinuxEdition.STANDARD,
                        virtualization: ec2.AmazonLinuxVirt.HVM,
                        storage: ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
                        cpuType: ec2.AmazonLinuxCpuType.X86_64,
                    });
                    break;
                default:
                    parentImage = ec2.MachineImage.lookup(this.getLookupCriteria(this.pipelineConfig.parent_image));
                    break;
            }
        }
        // Use a custom block device mapping if encryption is required.
        let blockDeviceMappings = this.createBlockDeviceMapping();
        this.recipe = new imagebuilder.CfnImageRecipe(this, 'ImageRecipe', {
            name: `${this.id}Recipe`,
            parentImage: image_id ? image_id : parentImage ? parentImage.getImage(this).imageId : null,
            version: this.getNextRecipeVersion(`${this.id}Recipe`),
            components: this.pipelineConfig.recipe.components.map((c) => ({
                componentArn: compMap[c.name].ref,
            })),
            blockDeviceMappings: blockDeviceMappings,
            tags: {
                ShortName: `${this.id}Recipe`,
            },
        });
        const builderRole = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'),
            ],
            inlinePolicies: this.getInlinePolicies(this.pipelineConfig),
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [builderRole.roleName],
        });
        this.topic = new sns.Topic(this, 'AmiPipelineTopic');
        this.infrastructure = new imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: `${this.id} - EC2 IB Infrastructure`,
            instanceProfileName: instanceProfile.ref,
            snsTopicArn: this.topic.topicArn,
            instanceTypes: this.pipelineConfig.instance_type ? [this.pipelineConfig.instance_type] : undefined,
            terminateInstanceOnFailure: !('terminate_on_failure' in this.pipelineConfig) || this.pipelineConfig.terminate_on_failure,
            subnetId: this.pipelineConfig.subnet_id,
            securityGroupIds: this.pipelineConfig.security_group_ids ? this.pipelineConfig.security_group_ids.split(',') : undefined,
        });
        this.distributionConfig = this.pipelineConfig.shared_with ? new imagebuilder.CfnDistributionConfiguration(this, 'DistributionConfig', {
            name: `${this.id} - Distribution`,
            distributions: this.pipelineConfig.shared_with.map((s) => ({
                region: s.region,
                amiDistributionConfiguration: {
                    Name: `${this.id} - {{ imagebuilder:buildDate }}`,
                    Description: `${this.id} Shared image`,
                    AmiTags: {
                        Name: this.id,
                    },
                    TargetAccountIds: s.account_ids,
                },
            })),
        }) : undefined;
        this.imagePipeline = new imagebuilder.CfnImagePipeline(this, 'AmiPipeline', {
            name: `${this.id} - AMI Pipeline`,
            imageRecipeArn: this.recipe.ref,
            infrastructureConfigurationArn: this.infrastructure.ref,
            distributionConfigurationArn: this.distributionConfig ? this.distributionConfig.ref : undefined,
        });
    }
    getInlinePolicies(pipelineConfig) {
        let result = {};
        const statements = pipelineConfig.policy_statements;
        if (statements) {
            for (let statement in statements) {
                result[statement] = new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            sid: statement,
                            actions: statements[statement].actions,
                            effect: iam.Effect.ALLOW,
                            resources: statements[statement].resources,
                        }),
                    ],
                });
            }
        }
        return result;
    }
    createBlockDeviceMapping() {
        const key = this.pipelineConfig.encrypted ? new kms.Key(this, 'PipelineEncryption', {
            description: `Image encryption for ${this.pipelineConfig.name}`,
            alias: `${utils.pascalize(this.pipelineConfig.name)}_key`,
        }) : null;
        let blockDeviceMappings = [
            {
                deviceName: this.pipelineConfig.parent_image.includes('AmazonLinux') ? '/dev/xvda' : '/dev/sda1',
                ebs: {
                    encrypted: key ? true : false,
                    volumeSize: this.pipelineConfig.disk_size ? this.pipelineConfig.disk_size : 8,
                    volumeType: 'gp2',
                    kmsKeyId: key === null || key === void 0 ? void 0 : key.keyArn,
                },
            },
        ];
        return blockDeviceMappings;
    }
    /**
     * @stability stable
     */
    getNextRecipeVersion(recipeName) {
        const recipeBumpCustomResource = new image_recipe_version_bump_1.ImageRecipeVersionBump(this, 'ImageRecipeVersionBump', recipeName);
        return recipeBumpCustomResource.nextVersion;
    }
    /**
     * @stability stable
     */
    createCodepipelineProject() {
        var _b;
        if (this.imagePipeline) {
            // Create CodePipeline project
            let imagebuilderBuild = new codebuild.PipelineProject(this, 'AmiPipelineBuild', {
                buildSpec: codebuild.BuildSpec.fromObject({
                    version: '0.2',
                    phases: {
                        install: {
                            commands: [
                                'apt-get update',
                                'apt-get install awscli -y',
                            ],
                        },
                        pre_build: {
                            commands: [
                                'curl -X POST -H \'Content-type: application/json\' --data "{\\"text\\":\\"$PIPELINE_NAME build has started.\\"}" $SLACK_WEBHOOK_URL',
                            ],
                        },
                        build: {
                            commands: 'aws imagebuilder start-image-pipeline-execution --image-pipeline-arn $IMAGE_PIPELINE_ARN',
                        },
                    },
                }),
                environment: {
                    buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
                    environmentVariables: {
                        IMAGE_PIPELINE_ARN: { value: this.imagePipeline.ref },
                        SLACK_WEBHOOK_URL: { value: this.slackConfig.slackWebhookUrl },
                        SLACK_CHANNEL: { value: this.slackConfig.channel },
                        SLACK_USERNAME: { value: this.slackConfig.username },
                        PIPELINE_NAME: { value: this.imagePipeline.name },
                    },
                },
            });
            imagebuilderBuild.addToRolePolicy(iam.PolicyStatement.fromJson({
                Sid: 'imagebuilderstart',
                Effect: 'Allow',
                Action: ['imagebuilder:StartImagePipelineExecution'],
                Resource: '*',
            }));
            const imageBuilderOutput = new codepipeline.Artifact('CdkBuildOutput');
            let sourceActions = this.sourceActionBuilder.createPipelineSources();
            const buildAction = new codepipeline_actions.CodeBuildAction({
                actionName: 'StartImageBuilder',
                project: imagebuilderBuild,
                input: (_b = sourceActions.find(a => a && a.sourceOutput)) === null || _b === void 0 ? void 0 : _b.sourceOutput,
                extraInputs: sourceActions.length > 1 ? sourceActions.slice(1).map(a => a.sourceOutput) : [],
                outputs: [imageBuilderOutput],
            });
            // Complete Pipeline Project
            this.codepipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                pipelineName: this.pipelineConfig.name,
                restartExecutionOnUpdate: true,
                stages: [
                    {
                        stageName: 'Source',
                        actions: sourceActions.map(c => c.action).filter((c) => c != null),
                    },
                    {
                        stageName: 'Build',
                        actions: [buildAction],
                    },
                ],
            });
        }
    }
    /**
     * @stability stable
     */
    getLookupCriteria(parentImage) {
        switch (parentImage) {
            case 'Ubuntu1804':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-bionic-18.04-amd64*',
                    owners: ['099720109477'],
                };
            case 'Ubuntu2004':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-focal-20.04-amd64*',
                    owners: ['099720109477'],
                };
            case 'CentOS7':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['cvugziknvmxgqna9noibqnnsy'],
                    },
                };
            case 'CentOS8':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['47k9ia2igxpcce2bzo8u3kj03'],
                    },
                };
            default:
                return {
                    name: 'amzn-linux',
                };
        }
    }
}
exports.AmiPipelineLib = AmiPipelineLib;
_a = JSII_RTTI_SYMBOL_1;
AmiPipelineLib[_a] = { fqn: "halloumi-ami-pipelines.AmiPipelineLib", version: "0.0.18" };
//# sourceMappingURL=data:application/json;base64,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