"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualGateway = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const gateway_route_1 = require("./gateway-route");
const mesh_1 = require("./mesh");
const utils_1 = require("./private/utils");
const virtual_gateway_listener_1 = require("./virtual-gateway-listener");
class VirtualGatewayBase extends cdk.Resource {
    /**
     * Utility method to add a new GatewayRoute to the VirtualGateway.
     *
     * @stability stable
     */
    addGatewayRoute(id, props) {
        return new gateway_route_1.GatewayRoute(this, id, {
            ...props,
            virtualGateway: this,
        });
    }
    /**
     * Grants the given entity `appmesh:StreamAggregatedResources`.
     *
     * @stability stable
     */
    grantStreamAggregatedResources(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['appmesh:StreamAggregatedResources'],
            resourceArns: [this.virtualGatewayArn],
        });
    }
}
/**
 * VirtualGateway represents a newly defined App Mesh Virtual Gateway.
 *
 * A virtual gateway allows resources that are outside of your mesh to communicate to resources that
 * are inside of your mesh.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html
 * @stability stable
 */
class VirtualGateway extends VirtualGatewayBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.virtualGatewayName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        /**
         * @stability stable
         */
        this.listeners = new Array();
        this.mesh = props.mesh;
        if (!props.listeners) {
            // Use listener default of http listener port 8080 if no listener is defined
            this.listeners.push(virtual_gateway_listener_1.VirtualGatewayListener.http().bind(this));
        }
        else {
            props.listeners.forEach(listener => this.listeners.push(listener.bind(this)));
        }
        const accessLogging = (_b = props.accessLog) === null || _b === void 0 ? void 0 : _b.bind(this);
        const node = new appmesh_generated_1.CfnVirtualGateway(this, 'Resource', {
            virtualGatewayName: this.physicalName,
            meshName: this.mesh.meshName,
            meshOwner: utils_1.renderMeshOwner(this.env.account, this.mesh.env.account),
            spec: {
                listeners: this.listeners.map(listener => listener.listener),
                backendDefaults: props.backendDefaults !== undefined
                    ? {
                        clientPolicy: {
                            tls: utils_1.renderTlsClientPolicy(this, (_c = props.backendDefaults) === null || _c === void 0 ? void 0 : _c.tlsClientPolicy),
                        },
                    }
                    : undefined,
                logging: accessLogging !== undefined ? {
                    accessLog: accessLogging.virtualGatewayAccessLog,
                } : undefined,
            },
        });
        this.virtualGatewayName = this.getResourceNameAttribute(node.attrVirtualGatewayName);
        this.virtualGatewayArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualGateway`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualGateway given an ARN.
     *
     * @stability stable
     */
    static fromVirtualGatewayArn(scope, id, virtualGatewayArn) {
        return new class extends VirtualGatewayBase {
            constructor() {
                super(...arguments);
                this.parsedArn = cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(virtualGatewayArn).resourceName);
                this.mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', cdk.Fn.select(0, this.parsedArn));
                this.virtualGatewayArn = virtualGatewayArn;
                this.virtualGatewayName = cdk.Fn.select(2, this.parsedArn);
            }
        }(scope, id);
    }
    /**
     * Import an existing VirtualGateway given its attributes.
     *
     * @stability stable
     */
    static fromVirtualGatewayAttributes(scope, id, attrs) {
        return new class extends VirtualGatewayBase {
            constructor() {
                super(...arguments);
                this.mesh = attrs.mesh;
                this.virtualGatewayName = attrs.virtualGatewayName;
                this.virtualGatewayArn = cdk.Stack.of(this).formatArn({
                    service: 'appmesh',
                    resource: `mesh/${attrs.mesh.meshName}/virtualGateway`,
                    resourceName: this.virtualGatewayName,
                });
            }
        }(scope, id);
    }
}
exports.VirtualGateway = VirtualGateway;
_a = JSII_RTTI_SYMBOL_1;
VirtualGateway[_a] = { fqn: "@aws-cdk/aws-appmesh.VirtualGateway", version: "1.125.0" };
//# sourceMappingURL=data:application/json;base64,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