"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const log_retention_1 = require("../lib/log-retention");
module.exports = {
    'log retention construct'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new log_retention_1.LogRetention(stack, 'MyLambda', {
            logGroupName: 'group',
            retention: logs.RetentionDays.ONE_MONTH,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'logs:PutRetentionPolicy',
                            'logs:DeleteRetentionPolicy',
                        ],
                        'Effect': 'Allow',
                        'Resource': '*',
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aServiceRoleDefaultPolicyADDA7DEB',
            'Roles': [
                {
                    'Ref': 'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aServiceRole9741ECFB',
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            'ServiceToken': {
                'Fn::GetAtt': [
                    'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aFD4BFC8A',
                    'Arn',
                ],
            },
            'LogGroupName': 'group',
            'RetentionInDays': 30,
        }));
        test.done();
    },
    'with imported role'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const role = iam.Role.fromRoleArn(stack, 'Role', 'arn:aws:iam::123456789012:role/CoolRole');
        // WHEN
        new log_retention_1.LogRetention(stack, 'MyLambda', {
            logGroupName: 'group',
            retention: logs.RetentionDays.ONE_MONTH,
            role,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'logs:PutRetentionPolicy',
                            'logs:DeleteRetentionPolicy',
                        ],
                        'Effect': 'Allow',
                        'Resource': '*',
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'RolePolicy72E7D967',
            'Roles': [
                'CoolRole',
            ],
        }));
        assert_1.expect(stack).to(assert_1.countResources('AWS::IAM::Role', 0));
        test.done();
    },
    'with RetentionPeriod set to Infinity'(test) {
        const stack = new cdk.Stack();
        new log_retention_1.LogRetention(stack, 'MyLambda', {
            logGroupName: 'group',
            retention: logs.RetentionDays.INFINITE,
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            RetentionInDays: assert_1.ABSENT,
        }));
        test.done();
    },
    'log group ARN is well formed and conforms'(test) {
        const stack = new cdk.Stack();
        const group = new log_retention_1.LogRetention(stack, 'MyLambda', {
            logGroupName: 'group',
            retention: logs.RetentionDays.ONE_MONTH,
        });
        const logGroupArn = group.logGroupArn;
        test.ok(logGroupArn.indexOf('logs') > -1, 'log group ARN is not as expected');
        test.ok(logGroupArn.indexOf('log-group') > -1, 'log group ARN is not as expected');
        test.ok(logGroupArn.endsWith(':*'), 'log group ARN is not as expected');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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