"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("@aws-cdk/aws-codepipeline");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * If and how the GitHub source action should be triggered
 */
var GitHubTrigger;
(function (GitHubTrigger) {
    GitHubTrigger["NONE"] = "None";
    GitHubTrigger["POLL"] = "Poll";
    GitHubTrigger["WEBHOOK"] = "WebHook";
})(GitHubTrigger = exports.GitHubTrigger || (exports.GitHubTrigger = {}));
/**
 * Source that is provided by a GitHub repository.
 */
class GitHubSourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: codepipeline.ActionCategory.SOURCE,
            owner: 'ThirdParty',
            provider: 'GitHub',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            repositoryName: this.variableExpression('RepositoryName'),
            branchName: this.variableExpression('BranchName'),
            authorDate: this.variableExpression('AuthorDate'),
            committerDate: this.variableExpression('CommitterDate'),
            commitId: this.variableExpression('CommitId'),
            commitMessage: this.variableExpression('CommitMessage'),
            commitUrl: this.variableExpression('CommitUrl'),
        };
    }
    bound(scope, stage, _options) {
        if (!this.props.trigger || this.props.trigger === GitHubTrigger.WEBHOOK) {
            new codepipeline.CfnWebhook(scope, 'WebhookResource', {
                authentication: 'GITHUB_HMAC',
                authenticationConfiguration: {
                    secretToken: this.props.oauthToken.toString(),
                },
                filters: [
                    {
                        jsonPath: '$.ref',
                        matchEquals: 'refs/heads/{Branch}',
                    },
                ],
                targetAction: this.actionProperties.actionName,
                targetPipeline: stage.pipeline.pipelineName,
                targetPipelineVersion: 1,
                registerWithThirdParty: true,
            });
        }
        return {
            configuration: {
                Owner: this.props.owner,
                Repo: this.props.repo,
                Branch: this.props.branch || "master",
                OAuthToken: this.props.oauthToken.toString(),
                PollForSourceChanges: this.props.trigger === GitHubTrigger.POLL,
            },
        };
    }
}
exports.GitHubSourceAction = GitHubSourceAction;
//# sourceMappingURL=data:application/json;base64,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