# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['grpc_argument_validator']

package_data = \
{'': ['*']}

install_requires = \
['grpcio', 'grpcio-status', 'protobuf']

setup_kwargs = {
    'name': 'grpc-argument-validator',
    'version': '0.2.2',
    'description': 'gRPC argument validator utility.',
    'long_description': '[![codecov](https://codecov.io/gh/messagebird/python-grpc-argument-validator/branch/master/graph/badge.svg)](https://codecov.io/gh/messagebird/python-grpc-argument-validator)\n[![PyPI](https://img.shields.io/pypi/v/grpc-argument-validator.svg?color=blue)](https://pypi.org/project/python-grpc-argument-validator/)\n[![License](https://img.shields.io/github/license/messagebird/python-grpc-argument-validator)](https://opensource.org/licenses/BSD-3-Clause)\n[![Docs](https://img.shields.io/static/v1?label=Docs&message=Github%20Pages&color=blue)](https://messagebird.github.io/python-grpc-argument-validator/grpc_argument_validator/)\n[![tests](https://github.com/messagebird/python-grpc-argument-validator/workflows/Tests/badge.svg)](https://github.com/messagebird/python-grpc-argument-validator/actions?workflow=tests)\n\n\n# gRPC argument validator\ngRPC argument validator is a library that provides decorators to automatically validate arguments in requests to rpc methods.\n\n<!-- GETTING STARTED -->\n## Getting Started\n\nThis is an example of how you may give instructions on setting up your project locally.\nTo get a local copy up and running follow these simple example steps.\n\n### Installation\n\n#### From PyPI\n```sh\npip install grpc-argument-validator\n```\n\n#### From source\n\n- Install [`poetry`](https://python-poetry.org/docs/)\n\n- Clone repo\n\n```sh\ngit clone https://github.com/messagebird/python-grpc-argument-validator.git\n```\n\n- Install packages\n\n```sh\ncd python-grpc-argument-validator && poetry install\n```\n\n- Run the tests\n\n```sh\ncd src/tests\npoetry run python -m unittest\n```\n\n\n\n<!-- USAGE EXAMPLES -->\n## Quick Example\n```python\nfrom google.protobuf.descriptor import FieldDescriptor\nfrom grpc_argument_validator import validate_args\nfrom grpc_argument_validator import AbstractArgumentValidator, ValidationResult, ValidationContext\n\nclass PathValidator(AbstractArgumentValidator):\n\n    def check(self, name: str, value: Path, field_descriptor: FieldDescriptor, validation_context: ValidationContext) -> ValidationResult:\n        if len(value.points) > 5:\n            return ValidationResult(valid=True)\n        return ValidationResult(False, f"path for \'{name}\' should be at least five points long")\n\nclass RouteService(RouteCheckerServicer):\n    @validate_args(\n        non_empty=["tags", "tags[]", "path.points"],\n        validators={"path": PathValidator()},\n    )\n    def Create(self, request: Route, context: grpc.ServicerContext):\n        return BoolValue(value=True)\n```\n\n<!-- DOCUMENTATION -->\n## Documentation\n\nWe host the full API reference on [GitHub pages](https://messagebird.github.io/python-grpc-argument-validator/grpc_argument_validator/).\n\n### Argument field syntax\nTo specify which argument field should be validated, `grpc-argument-validator` expects strings that match the field names\nas defined in the protobufs. To access nested fields, use a dot (`.`).\n\nConsider the following protobuf definition:\n```protobuf\nsyntax = "proto3";\n\npackage routeguide;\n\nimport "google/protobuf/empty.proto";\nimport "google/protobuf/wrappers.proto";\n\nmessage Point {\n  int32 x = 1;\n  int32 y = 2;\n  google.protobuf.StringValue name = 3;\n}\n\nmessage Rectangle {\n  Point lo = 1;\n  Point hi = 2;\n}\n\nmessage Area {\n  Rectangle rectangle = 1;\n  google.protobuf.StringValue message = 2;\n  google.protobuf.BytesValue uuid = 3;\n}\n\nmessage Path {\n  repeated Point points = 1;\n}\n\nenum Planet {\n  PLANET_INVALID = 0;\n  PLANET_EARTH = 1;\n  PLANET_MARS = 2;\n}\n\nmessage PlanetValue {\n  Planet value = 1;\n}\n\nmessage Route {\n  Path path = 1;\n  google.protobuf.StringValue name = 2;\n  PlanetValue planet = 3;\n  repeated string tags = 4;\n}\n\nservice RouteService {\n  rpc CreateRoute(Route) returns (google.protobuf.Empty);\n  rpc CreateArea(Area) returns (google.protobuf.Empty);\n}\n```\n- If you want to validate the field `planet` in a `Route` proto, simply specify `"planet"` or equivalently `".planet"`.\n- If you want to validate the `value` field within the `name` field of a `Route` proto, use `"name.value"` or\nequivalently `".name.value"`.\n- If you want to apply a check to each element of a `repeated` field, append `[]` to the name of the field.\n- If you want to apply a check to the \'root proto\' (i.e. the request itself), use `"."` as the field path.\n\nTo clarify this, let\'s say that we know that both `planet` and `name.value` should have non-default values. We can then\ndecorate a method in our gRPC server as follows:\n\n```python\nimport grpc\nfrom google.protobuf.empty_pb2 import Empty\nfrom grpc_argument_validator import validate_args\nfrom tests.route_guide_protos.route_guide_pb2 import Route\nfrom tests.route_guide_protos.route_guide_pb2_grpc import RouteServiceServicer\n\n\nclass RouteServiceImpl(RouteServiceServicer):\n    @validate_args(non_empty=["planet", "name.value"])\n    def CreateRoute(self, request: Route, context: grpc.ServicerContext):\n        return Empty()\n```\n\nCalling the service with a default value for either `planet` or `name.value` will yield an `INVALID_ARGUMENT` status code\nwith further details on which fields violate the validation.\n\n### Validators\nThere are two kinds of validators you might consider:\n\n- There are predefined validators which we will cover shortly\n- Another option is to define your own validators\n\nIn the examples below, we have used exactly one validator + field path per `validate_args` decorator for clarity.\nFortunately, our API allows you to use multiple validators and fields!\n\n#### \'Has\' validator\nThe simplest of all predefined validators is the \'has\' validator which simply checks whether a `HasField` evaluates to\n`True`. This of course works in combination with nested fields.\n\nIn the example below, calling the `Create` endpoint without setting `Route.name` would result in an `INVALID_ARGUMENT`\nstatus.\n```python\n\nclass RouteServiceImpl(RouteServiceServicer):\n    @validate_args(has=["name"])\n    def CreateRoute(self, request: Route, context: grpc.ServicerContext):\n        return Empty()\n```\nRun this on a local machine and make a request with an invalid argument:\n```python\nwith grpc.insecure_channel("127.0.0.1:50051") as c:\n    route_client = RouteServiceStub(channel=c)\n    try:\n        route_client.CreateRoute(Route(tags=["tag"]))\n    except grpc.RpcError as e:\n        if isinstance(e, grpc.Call):\n            print(e.details())\n```\nThe following will be printed:\n`must have \'name\'`\n\n#### UUID validator\nAnother common use-case is the validation of UUIDs. You can enlist the fields that should be UUIDs (represented as\n16 bytes) with the `uuids` argument:\n```python\nclass RouteServiceImpl(RouteServiceServicer):\n    @validate_args(uuids=["uuid.value"])\n    def CreateArea(self, request: Area, context: grpc.ServicerContext):\n        return Empty()\n```\nThe client side might violate the UUID requirement as follows:\n```python\nwith grpc.insecure_channel("127.0.0.1:50051") as c:\n    route_client = RouteServiceStub(channel=c)\n    try:\n        route_client.CreateArea(Area(uuid=BytesValue(value="not a uuid".encode())))\n    except grpc.RpcError as e:\n        if isinstance(e, grpc.Call):\n            print(e.details())\n```\nThis will print `\'uuid.value\' must be a valid UUID`.\n\n#### Non-default validator\nFor fields that should have a non-default value, such as\n[enums](https://developers.google.com/protocol-buffers/docs/style#enums), we have provided the `non_default` argument:\n```python\nclass RouteServiceImpl(RouteServiceServicer):\n    @validate_args(non_default=["planet.value"])\n    def CreateRoute(self, request: Route, context: grpc.ServicerContext):\n        return Empty()\n```\nThe client side may violate this as follows:\n```python\nwith grpc.insecure_channel("127.0.0.1:50051") as c:\n    route_client = RouteServiceStub(channel=c)\n    try:\n        route_client.CreateRoute(Route(planet=PlanetValue()))\n    except grpc.RpcError as e:\n        if isinstance(e, grpc.Call):\n            print(e.details())\n```\nWhich will print `\'planet.value\' must have non-default value`.\n\n#### Non-empty validator\nWe provide a \'non-\'empty validator which can be used to ensure that a `repeated` field has more than zero elements.\n```python\nclass RouteServiceImpl(RouteServiceServicer):\n    @validate_args(non_empty=["tags"])\n    def CreateRoute(self, request: Route, context: grpc.ServicerContext):\n        return Empty()\n```\nWhich can be violated as follows:\n\n```python\nwith grpc.insecure_channel("127.0.0.1:50051") as c:\n    route_client = RouteServiceStub(channel=c)\n    try:\n        route_client.CreateRoute(Route(tags=[]))\n    except grpc.RpcError as e:\n        if isinstance(e, grpc.Call):\n            print(e.details())\n```\nWhich will print `\'tags\' must be non-empty`.\n\n#### Regexp validator\nFinally, we have the regexp validator that can be used to check whether a string field matches a regular expression.\n```python\nclass RouteServiceImpl(RouteServiceServicer):\n    @validate_args(validators={"message.value": RegexpValidator(pattern=r"\\d+")})\n    def CreateArea(self, request: Area, context: grpc.ServicerContext):\n        return Empty()\n```\n\n```python\nwith grpc.insecure_channel("127.0.0.1:50051") as c:\n    route_client = RouteServiceStub(channel=c)\n    try:\n        route_client.CreateArea(Area(message=StringValue(value="hello world")))\n    except grpc.RpcError as e:\n        if isinstance(e, grpc.Call):\n            print(e.details())\n```\nWhich will print `\'message.value\' must match regexp pattern: \\d+`.\n\n#### Custom validators\nYou can also write custom validators to flexibily handle your use-case. You need to derive a class from\n`AbstractArgumentValidator` and implement its `check` method. The example below shows how to implement a simple\nvalidator for checking that a path has 5 points. You can provide such custom validators through a `dict` that\nmaps a field path to a validator:\n```python\nfrom grpc_argument_validator import AbstractArgumentValidator\nfrom grpc_argument_validator import ValidationContext\nfrom grpc_argument_validator import ValidationResult\nfrom google.protobuf.descriptor import FieldDescriptor\n\nfrom examples.route_guide_pb2 import Path\n\nclass PathValidator(AbstractArgumentValidator):\n    def check(\n        self, name: str, value: Path, field_descriptor: FieldDescriptor, validation_context: ValidationContext\n    ) -> ValidationResult:\n        if len(value.points) > 5:\n            return ValidationResult(valid=True)\n        return ValidationResult(False, f"path for \'{name}\' should be at least five points long")\n\n\nclass RouteServiceImpl(RouteServiceServicer):\n\n    @validate_args(validators={"path": PathValidator()})\n    def CreateRoute(self, request: Route, context: grpc.ServicerContext):\n        return Empty()\n```\n\n### Optional vs. required validators\nFor each of the built-in validators (except for the `has` validator), `validate_args` has not one but two keyword\narguments. One of those is prepended with `optional_`. This means that apart from `uuid`, `non_default` and\n`non_empty` we also have `optional_uuid`, `optional_non_default` and `optional_non_empty`. The behavior is slightly\ndifferent: for any of the `optional_*` validators, it is OK if the field is not contained by the incoming request.\nSometimes fields are simply optional, and you only want to validate them _if_ they are present.\n\nSince it is also common that fields are _not optional_, we also provide the required validators (without `optional_*`)\nfor which [`HasField`](https://googleapis.dev/python/protobuf/latest/google/protobuf/message.html#google.protobuf.message.Message.HasField)\nmust evaluate to `True` for that field and all preceding fields in the protos hierarchy.\n\nThe custom validator counterparts are `validators` and `optional_validators`. Each takes a `dict` with a mapping of\nfield paths to validators. These can be used for validators that might be preconfigured such as the `RegexpValidator`\nor for customer validators.\n\n### Streaming requests\nYou can also use the validators for streaming requests. Since streaming requests might not all look the same in a\nsingle stream (e.g. the first request might have metadata describing the remainder of the stream), we provide a\nstreaming request index in a `ValidationContext` that is passed to an `AbstractArgumentValidator`.\n\nHere\'s an example of how that could be used:\n```python\nclass StreamingPathValidator(AbstractArgumentValidator):\n    def __init__(self, first_number_of_points: int, second_number_of_points: int):\n        self._first_number_of_points = first_number_of_points\n        self._second_number_of_points = second_number_of_points\n\n    def check(\n        self, name: str, value: Any, field_descriptor: FieldDescriptor, validation_context: ValidationContext\n    ) -> ValidationResult:\n        if not validation_context.is_streaming:\n            return ValidationResult(False, "request must be a streaming request")\n\n        if validation_context.streaming_message_index == 0:\n            if len(value.points) != self._first_number_of_points:\n                return ValidationResult(False, f"first path should have {self._first_number_of_points} points")\n\n        if validation_context.streaming_message_index == 1:\n            if len(value.points) != self._second_number_of_points:\n                return ValidationResult(False, f"second path should have {self._second_number_of_points} points")\n\n        return ValidationResult(True)\n```\n\n### Enabling rich error details\nTo enable [richer error responses](https://cloud.google.com/apis/design/errors#error_model) where each violation is\ncontained in a\n[`BadRequest` proto](https://github.com/googleapis/googleapis/blob/master/google/rpc/error_details.proto), you can use\n```python\nfrom grpc_argument_validator import ArgumentValidatorConfig\n\nArgumentValidatorConfig.set_rich_grpc_errors(enabled=True)\n```\n\nNow, your client-side can parse the error details as follows:\n```python\ndef extract_error_details(err):\n    status_proto = status_pb2.Status()\n\n    for metadatum in err.trailing_metadata():\n        if isinstance(metadatum, _Metadatum):\n            if metadatum.key == "grpc-status-details-bin":\n                status_proto.MergeFromString(metadatum.value)\n\n    unpacked = [_unpack_error_detail(det) for det in status_proto.details]\n    return unpacked\n\ndef _unpack_error_detail(grpc_detail):\n    val = error_details_pb2.BadRequest()\n    grpc_detail.Unpack(val)\n    return val\n\nwith grpc.insecure_channel("127.0.0.1:50051") as c:\n    route_client = RouteServiceStub(channel=c)\n    try:\n        route_client.CreateArea(Area(message=StringValue(value="hello world")))\n    except grpc.RpcError as e:\n        error_details = extract_error_details(e)\n        print(error_details)\n```\n\n<!-- CONTRIBUTING -->\n## Contributing\n\nContributions are what make the open source community such an amazing place to be learn, inspire, and create. Any contributions you make are **greatly appreciated**.\n\n1. Fork the Project\n2. Create your Feature Branch (`git checkout -b feature/AmazingFeature`)\n3. Commit your Changes (`git commit -m \'Add some AmazingFeature\'`)\n4. Push to the Branch (`git push origin feature/AmazingFeature`)\n5. Open a Pull Request\n\n### Generating HTML Documentation\nGenerate the docs by running:\n```sh\npdoc --html -o docs src/grpc_argument_validator\n```\n\n\n<!-- LICENSE -->\n## License\n\nDistributed under The BSD 3-Clause License. Copyright (c) 2021, MessageBird\n',
    'author': 'Jos van de Wolfshaar',
    'author_email': 'jos@messagebird.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/messagebird/python-grpc-argument-validator',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
