try:
    import thot  # noqa: F401
except ImportError:
    raise RuntimeError(
        'sil-machine must be installed with the "thot" extra in order to use the machine.translation.thot package.'
    )

from typing import Optional, Union

from .thot_fast_align_word_alignment_model import ThotFastAlignWordAlignmentModel
from .thot_hmm_word_alignment_model import ThotHmmWordAlignmentModel
from .thot_ibm1_word_alignment_model import ThotIbm1WordAlignmentModel
from .thot_ibm2_word_alignment_model import ThotIbm2WordAlignmentModel
from .thot_ibm3_word_alignment_model import ThotIbm3WordAlignmentModel
from .thot_ibm4_word_alignment_model import ThotIbm4WordAlignmentModel
from .thot_symmetrized_word_alignment_model import ThotSymmetrizedWordAlignmentModel
from .thot_word_alignment_model import ThotWordAlignmentModel
from .thot_word_alignment_model_trainer import ThotWordAlignmentModelTrainer
from .thot_word_alignment_model_type import ThotWordAlignmentModelType
from .thot_word_alignment_parameters import ThotWordAlignmentParameters


def create_thot_word_alignment_model(type: Union[str, int]) -> Optional[ThotWordAlignmentModel]:
    if isinstance(type, str):
        type = ThotWordAlignmentModelType[type.upper()]
    if type == ThotWordAlignmentModelType.FAST_ALIGN:
        return ThotFastAlignWordAlignmentModel()
    if type == ThotWordAlignmentModelType.IBM1:
        return ThotIbm1WordAlignmentModel()
    if type == ThotWordAlignmentModelType.IBM2:
        return ThotIbm2WordAlignmentModel()
    if type == ThotWordAlignmentModelType.HMM:
        return ThotHmmWordAlignmentModel()
    if type == ThotWordAlignmentModelType.IBM3:
        return ThotIbm3WordAlignmentModel()
    if type == ThotWordAlignmentModelType.IBM4:
        return ThotIbm4WordAlignmentModel()
    return None


def create_thot_symmetrized_word_alignment_model(type: Union[int, str]) -> Optional[ThotSymmetrizedWordAlignmentModel]:
    direct_model = create_thot_word_alignment_model(type)
    inverse_model = create_thot_word_alignment_model(type)
    if direct_model is None or inverse_model is None:
        return None
    return ThotSymmetrizedWordAlignmentModel(direct_model, inverse_model)


__all__ = [
    "ThotFastAlignWordAlignmentModel",
    "ThotHmmWordAlignmentModel",
    "ThotIbm1WordAlignmentModel",
    "ThotIbm2WordAlignmentModel",
    "ThotIbm3WordAlignmentModel",
    "ThotIbm4WordAlignmentModel",
    "ThotSymmetrizedWordAlignmentModel",
    "ThotWordAlignmentModel",
    "ThotWordAlignmentModelTrainer",
    "ThotWordAlignmentModelType",
    "ThotWordAlignmentParameters",
    "create_thot_symmetrized_word_alignment_model",
    "create_thot_word_alignment_model",
]
