#!/usr/bin/python

# Imports
from gevent.pywsgi import WSGIServer
from hackattack_awa_matrix import create_app
import logging
from logging.handlers import RotatingFileHandler
from pathlib import Path, WindowsPath
import configparser
import os
import inspect
import sys
import argparse


# Global variables

# Class declarations

# Function declarations


LOG_LEVEL = None
LOG_FILE_PATH = None
LISTEN_IP = None
LISTEN_PORT = None
DATABASE_LOCATION = None
FLASK_SECRET = None


def read_config(config_file_path):
    global LOG_LEVEL
    global LOG_FILE_PATH
    global LISTEN_IP
    global LISTEN_PORT
    global DATABASE_LOCATION
    global FLASK_SECRET

    config = configparser.ConfigParser()
    config.read(config_file_path)

    current_dir = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
    default_log_file = os.path.join(current_dir, 'hackattack_awa_matrix.log')

    LOG_LEVEL = config['DEFAULT']['LOG_LEVEL'] if 'LOG_LEVEL' in config['DEFAULT'] else None
    LOG_FILE_PATH = config['DEFAULT']['LOG_FILE_PATH'] if 'LOG_FILE_PATH' in config['DEFAULT'] else default_log_file
    LISTEN_IP = config['DEFAULT']['LISTEN_IP'] if 'LISTEN_IP' in config['DEFAULT'] else None
    LISTEN_PORT = int(config['DEFAULT']['LISTEN_PORT']) if 'LISTEN_PORT' in config['DEFAULT'] else None
    DATABASE_LOCATION = config['DEFAULT']['DATABASE_LOCATION'] if 'DATABASE_LOCATION' in config['DEFAULT'] else None
    FLASK_SECRET = bytes.fromhex(config['DEFAULT']['FLASK_SECRET']) if 'FLASK_SECRET' in config['DEFAULT'] else None


def make_sure_awa_matrix_dir_exists_in_home():
    home_path = Path.home()
    home_path_str = str(home_path)

    if type(home_path) is WindowsPath:
        home_path_str += "\\.hackattack_awa_matrix\\"
    else:
        home_path_str += "/.hackattack_awa_matrix/"

    if not os.path.exists(home_path_str):
        os.makedirs(home_path_str)

    return home_path_str


def write_default_config_file(config_file_location):
    home_path_str = make_sure_awa_matrix_dir_exists_in_home()

    default_config_content = [
            "[DEFAULT]",
            "",
            "# SPECIFY WHERE TO LISTEN",
            "LISTEN_IP=127.0.0.1",
            "LISTEN_PORT=5000",
            "",
            "## If you want the database to be loaded and stored to a specific directory, enter the path to the file below",
            "DATABASE_LOCATION=" + str(home_path_str) + "my_awareness_db.sqlite",
            "",
            "## Enter a specific flask secret below. Will be autogenerated on first run, if it is not set!",
            "FLASK_SECRET=" + str(FLASK_SECRET.hex()),
            "",
            "## Configure logging (Default values below)",
            "LOG_LEVEL=info",
            "LOG_FILE_PATH=" + str(home_path_str) + "hackattack_awa_matrix.log"
        ]
    with open(config_file_location, 'w') as f:
        f.write('\n'.join(default_config_content))
        f.close()

def configure_application_logging():
    if LOG_FILE_PATH is not None and type(LOG_FILE_PATH) is str:
        formatter = logging.Formatter('%(asctime)s %(levelname)s %(message)s')
        # noinspection PyTypeChecker
        handler = RotatingFileHandler(LOG_FILE_PATH + ".application.log", maxBytes=10000000, backupCount=5)
        handler.setFormatter(formatter)

        logger = logging.getLogger("HACKATTACK Awareness Matrix - Application Log")
        if LOG_LEVEL == "error":
            logger.setLevel(logging.ERROR)
        elif LOG_LEVEL == "warning":
            logger.setLevel(logging.WARNING)
        elif LOG_LEVEL == "debug":
            logger.setLevel(logging.DEBUG)
        else:
            logger.setLevel(logging.INFO)
        logger.addHandler(handler)

        return logger
    return None


def configure_webserver_logging():
    if LOG_FILE_PATH is not None and type(LOG_FILE_PATH) is str:
        formatter = logging.Formatter('%(asctime)s %(levelname)s %(message)s')
        # noinspection PyTypeChecker
        handler = RotatingFileHandler(LOG_FILE_PATH + ".webserver.log", maxBytes=10000000, backupCount=5)
        handler.setFormatter(formatter)

        logger = logging.getLogger("HACKATTACK Awareness Matrix - Webserver Log")
        if LOG_LEVEL == "error":
            logger.setLevel(logging.ERROR)
        elif LOG_LEVEL == "warning":
            logger.setLevel(logging.WARNING)
        elif LOG_LEVEL == "debug":
            logger.setLevel(logging.DEBUG)
        else:
            logger.setLevel(logging.INFO)
        logger.addHandler(handler)

        return logger
    return None


def check_config():
    if LISTEN_IP is None or LISTEN_PORT is None or LOG_LEVEL is None or LOG_FILE_PATH is None or FLASK_SECRET is None:
        print("Config not properly initialized. Aborting")
        sys.exit(1)


def main():
    global FLASK_SECRET
    home_path = make_sure_awa_matrix_dir_exists_in_home()
    default_dir = home_path + '.env'
    parser = argparse.ArgumentParser()
    parser.add_argument("-c", "--config-file", type=str, default=default_dir,
                        help="Specify a config file to use. Default is .env file in current directory. The file will be created with a default config if it does not exist.")
    args = parser.parse_args()

    if args.config_file is not None and type(args.config_file) is str:
        if os.path.exists(args.config_file):
            print("Reading file from: " + str(args.config_file))
            read_config(args.config_file)
            check_config()
            if FLASK_SECRET is None:
                FLASK_SECRET = os.urandom(16)

        else:
            if FLASK_SECRET is None:
                FLASK_SECRET = os.urandom(16)
            print("Writing default config to: " + str(args.config_file))
            write_default_config_file(args.config_file)
            read_config(args.config_file)
            check_config()

        webserver_logger = configure_webserver_logging()
        application_logger = configure_application_logging()
        http_server = WSGIServer((LISTEN_IP, LISTEN_PORT), create_app(FLASK_SECRET, DATABASE_LOCATION, application_logger), log=webserver_logger)
        print("Server started at: http://" + str(LISTEN_IP) + ":" + str(LISTEN_PORT) + "  -  Open this URL in a web-browser!")
        print("Use <CTRL + C> to exit the server")
        http_server.serve_forever()

    else:
        print("Parameter config-file was not read successfully. Aborting now!")


# Main body
if __name__ == '__main__':
    main()
