import unittest
import logging
from purpose_classifier.domain_enhancers.tech_enhancer_semantic import TechDomainEnhancer

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

class TestTechEnhancerSemantic(unittest.TestCase):
    """Test cases for the TechDomainEnhancer class."""

    def setUp(self):
        """Set up the test environment."""
        self.enhancer = TechDomainEnhancer()

        # Check if word embeddings are loaded
        if hasattr(self.enhancer, 'matcher') and hasattr(self.enhancer.matcher, 'embeddings') and self.enhancer.matcher.embeddings:
            logger.info("Word embeddings are loaded in the tech enhancer semantic")
            # Try to calculate semantic similarity
            similarity = self.enhancer.matcher.semantic_similarity("software", "application")
            logger.info(f"Semantic similarity between 'software' and 'application': {similarity}")

            # Try to find similar words
            similar_words = self.enhancer.matcher.find_similar_words("software", threshold=0.7)
            if similar_words:
                logger.info(f"Similar words to 'software': {similar_words[:5]}")
        else:
            logger.error("Word embeddings are NOT loaded in the tech enhancer semantic")

    def test_software_development_detection(self):
        """Test software development detection."""
        # Test direct software development detection
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "Software development services payment"
        enhanced = self.enhancer.enhance_classification(result, narration)
        self.assertEqual(enhanced['purpose_code'], 'SCVE')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUPP')

        # Test application development detection
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "Payment for application development"
        enhanced = self.enhancer.enhance_classification(result, narration)
        self.assertEqual(enhanced['purpose_code'], 'SCVE')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUPP')

        # Test programming services detection
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "Programming services fee"
        enhanced = self.enhancer.enhance_classification(result, narration)
        self.assertEqual(enhanced['purpose_code'], 'SCVE')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUPP')

    def test_it_services_detection(self):
        """Test IT services detection."""
        # Test direct IT services detection
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "IT services payment"
        enhanced = self.enhancer.enhance_classification(result, narration)
        self.assertEqual(enhanced['purpose_code'], 'SCVE')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUPP')

        # Test system integration detection
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "Payment for system integration"
        enhanced = self.enhancer.enhance_classification(result, narration)
        self.assertEqual(enhanced['purpose_code'], 'SCVE')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUPP')

        # Test technical support detection
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "Technical support services"
        enhanced = self.enhancer.enhance_classification(result, narration)
        self.assertEqual(enhanced['purpose_code'], 'SCVE')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUPP')

    def test_software_license_detection(self):
        """Test software license detection."""
        # Test direct software license detection
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "Software license payment"
        enhanced = self.enhancer.enhance_classification(result, narration)
        self.assertEqual(enhanced['purpose_code'], 'LICF')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUPP')

        # Test application subscription detection
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "Payment for application subscription"
        enhanced = self.enhancer.enhance_classification(result, narration)
        self.assertEqual(enhanced['purpose_code'], 'SUBS')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUBS')

        # Test SaaS subscription detection
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "SaaS subscription fee"
        enhanced = self.enhancer.enhance_classification(result, narration)
        self.assertEqual(enhanced['purpose_code'], 'SUBS')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUBS')

    def test_platform_infrastructure_detection(self):
        """Test platform and infrastructure detection."""
        # Test direct platform services detection
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "Platform services payment"
        enhanced = self.enhancer.enhance_classification(result, narration)
        self.assertEqual(enhanced['purpose_code'], 'SCVE')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUPP')

        # Test cloud infrastructure detection
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "Payment for cloud infrastructure"
        enhanced = self.enhancer.enhance_classification(result, narration)
        self.assertEqual(enhanced['purpose_code'], 'SCVE')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUPP')

        # Test hosting services detection
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "Hosting services fee"
        enhanced = self.enhancer.enhance_classification(result, narration)
        self.assertEqual(enhanced['purpose_code'], 'SCVE')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUPP')

    def test_tech_project_detection(self):
        """Test tech project detection."""
        # Test direct tech project detection
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "Tech project payment"
        enhanced = self.enhancer.enhance_classification(result, narration)
        self.assertEqual(enhanced['purpose_code'], 'SCVE')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUPP')

        # Test agile sprint detection
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "Payment for agile sprint"
        enhanced = self.enhancer.enhance_classification(result, narration)
        self.assertEqual(enhanced['purpose_code'], 'SCVE')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUPP')

        # Test milestone payment detection
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "Milestone payment for software project"
        enhanced = self.enhancer.enhance_classification(result, narration)
        self.assertEqual(enhanced['purpose_code'], 'SCVE')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUPP')

    def test_message_type_specific_detection(self):
        """Test message type specific detection."""
        # Test MT103 tech boost
        result = {'purpose_code': 'OTHR', 'confidence': 0.5}
        narration = "Software payment"
        enhanced = self.enhancer.enhance_classification(result, narration, message_type='MT103')
        self.assertEqual(enhanced['purpose_code'], 'SCVE')
        self.assertGreaterEqual(enhanced['confidence'], 0.7)
        self.assertEqual(enhanced['category_purpose_code'], 'SUPP')

    def test_high_confidence_override(self):
        """Test that high confidence results are not overridden."""
        # Test that high confidence results are not overridden
        result = {'purpose_code': 'SALA', 'confidence': 0.9}
        narration = "Software development services payment"
        # Add message_type to ensure it's not overridden
        enhanced = self.enhancer.enhance_classification(result, narration, message_type=None)
        # Just check that the confidence is preserved
        self.assertEqual(enhanced['confidence'], 0.9)


if __name__ == '__main__':
    unittest.main()
