"""The logger module is essential for logging an experiment important numbers
doing a training.
"""
import io
import os
import sys
import typing

from energinetml.settings import DEFAULT_LOG_ENCODING
from energinetml.settings import DEFAULT_LOG_FILENAME
from energinetml.settings import DEFAULT_RELATIVE_ARTIFACT_PATH


class ConsoleLogger:
    """This class enables logging of stdout and stderr.
    The class is used during local trainings and finally uploaded to the azure ml
    expirment.

    Args:
        console (io.TextIOWrapper, optional): Possible values
            [sys.stdout, sys.stderr]. Defaults to sys.stdout.
        path (str, optional): Relative path for the log file.
            Defaults to "./outputs".

    Returns:
        object: A console logger object which can be used to overwrite
            the default sys.std* object.
    """

    def __init__(
        self,
        name: str,
        console: io.TextIOWrapper = sys.stdout,
    ):
        """Init ConsoleLogger with either sys.stdout or sys.stderr."""
        self.console = console

        file_prefix, ext = DEFAULT_LOG_FILENAME.split(".")
        self.filename = f"{file_prefix}_{name}.{ext}"

        self.filepath, self.log = self._init_log_file(self.filename)

    def _init_log_file(self, filename: str) -> typing.Tuple[str, io.TextIOWrapper]:
        """Performs Initialization of the log file and further enrich the object."""
        os.makedirs(DEFAULT_RELATIVE_ARTIFACT_PATH, exist_ok=True)

        filepath = f"{DEFAULT_RELATIVE_ARTIFACT_PATH}/{filename}"
        return filepath, open(filepath, "w", encoding=DEFAULT_LOG_ENCODING)

    def isatty(self) -> bool:
        """A requried function for sys.stdout and sys.stderr.

        Returns:
            bool: This function will only return False.
        """
        return False

    def write(self, message: str) -> None:
        """A requried function for sys.stdout and sys.stderr.
        The function will both print to file and to console.

        Args:
            message (str): The string which needs to be appended to the log.
        """
        self.log.write(message)
        self.console.write(message)

    def flush(self):
        """The function flush the io buffer.
        This function is called before uploading to azure ml.
        """
        self.log.flush()


class MetricsLogger:
    """
    TODO
    """

    def echo(self, s):
        """
        :param str s:
        """
        raise NotImplementedError

    def log(self, name, value):
        """
        :param str name:
        :param typing.Any value:
        """
        raise NotImplementedError

    def tag(self, key, value):
        """
        :param str key:
        :param str value:
        """
        raise NotImplementedError

    def dataframe(self, name, df):
        """
        :param str name:
        :param pandas.DataFrame df:
        """
        raise NotImplementedError
