"""This is lia's main program.

:author: Julian M. Kleber
"""
import os
import subprocess

import click
from click import echo
from emoji import emojize
from lia.check_git.verify_branch import verify_branch


@click.group()
def spells() -> None:
    """Collection for Lia's spells.

    For more info ask for help on each specific spell.
    """
    pass  # pragma: no cover


@click.command()
@click.argument("packagename")
@click.option("-t", default="y", help="If test should be run")
def heal(packagename: str, t: str) -> None:  # pragma: no cover
    """One of Lias most basic spells. It lints, and typechecks the code
    specified in the path. To test use the -t option. Lia only supports pytest.

    The heal function is a wrapper for the following commands:
        - black
        - autopep8
        - flake8 (with E9, F63, F7 and F82)
        - mypy --strict
       It also runs pylint with the parseable output format to
       make it easier to integrate into CI systems like
       Woodpecker or Codeberg CI.

    :param packagename:str: Used to specify the package name.
    :param o:str: Used to specify if the user wants to run tests or not.

    :doc-author: Julian M. Kleber
    """
    say_message()
    verify_branch()
    assert t in ["y", "n"], "Plase specify -t as y/n"
    assert os.path.isdir(packagename)
    if not packagename.endswith("/"):
        packagename += "/"
    subprocess.run(["pip freeze > requirements.txt"], shell=True, check=True)
    subprocess.run(["black " + packagename], shell=True, check=True)
    subprocess.run(
        [
            f"find . -type f -wholename '{packagename}/*.py' "
            "-exec sed --in-place 's/[[:space:]]\+$//' "
            + "{} \+ #sanitize trailing whitespace"
        ],
        shell=True,
        check=True,
    )
    subprocess.run(
        [f"autopep8 --in-place --recursive {packagename}"], shell=True, check=True
    )
    subprocess.run(
        [
            f"python -m flake8 {packagename} --count --select=E9,F63,F7,F82"
            " --show-source --statistics"
        ],
        shell=True,
        check=True,
    )
    try:
        subprocess.run(
            [f"mypy --strict {packagename}"], shell=True, check=True)
    except Exception as exc:
        print(str(exc))

    try:
        subprocess.run(
            [f"python -m pylint -f parseable {packagename}"], shell=True, check=True
        )
    except Exception as exc:
        print(str(exc))
    subprocess.run(
        [f"prettify-py format-dir {packagename}"], shell=True, check=True)
    if t == "y":
        try:
            subprocess.run(["pytest tests/"], shell=True, check=True)
        except Exception as exc:
            print(str(exc))
    say_end_message()


@click.command()
@click.option("-t", default="y", help="If test should be run")
def deploy(t: str) -> None:
    """Deployment routine.

    :author: Julian M. Kleber
    """
    say_message()
    verify_branch()
    if t == "y":
        out = subprocess.run(
            ["python -m pytest tests/"], shell=True, check=True, capture_output=True
        )
        try:
            assert "FAILED" not in str(out)
        except:
            print(out)
        finally:
            print("Done testing.")
    if os.path.isdir("dist") is True:
        subprocess.run(["rm -r dist "], shell=True, check=True)
    if os.path.isdir("build") is True:
        subprocess.run(["rm -r build"], shell=True, check=True)
    subprocess.run(["python3 -m build"], shell=True, check=True)
    subprocess.run(["twine check dist/*"], shell=True, check=True)
    subprocess.run(["python3 -m twine upload dist/*"], shell=True, check=True)
    say_end_message()


def say_message() -> None:
    """Console message for information about the project."""

    anatomical_heart = emojize(":anatomical_heart:")
    mechanical_arm = emojize(":mechanical_arm:")
    call_me_hand = emojize(":call_me_hand:")
    sailing_ship = emojize(":sailboat:")

    message = (
        f"{anatomical_heart} Lia is an open-source CI/CD tool\n"
        f"that eliminates the Fel {mechanical_arm}\n"
        f"Copyright © 2023 Soul Twin Studios {call_me_hand}\n"
        f"A subsidiary of SailBlack Inc. {sailing_ship}"
    )
    echo(message=message)


def say_end_message() -> None:
    """Goodbye message."""
    anatomical_heart = emojize(":anatomical_heart:")
    mechanical_arm = emojize(":mechanical_arm:")
    call_me_hand = emojize(":call_me_hand:")
    sailing_ship = emojize(":sailboat:")

    message = (
        f"Thank you for using Lia.\n"
        f"{anatomical_heart} Lia is an open-source CI/CD tool\n"
        f"that eliminates the Fel {mechanical_arm}\n"
        f"Copyright © 2023 Soul Twin Studios {call_me_hand}\n"
        f"A subsidiary of SailBlack Inc. {sailing_ship}\n"
        f"For more nice things follow us on www.codeberg.org/sail.black"
    )
    echo(message=message)


spells.add_command(heal)
spells.add_command(deploy)

if __name__ == "__main__":
    spells()
