"""
Комплексный тест для проверки gRPC-методов MSP и Vision через gRPCSync.

Этот модуль содержит тесты для проверки работы gRPC-сервисов:
- MSP (Multiwii Serial Protocol)
- Vision (компьютерное зрение)

Использует pytest для организации тестов и структурированной проверки.
"""

import time

import grpc
import pytest

from ara_api._utils import Altitude, Vector3, get_request, gRPCSync, vector2


class TestGRPCIntegration:
    """Тесты для проверки интеграции MSP и Vision gRPC-сервисов."""

    @pytest.fixture(scope="class")
    def grpc_sync(self) -> gRPCSync:
        """Создает экземпляр gRPCSync для тестов."""
        return gRPCSync()

    def test_grpc_sync_initialization(self, grpc_sync: gRPCSync) -> None:
        """Тест инициализации gRPCSync."""
        assert grpc_sync is not None
        assert hasattr(grpc_sync, "_msp_stub")
        assert hasattr(grpc_sync, "_vision_stub")

    # MSP Tests
    def test_msp_get_raw_imu(self, grpc_sync: gRPCSync) -> None:
        """Тест получения данных IMU через MSP."""
        try:
            result = grpc_sync.msp_get_raw_imu()
            print(f"MSP IMU: {result}")
            # Результат может быть None если сервис недоступен
            assert result is None or hasattr(result, "gyro")
        except grpc.RpcError as e:
            pytest.skip(f"MSP service not available: {e}")

    def test_msp_get_motor(self, grpc_sync: gRPCSync) -> None:
        """Тест получения данных моторов через MSP."""
        try:
            result = grpc_sync.msp_get_motor()
            print(f"MSP Motor: {result}")
            assert result is None or hasattr(result, "data")
        except grpc.RpcError as e:
            pytest.skip(f"MSP service not available: {e}")

    def test_msp_get_attitude(self, grpc_sync: gRPCSync) -> None:
        """Тест получения данных ориентации через MSP."""
        try:
            result = grpc_sync.msp_get_attitude()
            print(f"MSP Attitude: {result}")
            assert result is None or hasattr(result, "roll")
        except grpc.RpcError as e:
            pytest.skip(f"MSP service not available: {e}")

    def test_msp_get_altitude(self, grpc_sync: gRPCSync) -> None:
        """Тест получения данных высоты через MSP."""
        try:
            result = grpc_sync.msp_get_altitude()
            print(f"MSP Altitude: {result}")
            assert result is None or hasattr(result, "altitude")
        except grpc.RpcError as e:
            pytest.skip(f"MSP service not available: {e}")

    def test_msp_get_sonar(self, grpc_sync: gRPCSync) -> None:
        """Тест получения данных сонара через MSP."""
        try:
            result = grpc_sync.msp_get_sonar()
            print(f"MSP Sonar: {result}")
            assert result is None or hasattr(result, "distance")
        except grpc.RpcError as e:
            pytest.skip(f"MSP service not available: {e}")

    def test_msp_get_optical_flow(self, grpc_sync: gRPCSync) -> None:
        """Тест получения данных оптического потока через MSP."""
        try:
            result = grpc_sync.msp_get_optical_flow()
            print(f"MSP Optical Flow: {result}")
            assert result is None or hasattr(result, "quality")
        except grpc.RpcError as e:
            pytest.skip(f"MSP service not available: {e}")

    def test_msp_get_position(self, grpc_sync: gRPCSync) -> None:
        """Тест получения данных позиции через MSP."""
        try:
            result = grpc_sync.msp_get_position()
            print(f"MSP Position: {result}")
            assert result is None or hasattr(result, "x")
        except grpc.RpcError as e:
            pytest.skip(f"MSP service not available: {e}")

    def test_msp_get_analog(self, grpc_sync: gRPCSync) -> None:
        """Тест получения аналоговых данных через MSP."""
        try:
            result = grpc_sync.msp_get_analog()
            print(f"MSP Analog: {result}")
            assert result is None or hasattr(result, "voltage")
        except grpc.RpcError as e:
            pytest.skip(f"MSP service not available: {e}")

    # Vision Tests
    def test_vision_get_image(self, grpc_sync: gRPCSync) -> None:
        """Тест получения изображения через Vision."""
        try:
            result = grpc_sync.vision_get_image()
            print(f"Vision Image: {result}")
            assert result is None or hasattr(result, "data")
        except grpc.RpcError as e:
            pytest.skip(f"Vision service not available: {e}")

    def test_vision_get_image_stream(self, grpc_sync: gRPCSync) -> None:
        """Тест получения потока изображений через Vision."""
        try:
            result = grpc_sync.vision_get_image_stream()
            print(f"Vision Image Stream: {result}")
            assert result is None or hasattr(result, "frames")
        except grpc.RpcError as e:
            pytest.skip(f"Vision service not available: {e}")

    def test_vision_get_aruco(self, grpc_sync: gRPCSync) -> None:
        """Тест получения ArUco маркеров через Vision."""
        try:
            result = grpc_sync.vision_get_aruco()
            print(f"Vision ArUco: {result}")
            assert result is None or hasattr(result, "markers")
        except grpc.RpcError as e:
            pytest.skip(f"Vision service not available: {e}")

    def test_vision_get_qr_code(self, grpc_sync: gRPCSync) -> None:
        """Тест получения QR-кодов через Vision."""
        try:
            result = grpc_sync.vision_get_qr_code()
            print(f"Vision QR Code: {result}")
            assert result is None or hasattr(result, "codes")
        except grpc.RpcError as e:
            pytest.skip(f"Vision service not available: {e}")

    def test_vision_get_blob(self, grpc_sync: gRPCSync) -> None:
        """Тест получения blob объектов через Vision."""
        try:
            result = grpc_sync.vision_get_blob()
            print(f"Vision Blob: {result}")
            assert result is None or hasattr(result, "blobs")
        except grpc.RpcError as e:
            pytest.skip(f"Vision service not available: {e}")

    def test_nav_takeooff(self, grpc_sync: gRPCSync) -> None:
        """Тест взлета через сервис навигации."""
        try:
            metadata = [
                ("source", "test"),
                ("client-id", "grpc-sync"),
                ("process-id", str(os.getpid())),
                ("timestamp", str(time.time())),
            ]
            result = grpc_sync.nav_cmd_takeoff(
                Altitude(alt=1.0).grpc, metadata
            )
            print(f"Navigation Takeoff: {result}")
            assert result is None or hasattr(result, "status")
        except grpc.RpcError as e:
            pytest.skip(f"Navigation service not available: {e}")

    def test_nav_land(self, grpc_sync: gRPCSync) -> None:
        """Тест посадки через сервис навигации."""
        try:
            metadata = [
                ("source", "test"),
                ("client-id", "grpc-sync"),
                ("process-id", str(os.getpid())),
                ("timestamp", str(time.time())),
            ]
            result = grpc_sync.nav_cmd_land(get_request(), metadata)
            print(f"Navigation Land: {result}")
            assert result is None or hasattr(result, "status")
        except grpc.RpcError as e:
            pytest.skip(f"Navigation service not available: {e}")

    def test_nav_move(self, grpc_sync: gRPCSync) -> None:
        """Тест перемещения по точкам через сервис навигации."""
        try:
            metadata = [
                ("source", "test"),
                ("client-id", "grpc-sync"),
                ("process-id", str(os.getpid())),
                ("timestamp", str(time.time())),
            ]
            result = grpc_sync.nav_cmd_move(
                Vector3(1.0, 0.0, 0.0).grpc, metadata
            )
            print(f"Navigation Move: {result}")
            assert result is None or hasattr(result, "status")
        except grpc.RpcError as e:
            pytest.skip(f"Navigation service not available: {e}")

    def test_nav_velocity(self, grpc_sync: gRPCSync) -> None:
        """Тест перемещения по скорости через сервис навигации."""
        try:
            metadata = [
                ("source", "test"),
                ("client-id", "grpc-sync"),
                ("process-id", str(os.getpid())),
                ("timestamp", str(time.time())),
            ]
            result = grpc_sync.nav_cmd_velocity(
                vector2(x=1.0, y=0.0), metadata
            )
            print(f"Navigation Velocity: {result}")
            assert result is None or hasattr(result, "status")
        except grpc.RpcError as e:
            pytest.skip(f"Navigation service not available: {e}")

    def test_nav_altitude(self, grpc_sync: gRPCSync) -> None:
        """Тест перемещения по высоте через сервис навигации."""
        try:
            metadata = [
                ("source", "test"),
                ("client-id", "grpc-sync"),
                ("process-id", str(os.getpid())),
                ("timestamp", str(time.time())),
            ]
            result = grpc_sync.nav_cmd_altitude(
                Altitude(alt=1.0).grpc, metadata
            )
            print(f"Navigation Altitude: {result}")
            assert result is None or hasattr(result, "status")
        except grpc.RpcError as e:
            pytest.skip(f"Navigation service not available: {e}")


if __name__ == "__main__":
    """Запуск тестов напрямую."""
    print("Running MSP, Navigation and Vision gRPC Integration Tests...")

    import argparse

    parser = argparse.ArgumentParser(
        description="Run gRPC integration tests for MSP and Vision services."
    )

    parser.add_argument(
        "--vision", action="store_true", help="Run Vision service tests only"
    )

    parser.add_argument(
        "--msp", action="store_true", help="Run MSP service tests only"
    )

    parser.add_argument(
        "--nav", action="store_true", help="Run Navigation service tests only"
    )
    parser.add_argument(
        "--all",
        action="store_true",
        help="Run all tests (MSP, Vision, Navigation)",
    )

    args = parser.parse_args()

    # Создаем экземпляр тестового класса
    test_instance = TestGRPCIntegration()
    grpc_sync = gRPCSync()

    if args.msp or args.all:
        print("\n=== Testing MSP Services ===")
        test_instance.test_msp_get_position(grpc_sync)
        test_instance.test_msp_get_attitude(grpc_sync)
        test_instance.test_msp_get_altitude(grpc_sync)
        test_instance.test_msp_get_raw_imu(grpc_sync)
        test_instance.test_msp_get_motor(grpc_sync)
        test_instance.test_msp_get_sonar(grpc_sync)
        test_instance.test_msp_get_optical_flow(grpc_sync)
        test_instance.test_msp_get_analog(grpc_sync)

    if args.nav or args.all:
        print("\n=== Testing Navigation Services ===")
        # test_instance.test_nav_takeooff(grpc_sync)
        # test_instance.test_nav_land(grpc_sync)
        # test_instance.test_nav_move(grpc_sync)
        # test_instance.test_nav_velocity(grpc_sync)
        test_instance.test_nav_altitude(grpc_sync)

    if args.vision or args.all:
        print("\n=== Testing Vision Services ===")
        test_instance.test_vision_get_image(grpc_sync)
        test_instance.test_vision_get_aruco(grpc_sync)
        test_instance.test_vision_get_blob(grpc_sync)
        test_instance.test_vision_get_qr_code(grpc_sync)

    print("\n✓ All MSP and Vision tests completed!")
