#!/usr/bin/env python3
from typing import Any, Dict, List, Tuple, Union

import numpy as np
import plotly.graph_objects as go

from ara_api._utils import (
    ARAException,
    Path,
    PathSegment,
    Vector3,
)
from ara_api._utils.config import (
    VPT_GRID_SPACING,
    VPT_LAYOUT_BG_COLOR,
    VPT_LAYOUT_BORDER_COLOR,
    VPT_LAYOUT_X,
    VPT_LAYOUT_XANCHOR,
    VPT_LAYOUT_Y,
    VPT_LAYOUT_YANCHOR,
    VPT_LINE_COLOR,
    VPT_LINE_WIDTH,
    VPT_SHOW_WAYPOINTS,
    VPT_WAYPOINT_COLOR,
    VPT_WAYPOINT_SIZE,
)


class VisualPathPlanning:
    def __init__(
        self,
        title: str = "Визуальное планирование пути квадрокоптера",
        workspace_bounds: Tuple[
            Tuple[float, float], Tuple[float, float], Tuple[float, float]
        ] = ((-10, 10), (-10, 10), (0, 20)),
    ):
        """
        Инициализация визуализатора планирования пути.

        Args:
            title: Заголовок графика
            workspace_bounds: Границы рабочей области ((xmin, xmax), (ymin, ymax), (zmin, zmax))
            settings: Настройки визуализации
        """
        self.title = title
        self.workspace_bounds = workspace_bounds
        self.fig = go.Figure()
        self.path = Path()
        self.cash_paths: List[Dict[str, Any]] = []
        self.obstacles: List[Dict[str, Any]] = []
        self.waypoints: List[Dict[str, Any]] = []

        self._setup_layout()

    def _setup_layout(self) -> None:
        self.fig.update_layout(
            title=self.title,
            scene=dict(
                xaxis_title="X (м)",
                yaxis_title="Y (м)",
                zaxis_title="Z (м)",
                xaxis=dict(range=self.workspace_bounds[0]),
                yaxis=dict(range=self.workspace_bounds[1]),
                zaxis=dict(range=self.workspace_bounds[2]),
            ),
            margin=dict(l=0, r=0, b=0, t=50),
            legend=dict(
                xanchor=VPT_LAYOUT_XANCHOR,
                x=VPT_LAYOUT_X,
                yanchor=VPT_LAYOUT_YANCHOR,
                y=VPT_LAYOUT_Y,
                bgcolor=VPT_LAYOUT_BG_COLOR,
                bordercolor=VPT_LAYOUT_BORDER_COLOR,
            ),
        )

    def _create_box_vertices(
        self, min_point: Vector3, max_point: Vector3
    ) -> np.ndarray:
        """
        Создает вершины параллелепипеда из минимальной и максимальной точки.

        Args:
            min_point: Минимальная точка (Vector3)
            max_point: Максимальная точка (Vector3)

        Returns:
            Массив вершин параллелепипеда.
        """
        return np.array(
            [
                [min_point.x, min_point.y, min_point.z],  # 0
                [max_point.x, min_point.y, min_point.z],  # 1
                [max_point.x, max_point.y, min_point.z],  # 2
                [min_point.x, max_point.y, min_point.z],  # 3
                [min_point.x, min_point.y, max_point.z],  # 4
                [max_point.x, min_point.y, max_point.z],  # 5
                [max_point.x, max_point.y, max_point.z],  # 6
                [min_point.x, max_point.y, max_point.z],  # 7
            ]
        )

    def _create_box_faces(self) -> np.ndarray:
        """Создание граней параллелепипеда."""
        return np.array(
            [
                [0, 1, 2],
                [0, 2, 3],  # нижняя грань
                [4, 7, 6],
                [4, 6, 5],  # верхняя грань
                [0, 4, 5],
                [0, 5, 1],  # передняя грань
                [2, 6, 7],
                [2, 7, 3],  # задняя грань
                [0, 3, 7],
                [0, 7, 4],  # левая грань
                [1, 5, 6],
                [1, 6, 2],  # правая грань
            ]
        )

    def add_path(
        self,
        path: Union[Path, List[PathSegment]],
        name: str = "Траектория",
    ) -> None:
        if isinstance(path, Path):
            self.path = path
        elif isinstance(path, list):
            self.path = Path(segments=path)
        else:
            raise ARAException(
                "Path must be a Path object or a list of PathSegment."
            )

        x_coords = [
            self.path.segments[i].start.x
            for i in range(len(self.path.segments))
        ]
        x_coords.append(self.path.segments[-1].end.x)

        y_coords = [
            self.path.segments[i].start.y
            for i in range(len(self.path.segments))
        ]
        y_coords.append(self.path.segments[-1].end.y)

        z_coords = [
            self.path.segments[i].start.z
            for i in range(len(self.path.segments))
        ]
        z_coords.append(self.path.segments[-1].end.z)

        self.fig.add_trace(
            go.Scatter3d(
                x=x_coords,
                y=y_coords,
                z=z_coords,
                mode="lines+markers" if VPT_SHOW_WAYPOINTS else "lines",
                line=dict(color=VPT_LINE_COLOR, width=VPT_LINE_WIDTH),
                marker=dict(
                    size=VPT_WAYPOINT_SIZE,
                    color=VPT_WAYPOINT_COLOR,
                )
                if VPT_SHOW_WAYPOINTS
                else None,
                name=name,
                hovertemplate="<b>%{text}</b><br>X: %{x:.2f}<br>Y: %{y:.2f}<br>Z: %{z:.2f}<extra></extra>",
                text=[f"Точка {i + 1}" for i in range(len(x_coords))],
            )
        )

        self.cash_paths.append(
            {
                "name": name,
                "x": x_coords,
                "y": y_coords,
                "z": z_coords,
                "path_object": self.path,
            }
        )

    def add_obstacle_box(
        self,
        min_point: Vector3,
        max_point: Vector3,
        name: str = "Препятствие",
        color: str = "red",
        opacity: float = 0.3,
    ) -> None:
        vertices = self._create_box_vertices(min_point, max_point)
        faces = self._create_box_faces()

        self.fig.add_trace(
            go.Mesh3d(
                x=vertices[:, 0],
                y=vertices[:, 1],
                z=vertices[:, 2],
                i=faces[:, 0],
                j=faces[:, 1],
                k=faces[:, 2],
                color=color,
                opacity=opacity,
                name=name,
                hovertemplate=f"<b>{name}</b><br>Препятствие<extra></extra>",
            )
        )

        self.obstacles.append(
            {
                "name": name,
                "min": min_point,
                "max": max_point,
                "color": color,
                "opacity": opacity,
            }
        )

    def add_waypoint(
        self,
        point: Union[Vector3, Tuple[float, float, float]],
        name: str = "Точка",
        color: str = "blue",
        size: float = 10.0,
        symbol: str = "circle",
    ) -> None:
        self.fig.add_trace(
            go.Scatter3d(
                x=[point[0] if isinstance(point, tuple) else point.x],
                y=[point[1] if isinstance(point, tuple) else point.y],
                z=[point[2] if isinstance(point, tuple) else point.z],
                mode="markers",
                marker=dict(
                    size=size,
                    color=color,
                    symbol=symbol,
                ),
                name=name,
                hovertemplate=f"<b>{name}</b><br>X:%{point.x if isinstance(point, Vector3) else point[0]:.2f}<br>Y: %{point.y if isinstance(point, Vector3) else point[1]:.2f}<br>Z: %{point.z if isinstance(point, Vector3) else point[2]:.2f}<extra></extra>",
            )
        )

        self.waypoints.append(
            {
                "name": name,
                "point": point
                if isinstance(point, Vector3)
                else Vector3(*point),
                "color": color,
                "size": size,
                "symbol": symbol,
            }
        )

    def add_grid(
        self,
        spacing: float = 1.0,
        color: str = "lightgray",
        opacity: float = 0.2,
    ) -> None:
        x_min, x_max = self.workspace_bounds[0]
        y_min, y_max = self.workspace_bounds[1]
        z_min, z_max = self.workspace_bounds[2]

        x_range = np.arange(x_min, x_max + VPT_GRID_SPACING, VPT_GRID_SPACING)
        y_range = np.arange(y_min, y_max + VPT_GRID_SPACING, VPT_GRID_SPACING)
        z_range = np.arange(z_min, z_max + VPT_GRID_SPACING, VPT_GRID_SPACING)

        for z in z_range:
            for x in x_range:
                self.fig.add_trace(
                    go.Scatter3d(
                        x=[x, x],
                        y=[y_min, y_max],
                        z=[z, z],
                        mode="lines",
                        line=dict(color=color, width=1),
                        opacity=opacity,
                        showlegend=False,
                        hoverinfo="skip",
                    )
                )
            for y in y_range:
                self.fig.add_trace(
                    go.Scatter3d(
                        x=[x_min, x_max],
                        y=[y, y],
                        z=[z, z],
                        mode="lines",
                        line=dict(color=color, width=1),
                        opacity=opacity,
                        showlegend=False,
                        hoverinfo="skip",
                    )
                )

    def analyze_path_quality(
        self, path: Union[Path, List[PathSegment]]
    ) -> Dict[str, float]:
        if isinstance(path, Path):
            self.path = path
        elif isinstance(path, list):
            self.path = Path(segments=path)
        else:
            raise ARAException(
                "Path must be a Path object or a list of PathSegment."
            )

        x_coords = [
            self.path.segments[i].start.x
            for i in range(len(self.path.segments))
        ]
        x_coords.append(self.path.segments[-1].end.x)

        y_coords = [
            self.path.segments[i].start.y
            for i in range(len(self.path.segments))
        ]
        y_coords.append(self.path.segments[-1].end.y)

        z_coords = [
            self.path.segments[i].start.z
            for i in range(len(self.path.segments))
        ]
        z_coords.append(self.path.segments[-1].end.z)

        total_distance: float = 0.0
        for i in range(1, len(self.path.segments)):
            dx = x_coords[i] - x_coords[i - 1]
            dy = y_coords[i] - y_coords[i - 1]
            dz = z_coords[i] - z_coords[i - 1]
            total_distance += np.sqrt(dx**2 + dy**2 + dz**2)

        smoothness: float = 0.0
        if len(self.path.segments) > 2:
            for i in range(1, len(self.path.segments) - 1):
                v1 = np.array(
                    [
                        x_coords[i] - x_coords[i - 1],
                        y_coords[i] - y_coords[i - 1],
                        z_coords[i] - z_coords[i - 1],
                    ]
                )
                v2 = np.array(
                    [
                        x_coords[i + 1] - x_coords[i],
                        y_coords[i + 1] - y_coords[i],
                        z_coords[i + 1] - z_coords[i],
                    ]
                )

                if np.linalg.norm(v1) > 0 and np.linalg.norm(v2) > 0:
                    cos_angle = np.dot(v1, v2) / (
                        np.linalg.norm(v1) * np.linalg.norm(v2)
                    )
                    cos_angle = np.clip(
                        cos_angle, -1.0, 1.0
                    )  # Ограничение для acos
                    angle = np.arccos(cos_angle)
                    smoothness += angle

        elevation_change: float = max(z_coords) - min(z_coords)

        return {
            "total_distance": total_distance,
            "smoothness": smoothness,
            "elevation_change": elevation_change,
            "num_waypoints": len(self.path.segments),
        }

    def show(self, width: int = 1000, height: int = 700) -> None:
        self.fig.update_layout(width=width, height=height)
        self.fig.show()

    def clear(self):
        self.fig = go.Figure()
        self.path.clear()
        self.obstacles.clear()
        self.waypoints.clear()
        self._setup_layout()
