import ast
import importlib
import os
import warnings
from pathlib import Path
from typing import Any, Dict, List, Tuple


class ImportChecker:
    """Класс для проверки импортов модулей Python."""

    def __init__(self, project_root: Path) -> None:
        """
        Инициализация проверщика импортов.

        Args:
            project_root: Корневая папка проекта.
        """
        self.project_root = project_root
        self.ara_api_root = project_root / "ara_api"
        self.failed_imports: List[Tuple[str, str]] = []
        self.successful_imports: List[str] = []
        self.module_exports: Dict[str, List[str]] = {}

    def find_python_modules(self) -> List[Path]:
        """
        Находит все Python модули в проекте (папки с __init__.py).

        Returns:
            Список путей к модулям.
        """
        modules = []

        # Ищем все __init__.py файлы в ara_api
        for init_file in self.ara_api_root.rglob("__init__.py"):
            module_dir = init_file.parent
            # Исключаем __pycache__ и другие служебные папки
            if "__pycache__" not in str(module_dir):
                modules.append(module_dir)

        return sorted(modules)

    def path_to_module_name(self, module_path: Path) -> str:
        """
        Преобразует путь к модулю в имя для импорта.

        Args:
            module_path: Путь к модулю.

        Returns:
            Имя модуля для импорта.
        """
        # Получаем относительный путь от корня проекта
        rel_path = module_path.relative_to(self.project_root)
        # Заменяем слеши на точки
        module_name = str(rel_path).replace(os.sep, ".")
        return module_name

    def get_module_exports(self, module_path: Path) -> List[str]:
        """
        Получает список экспортируемых символов из __init__.py.

        Args:
            module_path: Путь к модулю.

        Returns:
            Список экспортируемых символов.
        """
        init_file = module_path / "__init__.py"
        exports = []

        try:
            with open(init_file, "r", encoding="utf-8") as f:
                content = f.read()

            # Парсим AST для поиска __all__ и импортов
            tree = ast.parse(content)

            for node in ast.walk(tree):
                # Ищем __all__
                if isinstance(node, ast.Assign) and any(
                    isinstance(target, ast.Name) and target.id == "__all__"
                    for target in node.targets
                ):
                    if isinstance(node.value, (ast.List, ast.Tuple)):
                        for elt in node.value.elts:
                            if isinstance(elt, ast.Str):
                                exports.append(elt.s)
                            elif isinstance(elt, ast.Constant) and isinstance(
                                elt.value, str
                            ):
                                exports.append(elt.value)

                # Ищем from ... import statements
                elif isinstance(node, ast.ImportFrom):
                    for alias in node.names:
                        if alias.name != "*":
                            exports.append(alias.asname or alias.name)

                # Ищем обычные import statements
                elif isinstance(node, ast.Import):
                    for alias in node.names:
                        exports.append(
                            alias.asname or alias.name.split(".")[-1]
                        )

        except Exception as e:
            print(f"Ошибка при анализе {init_file}: {e}")

        return exports

    def test_module_import(self, module_name: str) -> Tuple[bool, str]:
        """
        Тестирует импорт модуля.

        Args:
            module_name: Имя модуля для импорта.

        Returns:
            Tuple с результатом (успех, сообщение об ошибке).
        """
        try:
            # Подавляем предупреждения при импорте
            with warnings.catch_warnings():
                warnings.simplefilter("ignore")
                module = importlib.import_module(module_name)

            # Проверяем, что модуль действительно загружен
            if module is None:
                return False, "Модуль загружен как None"

            return True, ""

        except Exception as e:
            error_msg = f"{type(e).__name__}: {str(e)}"
            return False, error_msg

    def test_module_attributes(
        self, module_name: str, expected_exports: List[str]
    ) -> List[str]:
        """
        Проверяет наличие ожидаемых атрибутов в модуле.

        Args:
            module_name: Имя модуля.
            expected_exports: Ожидаемые экспортируемые атрибуты.

        Returns:
            Список отсутствующих атрибутов.
        """
        try:
            with warnings.catch_warnings():
                warnings.simplefilter("ignore")
                module = importlib.import_module(module_name)

            missing_attrs = []
            for attr_name in expected_exports:
                if not hasattr(module, attr_name):
                    missing_attrs.append(attr_name)

            return missing_attrs

        except Exception:
            return expected_exports  # Если импорт не удался, все атрибуты считаются отсутствующими

    def run_all_tests(self) -> Dict[str, Any]:
        """
        Запускает все тесты импортов.

        Returns:
            Словарь с результатами тестирования.
        """
        modules = self.find_python_modules()
        results = {
            "total_modules": len(modules),
            "successful_imports": 0,
            "failed_imports": 0,
            "modules_with_missing_attrs": 0,
            "details": [],
        }

        print(f"Найдено {len(modules)} модулей для тестирования...")

        for module_path in modules:
            module_name = self.path_to_module_name(module_path)
            expected_exports = self.get_module_exports(module_path)

            # Тестируем импорт
            import_success, error_msg = self.test_module_import(module_name)

            module_result = {
                "module_name": module_name,
                "module_path": str(module_path),
                "import_success": import_success,
                "error_message": error_msg,
                "expected_exports": expected_exports,
                "missing_attributes": [],
            }

            if import_success:
                results["successful_imports"] += 1
                self.successful_imports.append(module_name)

                # Проверяем атрибуты только если импорт успешен
                if expected_exports:
                    missing_attrs = self.test_module_attributes(
                        module_name, expected_exports
                    )
                    module_result["missing_attributes"] = missing_attrs

                    if missing_attrs:
                        results["modules_with_missing_attrs"] += 1

            else:
                results["failed_imports"] += 1
                self.failed_imports.append((module_name, error_msg))

            results["details"].append(module_result)

        return results
