#!/usr/bin/env python3
"""
Тестирование интеграции MPPI контроллера с глобальным планировщиком.

Этот тест демонстрирует, как MPPI контроллер следует по траектории,
созданной глобальным планировщиком.
"""

import math
import os
import sys
import time
from typing import Dict, List

import plotly.graph_objects as go
import plotly.subplots as sp

# Добавляем корневую директорию проекта в путь
sys.path.insert(0, os.path.join(os.path.dirname(__file__), "../../"))

from ara_api._core.services.nav.controller import (
    MPPIController,
    create_mppi_state,
)
from ara_api._core.services.nav.planner import GlobalNavigationPlanner
from ara_api._utils import MPPIState, ObstacleBox, Path, Vector3


def extract_waypoints_from_path(path: Path) -> List[Vector3]:
    """Извлекает промежуточные точки из пути планировщика."""
    waypoints = []

    if not path.segments:
        return waypoints

    # Добавляем начальную точку первого сегмента
    waypoints.append(path.segments[0].start)

    # Добавляем конечные точки всех сегментов
    for segment in path.segments:
        waypoints.append(segment.end)

    return waypoints


def find_next_waypoint(
    current_position: Vector3, waypoints: List[Vector3], tolerance: float = 0.3
) -> tuple[int, Vector3]:
    """
    Находит следующую промежуточную цель для MPPI контроллера.

    Args:
        current_position: Текущая позиция дрона
        waypoints: Список промежуточных точек
        tolerance: Допуск достижения точки

    Returns:
        Tuple из (индекс следующей точки, позиция следующей точки)
    """
    for i, waypoint in enumerate(waypoints):
        distance = math.sqrt(
            (current_position.x - waypoint.x) ** 2
            + (current_position.y - waypoint.y) ** 2
            + (current_position.z - waypoint.z) ** 2
        )

        if distance > tolerance:
            return i, waypoint

    # Если все промежуточные точки достигнуты, возвращаем последнюю
    return len(waypoints) - 1, waypoints[-1]


def test_mppi_waypoint_following():
    """Тестирует следование MPPI контроллера по промежуточным целям от РЕАЛЬНОГО GlobalPlanner."""
    print(
        "🎯 Тестирование MPPI контроллера с промежуточными целями от РЕАЛЬНОГО GlobalPlanner"
    )
    print("=" * 80)

    # Создаем препятствия (те же что и для планировщика)
    obstacles = [
        ObstacleBox(
            min_point=Vector3(1.2, 1.2, 0.0), max_point=Vector3(1.4, 1.4, 2.5)
        ),
        ObstacleBox(
            min_point=Vector3(0.8, 1.8, 0.0), max_point=Vector3(1.0, 2.0, 2.5)
        ),  # Дополнительное препятствие
    ]

    # ПРАВИЛЬНО: Используем РЕАЛЬНЫЙ GlobalPlanner для создания пути
    start_position = Vector3(0.5, 0.5, 1.0)
    goal_position = Vector3(2.5, 2.5, 1.5)

    global_planner = GlobalNavigationPlanner()
    print("🗺️ Запрашиваем путь у GlobalPlanner...")

    # Получаем РЕАЛЬНЫЙ путь от планировщика
    reference_path = global_planner.plan_path(
        start=start_position, goal=goal_position, obstacles=obstacles
    )

    # Извлекаем waypoints из РЕАЛЬНОГО пути планировщика
    carrot_waypoints = extract_waypoints_from_path(reference_path)

    print(
        f"🗺️ GlobalPlanner создал путь с {len(reference_path.segments)} сегментами"
    )
    print(
        f"📍 Извлечено {len(carrot_waypoints)} промежуточных точек из реального пути"
    )

    # Инициализируем MPPI контроллер
    controller = MPPIController(obstacles=obstacles)

    # Начальное состояние
    initial_state = create_mppi_state(carrot_waypoints[0])
    current_state = initial_state

    # Извлекаем промежуточные точки (исключая стартовую)
    waypoints = carrot_waypoints[1:]  # Начинаем с первой промежуточной точки
    current_waypoint_index = 0
    current_target = waypoints[current_waypoint_index]

    print(
        f"🚁 Начальная позиция: ({current_state.position.x:.1f}, {current_state.position.y:.1f}, {current_state.position.z:.1f})"
    )
    print(
        f"🎯 Первая промежуточная цель: ({current_target.x:.1f}, {current_target.y:.1f}, {current_target.z:.1f})"
    )

    # История для анализа
    state_history = [current_state]
    control_history = []
    target_history = [current_target]  # История смены целей
    waypoint_reached_steps = []  # Шаги, на которых достигались промежуточные точки

    dt = 0.1  # 10 Гц симуляция
    max_steps = 300  # Увеличиваем количество шагов
    waypoint_tolerance = 0.25  # Допуск достижения промежуточной точки

    start_time = time.time()

    for step in range(max_steps):
        # Создаем целевое состояние для текущей промежуточной точки
        target_state = create_mppi_state(current_target)

        # Вычисляем управление с использованием текущей промежуточной цели
        control = controller.compute_control(
            current_state, reference_path, target_state
        )
        control_history.append(control)

        # Симулируем движение дрона
        new_position = Vector3(
            current_state.position.x + control.linear_velocity.x * dt,
            current_state.position.y + control.linear_velocity.y * dt,
            current_state.position.z + control.linear_velocity.z * dt,
        )

        new_yaw = current_state.yaw + control.angular_velocity * dt

        # Простая инерция
        new_velocity = Vector3(
            current_state.velocity.x * 0.9 + control.linear_velocity.x * 0.1,
            current_state.velocity.y * 0.9 + control.linear_velocity.y * 0.1,
            current_state.velocity.z * 0.9 + control.linear_velocity.z * 0.1,
        )

        current_state = MPPIState(
            position=new_position,
            velocity=new_velocity,
            yaw=new_yaw,
            yaw_rate=control.angular_velocity,
            timestamp=time.time(),
        )

        state_history.append(current_state)

        # Проверяем достижение текущей промежуточной цели
        distance_to_target = math.sqrt(
            (current_state.position.x - current_target.x) ** 2
            + (current_state.position.y - current_target.y) ** 2
            + (current_state.position.z - current_target.z) ** 2
        )

        # Если достигли промежуточной точки, переключаемся на следующую
        if distance_to_target < waypoint_tolerance:
            waypoint_reached_steps.append(step)
            print(
                f"✅ Промежуточная цель {current_waypoint_index + 1} достигнута на шаге {step}!"
            )

            current_waypoint_index += 1
            if current_waypoint_index < len(waypoints):
                current_target = waypoints[current_waypoint_index]
                target_history.append(current_target)
                print(
                    f"🎯 Новая цель: ({current_target.x:.1f}, {current_target.y:.1f}, {current_target.z:.1f})"
                )
            else:
                print(
                    f"🏁 Все промежуточные цели достигнуты! Финальная цель достигнута на шаге {step}!"
                )
                break

        if step % 20 == 0:
            print(
                f"📊 Шаг {step}: позиция ({current_state.position.x:.2f}, {current_state.position.y:.2f}, {current_state.position.z:.2f}), "
                f"до цели {current_waypoint_index + 1}: {distance_to_target:.2f}м"
            )

    total_time = time.time() - start_time

    # Финальная статистика
    final_target = waypoints[-1]
    final_distance = math.sqrt(
        (current_state.position.x - final_target.x) ** 2
        + (current_state.position.y - final_target.y) ** 2
        + (current_state.position.z - final_target.z) ** 2
    )

    print("\n" + "=" * 80)
    print("📊 РЕЗУЛЬТАТЫ ТЕСТИРОВАНИЯ WAYPOINT FOLLOWING:")
    print(f"⏱️ Общее время: {total_time:.2f}с")
    print(f"🔢 Количество шагов: {len(state_history)}")
    print(
        f"🎯 Промежуточных целей достигнуто: {len(waypoint_reached_steps)}/{len(waypoints)}"
    )
    print(f"📏 Финальная ошибка позиции: {final_distance:.3f}м")
    print(f"⚡ Частота управления: {len(control_history) / total_time:.1f} Гц")

    # Анализ времени достижения промежуточных точек
    if waypoint_reached_steps:
        avg_time_per_waypoint = total_time / len(waypoint_reached_steps)
        print(
            f"⏱️ Среднее время на промежуточную точку: {avg_time_per_waypoint:.2f}с"
        )

    # Анализ следования по траектории
    path_following_errors = []
    for state in state_history:
        min_distance = float("inf")
        for segment in reference_path.segments:
            dist_to_start = math.sqrt(
                (state.position.x - segment.start.x) ** 2
                + (state.position.y - segment.start.y) ** 2
                + (state.position.z - segment.start.z) ** 2
            )
            dist_to_end = math.sqrt(
                (state.position.x - segment.end.x) ** 2
                + (state.position.y - segment.end.y) ** 2
                + (state.position.z - segment.end.z) ** 2
            )
            min_distance = min(min_distance, dist_to_start, dist_to_end)
        path_following_errors.append(min_distance)

    avg_path_error = sum(path_following_errors) / len(path_following_errors)
    max_path_error = max(path_following_errors)

    print(f"🛤️ Средняя ошибка следования по траектории: {avg_path_error:.3f}м")
    print(f"🛤️ Максимальная ошибка следования: {max_path_error:.3f}м")

    # Оценка успешности
    success = (len(waypoint_reached_steps) == len(waypoints)) and (
        final_distance < waypoint_tolerance
    )

    if success:
        print(
            "✅ ТЕСТ УСПЕШЕН! MPPI контроллер корректно следует по промежуточным целям от CarrotPlanner."
        )
    else:
        print("❌ ТЕСТ НЕУДАЧЕН! Не все промежуточные цели достигнуты.")

    return {
        "success": success,
        "final_distance": final_distance,
        "total_time": total_time,
        "waypoints_reached": len(waypoint_reached_steps),
        "total_waypoints": len(waypoints),
        "avg_path_error": avg_path_error,
        "max_path_error": max_path_error,
        "control_frequency": len(control_history) / total_time,
        # Добавляем данные для визуализации
        "state_history": state_history,
        "control_history": control_history,
        "path_following_errors": path_following_errors,
        "reference_path": reference_path,
        "obstacles": obstacles,
        "waypoints": carrot_waypoints,
        "target_history": target_history,
        "waypoint_reached_steps": waypoint_reached_steps,
        "goal_position": final_target,
    }


def plot_trajectory_3d(
    state_history: List[MPPIState],
    reference_path: Path,
    obstacles: List[ObstacleBox],
    goal_position: Vector3,
    save_path: str = "mppi_trajectory_3d.html",
) -> None:
    """
    Создает 3D график траектории MPPI контроллера с использованием Plotly.

    Args:
        state_history: История состояний дрона
        reference_path: Референсная траектория
        obstacles: Список препятствий
        goal_position: Целевая позиция
        save_path: Путь для сохранения HTML файла
    """
    fig = go.Figure()

    # Извлекаем координаты траектории дрона
    drone_x = [state.position.x for state in state_history]
    drone_y = [state.position.y for state in state_history]
    drone_z = [state.position.z for state in state_history]

    # Добавляем траекторию дрона
    fig.add_trace(
        go.Scatter3d(
            x=drone_x,
            y=drone_y,
            z=drone_z,
            mode="lines+markers",
            marker=dict(
                size=3,
                color=list(range(len(drone_x))),
                colorscale="Viridis",
                colorbar=dict(title="Время"),
                showscale=True,
            ),
            line=dict(width=4, color="blue"),
            name="Траектория дрона",
            hovertemplate="<b>Позиция дрона</b><br>"
            + "X: %{x:.2f}м<br>"
            + "Y: %{y:.2f}м<br>"
            + "Z: %{z:.2f}м<br>"
            + "<extra></extra>",
        )
    )

    # Добавляем начальную точку
    fig.add_trace(
        go.Scatter3d(
            x=[drone_x[0]],
            y=[drone_y[0]],
            z=[drone_z[0]],
            mode="markers",
            marker=dict(size=10, color="green"),
            name="Старт",
            hovertemplate="<b>Стартовая позиция</b><br>"
            + "X: %{x:.2f}м<br>"
            + "Y: %{y:.2f}м<br>"
            + "Z: %{z:.2f}м<br>"
            + "<extra></extra>",
        )
    )

    # Добавляем конечную точку
    fig.add_trace(
        go.Scatter3d(
            x=[goal_position.x],
            y=[goal_position.y],
            z=[goal_position.z],
            mode="markers",
            marker=dict(size=10, color="red"),
            name="Цель",
            hovertemplate="<b>Целевая позиция</b><br>"
            + "X: %{x:.2f}м<br>"
            + "Y: %{y:.2f}м<br>"
            + "Z: %{z:.2f}м<br>"
            + "<extra></extra>",
        )
    )

    # Добавляем референсную траекторию
    ref_x, ref_y, ref_z = [], [], []
    for segment in reference_path.segments:
        ref_x.extend([segment.start.x, segment.end.x, None])
        ref_y.extend([segment.start.y, segment.end.y, None])
        ref_z.extend([segment.start.z, segment.end.z, None])

    fig.add_trace(
        go.Scatter3d(
            x=ref_x,
            y=ref_y,
            z=ref_z,
            mode="lines",
            line=dict(width=6, color="orange", dash="dash"),
            name="Референсная траектория",
            hovertemplate="<b>Референсная траектория</b><br>"
            + "X: %{x:.2f}м<br>"
            + "Y: %{y:.2f}м<br>"
            + "Z: %{z:.2f}м<br>"
            + "<extra></extra>",
        )
    )

    # Добавляем препятствия как кубы
    for i, obstacle in enumerate(obstacles):
        # Создаем вершины куба
        vertices = [
            [obstacle.min_point.x, obstacle.min_point.y, obstacle.min_point.z],
            [obstacle.max_point.x, obstacle.min_point.y, obstacle.min_point.z],
            [obstacle.max_point.x, obstacle.max_point.y, obstacle.min_point.z],
            [obstacle.min_point.x, obstacle.max_point.y, obstacle.min_point.z],
            [obstacle.min_point.x, obstacle.min_point.y, obstacle.max_point.z],
            [obstacle.max_point.x, obstacle.min_point.y, obstacle.max_point.z],
            [obstacle.max_point.x, obstacle.max_point.y, obstacle.max_point.z],
            [obstacle.min_point.x, obstacle.max_point.y, obstacle.max_point.z],
        ]

        # Создаем грани куба
        faces = [
            [0, 1, 2, 3],  # нижняя грань
            [4, 5, 6, 7],  # верхняя грань
            [0, 1, 5, 4],  # передняя грань
            [2, 3, 7, 6],  # задняя грань
            [0, 3, 7, 4],  # левая грань
            [1, 2, 6, 5],  # правая грань
        ]

        for face in faces:
            face_vertices = [vertices[j] for j in face] + [vertices[face[0]]]
            x_coords = [v[0] for v in face_vertices]
            y_coords = [v[1] for v in face_vertices]
            z_coords = [v[2] for v in face_vertices]

            fig.add_trace(
                go.Scatter3d(
                    x=x_coords,
                    y=y_coords,
                    z=z_coords,
                    mode="lines",
                    line=dict(width=2, color="red"),
                    showlegend=False if i > 0 or face != faces[0] else True,
                    name="Препятствия"
                    if i == 0 and face == faces[0]
                    else None,
                    hovertemplate="<b>Препятствие</b><br>"
                    + "X: %{x:.2f}м<br>"
                    + "Y: %{y:.2f}м<br>"
                    + "Z: %{z:.2f}м<br>"
                    + "<extra></extra>",
                )
            )

    # Настройка макета
    fig.update_layout(
        title={
            "text": "MPPI Контроллер: Следование по траектории",
            "x": 0.5,
            "xanchor": "center",
            "font": {"size": 20},
        },
        scene=dict(
            xaxis_title="X (метры)",
            yaxis_title="Y (метры)",
            zaxis_title="Z (метры)",
            camera=dict(eye=dict(x=1.5, y=1.5, z=1.5)),
            aspectmode="cube",
        ),
        showlegend=True,
        width=1000,
        height=800,
    )

    # Сохраняем график
    fig.write_html(save_path)
    print(f"📊 3D график сохранен: {save_path}")


def plot_trajectory_analysis(
    state_history: List[MPPIState],
    control_history: List,
    path_following_errors: List[float],
    goal_position: Vector3,
    save_path: str = "mppi_analysis.html",
) -> None:
    """
    Создает детальный анализ производительности MPPI контроллера.

    Args:
        state_history: История состояний дрона
        control_history: История команд управления
        path_following_errors: Ошибки следования по траектории
        goal_position: Целевая позиция
        save_path: Путь для сохранения HTML файла
    """
    # Создаем подграфики
    fig = sp.make_subplots(
        rows=3,
        cols=2,
        subplot_titles=(
            "Позиция во времени",
            "Скорость во времени",
            "Команды управления",
            "Ошибка следования по траектории",
            "Расстояние до цели",
            "Траектория в плоскости XY",
        ),
        specs=[
            [{"secondary_y": False}, {"secondary_y": False}],
            [{"secondary_y": False}, {"secondary_y": False}],
            [{"secondary_y": False}, {"secondary_y": False}],
        ],
    )

    # Извлекаем данные
    times = list(range(len(state_history)))
    positions_x = [state.position.x for state in state_history]
    positions_y = [state.position.y for state in state_history]
    positions_z = [state.position.z for state in state_history]

    velocities_x = [state.velocity.x for state in state_history]
    velocities_y = [state.velocity.y for state in state_history]
    velocities_z = [state.velocity.z for state in state_history]

    # Команды управления
    if control_history:
        control_times = list(range(len(control_history)))
        linear_vx = [control.linear_velocity.x for control in control_history]
        linear_vy = [control.linear_velocity.y for control in control_history]
        linear_vz = [control.linear_velocity.z for control in control_history]

    # Расстояние до цели
    distances_to_goal = [
        math.sqrt(
            (state.position.x - goal_position.x) ** 2
            + (state.position.y - goal_position.y) ** 2
            + (state.position.z - goal_position.z) ** 2
        )
        for state in state_history
    ]

    # График 1: Позиция во времени
    fig.add_trace(
        go.Scatter(x=times, y=positions_x, name="X", line=dict(color="red")),
        row=1,
        col=1,
    )
    fig.add_trace(
        go.Scatter(x=times, y=positions_y, name="Y", line=dict(color="green")),
        row=1,
        col=1,
    )
    fig.add_trace(
        go.Scatter(x=times, y=positions_z, name="Z", line=dict(color="blue")),
        row=1,
        col=1,
    )

    # График 2: Скорость во времени
    fig.add_trace(
        go.Scatter(
            x=times,
            y=velocities_x,
            name="Vx",
            line=dict(color="red", dash="dash"),
        ),
        row=1,
        col=2,
    )
    fig.add_trace(
        go.Scatter(
            x=times,
            y=velocities_y,
            name="Vy",
            line=dict(color="green", dash="dash"),
        ),
        row=1,
        col=2,
    )
    fig.add_trace(
        go.Scatter(
            x=times,
            y=velocities_z,
            name="Vz",
            line=dict(color="blue", dash="dash"),
        ),
        row=1,
        col=2,
    )

    # График 3: Команды управления
    if control_history:
        fig.add_trace(
            go.Scatter(
                x=control_times,
                y=linear_vx,
                name="Cmd Vx",
                line=dict(color="orange"),
            ),
            row=2,
            col=1,
        )
        fig.add_trace(
            go.Scatter(
                x=control_times,
                y=linear_vy,
                name="Cmd Vy",
                line=dict(color="purple"),
            ),
            row=2,
            col=1,
        )
        fig.add_trace(
            go.Scatter(
                x=control_times,
                y=linear_vz,
                name="Cmd Vz",
                line=dict(color="brown"),
            ),
            row=2,
            col=1,
        )

    # График 4: Ошибка следования по траектории
    fig.add_trace(
        go.Scatter(
            x=times,
            y=path_following_errors,
            name="Ошибка пути",
            line=dict(color="red"),
        ),
        row=2,
        col=2,
    )

    # График 5: Расстояние до цели
    fig.add_trace(
        go.Scatter(
            x=times,
            y=distances_to_goal,
            name="Расстояние до цели",
            line=dict(color="black"),
        ),
        row=3,
        col=1,
    )

    # График 6: Траектория в плоскости XY
    fig.add_trace(
        go.Scatter(
            x=positions_x,
            y=positions_y,
            mode="lines+markers",
            name="Траектория XY",
            line=dict(color="blue"),
            marker=dict(size=2),
        ),
        row=3,
        col=2,
    )
    fig.add_trace(
        go.Scatter(
            x=[goal_position.x],
            y=[goal_position.y],
            mode="markers",
            name="Цель",
            marker=dict(size=10, color="red"),
        ),
        row=3,
        col=2,
    )

    # Обновляем макет
    fig.update_layout(
        title={
            "text": "MPPI Контроллер: Детальный анализ производительности",
            "x": 0.5,
            "xanchor": "center",
            "font": {"size": 20},
        },
        height=1200,
        showlegend=True,
    )

    # Подписи осей
    fig.update_xaxes(title_text="Время (шаги)", row=1, col=1)
    fig.update_yaxes(title_text="Позиция (м)", row=1, col=1)

    fig.update_xaxes(title_text="Время (шаги)", row=1, col=2)
    fig.update_yaxes(title_text="Скорость (м/с)", row=1, col=2)

    fig.update_xaxes(title_text="Время (шаги)", row=2, col=1)
    fig.update_yaxes(title_text="Команда (м/с)", row=2, col=1)

    fig.update_xaxes(title_text="Время (шаги)", row=2, col=2)
    fig.update_yaxes(title_text="Ошибка (м)", row=2, col=2)

    fig.update_xaxes(title_text="Время (шаги)", row=3, col=1)
    fig.update_yaxes(title_text="Расстояние (м)", row=3, col=1)

    fig.update_xaxes(title_text="X (м)", row=3, col=2)
    fig.update_yaxes(title_text="Y (м)", row=3, col=2)

    # Сохраняем график
    fig.write_html(save_path)
    print(f"📊 Анализ производительности сохранен: {save_path}")


def plot_waypoint_trajectory_3d(
    state_history: List[MPPIState],
    reference_path: Path,
    obstacles: List[ObstacleBox],
    waypoints: List[Vector3],
    target_history: List[Vector3],
    waypoint_reached_steps: List[int],
    goal_position: Vector3,
    save_path: str = "mppi_waypoint_trajectory_3d.html",
) -> None:
    """
    Создает 3D график траектории MPPI контроллера с waypoints с использованием Plotly.
    Показывает промежуточные цели и моменты их достижения.

    Args:
        state_history: История состояний дрона
        reference_path: Референсная траектория от глобального планировщика
        obstacles: Список препятствий
        waypoints: Промежуточные точки
        target_history: История смены целей
        waypoint_reached_steps: Шаги, на которых достигались промежуточные точки
        goal_position: Финальная целевая позиция
        save_path: Путь для сохранения HTML файла
    """
    fig = go.Figure()

    # Траектория дрона
    x_pos = [state.position.x for state in state_history]
    y_pos = [state.position.y for state in state_history]
    z_pos = [state.position.z for state in state_history]

    fig.add_trace(
        go.Scatter3d(
            x=x_pos,
            y=y_pos,
            z=z_pos,
            mode="lines+markers",
            name="MPPI Траектория",
            line=dict(color="blue", width=4),
            marker=dict(size=2),
        )
    )

    # Референсная траектория (сегменты от планировщика)
    for i, segment in enumerate(reference_path.segments):
        fig.add_trace(
            go.Scatter3d(
                x=[segment.start.x, segment.end.x],
                y=[segment.start.y, segment.end.y],
                z=[segment.start.z, segment.end.z],
                mode="lines",
                name=f"Reference Seg {i + 1}" if i == 0 else None,
                line=dict(color="green", width=2, dash="dash"),
                showlegend=(i == 0),
            )
        )

    # Waypoints (промежуточные цели)
    waypoint_x = [wp.x for wp in waypoints]
    waypoint_y = [wp.y for wp in waypoints]
    waypoint_z = [wp.z for wp in waypoints]

    fig.add_trace(
        go.Scatter3d(
            x=waypoint_x,
            y=waypoint_y,
            z=waypoint_z,
            mode="markers",
            name="Waypoints",
            marker=dict(
                size=8,
                color="orange",
                symbol="diamond",
            ),
        )
    )

    # Моменты достижения waypoints
    for i, step in enumerate(waypoint_reached_steps):
        if step < len(state_history):
            state = state_history[step]
            fig.add_trace(
                go.Scatter3d(
                    x=[state.position.x],
                    y=[state.position.y],
                    z=[state.position.z],
                    mode="markers",
                    name=f"Достигнута цель {i + 1}" if i == 0 else None,
                    marker=dict(
                        size=6,  # Уменьшили размер с 12 до 6
                        color="red",
                        symbol="x",
                    ),
                    showlegend=(i == 0),
                )
            )

    # Препятствия
    for i, obstacle in enumerate(obstacles):
        # Создаем куб препятствия
        x_min, x_max = obstacle.min_point.x, obstacle.max_point.x
        y_min, y_max = obstacle.min_point.y, obstacle.max_point.y
        z_min, z_max = obstacle.min_point.z, obstacle.max_point.z

        # Вершины куба
        vertices = [
            [x_min, y_min, z_min],
            [x_max, y_min, z_min],
            [x_max, y_max, z_min],
            [x_min, y_max, z_min],
            [x_min, y_min, z_max],
            [x_max, y_min, z_max],
            [x_max, y_max, z_max],
            [x_min, y_max, z_max],
        ]

        # Рёбра куба
        edges = [
            [0, 1],
            [1, 2],
            [2, 3],
            [3, 0],  # нижняя грань
            [4, 5],
            [5, 6],
            [6, 7],
            [7, 4],  # верхняя грань
            [0, 4],
            [1, 5],
            [2, 6],
            [3, 7],  # вертикальные рёбра
        ]

        for j, edge in enumerate(edges):
            v1, v2 = vertices[edge[0]], vertices[edge[1]]
            fig.add_trace(
                go.Scatter3d(
                    x=[v1[0], v2[0]],
                    y=[v1[1], v2[1]],
                    z=[v1[2], v2[2]],
                    mode="lines",
                    name=f"Препятствие {i + 1}" if j == 0 else None,
                    line=dict(color="red", width=3),
                    showlegend=(j == 0),
                )
            )

    # Стартовая и финальная позиции
    start_pos = state_history[0].position
    fig.add_trace(
        go.Scatter3d(
            x=[start_pos.x],
            y=[start_pos.y],
            z=[start_pos.z],
            mode="markers",
            name="Старт",
            marker=dict(size=15, color="green", symbol="circle"),
        )
    )

    fig.add_trace(
        go.Scatter3d(
            x=[goal_position.x],
            y=[goal_position.y],
            z=[goal_position.z],
            mode="markers",
            name="Финиш",
            marker=dict(size=15, color="red", symbol="square"),
        )
    )

    # Настройка макета
    fig.update_layout(
        title="MPPI Waypoint Following - 3D Траектория",
        scene=dict(
            xaxis_title="X (м)",
            yaxis_title="Y (м)",
            zaxis_title="Z (м)",
            aspectmode="cube",
        ),
        legend=dict(x=0.7, y=0.9),
    )

    # Сохраняем график
    fig.write_html(save_path)
    print(f"📊 3D график waypoint траектории сохранен: {save_path}")


def plot_multiple_scenarios_3d(
    scenarios_data: List[Dict],
    save_path: str = "mppi_multiple_scenarios_3d.html",
) -> None:
    """
    Создает 3D график с несколькими сценариями на одной странице.

    Args:
        scenarios_data: Список результатов сценариев
        save_path: Путь для сохранения HTML файла
    """
    # Определяем цвета для разных сценариев
    scenario_colors = ["blue", "green", "purple", "orange", "red", "brown"]

    fig = go.Figure()

    for idx, scenario in enumerate(scenarios_data):
        color = scenario_colors[idx % len(scenario_colors)]
        scenario_name = scenario["scenario_name"]

        # Траектория дрона для текущего сценария
        x_pos = [state.position.x for state in scenario["state_history"]]
        y_pos = [state.position.y for state in scenario["state_history"]]
        z_pos = [state.position.z for state in scenario["state_history"]]

        fig.add_trace(
            go.Scatter3d(
                x=x_pos,
                y=y_pos,
                z=z_pos,
                mode="lines+markers",
                name=f"{scenario_name} - Траектория",
                line=dict(color=color, width=4),
                marker=dict(size=2),
            )
        )

        # Референсная траектория (сегменты от планировщика)
        for i, segment in enumerate(scenario["reference_path"].segments):
            fig.add_trace(
                go.Scatter3d(
                    x=[segment.start.x, segment.end.x],
                    y=[segment.start.y, segment.end.y],
                    z=[segment.start.z, segment.end.z],
                    mode="lines",
                    name=f"{scenario_name} - Reference" if i == 0 else None,
                    line=dict(color=color, width=2, dash="dash"),
                    showlegend=(i == 0),
                )
            )

        # Waypoints для текущего сценария
        waypoint_x = [wp.x for wp in scenario["waypoints"]]
        waypoint_y = [wp.y for wp in scenario["waypoints"]]
        waypoint_z = [wp.z for wp in scenario["waypoints"]]

        fig.add_trace(
            go.Scatter3d(
                x=waypoint_x,
                y=waypoint_y,
                z=waypoint_z,
                mode="markers",
                name=f"{scenario_name} - Waypoints",
                marker=dict(
                    size=6,
                    color=color,
                    symbol="diamond",
                    line=dict(width=1, color="black"),
                ),
            )
        )

        # Моменты достижения waypoints для текущего сценария
        for i, step in enumerate(scenario["waypoint_reached_steps"]):
            if step < len(scenario["state_history"]):
                state = scenario["state_history"][step]
                fig.add_trace(
                    go.Scatter3d(
                        x=[state.position.x],
                        y=[state.position.y],
                        z=[state.position.z],
                        mode="markers",
                        name=f"{scenario_name} - Достигнуто"
                        if i == 0
                        else None,
                        marker=dict(
                            size=4,  # Еще меньше размер
                            color="red",
                            symbol="x",
                        ),
                        showlegend=(i == 0),
                    )
                )

        # Препятствия для текущего сценария
        for i, obstacle in enumerate(scenario["obstacles"]):
            # Создаем куб препятствия
            x_min, x_max = obstacle.min_point.x, obstacle.max_point.x
            y_min, y_max = obstacle.min_point.y, obstacle.max_point.y
            z_min, z_max = obstacle.min_point.z, obstacle.max_point.z

            # Вершины куба
            vertices = [
                [x_min, y_min, z_min],
                [x_max, y_min, z_min],
                [x_max, y_max, z_min],
                [x_min, y_max, z_min],
                [x_min, y_min, z_max],
                [x_max, y_min, z_max],
                [x_max, y_max, z_max],
                [x_min, y_max, z_max],
            ]

            # Рёбра куба
            edges = [
                [0, 1],
                [1, 2],
                [2, 3],
                [3, 0],  # нижняя грань
                [4, 5],
                [5, 6],
                [6, 7],
                [7, 4],  # верхняя грань
                [0, 4],
                [1, 5],
                [2, 6],
                [3, 7],  # вертикальные рёбра
            ]

            for j, edge in enumerate(edges):
                v1, v2 = vertices[edge[0]], vertices[edge[1]]
                fig.add_trace(
                    go.Scatter3d(
                        x=[v1[0], v2[0]],
                        y=[v1[1], v2[1]],
                        z=[v1[2], v2[2]],
                        mode="lines",
                        name=f"{scenario_name} - Препятствие"
                        if i == 0 and j == 0
                        else None,
                        line=dict(color="red", width=3),
                        showlegend=(i == 0 and j == 0),
                    )
                )

        # Стартовая и финальная позиции для текущего сценария
        start_pos = scenario["state_history"][0].position
        fig.add_trace(
            go.Scatter3d(
                x=[start_pos.x],
                y=[start_pos.y],
                z=[start_pos.z],
                mode="markers",
                name=f"{scenario_name} - Старт",
                marker=dict(
                    size=10,
                    color="green",
                    symbol="circle",
                    line=dict(width=2, color="black"),
                ),
            )
        )

        fig.add_trace(
            go.Scatter3d(
                x=[scenario["goal_position"].x],
                y=[scenario["goal_position"].y],
                z=[scenario["goal_position"].z],
                mode="markers",
                name=f"{scenario_name} - Финиш",
                marker=dict(
                    size=10,
                    color="red",
                    symbol="square",
                    line=dict(width=2, color="black"),
                ),
            )
        )

    # Настройка макета
    fig.update_layout(
        title="MPPI Multiple Scenarios - 3D Траектории",
        scene=dict(
            xaxis_title="X (м)",
            yaxis_title="Y (м)",
            zaxis_title="Z (м)",
            aspectmode="cube",
        ),
        legend=dict(x=0.02, y=0.98, bgcolor="rgba(255,255,255,0.8)"),
        width=1200,
        height=800,
    )

    # Сохраняем график
    fig.write_html(save_path)
    print(f"📊 3D график всех сценариев сохранен: {save_path}")


def test_scenario_square_frame():
    """Сценарий 1: Полет через квадратную рамку в воздухе - ВЕРТИКАЛЬНАЯ рамка в плоскости ZY."""
    print("📦 Сценарий 1: Полет через ВЕРТИКАЛЬНУЮ квадратную рамку в воздухе")

    # Создаем ВЕРТИКАЛЬНУЮ квадратную рамку в плоскости ZY (x=1.5)
    # Рамка стоит вертикально, отверстие в центре y=1.5, z=2.0
    frame_x = 1.5  # Постоянная координата X для всей рамки
    obstacles = [
        # Левая стенка рамки (нижняя часть по Y)
        ObstacleBox(
            min_point=Vector3(frame_x - 0.1, 1.0, 1.6),
            max_point=Vector3(frame_x + 0.1, 1.3, 2.4),
        ),
        # Правая стенка рамки (верхняя часть по Y)
        ObstacleBox(
            min_point=Vector3(frame_x - 0.1, 1.7, 1.6),
            max_point=Vector3(frame_x + 0.1, 2.0, 2.4),
        ),
        # Нижняя перекладина рамки
        ObstacleBox(
            min_point=Vector3(frame_x - 0.1, 1.3, 1.6),
            max_point=Vector3(frame_x + 0.1, 1.7, 1.8),
        ),
        # Верхняя перекладина рамки
        ObstacleBox(
            min_point=Vector3(frame_x - 0.1, 1.3, 2.2),
            max_point=Vector3(frame_x + 0.1, 1.7, 2.4),
        ),
    ]

    start_position = Vector3(0.5, 1.5, 2.0)  # Перед рамкой в воздухе
    goal_position = Vector3(2.5, 1.5, 2.0)  # За рамкой в воздухе

    return run_waypoint_scenario(
        obstacles=obstacles,
        start=start_position,
        goal=goal_position,
        scenario_name="Square Frame",
    )


def test_scenario_circular_frame():
    """Сценарий 2: Полет через круглую рамку - ВЕРТИКАЛЬНАЯ рамка в плоскости ZY."""
    print("🔵 Сценарий 2: Полет через ВЕРТИКАЛЬНУЮ круглую рамку")

    # Создаем ВЕРТИКАЛЬНУЮ круглую рамку в плоскости ZY (x=1.5)
    frame_x = 1.5  # Постоянная координата X для всей рамки
    center_y, center_z = 1.5, 2.0  # Центр отверстия в плоскости YZ
    obstacles = []

    # Создаем препятствия по кругу в вертикальной плоскости YZ
    for i in range(8):
        angle = i * 2 * math.pi / 8  # каждые 45 градусов
        radius = 0.5  # Радиус рамки
        y = center_y + radius * math.cos(angle)
        z = center_z + radius * math.sin(angle)
        obstacles.append(
            ObstacleBox(
                min_point=Vector3(frame_x - 0.1, y - 0.08, z - 0.08),
                max_point=Vector3(frame_x + 0.1, y + 0.08, z + 0.08),
            )
        )

    start_position = Vector3(0.3, 1.5, 2.0)  # Дальше от препятствий
    goal_position = Vector3(2.7, 1.5, 2.0)  # Дальше от препятствий

    return run_waypoint_scenario(
        obstacles=obstacles,
        start=start_position,
        goal=goal_position,
        scenario_name="Circular Frame",
    )


def test_scenario_spiral_path():
    """Сценарий 3: Движение по спирали вверх через вертикальные препятствия."""
    print(
        "🌀 Сценарий 3: Движение по спирали вверх с вертикальными препятствиями"
    )

    # Создаем умеренные препятствия, заставляющие дрон подниматься по спирали
    # но не слишком сложные для планировщика
    obstacles = [
        # Центральное препятствие внизу - блокирует прямой путь
        ObstacleBox(
            min_point=Vector3(1.3, 1.3, 0.7), max_point=Vector3(1.7, 1.7, 1.3)
        ),
        # Боковое препятствие на средней высоте
        ObstacleBox(
            min_point=Vector3(1.8, 1.0, 1.5), max_point=Vector3(2.2, 1.4, 1.9)
        ),
    ]

    start_position = Vector3(0.5, 0.5, 0.5)  # Начинаем внизу
    goal_position = Vector3(2.5, 2.5, 2.5)  # Поднимаемся по спирали вверх

    return run_waypoint_scenario(
        obstacles=obstacles,
        start=start_position,
        goal=goal_position,
        scenario_name="Spiral Ascent",
    )


def run_waypoint_scenario(obstacles, start, goal, scenario_name):
    """
    Запускает сценарий тестирования с waypoints.

    Args:
        obstacles: Список препятствий
        start: Стартовая позиция
        goal: Целевая позиция
        scenario_name: Название сценария

    Returns:
        Словарь с результатами тестирования
    """
    global_planner = GlobalNavigationPlanner()
    print(f"🗺️ Запрашиваем путь у GlobalPlanner для {scenario_name}...")

    # Получаем РЕАЛЬНЫЙ путь от планировщика
    reference_path = global_planner.plan_path(
        start=start, goal=goal, obstacles=obstacles
    )

    # Извлекаем waypoints из РЕАЛЬНОГО пути планировщика
    waypoints = extract_waypoints_from_path(reference_path)

    print(
        f"🗺️ GlobalPlanner создал путь с {len(reference_path.segments)} сегментами"
    )
    print(
        f"📍 Извлечено {len(waypoints)} промежуточных точек из реального пути"
    )

    # Инициализируем MPPI контроллер
    controller = MPPIController(obstacles=obstacles)

    # Начальное состояние
    initial_state = create_mppi_state(waypoints[0])
    current_state = initial_state

    # Извлекаем промежуточные точки (исключая стартовую)
    target_waypoints = waypoints[1:]  # Начинаем с первой промежуточной точки
    current_waypoint_index = 0
    current_target = target_waypoints[current_waypoint_index]

    print(
        f"🚁 Начальная позиция: ({current_state.position.x:.1f}, {current_state.position.y:.1f}, {current_state.position.z:.1f})"
    )
    print(
        f"🎯 Первая промежуточная цель: ({current_target.x:.1f}, {current_target.y:.1f}, {current_target.z:.1f})"
    )

    # История для анализа
    state_history = [current_state]
    control_history = []
    target_history = [current_target]
    waypoint_reached_steps = []

    dt = 0.1  # 10 Гц симуляция
    max_steps = 500  # Увеличиваем количество шагов для сложных сценариев
    waypoint_tolerance = 0.3  # Немного увеличиваем толерантность

    start_time = time.time()

    for step in range(max_steps):
        # Создаем целевое состояние для текущей промежуточной точки
        target_state = create_mppi_state(current_target)

        # Вычисляем управление с использованием текущей промежуточной цели
        control = controller.compute_control(
            current_state, reference_path, target_state
        )
        control_history.append(control)

        # Симулируем движение дрона
        new_position = Vector3(
            current_state.position.x + control.linear_velocity.x * dt,
            current_state.position.y + control.linear_velocity.y * dt,
            current_state.position.z + control.linear_velocity.z * dt,
        )

        new_yaw = current_state.yaw + control.angular_velocity * dt

        # Простая инерция
        new_velocity = Vector3(
            current_state.velocity.x * 0.9 + control.linear_velocity.x * 0.1,
            current_state.velocity.y * 0.9 + control.linear_velocity.y * 0.1,
            current_state.velocity.z * 0.9 + control.linear_velocity.z * 0.1,
        )

        current_state = MPPIState(
            position=new_position,
            velocity=new_velocity,
            yaw=new_yaw,
            yaw_rate=control.angular_velocity,
            timestamp=time.time(),
        )

        state_history.append(current_state)

        # Проверяем достижение текущей промежуточной цели
        distance_to_target = math.sqrt(
            (current_state.position.x - current_target.x) ** 2
            + (current_state.position.y - current_target.y) ** 2
            + (current_state.position.z - current_target.z) ** 2
        )

        # Если достигли промежуточной точки, переключаемся на следующую
        if distance_to_target < waypoint_tolerance:
            waypoint_reached_steps.append(step)

            current_waypoint_index += 1
            if current_waypoint_index < len(target_waypoints):
                current_target = target_waypoints[current_waypoint_index]
                target_history.append(current_target)
            else:
                print(
                    f"🏁 {scenario_name}: Все промежуточные цели достигнуты на шаге {step}!"
                )
                break

        if step % 50 == 0:  # Реже выводим информацию для краткости
            print(
                f"📊 {scenario_name} - Шаг {step}: до цели {distance_to_target:.2f}м"
            )

    total_time = time.time() - start_time

    # Анализ следования по траектории
    path_following_errors = []
    for state in state_history:
        min_distance = float("inf")
        for segment in reference_path.segments:
            dist_to_start = math.sqrt(
                (state.position.x - segment.start.x) ** 2
                + (state.position.y - segment.start.y) ** 2
                + (state.position.z - segment.start.z) ** 2
            )
            dist_to_end = math.sqrt(
                (state.position.x - segment.end.x) ** 2
                + (state.position.y - segment.end.y) ** 2
                + (state.position.z - segment.end.z) ** 2
            )
            min_distance = min(min_distance, dist_to_start, dist_to_end)
        path_following_errors.append(min_distance)

    # Финальная статистика
    final_target = target_waypoints[-1]
    final_distance = math.sqrt(
        (current_state.position.x - final_target.x) ** 2
        + (current_state.position.y - final_target.y) ** 2
        + (current_state.position.z - final_target.z) ** 2
    )

    success = (len(waypoint_reached_steps) == len(target_waypoints)) and (
        final_distance < waypoint_tolerance
    )

    print(
        f"✅ {scenario_name}: {'УСПЕХ' if success else 'НЕУДАЧА'} - {len(waypoint_reached_steps)}/{len(target_waypoints)} целей"
    )

    return {
        "scenario_name": scenario_name,
        "success": success,
        "final_distance": final_distance,
        "total_time": total_time,
        "waypoints_reached": len(waypoint_reached_steps),
        "total_waypoints": len(target_waypoints),
        "avg_path_error": sum(path_following_errors)
        / len(path_following_errors),
        "max_path_error": max(path_following_errors),
        "control_frequency": len(control_history) / total_time,
        "state_history": state_history,
        "control_history": control_history,
        "path_following_errors": path_following_errors,
        "reference_path": reference_path,
        "obstacles": obstacles,
        "waypoints": waypoints,
        "target_history": target_history,
        "waypoint_reached_steps": waypoint_reached_steps,
        "goal_position": final_target,
    }


def main():
    """Главная функция тестирования всех сценариев."""
    print("🚀 Запуск тестирования MPPI контроллера с 3-мя сценариями рамок")
    print(
        "🎯 Цель: проверить пролет дрона через квадратную рамку, круглую рамку и спиральный путь"
    )
    print("=" * 80)

    try:
        # Сценарий 1: Квадратная рамка в воздухе
        print("\n" + "=" * 60)
        print("📦 СЦЕНАРИЙ 1: Пролет через квадратную рамку")
        print("=" * 60)
        result1 = test_scenario_square_frame()

        # Создание отдельного графика для квадратной рамки
        print("� Создание графика для квадратной рамки...")
        plot_waypoint_trajectory_3d(
            result1["state_history"],
            result1["reference_path"],
            result1["obstacles"],
            result1["waypoints"],
            result1["target_history"],
            result1["waypoint_reached_steps"],
            result1["goal_position"],
            save_path="mppi_square_frame_3d.html",
        )

        # Сценарий 2: Круглая рамка в воздухе
        print("\n" + "=" * 60)
        print("🔵 СЦЕНАРИЙ 2: Пролет через круглую рамку")
        print("=" * 60)
        result2 = test_scenario_circular_frame()

        # Создание отдельного графика для круглой рамки
        print("📊 Создание графика для круглой рамки...")
        plot_waypoint_trajectory_3d(
            result2["state_history"],
            result2["reference_path"],
            result2["obstacles"],
            result2["waypoints"],
            result2["target_history"],
            result2["waypoint_reached_steps"],
            result2["goal_position"],
            save_path="mppi_circular_frame_3d.html",
        )

        # Сценарий 3: Спиральный путь
        print("\n" + "=" * 60)
        print("🌀 СЦЕНАРИЙ 3: Спиральный путь")
        print("=" * 60)
        result3 = test_scenario_spiral_path()

        # Создание отдельного графика для спирального пути
        print("📊 Создание графика для спирального пути...")
        plot_waypoint_trajectory_3d(
            result3["state_history"],
            result3["reference_path"],
            result3["obstacles"],
            result3["waypoints"],
            result3["target_history"],
            result3["waypoint_reached_steps"],
            result3["goal_position"],
            save_path="mppi_spiral_path_3d.html",
        )

        # Общий отчет по всем сценариям
        print("\n" + "=" * 80)
        print("🏆 ФИНАЛЬНЫЙ ОТЧЕТ ПО ВСЕМ СЦЕНАРИЯМ:")
        print("=" * 80)

        scenarios = [result1, result2, result3]
        scenario_names = [
            "Квадратная рамка",
            "Круглая рамка",
            "Спиральный путь",
        ]

        total_success = 0

        for i, (scenario, name) in enumerate(
            zip(scenarios, scenario_names), 1
        ):
            print(f"\n📊 Сценарий {i}: {name}")
            print(f"   ✅ Успех: {'ДА' if scenario['success'] else 'НЕТ'}")
            print(f"   📏 Точность: {scenario['final_distance']:.3f}м")
            print(
                f"   ⚡ Производительность: {scenario['control_frequency']:.1f} Гц"
            )
            print(
                f"   🎯 Промежуточных целей: {scenario['waypoints_reached']}/{scenario['total_waypoints']}"
            )
            print(
                f"   🛤️ Качество следования: {scenario['avg_path_error']:.3f}м"
            )

            if scenario["success"]:
                total_success += 1

        print("\n🎯 СВОДНАЯ СТАТИСТИКА:")
        print(f"   📊 Успешных сценариев: {total_success}/3")

        print("\n📊 Созданные графики:")
        print("   📦 Квадратная рамка: mppi_square_frame_3d.html")
        print("   🔵 Круглая рамка: mppi_circular_frame_3d.html")
        print("   🌀 Спиральный путь: mppi_spiral_path_3d.html")

        if total_success == 3:
            print(
                "\n🎉 ВСЕ СЦЕНАРИИ УСПЕШНЫ! MPPI контроллер отлично справляется с пролетом через рамки!"
            )
        elif total_success >= 2:
            print(
                f"\n✅ Большинство сценариев успешны ({total_success}/3). MPPI контроллер работает хорошо!"
            )
        else:
            print(
                f"\n⚠️ Только {total_success}/3 сценариев успешны. Требуется доработка."
            )

    except Exception as e:
        print(f"❌ Ошибка тестирования: {e}")
        import traceback

        traceback.print_exc()


if __name__ == "__main__":
    main()
