#!/usr/bin/env python3
"""
Тестовый скрипт для Carrot-планировщика в реальных условиях.

Данный скрипт тестирует алгоритм Carrot в клетке 3x3 метра с дроном 30x30 см:
- Прямые перелеты в ограниченном пространстве
- Обход препятствий в тесных условиях
- Навигация с учетом размеров дрона
- Точные маневры в углах клетки
- Вертикальные маневры в низком потолке

Физические ограничения:
- Клетка: 3.0 x 3.0 x 2.5 метра
- Дрон: 0.3 x 0.3 x 0.15 метра
- Безопасный зазор: 0.2 метра от стен
- Рабочая зона: 2.6 x 2.6 x 2.0 метра
"""

import os
import sys
from typing import Any, Dict, List, Tuple

import numpy as np

from test.subutils.visual_path_planning import VisualPathPlanning

# Добавляем корневую директорию проекта в путь
sys.path.insert(0, os.path.join(os.path.dirname(__file__), "../../"))

from ara_api._core.services.nav.planner.global_planner import (
    GlobalNavigationPlanner,
)
from ara_api._utils import (
    ObstacleBox,
    Vector3,
)


class CarrotPlannerTestSuite:
    """Набор тестов для Carrot-планировщика с визуализацией в клетке 3x3м."""

    def __init__(self):
        """Инициализация тестового набора."""
        self.planner = GlobalNavigationPlanner()
        self.test_results: List[Dict[str, Any]] = []

        # Физические параметры клетки и дрона
        self.cage_size = 3.0  # Размер клетки в метрах
        self.cage_height = 2.5  # Высота клетки в метрах
        self.drone_size = 0.3  # Размер дрона в метрах (30 см)
        self.safety_margin = 0.2  # Безопасный отступ от стен

        # Рабочая зона с учетом безопасности (от 0 до максимума)
        self.work_size = (
            self.cage_size - self.safety_margin
        )  # 2.8м (от 0.2 до 3.0)
        self.work_height = (
            self.cage_height - self.safety_margin
        )  # 2.3м (от 0.2 до 2.5)

        # Границы рабочего пространства (от точки старта дрона)
        self.workspace_bounds = (
            (self.safety_margin, self.work_size),  # X: 0.2 до 2.8
            (self.safety_margin, self.work_size),  # Y: 0.2 до 2.8
            (self.safety_margin, self.work_height),  # Z: 0.2 до 2.3
        )

    def create_test_scenario(
        self,
        name: str,
        start: Vector3,
        goal: Vector3,
        obstacles: List[ObstacleBox],
        description: str = "",
        workspace_bounds: Tuple[
            Tuple[float, float], Tuple[float, float], Tuple[float, float]
        ] = None,
    ) -> Dict[str, Any]:
        """
        Создает тестовый сценарий для клетки 3x3м.

        Args:
            name: Название теста
            start: Начальная точка
            goal: Целевая точка
            obstacles: Список препятствий
            description: Описание теста
            workspace_bounds: Границы рабочего пространства (по умолчанию - клетка 3x3м)

        Returns:
            Словарь с данными тестового сценария
        """
        if workspace_bounds is None:
            workspace_bounds = self.workspace_bounds

        return {
            "name": name,
            "start": start,
            "goal": goal,
            "obstacles": obstacles,
            "description": description,
            "workspace_bounds": workspace_bounds,
            "result": None,
            "path": None,
            "error": None,
            "metrics": None,
        }

    def run_test_scenario(self, scenario: Dict[str, Any]) -> Dict[str, Any]:
        """
        Выполняет один тестовый сценарий.

        Args:
            scenario: Тестовый сценарий

        Returns:
            Обновленный сценарий с результатами
        """
        print(f"\n🧪 Тест: {scenario['name']}")
        print(f"📝 Описание: {scenario['description']}")
        print(
            f"🎯 Старт: ({scenario['start'].x:.1f}, {scenario['start'].y:.1f}, {scenario['start'].z:.1f})"
        )
        print(
            f"🏁 Цель: ({scenario['goal'].x:.1f}, {scenario['goal'].y:.1f}, {scenario['goal'].z:.1f})"
        )
        print(f"🚧 Препятствий: {len(scenario['obstacles'])}")

        try:
            # Планирование пути
            path = self.planner.plan_path(
                start=scenario["start"],
                goal=scenario["goal"],
                obstacles=scenario["obstacles"],
            )

            scenario["result"] = "SUCCESS"
            scenario["path"] = path
            scenario["error"] = None

            # Анализ качества пути
            visualizer = VisualPathPlanning(
                title=f"Тест: {scenario['name']}",
                workspace_bounds=scenario["workspace_bounds"],
            )
            scenario["metrics"] = visualizer.analyze_path_quality(path)

            print(f"✅ Успех! Найден путь из {len(path.segments)} сегментов")
            print(
                f"📊 Общая длина: {scenario['metrics']['total_distance']:.2f}м"
            )
            print(f"📊 Плавность: {scenario['metrics']['smoothness']:.3f} рад")
            print(
                f"📊 Изменение высоты: {scenario['metrics']['elevation_change']:.2f}м"
            )

        except Exception as e:
            scenario["result"] = "FAILED"
            scenario["path"] = None
            scenario["error"] = str(e)
            scenario["metrics"] = None

            print(f"❌ Ошибка: {e}")

        return scenario

    def visualize_scenario(
        self, scenario: Dict[str, Any], show: bool = True
    ) -> VisualPathPlanning:
        """
        Визуализирует результаты тестового сценария.

        Args:
            scenario: Тестовый сценарий с результатами
            show: Показывать ли график сразу

        Returns:
            Объект визуализатора
        """
        visualizer = VisualPathPlanning(
            title=f"Тест Carrot-планировщика: {scenario['name']}",
            workspace_bounds=scenario["workspace_bounds"],
        )

        # Добавляем стартовую и целевую точки
        visualizer.add_waypoint(
            scenario["start"],
            name="Старт",
            color="green",
            size=12,
            symbol="diamond",
        )
        visualizer.add_waypoint(
            scenario["goal"], name="Цель", color="red", size=12, symbol="cross"
        )

        # Добавляем препятствия
        for i, obstacle in enumerate(scenario["obstacles"]):
            visualizer.add_obstacle_box(
                min_point=obstacle.min_point,
                max_point=obstacle.max_point,
                name=f"Препятствие {i + 1}",
                color="red",
                opacity=0.4,
            )

        # Добавляем путь, если он найден
        if scenario["path"] is not None:
            visualizer.add_path(scenario["path"], name="Carrot-путь")

        if show:
            visualizer.show()

        return visualizer

    def generate_test_scenarios(self) -> List[Dict[str, Any]]:
        """
        Генерирует набор тестовых сценариев.

        Returns:
            Список тестовых сценариев
        """
        scenarios = []

        # Тест 1: Прямой перелет в клетке
        scenarios.append(
            self.create_test_scenario(
                name="Прямой полет в клетке",
                start=Vector3(0.5, 0.5, 0.5),
                goal=Vector3(2.5, 2.5, 1.5),
                obstacles=[],
                description="Простейший перелет по диагонали клетки без препятствий",
            )
        )

        # Тест 2: Препятствие в центре клетки
        scenarios.append(
            self.create_test_scenario(
                name="Обход центрального препятствия",
                start=Vector3(0.5, 1.5, 1.0),
                goal=Vector3(2.5, 1.5, 1.0),
                obstacles=[
                    ObstacleBox(
                        min_point=Vector3(1.3, 1.3, 0.0),
                        max_point=Vector3(1.7, 1.7, 1.5),
                    )
                ],
                description="Обход малого препятствия в центре клетки - тест маневренности",
            )
        )

        # Тест 3: Узкий проход
        scenarios.append(
            self.create_test_scenario(
                name="Узкий проход",
                start=Vector3(0.5, 1.5, 1.0),
                goal=Vector3(2.5, 1.5, 1.0),
                obstacles=[
                    ObstacleBox(
                        min_point=Vector3(1.4, 0.2, 0.0),
                        max_point=Vector3(1.6, 1.2, 2.0),
                    ),
                    ObstacleBox(
                        min_point=Vector3(1.4, 1.8, 0.0),
                        max_point=Vector3(1.6, 2.8, 2.0),
                    ),
                ],
                description="Прохождение через узкий коридор шириной 60см",
            )
        )

        # Тест 4: Угловой маневр
        scenarios.append(
            self.create_test_scenario(
                name="Угловой маневр",
                start=Vector3(0.3, 0.3, 0.5),
                goal=Vector3(2.7, 2.7, 2.0),
                obstacles=[
                    ObstacleBox(
                        min_point=Vector3(1.2, 1.2, 0.0),
                        max_point=Vector3(1.8, 1.8, 1.2),
                    ),
                ],
                description="Маневр из угла в угол с малым препятствием в центре",
            )
        )

        # Тест 5: Высокое препятствие
        scenarios.append(
            self.create_test_scenario(
                name="Высокое препятствие",
                start=Vector3(0.7, 1.5, 0.5),
                goal=Vector3(2.3, 1.5, 0.5),
                obstacles=[
                    ObstacleBox(
                        min_point=Vector3(1.3, 1.3, 0.0),
                        max_point=Vector3(1.7, 1.7, 1.8),
                    ),
                ],
                description="Обход высокого препятствия - тест вертикального планирования",
            )
        )

        # Тест 6: Поле препятствий
        scenarios.append(
            self.create_test_scenario(
                name="Поле препятствий",
                start=Vector3(0.4, 0.4, 1.0),
                goal=Vector3(2.6, 2.6, 1.0),
                obstacles=[
                    ObstacleBox(
                        min_point=Vector3(0.8, 1.2, 0.0),
                        max_point=Vector3(1.0, 1.4, 1.2),
                    ),
                    ObstacleBox(
                        min_point=Vector3(1.4, 0.7, 0.0),
                        max_point=Vector3(1.6, 0.9, 1.0),
                    ),
                    ObstacleBox(
                        min_point=Vector3(2.0, 1.8, 0.0),
                        max_point=Vector3(2.2, 2.0, 1.3),
                    ),
                ],
                description="Навигация через поле из малых препятствий",
            )
        )

        # Тест 7: Полет по границе клетки
        scenarios.append(
            self.create_test_scenario(
                name="Полет по границе",
                start=Vector3(0.3, 0.3, 0.5),
                goal=Vector3(2.7, 2.7, 2.0),
                obstacles=[
                    # Препятствие, заставляющее лететь по краю клетки
                    ObstacleBox(
                        min_point=Vector3(1.0, 1.0, 0.0),
                        max_point=Vector3(2.0, 2.0, 1.5),
                    ),
                ],
                description="Экстремальный маневр вдоль границ клетки",
            )
        )

        return scenarios

    def run_all_tests(
        self, visualize: bool = True, interactive: bool = True
    ) -> None:
        """
        Запускает все тесты и выводит результаты.

        Args:
            visualize: Показывать ли визуализацию для каждого теста
            interactive: Ждать ли подтверждения пользователя между тестами
        """
        print("🚀 Запуск тестового набора Carrot-планировщика")
        print("🏠 Условия тестирования: клетка 3x3x2.5м, дрон 30x30см")
        print("=" * 60)

        scenarios = self.generate_test_scenarios()
        success_count = 0

        for i, scenario in enumerate(scenarios, 1):
            print(f"\n[{i}/{len(scenarios)}]", end=" ")

            # Выполняем тест
            result_scenario = self.run_test_scenario(scenario)
            self.test_results.append(result_scenario)

            if result_scenario["result"] == "SUCCESS":
                success_count += 1

            # Визуализация
            if visualize:
                try:
                    self.visualize_scenario(result_scenario, show=True)
                except Exception as e:
                    print(f"⚠️  Ошибка визуализации: {e}")

            # Интерактивная пауза
            if interactive and i < len(scenarios):
                input("\n📱 Нажмите Enter для продолжения...")

        # Итоговая статистика
        self.print_summary(success_count, len(scenarios))

    def print_summary(self, success_count: int, total_count: int) -> None:
        """
        Выводит итоговую статистику тестов.

        Args:
            success_count: Количество успешных тестов
            total_count: Общее количество тестов
        """
        print("\n" + "=" * 60)
        print("📊 ИТОГОВАЯ СТАТИСТИКА")
        print("=" * 60)
        print(f"✅ Успешных тестов: {success_count}/{total_count}")
        print(
            f"❌ Неудачных тестов: {total_count - success_count}/{total_count}"
        )
        print(f"📈 Процент успеха: {(success_count / total_count) * 100:.1f}%")

        if success_count > 0:
            successful_results = [
                r for r in self.test_results if r["result"] == "SUCCESS"
            ]
            avg_distance = np.mean(
                [r["metrics"]["total_distance"] for r in successful_results]
            )
            avg_smoothness = np.mean(
                [r["metrics"]["smoothness"] for r in successful_results]
            )
            avg_segments = np.mean(
                [r["metrics"]["num_waypoints"] for r in successful_results]
            )

            print(f"\n📏 Средняя длина пути: {avg_distance:.2f}м")
            print(f"🌊 Средняя плавность: {avg_smoothness:.3f} рад")
            print(f"🔗 Среднее количество сегментов: {avg_segments:.1f}")

            # Анализ практичности для клетки 3x3м
            print("🏠 АНАЛИЗ ПРАКТИЧНОСТИ ДЛЯ КЛЕТКИ 3x3м:")
            print(
                f"📐 Клетка: {self.cage_size}x{self.cage_size}x{self.cage_height}м"
            )
            print(f"🚁 Дрон: {self.drone_size}x{self.drone_size}м")
            print(f"🛡️  Безопасный отступ: {self.safety_margin}м")
            print(
                f"⚡ Рабочая зона: {self.work_size:.1f}x{self.work_size:.1f}x{self.work_height:.1f}м"
            )

            # Оценка эффективности путей
            max_theoretical_distance = np.sqrt(
                self.work_size**2 + self.work_size**2 + self.work_height**2
            )
            efficiency = (avg_distance / max_theoretical_distance) * 100
            print(
                f"📊 Эффективность путей: {efficiency:.1f}% от максимальной диагонали"
            )

            # Практические рекомендации
            if avg_distance < 1.0:
                print("💡 Короткие маршруты - подходит для точных маневров")
            elif avg_distance < 2.0:
                print("💡 Средние маршруты - оптимально для клетки")
            else:
                print("⚠️  Длинные маршруты - может потребоваться оптимизация")

            if avg_smoothness < 0.5:
                print("✨ Плавные траектории - безопасно для дрона")
            elif avg_smoothness < 1.0:
                print("📈 Умеренная сложность траекторий")
            else:
                print("⚠️  Резкие повороты - требуется осторожность")

        print("\n📋 ДЕТАЛИ ТЕСТОВ:")
        for result in self.test_results:
            status_icon = "✅" if result["result"] == "SUCCESS" else "❌"
            print(f"{status_icon} {result['name']}: {result['result']}")
            if result["error"]:
                print(f"   └─ Ошибка: {result['error']}")


def main():
    """Главная функция для запуска тестов."""
    print("🔧 Инициализация тестового набора Carrot-планировщика...")

    try:
        # Создаем тестовый набор
        test_suite = CarrotPlannerTestSuite()

        # Запускаем все тесты
        test_suite.run_all_tests(
            visualize=True,  # Показывать визуализацию
            interactive=True,  # Интерактивный режим
        )

    except KeyboardInterrupt:
        print("\n\n⏹️  Тестирование прервано пользователем")
    except Exception as e:
        print(f"\n\n💥 Критическая ошибка: {e}")
        import traceback

        traceback.print_exc()


if __name__ == "__main__":
    main()
