"""QR code data class"""

import time
from dataclasses import dataclass, field
from typing import List, Union, overload

from ara_api._utils.communication import qr_grpc


@dataclass
class QRCode:
    """Represents a detected QR code"""

    grpc: qr_grpc = field(default_factory=lambda: qr_grpc())
    json: dict = field(
        default_factory=lambda: {
            "data": "",
            "position": {"x": 0.0, "y": 0.0},
            # Extended data not in proto but useful for processing
            "corners": [],
            "center": [0.0, 0.0],
            "timestamp": time.time(),
            "confidence": 0.0,
        }
    )

    def __post_init__(self) -> None:
        if self.json["corners"] and not self.json["center"]:
            self.json["center"] = self._calculate_center()

    def _calculate_center(self) -> List[float]:
        """Calculate the center point from corners"""
        if not self.json["corners"] or len(self.json["corners"]) != 4:
            return [0.0, 0.0]

        x = sum(corner[0] for corner in self.json["corners"]) / 4
        y = sum(corner[1] for corner in self.json["corners"]) / 4
        return [x, y]

    def __repr__(self) -> str:
        return (
            f"QRCode(data='{self.json['data']}',"
            f" position={self.json['position']})"
        )

    @overload
    def sync(self, grpc: qr_grpc) -> None: ...

    @overload
    def sync(self, json: dict) -> None: ...

    def sync(self, data: Union[qr_grpc, dict]) -> None:
        """Update both gRPC and JSON representations with new values

        Args:
            data: Either a gRPC QR code object or a JSON dictionary
        """
        if isinstance(data, qr_grpc):
            # Update from gRPC object
            self.grpc.CopyFrom(data)

            # Update JSON
            self.json["data"] = data.data
            if hasattr(data, "position"):
                self.json["position"] = {
                    "x": data.position.x,
                    "y": data.position.y,
                }

        elif isinstance(data, dict):
            # Update from JSON dict
            self.json.update(data)

            # Update gRPC
            self.grpc.data = self.json.get("data", "")

            # Handle position
            if "position" in self.json:
                self.grpc.position.x = self.json["position"].get("x", 0.0)
                self.grpc.position.y = self.json["position"].get("y", 0.0)
