# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['torch_points3d',
 'torch_points3d.applications',
 'torch_points3d.core',
 'torch_points3d.core.base_conv',
 'torch_points3d.core.common_modules',
 'torch_points3d.core.data_transform',
 'torch_points3d.core.initializer',
 'torch_points3d.core.losses',
 'torch_points3d.core.regularizer',
 'torch_points3d.core.schedulers',
 'torch_points3d.core.spatial_ops',
 'torch_points3d.datasets',
 'torch_points3d.datasets.classification',
 'torch_points3d.datasets.object_detection',
 'torch_points3d.datasets.panoptic',
 'torch_points3d.datasets.registration',
 'torch_points3d.datasets.segmentation',
 'torch_points3d.datasets.segmentation.forward',
 'torch_points3d.metrics',
 'torch_points3d.metrics.box_detection',
 'torch_points3d.models',
 'torch_points3d.models.base_architectures',
 'torch_points3d.models.object_detection',
 'torch_points3d.models.panoptic',
 'torch_points3d.models.registration',
 'torch_points3d.models.segmentation',
 'torch_points3d.modules',
 'torch_points3d.modules.KPConv',
 'torch_points3d.modules.MinkowskiEngine',
 'torch_points3d.modules.PPNet',
 'torch_points3d.modules.PVCNN',
 'torch_points3d.modules.PointCNN',
 'torch_points3d.modules.PointNet',
 'torch_points3d.modules.RSConv',
 'torch_points3d.modules.RandLANet',
 'torch_points3d.modules.SparseConv3d',
 'torch_points3d.modules.SparseConv3d.nn',
 'torch_points3d.modules.VoteNet',
 'torch_points3d.modules.pointnet2',
 'torch_points3d.utils',
 'torch_points3d.utils.model_building_utils',
 'torch_points3d.visualization']

package_data = \
{'': ['*'],
 'torch_points3d.applications': ['conf/kpconv/*',
                                 'conf/pointnet2/*',
                                 'conf/rsconv/*',
                                 'conf/sparseconv3d/*'],
 'torch_points3d.datasets.object_detection': ['scannet_metadata/*'],
 'torch_points3d.datasets.registration': ['urls/*'],
 'torch_points3d.modules.KPConv': ['kernels/dispositions/*']}

install_requires = \
['gdown>=3.12.0,<4.0.0',
 'hydra-core>=0.11.2,<0.12.0',
 'matplotlib>=3.1,<4.0',
 'numba>=0.49.0,<0.50.0',
 'numpy>=1.18.2,<2.0.0',
 'open3d==0.10.0.0',
 'plyfile>=0.7.2,<0.8.0',
 'pytorch_metric_learning>=0.9.87.dev0,<0.10.0',
 'scikit-image>=0.16.2,<0.17.0',
 'tensorboard>=2.1,<3.0',
 'torch-cluster>=1.5.6,<2.0.0',
 'torch-geometric>=1.3.2,<2.0.0',
 'torch-points-kernels>=0.6.6,<0.7.0',
 'torch-scatter>=2.0.0,<3.0.0',
 'torch-sparse>=0.6.0,<0.7.0',
 'torch>=1.3.1,<2.0.0',
 'torchnet>=0.0.4,<0.0.5',
 'tqdm>=4.40,<5.0',
 'wandb>=0.8.18,<0.9.0']

setup_kwargs = {
    'name': 'torch-points3d',
    'version': '1.2.0',
    'description': 'Point Cloud Deep Learning Extension Library for PyTorch',
    'long_description': '<p align="center">\n  <img width="40%" src="https://raw.githubusercontent.com/nicolas-chaulet/torch-points3d/master/docs/logo.png" />\n</p>\n\n[![PyPI version](https://badge.fury.io/py/torch-points3d.svg)](https://badge.fury.io/py/torch-points3d) [![codecov](https://codecov.io/gh/nicolas-chaulet/torch-points3d/branch/master/graph/badge.svg)](https://codecov.io/gh/nicolas-chaulet/torch-points3d) [![Actions Status](https://github.com/nicolas-chaulet/torch-points3d/workflows/unittest/badge.svg)](https://github.com/nicolas-chaulet/torch-points3d/actions) [![Documentation Status](https://readthedocs.org/projects/torch-points3d/badge/?version=latest)](https://torch-points3d.readthedocs.io/en/latest/?badge=latest) [![slack](https://img.shields.io/badge/slack-tp3d-brightgreen)](https://torchgeometricco.slack.com/join/shared_invite/zt-hn9vter8-EQn4L4wLfE7PZEYbLMlw~Q#/)\n\nThis is a framework for running common deep learning models for point cloud analysis tasks against classic benchmark. It heavily relies on [Pytorch Geometric](https://pytorch-geometric.readthedocs.io/en/latest/notes/resources.html) and [Facebook Hydra](https://hydra.cc/).\n\nThe framework allows lean and yet complex model to be built with minimum effort and great reproducibility.\nIt also provide a high level API to democratize deep learning on pointclouds.\nSee our [paper](https://arxiv.org/pdf/2010.04642.pdf) at 3DV for an overview of the framework capacities and benchmarks of state-of-the-art networks.\n\n# Table of Contents\n\n   * [Overview](#overview)\n      * [Requirements](#requirements)\n      * [Project structure](#project-structure)\n      * [Methods currently implemented](#methods-currently-implemented)\n   * [Available Tasks](#available-tasks)\n   * [Available datasets](#available-datasets)\n      * [Segmentation](#segmentation)\n      * [Object detection and panoptic](#object-detection-and-panoptic)\n      * [Registration](#registration)\n      * [Classification](#classification)\n   * [3D Sparse convolution support](#3d-sparse-convolution-support)\n   * [Adding your model to the PretrainedRegistry.](#adding-your-model-to-the-pretrainedregistry)\n   * [Developer guidelines](#developer-guidelines)\n      * [Setting repo](#setting-repo)\n      * [Getting started: Train pointnet   on part segmentation task for dataset shapenet](#getting-started-train-pointnet-on-part-segmentation-task-for-dataset-shapenet)\n      * [Inference](#inference)\n         * [Inference script](#inference-script)\n         * [Containerizing your model with Docker](#containerizing-your-model-with-docker)\n      * [Profiling](#profiling)\n      * [Troubleshooting](#troubleshooting)\n   * [Exploring your experiments](#exploring-your-experiments)\n   * [Contributing](#contributing)\n   * [Citing](#citing)\n\n# Overview\n## Requirements\n\n- CUDA 10 or higher (if you want GPU version)\n- Python 3.7 or higher + headers (python-dev)\n- PyTorch 1.5 or higher (1.4 and 1.3.1 should also be working but are not actively supported moving forward)\n- A Sparse convolution backend (optional) see [here](https://github.com/nicolas-chaulet/torch-points3d#3d-sparse-convolution-support) for installation instructions\n\nInstall with\n\n```bash\npip install torch\npip install torch-points3d\n```\n\n## Project structure\n\n```bash\n├─ benchmark               # Output from various benchmark runs\n├─ conf                    # All configurations for training nad evaluation leave there\n├─ notebooks               # A collection of notebooks that allow result exploration and network debugging\n├─ docker                  # Docker image that can be used for inference or training\n├─ docs                    # All the doc\n├─ eval.py                 # Eval script\n├─ find_neighbour_dist.py  # Script to find optimal #neighbours within neighbour search operations\n├─ forward_scripts         # Script that runs a forward pass on possibly non annotated data\n├─ outputs                 # All outputs from your runs sorted by date\n├─ scripts                 # Some scripts to help manage the project\n├─ torch_points3d\n    ├─ core                # Core components\n    ├─ datasets            # All code related to datasets\n    ├─ metrics             # All metrics and trackers\n    ├─ models              # All models\n    ├─ modules             # Basic modules that can be used in a modular way\n    ├─ utils               # Various utils\n    └─ visualization       # Visualization\n├─ test\n└─ train.py                # Main script to launch a training\n```\n\nAs a general philosophy we have split datasets and models by task. For example, datasets has five subfolders:\n\n- segmentation\n- classification\n- registration\n- object_detection\n- panoptic\n\nwhere each folder contains the dataset related to each task.\n\n## Methods currently implemented\n\n- **[PointNet](https://github.com/nicolas-chaulet/torch-points3d/blob/master/torch_points3d/modules/PointNet/modules.py#L54)** from Charles R. Qi _et al._: [PointNet: Deep Learning on Point Sets for 3D Classification and Segmentation](https://arxiv.org/abs/1612.00593) (CVPR 2017)\n- **[PointNet++](https://github.com/nicolas-chaulet/torch-points3d/tree/master/torch_points3d/modules/pointnet2)** from Charles from Charles R. Qi _et al._: [PointNet++: Deep Hierarchical Feature Learning on Point Sets in a Metric Space](https://arxiv.org/abs/1706.02413)\n- **[RSConv](https://github.com/nicolas-chaulet/torch-points3d/tree/master/torch_points3d/modules/RSConv)** from Yongcheng Liu _et al._: [Relation-Shape Convolutional Neural Network for Point Cloud Analysis](https://arxiv.org/abs/1904.07601) (CVPR 2019)\n- **[RandLA-Net](https://github.com/nicolas-chaulet/torch-points3d/tree/master/torch_points3d/modules/RandLANet)** from Qingyong Hu _et al._: [RandLA-Net: Efficient Semantic Segmentation of Large-Scale Point Clouds](https://arxiv.org/abs/1911.11236)\n- **[PointCNN](https://github.com/nicolas-chaulet/torch-points3d/tree/master/torch_points3d/modules/PointCNN)** from Yangyan Li _et al._: [PointCNN: Convolution On X-Transformed Points](https://arxiv.org/abs/1801.07791) (NIPS 2018)\n- **[KPConv](https://github.com/nicolas-chaulet/torch-points3d/tree/master/torch_points3d/modules/KPConv)** from Hugues Thomas _et al._: [KPConv: Flexible and Deformable Convolution for Point Clouds](https://arxiv.org/abs/1904.08889) (ICCV 2019)\n- **[MinkowskiEngine](https://github.com/nicolas-chaulet/torch-points3d/tree/master/torch_points3d/modules/MinkowskiEngine)** from Christopher Choy _et al._: [4D Spatio-Temporal ConvNets: Minkowski Convolutional Neural Networks](https://arxiv.org/abs/1904.08755) (CVPR19)\n- **[VoteNet](https://github.com/nicolas-chaulet/torch-points3d/tree/master/torch_points3d/models/object_detection/votenet.py)** from Charles R. Qi _et al._: [Deep Hough Voting for 3D Object Detection in Point Clouds](https://arxiv.org/abs/1904.09664) (ICCV 19)\n- **[FCGF](https://github.com/chrischoy/FCGF)** from Christopher Choy _et al._: [Fully Convolutional Geometric Features](https://node1.chrischoy.org/data/publications/fcgf/fcgf.pdf) (ICCV\'19)\n- **[PointGroup](https://github.com/Jia-Research-Lab/PointGroup)** from Li Jiang _et al._: [PointGroup: Dual-Set Point Grouping for 3D Instance Segmentation](https://arxiv.org/abs/2004.01658)\n- **[PPNet (PosPool)](https://github.com/zeliu98/CloserLook3D)** from Ze Liu _et al._: [A Closer Look at Local Aggregation Operators in Point Cloud Analysis](https://arxiv.org/pdf/2007.01294.pdf) (ECCV 2020)\n- **[TorchSparse](https://github.com/mit-han-lab/torchsparse)** from Haotian Tang _et al_: [Searching Efficient 3D Architectures with Sparse Point-Voxel Convolution](https://arxiv.org/abs/2007.16100)\n- **[PVCNN](https://github.com/mit-han-lab/pvcnn)** model for semantic segmentation from Zhijian Liu _et al_:[Point-Voxel CNN for Efficient 3D Deep Learning](https://arxiv.org/abs/1907.03739)\n\nPlease refer to our [documentation](https://torch-points3d.readthedocs.io/en/latest/src/api/models.html) for accessing some of those models directly from the API and see our example notebooks for [KPconv](https://colab.research.google.com/github/nicolas-chaulet/torch-points3d/blob/master/notebooks/PartSegmentationKPConv.ipynb) and [RSConv](https://colab.research.google.com/github/nicolas-chaulet/torch-points3d/blob/master/notebooks/ObjectClassificationRSConv.ipynb) for more details.\n\n# Available Tasks\n\n|               <h3> Tasks </h3>                |                            <h3> Examples </h3>                            |\n| :-------------------------------------------: | :-----------------------------------------------------------------------: |\n| <h3> Classification / Part Segmentation </h3> | <img src="docs/imgs/classification.png"  height="220"> <br />  |\n|            <h3> Segmentation </h3>            |  <img src="docs/imgs/semantic.png"  height="220"> <br />   |\n|          <h3> Object Detection </h3>          |   <img src="docs/imgs/objects.png" height="220" >    |\n|       <h3> Panoptic Segmentation </h3>        | <img src="docs/imgs/panoptic.png"  height="220"> |\n|            <h3> Registration </h3>            |     <img src="docs/imgs/registration.png" height="220">      |\n\n# Available datasets\n\n## Segmentation\n\n- **[Scannet](https://github.com/ScanNet/ScanNet)** from Angela Dai _et al._: [ScanNet: Richly-annotated 3D Reconstructions of Indoor Scenes](https://arxiv.org/abs/1702.04405)\n\n- **[S3DIS](http://buildingparser.stanford.edu/dataset.html)** from Iro Armeni _et al._: [Joint 2D-3D-Semantic Data for Indoor Scene Understanding](https://arxiv.org/abs/1702.01105)\n\n```\n* S3DIS 1x1\n* S3DIS Room\n* S3DIS Fused - Sphere | Cylinder\n```\n\n- **[Shapenet](https://www.shapenet.org/)** from Angel X. Chang _et al._: [ShapeNet: An Information-Rich 3D Model Repository](https://arxiv.org/abs/1512.03012)\n\n## Object detection and panoptic\n\n- **[Scannet](https://github.com/ScanNet/ScanNet)** from Angela Dai _et al._: [ScanNet: Richly-annotated 3D Reconstructions of Indoor Scenes](https://arxiv.org/abs/1702.04405)\n- **[S3DIS](http://buildingparser.stanford.edu/dataset.html)** from Iro Armeni _et al._: [Joint 2D-3D-Semantic Data for Indoor Scene Understanding](https://arxiv.org/abs/1702.01105)\n\n```\n* S3DIS Fused - Sphere | Cylinder\n```\n\n- **[SemanticKitti](http://semantic-kitti.org/)** from J. Behley _et al_: [SemanticKITTI: A Dataset for Semantic Scene Understanding of LiDAR Sequences](https://arxiv.org/abs/1904.01416)\n\n## Registration\n\n- **[3DMatch](http://3dmatch.cs.princeton.edu)** from Andy Zeng _et al._: [3DMatch: Learning Local Geometric Descriptors from RGB-D Reconstructions](https://arxiv.org/abs/1603.08182)\n\n- **[The IRALab Benchmark](https://github.com/iralabdisco/point_clouds_registration_benchmark)** from Simone Fontana _et al._:[A Benchmark for Point Clouds Registration Algorithms](https://arxiv.org/abs/2003.12841), which is composed of data from:\n    - [the ETH datasets](https://projects.asl.ethz.ch/datasets/doku.php?id=laserregistration:laserregistration);\n    - [the Canadian Planetary Emulation Terrain 3D Mapping datasets](http://asrl.utias.utoronto.ca/datasets/3dmap/index.html);\n    - [the TUM Vision Groud RGBD datasets](https://vision.in.tum.de/data/datasets/rgbd-dataset);\n    - [the KAIST Urban datasets](https://irap.kaist.ac.kr/dataset/).\n\n- **[Kitti odometry](http://www.cvlibs.net/datasets/kitti/eval_odometry.php)** with corrected poses (thanks to @humanpose1) from A. Geiger _et al_: [Are we ready for Autonomous Driving? The KITTI Vision Benchmark Suite](http://www.cvlibs.net/publications/Geiger2012CVPR.pdf)\n\n## Classification\n\n- **[ModelNet](https://modelnet.cs.princeton.edu)** from Zhirong Wu _et al._: [3D ShapeNets: A Deep Representation for Volumetric Shapes](https://people.csail.mit.edu/khosla/papers/cvpr2015_wu.pdf)\n\n\n# 3D Sparse convolution support\nWe currently support [Minkowski Engine](https://github.com/StanfordVL/MinkowskiEngine) and [torchsparse](https://github.com/mit-han-lab/torchsparse) as backends for sparse convolutions. Those packages need to be installed independently from Torch Points3d, please follow installation instructions and troubleshooting notes on the respective repositories. At the moment `torchsparse` demonstrates faster training and inference on GPU but comes with limited functionalities. For example, `MinkowskiEngine` can be used    **Please be aware that `torchsparse` is still in beta and does not support CPU processing only for example.**\n\nOnce you have setup one of those two sparse convolution framework you can start using are high level to define a unet backbone or simply an encoder:\n\n```python\nfrom torch_points3d.applications.sparseconv3d import SparseConv3d\n\nmodel = SparseConv3d("unet", input_nc=3, output_nc=5, num_layers=4, backbone="torchsparse") # minkowski by default\n```\n\nYou can also assemble your own networks by using the modules provided in `torch_points3d/modules/SparseConv3d/nn`. For example if you wish to use `torchsparse` backend you can do the following:\n```python\nimport torch_points3d.modules.SparseConv3d as sp3d\n\nsp3d.nn.set_backend("torchsparse")\nconv = sp3d.nn.Conv3d(10, 10)\nbn = sp3d.nn.BatchNorm(10)\n```\n\n# Adding your model to the PretrainedRegistry.\n\nThe `PretrainedRegistry` enables anyone to add their own pre-trained models and `re-create` them with only 2 lines of code for `finetunning` or `production` purposes.\n\n- `[You]` Launch your model training with [Wandb](https://www.wandb.com) activated (`wandb.log=True`)\n- `[TorchPoints3d]` Once the training finished, `TorchPoints3d` will upload your trained model within [our custom checkpoint](https://app.wandb.ai/nicolas/scannet/runs/1sd84bf1) to your wandb.\n- `[You]` Within [`PretainedRegistry`](https://github.com/nicolas-chaulet/torch-points3d/blob/master/torch_points3d/applications/pretrained_api.py#L31) class, add a `key-value pair` within its attribute `MODELS`. The `key` should be describe your model, dataset and training hyper-parameters (possibly the best model), the `value` should be the `url` referencing the `.pt` file on your wandb.\n\nExample: Key: `pointnet2_largemsg-s3dis-1` and URL value: `https://api.wandb.ai/files/loicland/benchmark-torch-points-3d-s3dis/1e1p0csk/pointnet2_largemsg.pt` for the `pointnet2_largemsg.pt` file.\nThe key desribes a `pointnet2 largemsg trained on s3dis fold 1`.\n\n- `[Anyone]` By using the `PretainedRegistry` class and by providing the `key`, the associated model weights will be `downloaded` and the pre-trained model will be `ready to use` with its transforms.\n\n```python\n[In]:\nfrom torch_points3d.applications.pretrained_api import PretainedRegistry\n\nmodel = PretainedRegistry.from_pretrained("pointnet2_largemsg-s3dis-1")\n\nprint(model.wandb)\nprint(model.print_transforms())\n\n[Out]:\n=================================================== WANDB URLS ======================================================\nWEIGHT_URL: https://api.wandb.ai/files/loicland/benchmark-torch-points-3d-s3dis/1e1p0csk/pointnet2_largemsg.pt\nLOG_URL: https://app.wandb.ai/loicland/benchmark-torch-points-3d-s3dis/runs/1e1p0csk/logs\nCHART_URL: https://app.wandb.ai/loicland/benchmark-torch-points-3d-s3dis/runs/1e1p0csk\nOVERVIEW_URL: https://app.wandb.ai/loicland/benchmark-torch-points-3d-s3dis/runs/1e1p0csk/overview\nHYDRA_CONFIG_URL: https://app.wandb.ai/loicland/benchmark-torch-points-3d-s3dis/runs/1e1p0csk/files/hydra-config.yaml\nOVERRIDES_URL: https://app.wandb.ai/loicland/benchmark-torch-points-3d-s3dis/runs/1e1p0csk/files/overrides.yaml\n======================================================================================================================\n\npre_transform = None\ntest_transform = Compose([\n    FixedPoints(20000, replace=True),\n    XYZFeature(axis=[\'z\']),\n    AddFeatsByKeys(rgb=True, pos_z=True),\n    Center(),\n    ScalePos(scale=0.5),\n])\ntrain_transform = Compose([\n    FixedPoints(20000, replace=True),\n    RandomNoise(sigma=0.001, clip=0.05),\n    RandomRotate((-180, 180), axis=2),\n    RandomScaleAnisotropic([0.8, 1.2]),\n    RandomAxesSymmetry(x=True, y=False, z=False),\n    DropFeature(proba=0.2, feature=\'rgb\'),\n    XYZFeature(axis=[\'z\']),\n    AddFeatsByKeys(rgb=True, pos_z=True),\n    Center(),\n    ScalePos(scale=0.5),\n])\nval_transform = Compose([\n    FixedPoints(20000, replace=True),\n    XYZFeature(axis=[\'z\']),\n    AddFeatsByKeys(rgb=True, pos_z=True),\n    Center(),\n    ScalePos(scale=0.5),\n])\ninference_transform = Compose([\n    FixedPoints(20000, replace=True),\n    XYZFeature(axis=[\'z\']),\n    AddFeatsByKeys(rgb=True, pos_z=True),\n    Center(),\n    ScalePos(scale=0.5),\n])\npre_collate_transform = Compose([\n    PointCloudFusion(),\n    SaveOriginalPosId,\n    GridSampling3D(grid_size=0.04, quantize_coords=False, mode=mean),\n])\n```\n\n\n# Developer guidelines\n\n## Setting repo\n\nWe use [Poetry](https://poetry.eustace.io/) for managing our packages.\nIn order to get started, clone this repositories and run the following command from the root of the repo\n\n```\npoetry install --no-root\n```\n\nThis will install all required dependencies in a new virtual environment.\n\nActivate the environment\n\n```bash\npoetry shell\n```\n\nYou can check that the install has been successful by running\n\n```bash\npython -m unittest -v\n```\n\nFor `pycuda` support (only needed for the registration tasks):\n\n```bash\npip install pycuda\n```\n\n## Getting started: Train pointnet++ on part segmentation task for dataset shapenet\n\n```bash\npoetry run python train.py task=segmentation model_type=pointnet2 model_name=pointnet2_charlesssg dataset=shapenet-fixed\n```\n\nAnd you should see something like that\n\n![logging](https://raw.githubusercontent.com/nicolas-chaulet/torch-points3d/master/docs/imgs/logging.png)\n\nThe [config](https://raw.githubusercontent.com/nicolas-chaulet/torch-points3d/master/conf/models/segmentation/pointnet2.yaml) for pointnet++ is a good example of how to define a model and is as follow:\n\n```yaml\n# PointNet++: Deep Hierarchical Feature Learning on Point Sets in a Metric Space (https://arxiv.org/abs/1706.02413)\n# Credit Charles R. Qi: https://github.com/charlesq34/pointnet2/blob/master/models/pointnet2_part_seg_msg_one_hot.py\n\npointnet2_onehot:\n  architecture: pointnet2.PointNet2_D\n  conv_type: \'DENSE\'\n  use_category: True\n  down_conv:\n    module_name: PointNetMSGDown\n    npoint: [1024, 256, 64, 16]\n    radii: [[0.05, 0.1], [0.1, 0.2], [0.2, 0.4], [0.4, 0.8]]\n    nsamples: [[16, 32], [16, 32], [16, 32], [16, 32]]\n    down_conv_nn:\n      [\n        [[FEAT, 16, 16, 32], [FEAT, 32, 32, 64]],\n        [[32 + 64, 64, 64, 128], [32 + 64, 64, 96, 128]],\n        [[128 + 128, 128, 196, 256], [128 + 128, 128, 196, 256]],\n        [[256 + 256, 256, 256, 512], [256 + 256, 256, 384, 512]],\n      ]\n  up_conv:\n    module_name: DenseFPModule\n    up_conv_nn:\n      [\n        [512 + 512 + 256 + 256, 512, 512],\n        [512 + 128 + 128, 512, 512],\n        [512 + 64 + 32, 256, 256],\n        [256 + FEAT, 128, 128],\n      ]\n    skip: True\n  mlp_cls:\n    nn: [128, 128]\n    dropout: 0.5\n```\n## Inference\n\n### Inference script\n\nWe provide a script for running a given pre trained model on custom data that may not be annotated. You will find an [example](https://github.com/nicolas-chaulet/torch-points3d/blob/master/forward_scripts/forward.py) of this for the part segmentation task on Shapenet. Just like for the rest of the codebase most of the customization happens through config files and the provided example can be extended to other datasets. You can also easily create your own from there. Going back to the part segmentation task, say you have a folder full of point clouds that you know are Airplanes, and you have the checkpoint of a model trained on Airplanes and potentially other classes, simply edit the [config.yaml](https://github.com/nicolas-chaulet/torch-points3d/blob/master/forward_scripts/conf/config.yaml) and [shapenet.yaml](https://github.com/nicolas-chaulet/torch-points3d/blob/master/forward_scripts/conf/dataset/shapenet.yaml) and run the following command:\n\n```bash\npython forward_scripts/forward.py\n```\n\nThe result of the forward run will be placed in the specified `output_folder` and you can use the [notebook](https://github.com/nicolas-chaulet/torch-points3d/blob/master/forward_scripts/notebooks/viz_shapenet.ipynb) provided to explore the results. Below is an example of the outcome of using a model trained on caps only to find the parts of airplanes and caps.\n\n![resexplore](https://raw.githubusercontent.com/nicolas-chaulet/torch-points3d/master/docs/imgs/inference_demo.gif)\n\n### Containerizing your model with Docker\n\nFinally, for people interested in deploying their models to production environments, we provide a [Dockerfile](https://github.com/nicolas-chaulet/torch-points3d/blob/master/docker/Dockerfile) as well as a [build script](https://github.com/nicolas-chaulet/torch-points3d/blob/master/docker/build.sh). Say you have trained a network for semantic segmentation that gave the weight `<outputfolder/weights.pt>`, the following command will build a docker image for you:\n\n```bash\ncd docker\n./build.sh outputfolder/weights.pt\n```\n\nYou can then use it to run a forward pass on a all the point clouds in `input_path` and generate the results in `output_path`\n\n```bash\ndocker run -v /test_data:/in -v /test_data/out:/out pointnet2_charlesssg:latest python3 forward_scripts/forward.py dataset=shapenet data.forward_category=Cap input_path="/in" output_path="/out"\n```\n\nThe `-v` option mounts a local directory to the container\'s file system. For example in the command line above, `/test_data/out` will be mounted at the location `/out`. As a consequence, all files written in `/out` will be available in the folder `/test_data/out` on your machine.\n\n## Profiling\n\nWe advice to use [`snakeviz`](https://jiffyclub.github.io/snakeviz/) and [`cProfile`](https://docs.python.org/2/library/profile.html)\n\nUse cProfile to profile your code\n\n```\npoetry run python -m cProfile -o {your_name}.prof train.py ... debugging.profiling=True\n```\n\nAnd visualize results using snakeviz.\n\n```\nsnakeviz {your_name}.prof\n```\n\nIt is also possible to use [`torch.utils.bottleneck`](https://pytorch.org/docs/stable/bottleneck.html)\n\n```\npython -m torch.utils.bottleneck /path/to/source/script.py [args]\n```\n\n## Troubleshooting\n\n### Cannot compile certain CUDA Kernels or seg faults while running the tests\n\nEnsure that at least PyTorch 1.4.0 is installed and verify that `cuda/bin` and `cuda/include` are in your `$PATH` and `$CPATH` respectively, e.g.:\n\n```\n$ python -c "import torch; print(torch.__version__)"\n>>> 1.4.0\n\n$ echo $PATH\n>>> /usr/local/cuda/bin:...\n\n$ echo $CPATH\n>>> /usr/local/cuda/include:...\n```\n\n### Undefined symbol / Updating Pytorch\n\nWhen we update the version of Pytorch that is used, the compiled packages need to be reinstalled, otherwise you will run into an error that looks like this:\n\n```\n... scatter_cpu.cpython-36m-x86_64-linux-gnu.so: undefined symbol: _ZN3c1012CUDATensorIdEv\n```\n\nThis can happen for the following libraries:\n\n- torch-points-kernels\n- torch-scatter\n- torch-cluster\n- torch-sparse\n\nAn easy way to fix this is to run the following command with the virtual env activated:\n\n```\npip uninstall torch-scatter torch-sparse torch-cluster torch-points-kernels -y\nrm -rf ~/.cache/pip\npoetry install\n```\n\n### CUDA kernel failed : no kernel image is available for execution on the device\n\nThis can happen when trying to run the code on a different GPU than the one used to compile the `torch-points-kernels` library. Uninstall `torch-points-kernels`, clear cache, and reinstall after setting the `TORCH_CUDA_ARCH_LIST` environment variable. For example, for compiling with a Tesla T4 (Turing 7.5) and running the code on a Tesla V100 (Volta 7.0) use:\n\n```\nexport TORCH_CUDA_ARCH_LIST="7.0;7.5"\n```\n\nSee [this useful chart](http://arnon.dk/matching-sm-architectures-arch-and-gencode-for-various-nvidia-cards/) for more architecture compatibility.\n\n### Cannot use wandb on Windows\n\nRaises `OSError: [WinError 6] The handle is invalid` / `wandb: ERROR W&B process failed to launch`\nWandb is currently broken on Windows (see [this issue](https://github.com/wandb/client/issues/862)), a workaround is to use the command line argument `wandb.log=false`\n\n\n# Exploring your experiments\n\nWe provide a [notebook](https://github.com/nicolas-chaulet/torch-points3d/blob/master/notebooks/dashboard.ipynb) based [pyvista](https://docs.pyvista.org/) and [panel](https://panel.holoviz.org/) that allows you to explore your past experiments visually. When using jupyter lab you will have to install an extension:\n\n```\njupyter labextension install @pyviz/jupyterlab_pyviz\n```\n\nRun through the notebook and you should see a dashboard starting that looks like the following:\n\n![dashboard](https://raw.githubusercontent.com/nicolas-chaulet/torch-points3d/master/docs/imgs/Dashboard_demo.gif)\n\n\n# Contributing\n\nContributions are welcome! The only asks are that you stick to the styling and that you add tests as you add more features!\n\nFor styling you can use [pre-commit hooks](https://ljvmiranda921.github.io/notebook/2018/06/21/precommits-using-black-and-flake8/) to help you:\n\n```\npre-commit install\n```\n\nA sequence of checks will be run for you and you may have to add the fixed files again to the stashed files.\n\nWhen it comes to docstrings we use [numpy style](https://numpydoc.readthedocs.io/en/latest/format.html) docstrings, for those who use\nVisual Studio Code, there is a great [extension](https://github.com/NilsJPWerner/autoDocstring) that can help with that. Install it and set the format to numpy and you should be good to go!\n\nFinaly, if you want to have a direct chat with us feel free to join our slack, just shoot us an email and we\'ll add you.\n\n# Citing\n\nIf you find our work useful, do not hesitate to cite it:\n\n```\n@inproceedings{\n  tp3d,\n  title={Torch-Points3D: A Modular Multi-Task Frameworkfor Reproducible Deep Learning on 3D Point Clouds},\n  author={Chaton, Thomas and Chaulet Nicolas and Horache, Sofiane and Landrieu, Loic},\n  booktitle={2020 International Conference on 3D Vision (3DV)},\n  year={2020},\n  organization={IEEE},\n  url = {\\url{https://github.com/nicolas-chaulet/torch-points3d}}\n}\n```\n\nand please also include a citation to the\n[models](https://github.com/nicolas-chaulet/torch-points3d#methods-currently-implemented)\nor the [datasets](https://github.com/nicolas-chaulet/torch-points3d#available-datasets) you have used in your experiments!\n',
    'author': 'Thomas Chaton',
    'author_email': 'thomas.chaton.ai@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
