import copy

from django.db.models import Model

from .constants import SUB_SCHEMA_NAME


class NotManagedModel(Model):
    """
    Abstract model used to define not managed django models that use
    database views as a data source.

    Warning: adding custom Meta class to the child model leads to an error
    """

    class Meta:
        abstract = True

    def __init_subclass__(cls, is_abstract_model=False):
        """
        Adds metadata to the class that makes it a not managed model with the data source equal to the
        database view name generated by BasePostgresView parent class
        """
        super().__init_subclass__()

        # This check is done to make sure abstract models don't have Meta attributes generated as that can
        # interfere with the children Meta
        if not is_abstract_model:
            cls.Meta = cls.generate_meta_data()

    @classmethod
    def generate_meta_data(cls):
        """
        Adds `managed=False` and `db_table` attributes to the Meta class of the model

        Returns:
            Meta class with injected managed=False and db_table attributes
        """
        meta = type("Meta", (), {})

        generated_view_name_with_schema = f'"{SUB_SCHEMA_NAME}"."{cls.name}"'

        meta.managed = False
        meta.db_table = generated_view_name_with_schema

        return meta

