#####################################################
#### Written By: SATYAKI DE                      ####
#### Written On: 22-Jul-2022                     ####
#### Modified On 30-Aug-2022                     ####
####                                             ####
#### Objective: This is the sub-class of the     ####
#### main class, where the source form tries to  ####
#### map with the supplied template form to      ####
#### accurately extract the text from the        ####
#### specific regions.                           ####
#####################################################

# import the necessary packages
import numpy as np
import imutils
import cv2

from .clsConfig import clsConfig as cf

class clsAlignTemplates:
	def __init__(self):
		self.sep = str(cf.conf['SEP'])
		self.Curr_Path = str(cf.conf['INIT_PATH'])

	def startAligning(self, image, template, maxFeatures=500, keepPercent=0.2, debug=False):
		try:
			# convert both the input image and template to grayscale
			imageGray = cv2.cvtColor(image, cv2.COLOR_BGR2GRAY)
			templateGray = cv2.cvtColor(template, cv2.COLOR_BGR2GRAY)

			# use ORB to detect keypoints and extract (binary) local
			# invariant features
			orb = cv2.ORB_create(maxFeatures)
			(kpsA, descsA) = orb.detectAndCompute(imageGray, None)
			(kpsB, descsB) = orb.detectAndCompute(templateGray, None)

			# match the features
			method = cv2.DESCRIPTOR_MATCHER_BRUTEFORCE_HAMMING
			matcher = cv2.DescriptorMatcher_create(method)
			matches = matcher.match(descsA, descsB, None)

			# sort the matches by their distance (the smaller the distance,
			# the "more similar" the features are)
			matches = sorted(matches, key=lambda x:x.distance)

			# keep only the top matches
			keep = int(len(matches) * keepPercent)
			matches = matches[:keep]

			# check to see if we should visualize the matched keypoints
			if debug:
				matchedVis = cv2.drawMatches(image, kpsA, template, kpsB,
					matches, None)
				matchedVis = imutils.resize(matchedVis, width=1000)
				cv2.imshow("Matched Keypoints", matchedVis)
				cv2.waitKey(0)

			# allocate memory for the keypoints (x,y-coordinates) from the
			# top matches -- we'll use these coordinates to compute our
			# homography matrix
			ptsA = np.zeros((len(matches), 2), dtype="float")
			ptsB = np.zeros((len(matches), 2), dtype="float")

			# loop over the top matches
			for (i, m) in enumerate(matches):
				# indicate that the two keypoints in the respective images
				# map to each other
				ptsA[i] = kpsA[m.queryIdx].pt
				ptsB[i] = kpsB[m.trainIdx].pt

			# compute the homography matrix between the two sets of matched
			# points
			(H, mask) = cv2.findHomography(ptsA, ptsB, method=cv2.RANSAC)

			# use the homography matrix to align the images
			(h, w) = template.shape[:2]
			aligned = cv2.warpPerspective(image, H, (w, h))

			# return the aligned image
			return aligned

		except Exception as e:
			x = str(e)
			print('Error: ', x)

			aligned = ''
			return aligned
