"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const cf = require("@aws-cdk/aws-cloudfront");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
test('Cloudfront Distribution Included', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            CustomErrorResponses: [
                {
                    ErrorCode: 403,
                    ResponseCode: 200,
                    ResponsePagePath: '/index.html',
                },
                {
                    ErrorCode: 404,
                    ResponseCode: 200,
                    ResponsePagePath: '/index.html',
                },
            ],
            DefaultCacheBehavior: {
                ViewerProtocolPolicy: 'redirect-to-https',
            },
            DefaultRootObject: 'index.html',
            HttpVersion: 'http2',
            IPV6Enabled: true,
            PriceClass: 'PriceClass_100',
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true,
            },
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        CanonicalUser: {
                            'Fn::GetAtt': [
                                'spaDeployOriginAccessIdentityEDA4C19C',
                                'S3CanonicalUserId',
                            ],
                        },
                    },
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'spaDeployWebsiteBucket1E4C4442',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'spaDeployWebsiteBucket1E4C4442',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }
            ],
        },
    }));
});
test('Cloudfront Distribution Included - with non-default error-doc cfg set', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            CustomErrorResponses: [
                {
                    ErrorCode: 403,
                    ResponseCode: 200,
                    ResponsePagePath: '/error.html',
                },
                {
                    ErrorCode: 404,
                    ResponseCode: 200,
                    ResponsePagePath: '/error.html',
                },
            ],
            DefaultCacheBehavior: {
                ViewerProtocolPolicy: 'redirect-to-https',
            },
            DefaultRootObject: 'index.html',
            HttpVersion: 'http2',
            IPV6Enabled: true,
            PriceClass: 'PriceClass_100',
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true,
            },
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        CanonicalUser: {
                            'Fn::GetAtt': [
                                'spaDeployOriginAccessIdentityEDA4C19C',
                                'S3CanonicalUserId',
                            ],
                        },
                    },
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'spaDeployWebsiteBucket1E4C4442',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'spaDeployWebsiteBucket1E4C4442',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }
            ],
        },
    }));
});
test('Cloudfront With Custom Cert and Aliases', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com'],
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.test.com',
            ],
            ViewerCertificate: {
                AcmCertificateArn: 'arn:1234',
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Basic Site Setup', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Basic Site Setup with Error Doc set', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Basic Site Setup, Encrypted Bucket', () => {
    const stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Cloudfront With Encrypted Bucket', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true });
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com'],
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.test.com',
            ],
            ViewerCertificate: {
                AcmCertificateArn: 'arn:1234',
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Cloudfront With Custom Defined Behaviors', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        cfBehaviors: [
            {
                isDefaultBehavior: true,
                allowedMethods: cf.CloudFrontAllowedMethods.ALL,
                forwardedValues: {
                    queryString: true,
                    cookies: { forward: 'all' },
                    headers: ['*'],
                },
            },
            {
                pathPattern: '/virtual-path',
                allowedMethods: cf.CloudFrontAllowedMethods.GET_HEAD,
                cachedMethods: cf.CloudFrontAllowedCachedMethods.GET_HEAD,
            },
        ],
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            CacheBehaviors: [
                {
                    AllowedMethods: ['GET', 'HEAD'],
                    CachedMethods: ['GET', 'HEAD'],
                    PathPattern: '/virtual-path',
                },
            ],
            DefaultCacheBehavior: {
                AllowedMethods: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
                ForwardedValues: {
                    Cookies: { Forward: 'all' },
                    Headers: ['*'],
                    QueryString: true,
                },
                TargetOriginId: 'origin1',
            },
        },
    }));
});
test('Cloudfront With Custom Security Policy', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com'],
        securityPolicy: cf.SecurityPolicyProtocol.TLS_V1_2_2019,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.test.com',
            ],
            ViewerCertificate: {
                AcmCertificateArn: 'arn:1234',
                SslSupportMethod: 'sni-only',
                MinimumProtocolVersion: 'TLSv1.2_2019',
            },
        },
    }));
});
test('Cloudfront With Custom SSL Method', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com'],
        sslMethod: cf.SSLMethod.VIP,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.test.com',
            ],
            ViewerCertificate: {
                AcmCertificateArn: 'arn:1234',
                SslSupportMethod: 'vip',
            },
        },
    }));
});
test('Basic Site Setup, IP Filter', () => {
    const stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true, ipFilter: true, ipList: ['1.1.1.1'] })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Condition: {
                        IpAddress: {
                            'aws:SourceIp': [
                                '1.1.1.1',
                            ],
                        },
                    },
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Create From Hosted Zone', () => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234',
        },
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.cdkspadeploy.com',
            ],
            ViewerCertificate: {
                SslSupportMethod: 'sni-only',
            },
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        CanonicalUser: {
                            'Fn::GetAtt': [
                                'spaDeployOriginAccessIdentityEDA4C19C',
                                'S3CanonicalUserId',
                            ],
                        },
                    },
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'spaDeployWebsiteBucket1E4C4442',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'spaDeployWebsiteBucket1E4C4442',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }
            ],
        },
    }));
});
test('Create From Hosted Zone with subdomain', () => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234',
        },
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        websiteFolder: 'website',
        subdomain: 'myhost',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'myhost.cdkspadeploy.com',
            ],
            ViewerCertificate: {
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Create From Hosted Zone with Error Bucket', () => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234',
        },
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.cdkspadeploy.com',
            ],
            ViewerCertificate: {
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Basic Site Setup, URL Output Enabled With Name', () => {
    const stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: true,
        exportWebsiteUrlName: exportName,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveOutput({
        exportName,
    }));
});
test('Basic Site Setup, URL Output Enabled With No Name', () => {
    const stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: true,
    });
    // THEN
    assert_1.expect(stack).notTo(assert_1.haveOutput({
        exportName,
    }));
});
test('Basic Site Setup, URL Output Not Enabled', () => {
    const stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: false,
    });
    // THEN
    assert_1.expect(stack).notTo(assert_1.haveOutput({
        exportName,
    }));
});
test('Basic Site Setup, Block Public Enabled', () => {
    const stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { ipFilter: true })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true,
        },
    }));
});
//# sourceMappingURL=data:application/json;base64,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