#include <exception>
#include <iostream>
#include <optional>
#include <tuple>

#include <CLI/CLI.hpp>

#include "ecole/instance/capacitated-facility-location.hpp"
#include "ecole/instance/combinatorial-auction.hpp"
#include "ecole/instance/independent-set.hpp"
#include "ecole/instance/set-cover.hpp"
#include "ecole/random.hpp"
#include "ecole/scip/type.hpp"

#include "bench-branching.hpp"
#include "benchmark.hpp"

using namespace ecole::benchmark;
using namespace ecole::instance;

/** Apply func on each element of the tuple. */
template <typename... Args, typename Func> void for_each(std::tuple<Args...>& t, Func&& func) {
	std::apply([&func](auto&&... t_elem) { ((func(std::forward<decltype(t_elem)>(t_elem))), ...); }, t);
}

void seed_model(ecole::scip::Model& model, ecole::RandomEngine& random_engine) {
	std::uniform_int_distribution<ecole::scip::Seed> seed_distrib{ecole::scip::min_seed, ecole::scip::max_seed};
	model.set_param("randomization/permuteconss", true);
	model.set_param("randomization/permutevars", true);
	model.set_param("randomization/permutationseed", seed_distrib(random_engine));
	model.set_param("randomization/randomseedshift", seed_distrib(random_engine));
	model.set_param("randomization/lpseed", seed_distrib(random_engine));
}

/** The generators used to benchmark branching dynamics. */
auto benchmark_branching(std::size_t n_instances, std::size_t n_nodes) {
	using GraphType = typename ecole::instance::IndependentSetGenerator::Parameters::GraphType;
	auto generators = std::tuple{
		SetCoverGenerator{{500, 1000}},                           // NOLINT(readability-magic-numbers)
		SetCoverGenerator{{1000, 1000}},                          // NOLINT(readability-magic-numbers)
		SetCoverGenerator{{2000, 1000}},                          // NOLINT(readability-magic-numbers)
		CombinatorialAuctionGenerator{{100, 500}},                // NOLINT(readability-magic-numbers)
		CombinatorialAuctionGenerator{{200, 1000}},               // NOLINT(readability-magic-numbers)
		CombinatorialAuctionGenerator{{300, 1500}},               // NOLINT(readability-magic-numbers)
		CapacitatedFacilityLocationGenerator{{100, 100}},         // NOLINT(readability-magic-numbers)
		CapacitatedFacilityLocationGenerator{{200, 100}},         // NOLINT(readability-magic-numbers)
		CapacitatedFacilityLocationGenerator{{400, 100}},         // NOLINT(readability-magic-numbers)
		IndependentSetGenerator{{500, GraphType::erdos_renyi}},   // NOLINT(readability-magic-numbers)
		IndependentSetGenerator{{1000, GraphType::erdos_renyi}},  // NOLINT(readability-magic-numbers)
		IndependentSetGenerator{{1500, GraphType::erdos_renyi}},  // NOLINT(readability-magic-numbers)
	};
	auto random_engine = ecole::spawn_random_engine();

	std::cout << BranchingResult::csv_title() << '\n';
	for (std::size_t i = 0; i < n_instances; ++i) {
		auto benchmark_and_print = [&](auto& gen) noexcept {
			try {
				auto model = gen.next();
				model.disable_presolve();
				model.disable_cuts();
				model.set_param("limits/totalnodes", n_nodes);
				seed_model(model, random_engine);
				std::cout << benchmark_branching(model).csv() << '\n';
			} catch (std::exception const& e) {
				std::cerr << "Error when benchmarking an instance: " << e.what() << '\n';
			}
		};
		for_each(generators, benchmark_and_print);
	}
}

int main(int argc, char** argv) {
	try {

		auto app = CLI::App{};
		app.failure_message(CLI::FailureMessage::help);
		auto n_instances = std::size_t{10};  // NOLINT(readability-magic-numbers)
		app.add_option(
			"--intances-per-generator,--ipg", n_instances, "Number of instances generated by each instance generator");
		auto n_nodes = std::size_t{100};  // NOLINT(readability-magic-numbers)
		app.add_option("--node-limit,--nl", n_nodes, "Limit the number of nodes in each run");
		auto seed = std::optional<ecole::Seed>{};
		app.add_option("--seed,-s", seed, "Global Ecole random seed");
		CLI11_PARSE(app, argc, argv);

		if (seed.has_value()) {
			ecole::seed(seed.value());
		}
		benchmark_branching(n_instances, n_nodes);

	} catch (std::exception const& e) {
		std::cerr << "An error occured: " << e.what() << '\n';
	}
}
