''' Backend for JSON.  '''
import json
import sys
from wrc.sema.ast import WCADocument, Guideline
from wrc.codegen.cg import CGDocument
from wrc.codegen.cghtml import simple_md2html

REGULATIONS_ROOT = "https://www.worldcubeassociation.org/regulations/"
COMMENTS = ('Generated by `{cmd}` The WRC maintains this file as part '
              'of the Regulations repository. Every time we update this '
              'file, we have to remember to do a migration. See '
              'https://github.com/thewca/worldcubeassociation.org/pull/1650.')


class WCADocumentJSON(CGDocument):
    ''' Implement a simple JSON generator from Regulations and Guidelines ASTs. '''

    name = "JSON"
    def __init__(self, versionhash, language, pdf):
        # We don't need it
        del language
        super(WCADocumentJSON, self).__init__(list)
        self.versionhash = versionhash
        self.urls = {'regulations': REGULATIONS_ROOT,
                     'guidelines': REGULATIONS_ROOT + "guidelines.html",
                     'pdf': pdf}

    def visitWCAStates(self, document):
        self.state_lists = []
        retval = super(WCADocumentJSON, self).visitWCAStates(document)
        self.codegen = {'title': document.title, 'version': document.version,
                        'version_hash': self.versionhash,
                        'text': document.text, 'states_lists': self.state_lists,
                        'comment': COMMENTS.format(cmd=" ".join(sys.argv))}
        return retval

    def visitStatesList(self, states_list):
        self.current_states = []
        retval = super(WCADocumentJSON, self).visitStatesList(states_list)
        self.state_lists.append({'title': states_list.title,
                                 'states': self.current_states})
        return retval

    def emit(self, regulations, guidelines):
        reg_list, guide_list = super(WCADocumentJSON, self).emit(regulations, guidelines)
        if len(guide_list) > 0:
            reg_list.extend(guide_list)
        return json.dumps(reg_list, indent=2, separators=(',', ': ')), ""


    def visitRule(self, reg):
        label = None
        url = "/regulations/"

        if isinstance(reg, Guideline):
            url += "guidelines.html"
            label = reg.labelname

        url += "#" + reg.number
        reg_dict = {
            'class': 'regulation',
            'id': reg.number,
            'content_html': simple_md2html(reg.text, self.urls),
            'url': url
        }
        if label:
            reg_dict.update({'guideline_label': label})
        self.codegen.append(reg_dict)
        retval = super(WCADocumentJSON, self).visitRule(reg)
        return retval

    def visitState(self, state):
        self.current_states.append({'class': 'state', 'iso2': state.state_id,
                                    'continent_id': state.continent_id,
                                    'name': state.name, 'info': state.info,
                                    'id': state.friendly_id})
        return True

